# declaration of function

set(VPVL2_BASE_DIR "${CMAKE_CURRENT_SOURCE_DIR}/..")
function(get_source_directory output dir)
  set(${output} "${VPVL2_BASE_DIR}/${dir}" PARENT_SCOPE)
endfunction()

function(get_build_directory output source_dir)
  string(TOLOWER ${CMAKE_BUILD_TYPE} CMAKE_BUILD_TYPE_TOLOWER)
  set(BUILD_DIRECTORY "build-${CMAKE_BUILD_TYPE_TOLOWER}")
  set(${output} "${${source_dir}}/${BUILD_DIRECTORY}" PARENT_SCOPE)
endfunction()

function(get_local_library_directory_named output source_dir lib_dir)
  string(TOLOWER ${CMAKE_BUILD_TYPE} CMAKE_BUILD_TYPE_TOLOWER)
  get_build_directory(build_dir ${source_dir})
  if(MSVC)
    set(${output} "${build_dir}/${lib_dir}/${CMAKE_BUILD_TYPE_TOLOWER}" PARENT_SCOPE)
  else()
    set(${output} "${build_dir}/${lib_dir}" PARENT_SCOPE)
  endif()
endfunction()

function(get_local_library_directory output source_dir)
  get_local_library_directory_named(output_to_reassign ${source_dir} "lib")
  set(${output} ${output_to_reassign} PARENT_SCOPE)
endfunction()

function(get_install_directory output dir)
  get_source_directory(source_dir ${dir})
  string(TOLOWER ${CMAKE_BUILD_TYPE} CMAKE_BUILD_TYPE_TOLOWER)
  set(build_dir "build-${CMAKE_BUILD_TYPE_TOLOWER}")
  set(${output} "${source_dir}/${build_dir}/install-root" PARENT_SCOPE)
endfunction()

function(vpvl2_set_library_properties target public_headers)
  if(WIN32 AND BUILD_SHARED_LIBS)
    set_target_properties(${target} PROPERTIES PREFIX "" SUFFIX .dll IMPORT_SUFFIX ${CMAKE_IMPORT_LIBRARY_SUFFIX})
  elseif(APPLE)
    # create as a framework if build on darwin environment
    if(BUILD_SHARED_LIBS AND FRAMEWORK)
      set_target_properties(${target} PROPERTIES FRAMEWORK true PROPERTIES PUBLIC_HEADER "${public_headers}")
    endif()
    set_target_properties(${target}
                          PROPERTIES
                          VERSION "${VPVL2_VERSION_MAJOR}.${VPVL2_VERSION_COMPAT}.${VPVL2_VERSION_MINOR}"
                          SOVERSION "${VPVL2_VERSION_MAJOR}.${VPVL2_VERSION_COMPAT}.0"
                          INSTALL_NAME_DIR "${CMAKE_INSTALL_PREFIX}/lib")
  endif()
endfunction()

function(vpvl2_set_warnings)
  if("${CMAKE_CXX_COMPILER_ID}" STREQUAL "Clang" OR "${CMAKE_CXX_COMPILER_ID}" STREQUAL "Clang" OR CMAKE_COMPILER_IS_GNUC)
  # set more warnings when clang or gcc is selected
    add_definitions(-W -Wall -Wextra -Wformat=2 -Wstrict-aliasing=2 -Wwrite-strings)
  elseif(MSVC)
    # disable some specified warnings on MSVC
    set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} /EHsc /wd4068 /wd4819" CACHE STRING "disable warnings of C4068 (for clang pragma) and C4819" FORCE)
    set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /EHsc /wd4068 /wd4355 /wd4819" CACHE STRING "disable warnings of C4068 (for clang pragma), C4355 (for glog) and C4819" FORCE)
    # disable _CRT_SECURE_NO_WARNINGS for surpressing warnings from vpvl2/Common.h
    add_definitions(-D_CRT_SECURE_NO_WARNINGS -D_CRT_SECURE_NO_DEPRECATE -D_SCL_SECURE_NO_WARNINGS)
  endif()
endfunction()

function(vpvl2_link_bullet target)
  if(NOT VPVL2_ENABLE_LAZY_LINK)
    target_link_libraries(${target} ${BULLET_DYNAMICS_LIB}
                                    ${BULLET_COLLISION_LIB}
                                    ${BULLET_SOFTBODY_LIB}
                                    ${BULLET_MULTITHREADED_LIB}
                                    ${BULLET_LINEARMATH_LIB})
  endif()
endfunction()

function(vpvl2_find_bullet)
  get_install_directory(BULLET_INSTALL_DIR "bullet-src")
  find_path(BULLET_INCLUDE_DIR NAMES btBulletCollisionCommon.h PATH_SUFFIXES include/bullet PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  find_library(BULLET_LINEARMATH_LIB NAMES LinearMath PATH_SUFFIXES lib64 lib32 lib PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  find_library(BULLET_COLLISION_LIB NAMES BulletCollision PATH_SUFFIXES lib64 lib32 lib PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  find_library(BULLET_DYNAMICS_LIB NAMES BulletDynamics PATH_SUFFIXES lib64 lib32 lib PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  find_library(BULLET_MULTITHREADED_LIB NAMES BulletMultiThreaded PATH_SUFFIXES lib64 lib32 lib PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  find_library(BULLET_SOFTBODY_LIB NAMES BulletSoftBody PATH_SUFFIXES lib64 lib32 lib PATHS ${BULLET_INSTALL_DIR} NO_DEFAULT_PATH)
  include_directories(${BULLET_INCLUDE_DIR})
endfunction()

function(vpvl2_link_assimp target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND (VPVL2_LINK_ASSIMP3 OR VPVL2_LINK_ASSIMP))
    target_link_libraries(${target} ${ASSIMP_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_assimp)
  if(VPVL2_LINK_ASSIMP3 OR VPVL2_LINK_ASSIMP)
    get_install_directory(ASSIMP_INSTALL_DIR "assimp-src")
    if(VPVL2_LINK_ASSIMP3)
      find_path(ASSIMP_INCLUDE_DIR NAMES assimp/Importer.hpp PATH_SUFFIXES include PATHS ${ASSIMP_INSTALL_DIR} NO_DEFAULT_PATH)
      find_library(ASSIMP_LIBRARY NAMES assimp assimpD PATH_SUFFIXES lib64 lib32 lib PATHS ${ASSIMP_INSTALL_DIR} NO_DEFAULT_PATH)
    elseif(VPVL2_LINK_ASSIMP)
      find_path(ASSIMP_INCLUDE_DIR NAMES assimp/assimp.h PATH_SUFFIXES include PATHS ${ASSIMP_INSTALL_DIR} NO_DEFAULT_PATH)
      find_library(ASSIMP_LIBRARY NAMES assimp PATH_SUFFIXES lib64 lib32 lib PATHS ${ASSIMP_INSTALL_DIR} NO_DEFAULT_PATH)
    endif()
    include_directories(${ASSIMP_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_vpvl target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_VPVL)
    target_link_libraries(${target} ${VPVL_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_vpvl)
  if(VPVL2_LINK_VPVL)
    get_install_directory(VPVL_INSTALL_DIR "libvpvl")
    find_path(VPVL_INCLUDE_DIR NAMES vpvl/vpvl.h PATH_SUFFIXES include PATHS ${VPVL_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(VPVL_LIBRARY NAMES vpvl PATH_SUFFIXES lib64 lib32 lib PATHS ${VPVL_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${VPVL_INCLUDE_DIR} ${VPVL_CONFIG_DIR})
  endif()
endfunction()

function(vpvl2_find_openmp)
  if(VPVL2_ENABLE_OPENMP)
    find_package(OpenMP)
    if(OPENMP_FOUND)
      set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} ${OpenMP_C_FLAGS}")
      set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} ${OpenMP_CXX_FLAGS}")
    endif()
  endif()
endfunction()

function(vpvl2_link_icu target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_STRING)
    target_link_libraries(${target} ${ICU_LIBRARY_I18N} ${ICU_LIBRARY_UC} ${ICU_LIBRARY_DATA})
  endif()
endfunction()

function(vpvl2_find_icu)
  if(VPVL2_ENABLE_EXTENSIONS_STRING)
    if(MSVC)
      get_source_directory(ICU_INSTALL_DIR "icu4c-src")
    else()
      get_install_directory(ICU_INSTALL_DIR "icu4c-src")
    endif()
    find_path(ICU_INCLUDE_DIR NAMES unicode/unistr.h PATH_SUFFIXES include PATHS ${ICU_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ICU_LIBRARY_DATA NAMES icudata icudt PATH_SUFFIXES lib64 lib32 lib PATHS ${ICU_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ICU_LIBRARY_I18N NAMES icui18n icuin PATH_SUFFIXES lib64 lib32 lib PATHS ${ICU_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ICU_LIBRARY_UC NAMES icuuc PATH_SUFFIXES lib64 lib32 lib PATHS ${ICU_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${ICU_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_tbb target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_INTEL_TBB)
    target_link_libraries(${target} ${TBB_LIBRARY} ${TBB_PROXY_LIBRARY} ${TBB_MALLOC_LIBRARY} ${TBB_MALLOC_PROXY_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_tbb)
  if(VPVL2_LINK_INTEL_TBB)
    get_source_directory(TBB_SOURCE_DIRECTORY "tbb-src")
    if(WIN32 AND CMAKE_BUILD_TYPE STREQUAL "Debug")
      find_library(TBB_LIBRARY NAMES tbb_debug PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      find_library(TBB_PROXY_LIBRARY NAMES tbbproxy_debug PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      find_library(TBB_MALLOC_LIBRARY NAMES tbbmalloc_debug PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      find_library(TBB_MALLOC_PROXY_LIBRARY NAMES tbbmalloc_proxy_debug PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
    else()
      find_library(TBB_LIBRARY NAMES tbb PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      find_library(TBB_MALLOC_LIBRARY NAMES tbbmalloc PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      if(WIN32)
        find_library(TBB_PROXY_LIBRARY NAMES tbbproxy PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
        find_library(TBB_MALLOC_PROXY_LIBRARY NAMES tbbmalloc_proxy PATH_SUFFIXES lib PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      endif()
    endif()
    find_path(TBB_INCLUDE_DIR tbb/tbb.h PATH_SUFFIXES include PATHS ${TBB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
    include_directories(${TBB_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_zlib target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_ARCHIVE)
    target_link_libraries(${target} ${ZLIB_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_zlib)
  if(VPVL2_ENABLE_EXTENSIONS_ARCHIVE)
    if(NOT APPLE)
      get_install_directory(ZLIB_INSTALL_DIR "zlib-src")
      find_path(ZLIB_INCLUDE_DIR NAMES zlib.h PATH_SUFFIXES include PATHS ${ZLIB_INSTALL_DIR} NO_DEFAULT_PATH)
      find_library(ZLIB_LIBRARY NAMES z zlibstaticd zlibstatic PATH_SUFFIXES lib64 lib32 lib PATHS ${ZLIB_INSTALL_DIR} NO_DEFAULT_PATH)
    else()
      get_source_directory(ZLIB_SOURCE_DIRECTORY "zlib-src")
      get_build_directory(ZLIB_BUILD_DIRECTORY ZLIB_SOURCE_DIRECTORY)
      get_local_library_directory(ZLIB_LIBRARY_LOCAL_DIR ZLIB_SOURCE_DIRECTORY)
      find_library(ZLIB_LIBRARY NAMES z PATHS ${ZLIB_LIBRARY_LOCAL_DIR} NO_DEFAULT_PATH)
      find_path(ZLIB_INCLUDE_DIR NAMES zlib.h PATHS ${ZLIB_SOURCE_DIRECTORY} NO_DEFAULT_PATH)
      find_path(ZLIB_INCLUDE_CONFIG_DIR NAMES zconf.h PATHS ${ZLIB_BUILD_DIRECTORY} NO_DEFAULT_PATH)
    endif()
    include_directories(${ZLIB_INCLUDE_DIR} ${ZLIB_INCLUDE_CONFIG_DIR})
  endif()
endfunction()

function(vpvl2_link_libxml2 target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_PROJECT)
    target_link_libraries(${target} ${LIBXML2_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_libxml2)
  if(VPVL2_ENABLE_EXTENSIONS_PROJECT)
    get_install_directory(LIBXML2_INSTALL_DIR "libxml2-src")
    find_path(LIBXML2_INCLUDE_DIR NAMES libxml/xmlwriter.h PATH_SUFFIXES include/libxml2 PATHS ${LIBXML2_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(LIBXML2_LIBRARY NAMES xml2 libxml2_a libxml2 PATH_SUFFIXES lib64 lib32 lib PATHS ${LIBXML2_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${LIBXML2_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_alsoft target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    target_link_libraries(${target} ${ALSOFT_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_alsoft)
  if(VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    get_install_directory(ALSOFT_INSTALL_DIR "openal-soft-src")
    if(WIN32)
      find_path(ALSOFT_INCLUDE_DIR NAMES al.h PATH_SUFFIXES include/AL PATHS ${ALSOFT_INSTALL_DIR} NO_DEFAULT_PATH)
    else()
      find_path(ALSOFT_INCLUDE_DIR NAMES OpenAL/al.h AL/al.h PATH_SUFFIXES include PATHS ${ALSOFT_INSTALL_DIR} NO_DEFAULT_PATH)
    endif()
    find_library(ALSOFT_LIBRARY NAMES OpenAL32 openal PATH_SUFFIXES lib64 lib32 lib PATHS ${ALSOFT_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${ALSOFT_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_alure target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    target_link_libraries(${target} ${ALURE_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_alure)
  if(VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    get_install_directory(ALURE_INSTALL_DIR "alure-src")
    find_path(ALURE_INCLUDE_DIR NAMES OpenAL/alure.h AL/alure.h PATH_SUFFIXES include PATHS ${ALURE_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ALURE_LIBRARY NAMES ALURE32-static alure-static ALURE32 alure PATH_SUFFIXES lib64 lib32 lib PATHS ${ALURE_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${ALURE_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_gl_runtime target)
  if(NOT VPVL2_ENABLE_LAZY_LINK)
    target_link_libraries(${target} ${OPENGL_gl_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_gl_runtime)
  if(VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    if(VPVL2_ENABLE_OSMESA)
      get_source_directory(MESA3D_SOURCE_DIR "mesa-src")
      find_library(MESA3D_MESA_LIBRARY NAMES mesa PATH_SUFFIXES "embed-darwin-x86_64/mesa" "darwin-x86_64/mesa" PATHS "${MESA3D_SOURCE_DIR}/build" NO_DEFAULT_PATH)
      find_library(MESA3D_OSMESA_LIBRARY NAMES osmesa PATH_SUFFIXES "embed-darwin-x86_64/mesa/drivers/osmesa" "darwin-x86_64/mesa/drivers/osmesa" PATHS "${MESA3D_SOURCE_DIR}/build" NO_DEFAULT_PATH)
      set(OPENGL_gl_LIBRARY "${OPENGL_gl_mesa_LIBRARY} ${OPENGL_gl_osmesa_LIBRARY}")
      find_path(OPENGL_INCLUDE_DIR GL/osmesa.h PATH_SUFFIXES include PATHS ${MESA3D_SOURCE_DIR} NO_DEFAULT_PATH)
    elseif(VPVL2_ENABLE_GLES2)
      find_path(OPENGL_INCLUDE_DIR gl2.h PATH_SUFFIXES include/OpenGLES2 include/GLES2 include)
      if(NOT VPVL2_PLATFORM_EMSCRIPTEN)
        find_library(OPENGL_gl_LIBRARY NAMES ppapi_gles2 GLESv2)
      endif()
    elseif(NOT VPVL2_ENABLE_LAZY_LINK)
      find_package(OpenGL REQUIRED)
    endif()
    include_directories(${OPENGL_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_cg_runtime target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_NVIDIA_CG AND VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    target_link_libraries(${target} ${CG_LIBRARY} ${CG_GL_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_cg_runtime)
  if(VPVL2_ENABLE_NVIDIA_CG AND VPVL2_ENABLE_EXTENSIONS_APPLICATIONCONTEXT)
    find_package(Cg REQUIRED)
    include_directories(${CG_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_cl_runtime target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_OPENCL)
    target_link_libraries(${target} ${OPENCL_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_cl_runtime)
  if(VPVL2_ENABLE_OPENCL)
    find_library(OPENCL_LIBRARY NAMES OpenCL)
    find_path(OPENCL_INCLUDE_DIR cl.h PATH_SUFFIXES include/OpenCL include/CL include)
    include_directories(${OPENCL_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_find_gli)
  get_source_directory(GLI_SOURCE_DIRECTORY "gli-src")
  find_path(GLI_INCLUDE_DIR NAMES "gli/gli.hpp" PATHS ${GLI_SOURCE_DIRECTORY})
  include_directories(${GLI_INCLUDE_DIR})
endfunction()

function(vpvl2_find_glm)
  get_source_directory(GLM_SOURCE_DIRECTORY "glm-src")
  find_path(GLM_INCLUDE_DIR NAMES "glm/glm.hpp" PATHS ${GLM_SOURCE_DIRECTORY})
  include_directories(${GLM_INCLUDE_DIR})
endfunction()

function(vpvl2_link_glog target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_GLOG)
    target_link_libraries(${target} ${GLOG_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_glog)
  if(VPVL2_LINK_GLOG)
    if(MSVC)
      get_source_directory(GLOG_INSTALL_DIR "glog-src")
      find_path(GLOG_INCLUDE_DIR NAMES glog/logging.h PATH_SUFFIXES src/windows PATHS ${GLOG_INSTALL_DIR} NO_DEFAULT_PATH)
      find_library(GLOG_LIBRARY NAMES libglog PATH_SUFFIXES ${CMAKE_BUILD_TYPE} PATHS ${GLOG_INSTALL_DIR} NO_DEFAULT_PATH)
    else()
      get_install_directory(GLOG_INSTALL_DIR "glog-src")
      find_path(GLOG_INCLUDE_DIR NAMES glog/logging.h PATH_SUFFIXES include PATHS ${GLOG_INSTALL_DIR} NO_DEFAULT_PATH)
      find_library(GLOG_LIBRARY NAMES glog PATH_SUFFIXES lib64 lib32 lib PATHS ${GLOG_INSTALL_DIR} NO_DEFAULT_PATH)
    endif()
    include_directories(${GLOG_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_hlslxc target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_ENABLE_EXTENSIONS_EFFECT)
    target_link_libraries(${target} ${HLSLXC_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_hlslxc)
  if(VPVL2_ENABLE_EXTENSIONS_EFFECT)
    get_source_directory(HLSLXC_SRC_DIR "hlslxc-src")
    find_path(HLSLXC_INCLUDE_DIR NAMES toGLSL.h PATHS ${HLSLXC_SRC_DIR} NO_DEFAULT_PATH)
    find_library(HLSLXC_LIBRARY NAMES libHLSLcc PATH_SUFFIXES lib PATHS ${HLSLXC_SRC_DIR} NO_DEFAULT_PATH)
    include_directories(${HLSLXC_INCLUDE_DIR} "${HLSLXC_INCLUDE_DIR}/cbstring")
  endif()
endfunction()

function(vpvl2_link_regal target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_REGAL)
    target_link_libraries(${target} ${REGAL_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_regal)
  if(VPVL2_LINK_REGAL)
    get_source_directory(REGAL_INSTALL_DIR "regal-src")
    find_path(REGAL_INCLUDE_DIR NAMES GL/Regal.h PATH_SUFFIXES include PATHS ${REGAL_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(REGAL_LIBRARY NAMES Regal PATH_SUFFIXES lib64 lib32 lib PATHS ${REGAL_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${REGAL_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_nvfx target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_NVFX)
    target_link_libraries(${target} ${NVFX_FXPARSER_LIBRARY} ${NVFX_FXLIBGL_LIBRARY} ${NVFX_FXLIB_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_nvfx)
  if(VPVL2_LINK_NVFX)
    get_install_directory(NVFX_INSTALL_DIR "nvfx-src")
    find_path(NVFX_INCLUDE_DIR NAMES FxParser.h PATH_SUFFIXES include PATHS ${NVFX_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(NVFX_FXLIB_LIBRARY NAMES FxLib PATH_SUFFIXES lib64 lib32 lib PATHS ${NVFX_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(NVFX_FXLIBGL_LIBRARY NAMES FxLibGL PATH_SUFFIXES lib64 lib32 lib PATHS ${NVFX_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(NVFX_FXPARSER_LIBRARY NAMES FxParser PATH_SUFFIXES lib64 lib32 lib PATHS ${NVFX_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${NVFX_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_atb target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_ATB)
    target_link_libraries(${target} ${ATB_LIBRARY})
    if(APPLE)
      find_library(COCOA_FRAMEWORK NAMES Cocoa)
      target_link_libraries(${target} ${COCOA_FRAMEWORK})
     endif()
  endif()
endfunction()

function(vpvl2_find_atb)
  if(VPVL2_LINK_ATB)
    get_source_directory(ATB_INSTALL_DIR "AntTweakBar-src")
    find_path(ATB_INCLUDE_DIR NAMES AntTweakBar.h PATH_SUFFIXES include PATHS ${ATB_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ATB_LIBRARY NAMES AntTweakBar PATH_SUFFIXES lib64 lib32 lib PATHS ${ATB_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${ATB_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_link_freeimage target)
  if(NOT VPVL2_ENABLE_LAZY_LINK AND VPVL2_LINK_FREEIMAGE)
    target_link_libraries(${target} ${FREEIMAGE_LIBRARY})
  endif()
endfunction()

function(vpvl2_find_freeimage)
  if(VPVL2_LINK_FREEIMAGE)
    find_path(FREEIMAGE_INCLUDE_DIR NAMES FreeImage.h PATH_SUFFIXES include)
    find_library(FREEIMAGE_LIBRARY NAMES freeimage PATH_SUFFIXESlib lib64 lib32 )
    include_directories(${FREEIMAGE_INCLUDE_DIR})
  endif()
endfunction()

function(vpvl2_create_executable target)
  if(target)
    add_dependencies(${target} ${VPVL2_PROJECT_NAME})
    target_link_libraries(${target} ${VPVL2_PROJECT_NAME})
    vpvl2_link_all(${target})
  endif()
endfunction()

function(vpvl2_add_qt_helpers)
  if(VPVL2_BUILD_QT_RENDERER OR VPVL2_LINK_QT)
    find_package(Qt5Core QUIET)
    if (Qt5Core_FOUND)
      qt5_add_resources(vpvl2qtcommon_rc_src "${CMAKE_CURRENT_SOURCE_DIR}/src/qt/resources/libvpvl2qtcommon.qrc")
    else()
      find_package(Qt4 4.8 REQUIRED QtCore QtGui QtOpenGL)
      include(${QT_USE_FILE})
      qt4_add_resources(vpvl2qtcommon_rc_src "${CMAKE_CURRENT_SOURCE_DIR}/src/qt/resources/libvpvl2qtcommon.qrc")
    endif()
    file(GLOB vpvl2qtcommon_sources_common "${CMAKE_CURRENT_SOURCE_DIR}/src/qt/common/*.cc")
    file(GLOB vpvl2qtcommon_sources_unzip "${CMAKE_CURRENT_SOURCE_DIR}/src/qt/unzip/*.c")
    file(GLOB vpvl2qtcommon_headers_extensions "${CMAKE_CURRENT_SOURCE_DIR}/include/vpvl2/extensions/details/*.h")
    file(GLOB vpvl2qtcommon_headers "${CMAKE_CURRENT_SOURCE_DIR}/include/vpvl2/qt/*.h")
    source_group("VPVL2 for Qt sources" FILES ${vpvl2qtcommon_sources_common}
                                              ${vpvl2qtcommon_sources_unzip}
                                              ${vpvl2qtcommon_headers_extensions}
                                              ${vpvl2qtcommon_headers})
    set(vpvl2_qt_sources "${CMAKE_CURRENT_SOURCE_DIR}/render/qt/main.cc"
                         "${CMAKE_CURRENT_SOURCE_DIR}/render/qt/UI.cc")
    include_directories("${CMAKE_CURRENT_SOURCE_DIR}/render/qt"
                        "${CMAKE_CURRENT_SOURCE_DIR}/include/vpvl2/qt")
    if(APPLE AND NOT Qt5Core_FOUND)
      find_library(COCOA_LIBRARY NAMES Cocoa)
      find_path(COCOA_INCLUDE_DIR NAMES Cocoa/Cocoa.h)
      file(GLOB vpvl2qtcommon_sources_osx "${CMAKE_CURRENT_SOURCE_DIR}/src/qt/osx/*.mm")
      include_directories(${COCOA_INCLUDE_DIR})
    endif()
    add_library(vpvl2qtcommon ${VPVL2_LIB_TYPE} ${vpvl2qtcommon_sources_common} ${vpvl2qtcommon_sources_unzip}
                                                ${vpvl2qtcommon_headers} ${vpvl2qtcommon_headers_extensions}
                                                ${vpvl2qtcommon_sources_osx} ${vpvl2qtcommon_rc_src})
    vpvl2_set_library_properties(vpvl2qtcommon ${vpvl2qtcommon_headers})
    target_link_libraries(vpvl2qtcommon ${VPVL2_PROJECT_NAME})
    if (Qt5Core_FOUND)
      qt5_use_modules(vpvl2qtcommon Concurrent Widgets)
    else()
      target_link_libraries(vpvl2qtcommon ${VPVL2_PROJECT_NAME} ${QT_LIBRARIES})
    endif()
    if(APPLE)
      target_link_libraries(vpvl2qtcommon ${COCOA_LIBRARY})
    endif()
    set(VPVL2QTCOMMON_OUTPUT_NAME "vpvl2qtcommon")
    set_target_properties(vpvl2qtcommon PROPERTIES OUTPUT_NAME ${VPVL2QTCOMMON_OUTPUT_NAME}
                                                   VERSION ${VPVL2_VERSION}
                                                   SOVERSION ${VPVL2_VERSION_COMPATIBLE})
    if(VPVL2_BUILD_QT_RENDERER)
      set(VPVL2_EXECUTABLE vpvl2_qt)
      add_executable(${VPVL2_EXECUTABLE} ${vpvl2_qt_sources} ${vpvl2_internal_headers})
      add_dependencies(${VPVL2_EXECUTABLE} vpvl2qtcommon)
      vpvl2_link_alsoft(${VPVL2_EXECUTABLE})
      vpvl2_link_alure(${VPVL2_EXECUTABLE})
      if (Qt5Core_FOUND)
        qt5_use_modules(${VPVL2_EXECUTABLE} OpenGL Concurrent Widgets)
        target_link_libraries(${VPVL2_EXECUTABLE} vpvl2qtcommon)
      else()
        target_link_libraries(${VPVL2_EXECUTABLE} ${VPVL2_PROJECT_NAME} vpvl2qtcommon ${QT_LIBRARIES})
      endif()
      vpvl2_create_executable(${VPVL2_EXECUTABLE})
    endif()
    if(VPVL2_ENABLE_TEST)
      get_source_directory(GTEST_INSTALL_DIR "gtest-src")
      get_source_directory(GMOCK_INSTALL_DIR "gmock-src")
      file(GLOB vpvl2_test_sources "${CMAKE_CURRENT_SOURCE_DIR}/test/*.cc")
      source_group("VPVL2 Test Case Classes" FILES ${vpvl2_test_sources})
      file(GLOB vpvl2_mock_headers "${CMAKE_CURRENT_SOURCE_DIR}/test/mock/*.h")
      source_group("VPVL2 Mock Classes" FILES ${vpvl2_mock_headers})
      file(GLOB gtest_source "${GTEST_INSTALL_DIR}/src/gtest-all.cc")
      file(GLOB gmock_source "${GMOCK_INSTALL_DIR}/src/gmock-all.cc")
      if(Qt5Core_FOUND)
        qt5_add_resources(vpvl2_test_qrc "${CMAKE_CURRENT_SOURCE_DIR}/test/fixtures.qrc")
      else()
        qt4_add_resources(vpvl2_test_qrc "${CMAKE_CURRENT_SOURCE_DIR}/test/fixtures.qrc")
      endif()
      add_executable(vpvl2_test ${vpvl2_test_sources} ${vpvl2_mock_headers} ${vpvl2_test_qrc} ${gtest_source} ${gmock_source})
      if(Qt5Core_FOUND)
        qt5_use_modules(vpvl2_test OpenGL Concurrent)
      endif()
      include_directories(${GTEST_INSTALL_DIR} "${GTEST_INSTALL_DIR}/include")
      include_directories(${GMOCK_INSTALL_DIR} "${GMOCK_INSTALL_DIR}/include")
      target_link_libraries(vpvl2_test vpvl2qtcommon ${VPVL2_PROJECT_NAME} ${QT_LIBRARIES})
      add_dependencies(vpvl2_test ${VPVL2_PROJECT_NAME} vpvl2qtcommon)
      vpvl2_link_all(vpvl2_test)
    endif()
  endif()
endfunction()

function(vpvl2_add_sdl_renderer)
  if(VPVL2_LINK_SDL2)
    get_install_directory(SDL_INSTALL_DIR "SDL2-src")
    find_library(SDL_LIBRARY NAMES SDL2 PATH_SUFFIXES lib PATHS ${SDL_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(SDL_MAIN_LIBRARY NAMES SDL2main PATH_SUFFIXES lib PATHS ${SDL_INSTALL_DIR} NO_DEFAULT_PATH)
    find_path(SDL_INCLUDE_DIR NAMES SDL.h PATH_SUFFIXES include/SDL2 PATHS ${SDL_INSTALL_DIR} NO_DEFAULT_PATH)
    add_definitions(${SDL_DEFINITIONS})
    include_directories(${SDL_INCLUDE_DIR} ${SDLIMAGE_INCLUDE_DIR})
    set(vpvl2_sdl_sources "render/sdl/main.cc")
    set(VPVL2_EXECUTABLE vpvl2_sdl)
    add_executable(${VPVL2_EXECUTABLE} ${vpvl2_sdl_sources})
    target_link_libraries(${VPVL2_EXECUTABLE} ${SDL_MAIN_LIBRARY} ${SDL_LIBRARY})# ${SDLIMAGE_LIBRARY})
    vpvl2_create_executable(${VPVL2_EXECUTABLE})
    if(WIN32)
      find_library(WIN32_WINMM_LIBRARY NAMES winmm)
      find_library(WIN32_VERSION_LIBRARY NAMES version)
      find_library(WIN32_IMM_LIBRARY NAMES imm32)
      target_link_libraries(${VPVL2_EXECUTABLE} ${WIN32_WINMM_LIBRARY} ${WIN32_IMM_LIBRARY} ${WIN32_VERSION_LIBRARY})
    endif()
  endif()
endfunction()

function(vpvl2_add_glfw_renderer)
  if(VPVL2_LINK_GLFW)
    get_install_directory(GLFW_INSTALL_DIR "glfw-src")
    find_path(GLFW_INCLUDE_DIR NAMES GLFW/glfw3.h PATH_SUFFIXES include PATHS ${GLFW_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(GLFW_LIBRARY NAMES glfw3 glfw PATH_SUFFIXES lib64 lib32 lib PATHS ${GLFW_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${GLFW_INCLUDE_DIR})
    set(vpvl2_glfw_sources "render/glfw/main.cc")
    set(VPVL2_EXECUTABLE vpvl2_glfw)
    add_executable(${VPVL2_EXECUTABLE} ${vpvl2_glfw_sources})
    if(APPLE)
      find_library(COCOA_LIBRARY NAMES Cocoa)
      find_library(IOKIT_LIBRARY NAMES IOKit)
      find_library(COREFOUNDATION_LIBRARY NAMES CoreFoundation)
      target_link_libraries(${VPVL2_EXECUTABLE} ${COCOA_LIBRARY} ${IOKIT_LIBRARY} ${COREFOUNDATION_LIBRARY})
    endif()
    target_link_libraries(${VPVL2_EXECUTABLE} ${GLFW_LIBRARY})
    vpvl2_create_executable(${VPVL2_EXECUTABLE})
  endif()
endfunction()

function(vpvl2_add_sfml_renderer)
  if(VPVL2_LINK_SFML)
    get_install_directory(SFML2_INSTALL_DIR "SFML2-src")
    find_library(SFML2_GRAPHICS_LIBRARY NAMES sfml-graphics sfml-graphics-d PATH_SUFFIXES lib64 lib32 lib PATHS ${SFML2_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(SFML2_SYSTEM_LIBRARY NAMES sfml-system sfml-system-d PATH_SUFFIXES lib64 lib32 lib PATHS ${SFML2_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(SFML2_WINDOW_LIBRARY NAMES sfml-window sfml-window-d PATH_SUFFIXES lib64 lib32 lib PATHS ${SFML2_INSTALL_DIR} NO_DEFAULT_PATH)
    find_path(SFML_INCLUDE_DIR SFML/System.hpp PATH_SUFFIXES include PATHS ${SFML2_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${SFML_INCLUDE_DIR})
    set(vpvl2_sfml_sources "render/sfml/main.cc")
    set(VPVL2_EXECUTABLE vpvl2_sfml)
    add_executable(${VPVL2_EXECUTABLE} ${vpvl2_sfml_sources})
    target_link_libraries(${VPVL2_EXECUTABLE} ${SFML2_GRAPHICS_LIBRARY}
                                              ${SFML2_WINDOW_LIBRARY}
                                              ${SFML2_SYSTEM_LIBRARY})
    if(APPLE AND NOT VPVL2_LINK_GLEW)
      find_library(COREFOUNDATION_FRAMEWORK NAMES CoreFoundation)
      find_path(COREFOUNDATION_INCLUDE_PATH "CoreFoundation/CoreFoundation.h")
      target_link_libraries(${VPVL2_EXECUTABLE} ${COREFOUNDATION_FRAMEWORK})
      include_directories(${COREFOUNDATION_INCLUDE_PATH})
    endif()
    vpvl2_create_executable(${VPVL2_EXECUTABLE})
  endif()
endfunction()

function(vpvl2_add_allegro_renderer)
  if(VPVL2_LINK_ALLEGRO)
    get_install_directory(ALLEGRO_INSTALL_DIR "allegro-src")
    find_library(ALLEGRO_LIBRARY NAMES allegro-static allegro allegro-debug-static allegro-debug PATH_SUFFIXES lib64 lib32 lib PATHS ${ALLEGRO_INSTALL_DIR} NO_DEFAULT_PATH)
    find_library(ALLEGRO_MAIN_LIBRARY NAMES allegro_main-static allegro_main allegro_main-debug-static allegro_main-debug PATH_SUFFIXES lib64 lib32 lib PATHS ${ALLEGRO_INSTALL_DIR} NO_DEFAULT_PATH)
    find_path(ALLEGRO_INCLUDE_DIR NAMES allegro5/allegro5.h PATH_SUFFIXES include PATHS ${ALLEGRO_INSTALL_DIR} NO_DEFAULT_PATH)
    include_directories(${ALLEGRO_INCLUDE_DIR})
    set(vpvl2_sfml_sources "render/allegro/main.cc")
    set(VPVL2_EXECUTABLE vpvl2_allegro)
    add_executable(${VPVL2_EXECUTABLE} ${vpvl2_sfml_sources})
    target_link_libraries(${VPVL2_EXECUTABLE} ${ALLEGRO_MAIN_LIBRARY} ${ALLEGRO_LIBRARY})
    if(APPLE)
      find_library(IOKIT_FRAMEWORK NAMES IOKit)
      target_link_libraries(${VPVL2_EXECUTABLE} ${IOKIT_FRAMEWORK})
    endif()
    vpvl2_create_executable(${VPVL2_EXECUTABLE})
  endif()
endfunction()

function(vpvl2_add_egl_renderer)
  if(VPVL2_LINK_EGL)
    find_path(EGL_INCLUDE_DIR NAMES EGL/egl.h)
    find_library(EGL_LIBRARY NAMES EGL)
    set(vpvl2_egl_sources "render/egl/main.cc")
    set(VPVL2_EXECUTABLE vpvl2_egl)
    add_executable(${VPVL2_EXECUTABLE} ${vpvl2_egl_sources})
    target_link_libraries(${VPVL2_EXECUTABLE} ${EGL_LIBRARY})
    include_directories(${EGL_INCLUDE_DIR})
    if(VPVL2_PLATFORM_RASPBERRY_PI)
      find_path(VCOS_INCLUDE_DIR vcos_platform_types.h)
      find_library(BCM_HOST_LIBRARY NAMES bcm_host)
      find_library(VCOS_LIBRARY NAMES vcos)
      find_library(VCHIQ_ARM_LIBRARY NAMES vchiq_arm)
      include_directories(${VCOS_INCLUDE_DIR})
      target_link_libraries(${VPVL2_EXECUTABLE} ${BCM_HOST_LIBRARY} ${VCOS_LIBRARY} ${VCHIQ_ARM_LIBRARY})
    endif()
    vpvl2_create_executable(${VPVL2_EXECUTABLE})
  endif()
endfunction()

function(vpvl2_find_all)
  vpvl2_find_vpvl()
  vpvl2_find_nvfx()
  vpvl2_find_regal()
  vpvl2_find_atb()
  vpvl2_find_hlslxc()
  vpvl2_find_alsoft()
  vpvl2_find_alure()
  vpvl2_find_freeimage()
  vpvl2_find_tbb()
  vpvl2_find_bullet()
  vpvl2_find_assimp()
  vpvl2_find_icu()
  vpvl2_find_glog()
  vpvl2_find_gli()
  vpvl2_find_glm()
  vpvl2_find_zlib()
  vpvl2_find_openmp()
  vpvl2_find_cg_runtime()
  vpvl2_find_cl_runtime()
  vpvl2_find_gl_runtime()
endfunction()

function(vpvl2_link_all target)
  vpvl2_link_vpvl(${target})
  vpvl2_link_nvfx(${target})
  vpvl2_link_regal(${target})
  vpvl2_link_atb(${target})
  vpvl2_link_hlslxc(${target})
  vpvl2_link_freeimage(${target})
  vpvl2_link_tbb(${target})
  vpvl2_link_bullet(${target})
  vpvl2_link_assimp(${target})
  vpvl2_link_icu(${target})
  vpvl2_link_glog(${target})
  vpvl2_link_zlib(${target})
  vpvl2_link_cg_runtime(${target})
  vpvl2_link_cl_runtime(${target})
  vpvl2_link_gl_runtime(${target})
endfunction()
# end of functions
