﻿using System;
using System.Globalization;
using System.Xml.Linq;
using Codeplex.Data;
using LinqToTwitter;

namespace Azyobuzi.UserStreamEx
{
    public class StoppedEventArgs : EventArgs
    {
        public StoppedEventArgs(StopReason reason, Exception ex)
        {
            this.Reason = reason;
            this.Exception = ex;
        }

        public StopReason Reason { private set; get; }
        public Exception Exception { private set; get; }
    }

    public enum StopReason
    {
        /// <summary>
        /// Twitterが切断
        /// </summary>
        CloseResponse,
        /// <summary>
        /// Stopメソッドが呼ばれた
        /// </summary>
        UserStop,
        /// <summary>
        /// その他の例外
        /// </summary>
        Error
    }

    public class ReceiveJsonEventArgs : EventArgs
    {
        public ReceiveJsonEventArgs(string line)
        {
            this.Line = line;
        }

        public string Line { private set; get; }
    }

    public class ReceiveFriendsEventArgs : ReceiveJsonEventArgs
    {
        public ReceiveFriendsEventArgs(string line)
            : base(line)
        {
            FriendIds = (string[])DynamicJson.Parse(line).friends;
        }

        public string[] FriendIds { private set; get; }
    }

    public class NewTweetEventArgs : ReceiveJsonEventArgs
    {
        public NewTweetEventArgs(string line)
            : base(line)
        {
            Status = Status.CreateStatus(line.JsonToXml());
        }

        public Status Status { private set; get; }
    }

    public class NewDirectMessageEventArgs : ReceiveJsonEventArgs
    {
        public NewDirectMessageEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line).direct_message;
            DirectMessage = new DirectMessage()
            {
                ID = (ulong)json.id,
                SenderID = (ulong)json.sender_id,
                Text = json.text,
                RecipientID = (ulong)json.recipient_id,
                CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture),
                SenderScreenName = json.sender_screen_name,
                RecipientScreenName = json.recipient_screen_name,
                Sender = User.CreateUser(((string)json.sender.ToString()).JsonToXml()),
                Recipient = User.CreateUser(((string)json.recipient.ToString()).JsonToXml())
            };
        }

        public DirectMessage DirectMessage { private set; get; }
    }

    public class DeleteStatusEventArgs : ReceiveJsonEventArgs
    {
        public DeleteStatusEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line).delete;
            if (!json.direct_message())
            {
                //ツイート
                StatusId = json.status.id_str;
                UserId = json.status.user_id_str;
                IsDirectMessage = false;
            }
            else
            {
                //DM
                StatusId = json.direct_message.id.ToString();
                UserId = json.direct_message.user_id.ToString();
                IsDirectMessage = true;
            }
        }

        public string StatusId { private set; get; }
        public string UserId { private set; get; }
        public bool IsDirectMessage { private set; get; }
    }

    public class ReceiveEventEventArgs : ReceiveJsonEventArgs
    {
        public ReceiveEventEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line);
            EventTypes eventType = EventTypes.Unknown;
            Enum.TryParse(((string)json.@event).Replace("_", ""), true, out eventType);
            EventType = eventType;
            CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture).ToLocalTime();
            Source = User.CreateUser(((string)json.source.ToString()).JsonToXml());
            Target = User.CreateUser(((string)json.target.ToString()).JsonToXml());
            if (json.target_object())
            {
                if (json.target_object.mode())
                {
                    //リスト
                    TargetList = List.CreateList(((string)json.target_object.ToString()).JsonToXml(), new XElement("lists_list"));
                }
                else
                {
                    //ツイート
                    TargetStatus = Status.CreateStatus(((string)json.target_object.ToString()).JsonToXml());
                }
            }
        }

        public EventTypes EventType { private set; get; }
        public DateTime CreatedAt { private set; get; }
        public User Source { private set; get; }
        public User Target { private set; get; }
        public Status TargetStatus { private set; get; }
        public List TargetList { private set; get; }
    }

    public enum EventTypes
    {
        Unknown,
        Favorite,
        Unfavorite,
        Follow,
        ListMemberAdded,
        ListMemberRemoved,
        Block,
        Unblock,
        UserUpdate,
        ListCreated,
        ListDestroyed
    }

    public class TrackLimitEventArgs : ReceiveJsonEventArgs
    {
        public TrackLimitEventArgs(string line)
            : base(line)
        {
            Track = (ulong)DynamicJson.Parse(line).limit.track;
        }

        public ulong Track { private set; get; }
    }
}
