﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Threading;
using Codeplex.Data;
using LinqToTwitter;

namespace Azyobuzi.UserStreamEx
{
    /// <summary>
    /// TwitterのUserStreamをイベントを使って処理します。
    /// </summary>
    public class UserStream
        : INotStartedUserStream, IStartedUserStream
    {
        /// <summary>
        /// <see cref="UserStream"/>クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="auth">認証済みの<see cref="LinqToTwitter.ITwitterAuthorizer"/></param>
        public UserStream(ITwitterAuthorizer auth)
        {
            this.auth = auth;
        }

        private ITwitterAuthorizer auth;
        private HttpWebRequest req;

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="track">検索キーワードっぽいもの。null可。</param>
        /// <param name="allReplies">friendsに対する返信も含めるかどうか</param>
        public IStartedUserStream Start(string track, bool allReplies)
        {
            if (req != null) req.Abort();
            
            var reqUri = new UriBuilder("https://userstream.twitter.com/2/user.json");
            var param = new Dictionary<string, string>();
            if (!string.IsNullOrEmpty(track)) param.Add("track", track);
            if (allReplies) param.Add("replies", "all");
            reqUri.Query = string.Join("&", param.Select(_ => string.Format("{0}={1}", _.Key, _.Value)));

            req = auth.Get(reqUri.ToString()) as HttpWebRequest;
            req.BeginGetResponse(result =>
            {
                try
                {
                    if (Started != null) Started(this, EventArgs.Empty);
                    var res = req.EndGetResponse(result);
                    using (var sr = new StreamReader(res.GetResponseStream()))
                    {
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();

                            if (StreamingWriter != null)
                            {
                                try
                                {
                                    StreamingWriter.WriteLine(line);
                                    StreamingWriter.Flush();
                                }
                                catch { }
                            }

                            if (string.IsNullOrWhiteSpace(line)) continue;

                            var t = new Thread(RaiseEvent);
                            t.Start(line);
                        }
                    }

                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(StopReason.CloseResponse, null));
                }
                catch (WebException ex)
                {
                    if (ex.Status == WebExceptionStatus.RequestCanceled)
                    {
                        if (Stopped != null)
                            Stopped(this, new StoppedEventArgs(StopReason.UserStop, ex));
                    }
                    else
                    {
                        throw;
                    }
                }
                catch (Exception ex)
                {
                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(StopReason.Error, ex));
                }
                req = null;
            }, null);

            return this;
        }

        private void RaiseEvent(object args)
        {
            var line = (string)args;
            var json = DynamicJson.Parse(line);
            if (json.friends())
            {
                if (ReceiveFriends != null) ReceiveFriends(this, new ReceiveFriendsEventArgs(line));
            }
            else if (json.delete())
            {
                if (DeleteStatus != null) DeleteStatus(this, new DeleteStatusEventArgs(line));
            }
            else if (json.direct_message())
            {
                if (NewDirectMessage != null) NewDirectMessage(this, new NewDirectMessageEventArgs(line));
            }
            else if (json.@event())
            {
                if (ReceiveEvent != null) ReceiveEvent(this, new ReceiveEventEventArgs(line));
            }
            else if (json.limit())
            {
                if (TrackLimit != null) TrackLimit(this, new TrackLimitEventArgs(line));
            }
            else if (json.text())
            {
                if (NewTweet != null) NewTweet(this, new NewTweetEventArgs(line));
            }
            else
            {
                if (ReceiveUnsupportedData != null) ReceiveUnsupportedData(this, new ReceiveJsonEventArgs(line));
            }
        }

        /// <summary>
        /// 切断してStoppedイベントを発生させます。
        /// </summary>
        public INotStartedUserStream Stop()
        {
            if (req != null) req.Abort();
            req = null;
            return this;
        }

        /// <summary>
        /// 指定した時間が経過したら切断するように設定します。
        /// </summary>
        /// <param name="waitTime">切断するまでの時間（ミリ秒単位）。0以下を指定するとすぐに切断します。</param>
        public ISetedTimeoutUserStream SetTimeout(int waitTime)
        {
            if (waitTime < 1)
            {
                Stop();
            }
            else
            {
                var t = new Thread(_ =>
                {
                    Thread.Sleep((int)_);
                    Stop();
                });
                t.IsBackground = true;
                t.Start(waitTime);
            }
            return this;
        }

        /// <summary>
        /// 取得したデータを書き込む<see cref="System.IO.TextWriter"/>
        /// </summary>
        public TextWriter StreamingWriter { set; get; }

        #region Events
        /// <summary>
        /// 接続を開始するときに発生します。
        /// </summary>
        public event EventHandler Started;
        /// <summary>
        /// 切断されたときに発生します。
        /// </summary>
        public event EventHandler<StoppedEventArgs> Stopped;
        /// <summary>
        /// friends(following)のID一覧を受け取ったときに発生します。通常は開始直後に一回だけ発生します。
        /// </summary>
        public event EventHandler<ReceiveFriendsEventArgs> ReceiveFriends;
        /// <summary>
        /// 新しいツイートを受け取ったときに発生します。
        /// </summary>
        public event EventHandler<NewTweetEventArgs> NewTweet;
        /// <summary>
        /// 新しいダイレクトメッセージを受信・送信したときに発生します。
        /// </summary>
        public event EventHandler<NewDirectMessageEventArgs> NewDirectMessage;
        /// <summary>
        /// ツイートまたはダイレクトメッセージが削除されたときに発生します。ハンドラでは直ちに表示から消すような処理をしてください。
        /// </summary>
        public event EventHandler<DeleteStatusEventArgs> DeleteStatus;
        /// <summary>
        /// イベントを受信したときに発生します。
        /// </summary>
        public event EventHandler<ReceiveEventEventArgs> ReceiveEvent;
        /// <summary>
        /// Twitter側で処理しきれないツイートがあったときに発生します。
        /// </summary>
        /// <remarks>詳細不明</remarks>
        public event EventHandler<TrackLimitEventArgs> TrackLimit;
        /// <summary>
        /// 対応していないデータを受け取ったときに発生します。
        /// </summary>
        public event EventHandler<ReceiveJsonEventArgs> ReceiveUnsupportedData;
        #endregion

        #region AddHandlerMethod
        /// <summary>
        /// <see cref="Started"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetStartedEvent(EventHandler handler)
        {
            Started += handler;
            return this;
        }

        /// <summary>
        /// <see cref="Stopped"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetStoppedEvent(EventHandler<StoppedEventArgs> handler)
        {
            Stopped += handler;
            return this;
        }

        /// <summary>
        /// <see cref="ReceiveFriends"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetReceiveFriendsEvent(EventHandler<ReceiveFriendsEventArgs> handler)
        {
            ReceiveFriends += handler;
            return this;
        }

        /// <summary>
        /// <see cref="NewTweet"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetNewTweetEvent(EventHandler<NewTweetEventArgs> handler)
        {
            NewTweet += handler;
            return this;
        }

        /// <summary>
        /// <see cref="NewDirectMessage"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetNewDirectMessageEvent(EventHandler<NewDirectMessageEventArgs> handler)
        {
            NewDirectMessage += handler;
            return this;
        }

        /// <summary>
        /// <see cref="DeleteStatus"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetDeleteStatusEvent(EventHandler<DeleteStatusEventArgs> handler)
        {
            DeleteStatus += handler;
            return this;
        }

        /// <summary>
        /// <see cref="ReceiveEvent"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetReceiveEventEvent(EventHandler<ReceiveEventEventArgs> handler)
        {
            ReceiveEvent += handler;
            return this;
        }

        /// <summary>
        /// <see cref="TrackLimit"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetTrackLimitEvent(EventHandler<TrackLimitEventArgs> handler)
        {
            TrackLimit += handler;
            return this;
        }

        /// <summary>
        /// <see cref="ReceiveUnsupportedData"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        public INotStartedUserStream SetReceiveUnsupportedDataEvent(EventHandler<ReceiveJsonEventArgs> handler)
        {
            ReceiveUnsupportedData += handler;
            return this;
        }
        #endregion
    }
}
