/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import net.morilib.automata.DFA;
import net.morilib.automata.DFAState;
import net.morilib.automata.NFA;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.automata.dfa.DFAs;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.util.TreeSectionMap;
import net.morilib.util.Tuple2;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/14
 */
public class DFABuilder implements NinaAction {

	static class DBS implements DFAState<Object, Object, Void> {

		TreeSectionMap<Interval, Object, DBS> edges;
		TreeSectionMap<Interval, Object, Object> mealy;
		boolean initial, accepted, moore;
		String nomFichier, label;

		PrintWriter out = null;
		int devoirFermer = 0;

		void linkAlphabet(DBS d, Object o, Object m) {
			if(edges == null) {
				edges = new TreeSectionMap<Interval, Object, DBS>(
						Interval.newPoint(o), d);
				mealy = new TreeSectionMap<Interval, Object, Object>(
						Interval.newPoint(o), m);
			} else if(edges.map(o) == null) {
				edges.insert(Interval.newPoint(o), d);
				mealy.insert(Interval.newPoint(o), m);
			} else {
				throw new NinaNondeterministicException(o.toString());
			}
		}

		void linkAlphabet(DBS d, Range o, Object m) {
			for(Interval v : o.intervals()) {
				if(edges == null) {
					edges = new TreeSectionMap<Interval, Object, DBS>(
							v, d);
					mealy = new TreeSectionMap<Interval, Object, Object>(
							v, m);
				} else {
					for(Tuple2<Interval, DBS> w : edges.entries()) {
						if(!o.independentOf(w.getA()) &&
								!d.equals(w.getB())) {
							throw new NinaNondeterministicException(
									o + "," + d + "," + w);
						}
					}
					edges.insert(v, d);
				}
			}
		}

		void sort(Object o) {
			try {
				if(nomFichier == null || nomFichier.equals("")) {
					return;
				} else if(out == null) {
					if(nomFichier.equals("/dev/stdout")) {
						out = new PrintWriter(System.out, true);
						devoirFermer = -1;
					} else if(nomFichier.equals("/dev/stderr")) {
						out = new PrintWriter(System.err, true);
						devoirFermer = -1;
					} else if(nomFichier.equals("/dev/null")) {
						out = null;
					} else {
						out = new PrintWriter(new FileOutputStream(
								nomFichier, devoirFermer > 0), true);
						devoirFermer = 1;
					}
				}
		
				if(out != null && o != null) {
					out.print(o);
					out.flush();
				}
				if(devoirFermer > 0)  out.close();
			} catch(IOException e) {
				// ignore it
				throw new RuntimeException(e);
			}
		}

		void sortMoore() {
			if(moore)  sort(label.trim());
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#go(java.lang.Object)
		 */
		@Override
		public DFAState<Object, Object, Void> go(Object a) {
			DFAState<Object, Object, Void> d;

			sort(mealy.map(a));
			if(edges == null || (d = edges.map(a)) == null) {
				return DFAs.<Object, Object, Void>deadState();
			} else {
				if(d instanceof DBS)  ((DBS)d).sortMoore();
				return d;
			}
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goInt(int)
		 */
		@Override
		public DFAState<Object, Object, Void> goInt(int x) {
			return go(Integer.valueOf(x));
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goChar(char)
		 */
		@Override
		public DFAState<Object, Object, Void> goChar(char x) {
			return go(Integer.valueOf(x));
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goBound(net.morilib.automata.TextBound)
		 */
		@Override
		public DFAState<Object, Object, Void> goBound(TextBound b) {
			return DFAs.deadState();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isInitialState()
		 */
		@Override
		public boolean isInitialState() {
			return initial;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#getAccepted()
		 */
		@Override
		public Set<Object> getAccepted() {
			return accepted ?
					Collections.<Object>singleton(label) :
						Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isDead()
		 */
		@Override
		public boolean isDead() {
			return edges == null;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isAccepted()
		 */
		@Override
		public boolean isAccepted() {
			return accepted;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#getAlphabets()
		 */
		@Override
		public Set<Object> getAlphabets() {
			Set<Object> r = new HashSet<Object>();
			Object a, b;
			int x;

			for(Interval v : edges.keys()) {
				a = v.getInfimumBound();
				b = v.getSupremumBound();
				if(a.equals(b)) {
					r.add(a);
				} else if(a instanceof Integer) {
					x = ((Integer)a).intValue();
					for(int i = x; v.contains(i); i++) {
						r.add(Integer.valueOf(i));
					}
				} else {
					throw new NinaParseException("invalidalphabet", a);
				}
			}
			return r;
		}

		@Override
		public Iterable<Interval> getAlphabetRanges() {
			return edges != null ?
					edges.keys() : Collections.<Interval>emptySet();
		}

		@Override
		public String toString() {
			return label;
		}

	};

	private DBS initial, vertex;
	private Object mealy;

	private DBS prendEtat(NinaEvent q) {
		DBS s;

		s = new DBS();
		s.nomFichier = q.getOption("out");
		s.moore = q.getOptionBoolean("moore");
		return s;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#labelAdded(net.morilib.nina.NinaEvent, java.lang.String, boolean)
	 */
	@Override
	public void labelAdded(NinaEvent q, NinaFrameReader rd,
			boolean accept) {
		StringWriter wr = new StringWriter();
		char[] a = new char[1024];
		int l;

		while((l = rd.read(a)) >= 0)  wr.write(a, 0, l);
		if(vertex == null) {
			// set initial state if it is not set
			vertex = prendEtat(q);
			vertex.initial = true;
			initial = vertex;
		}
		vertex.accepted = accept;
		vertex.label = wr.toString();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#link(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void link(NinaEvent q, Object ver) {
		DBS v = vertex;

		vertex = (ver != null) ? (DBS)ver : prendEtat(q);
		if(q.getEdge() instanceof Range) {
			v.linkAlphabet(vertex, (Range)q.getEdge(), mealy);
		} else {
			v.linkAlphabet(vertex, q.getEdge(), mealy);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getLabel()
	 */
	@Override
	public String getLabel() {
		return vertex.label;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getVertex()
	 */
	@Override
	public Object getVertex() {
		return vertex;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setVertex(java.lang.Object)
	 */
	@Override
	public void setVertex(Object o) {
		vertex = (DBS)o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#doneBlockSearching(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void doneBlockSearching(NinaEvent q) {
		vertex = (DBS)q.getScratch();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdge(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void setEdge(NinaEvent q, Object o) {
		q.setEdge(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeResource(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, java.lang.String)
	 */
	@Override
	public void setEdgeResource(NinaEvent q, NinaParser p, String s) {
		throw new NinaParseException("cannotaddnfa");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeNFA(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, net.morilib.automata.NFA)
	 */
	@Override
	public void setEdgeNFA(NinaEvent q, NinaParser p,
			NFA<Object, NFAState, Void> s) {
		throw new NinaParseException("cannotaddnfa");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#accept()
	 */
	@Override
	public Object accept() {
		return new DFA<Object, Object, Void>() {

			@Override
			public DFAState<Object, Object, Void> getInitialState() {
				initial.sortMoore();
				return initial;
			}

		};
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setMealyEdge(int)
	 */
	@Override
	public void setMealyEdge(int c) {
		mealy = Character.valueOf((char)c);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setMealyEdge(java.lang.Object)
	 */
	@Override
	public void setMealyEdge(Object o) {
		mealy = o;
	}

}
