/* JNetHack Copyright */
/* (c) Issei Numata 1994-2000                                      */
/* For 3.4-, Copyright (c) SHIRAKATA Kentaro, 2002-2020            */
/* JNetHack may be freely redistributed.  See license for details. */

#include <stdio.h>
#include <ctype.h>
#include "hack.h"
#ifdef POSIX_ICONV
#include <iconv.h>
#endif

int xputc(CHAR_P);
int xputc2(int, int);


#define EUC     0
#define SJIS    1
#define UTF8    2

#ifdef POSIX_ICONV
static const char* ccode[]={
    "EUC-JP-MS",
    "CP932",
    "UTF-8"
};
static const char* ccode_alt[]={
    "EUC-JP",
    "ShiftJIS",
    "UTF-8"
};
#endif

/* internal kcode */
/* IC=0 EUC */
/* IC=1 SJIS */
#define IC ((unsigned char)(""[0])==0x8a)

/* default input kcode */
#ifndef INPUT_KCODE
# if defined(MSDOS) || defined(WIN32)
#  define INPUT_KCODE SJIS
# else
#  define INPUT_KCODE UTF8
# endif
#endif

/* default output kcode */
#ifndef OUTPUT_KCODE
# if defined(MSDOS) || defined(WIN32)
#  define OUTPUT_KCODE SJIS
# else
#  define OUTPUT_KCODE UTF8
# endif
#endif

#ifdef WIN32_PLATFORM_PSPC 
#define putchar(x) {}
#endif

static int      output_kcode = OUTPUT_KCODE;
static int      input_kcode = INPUT_KCODE;
#ifdef POSIX_ICONV
static iconv_t  output_dsc = 0;
static iconv_t  input_dsc = 0;
#endif

/*
**      Kanji code library....
*/

/*
 * (1)ǂ𒲂ׂ
 */
int
is_kanji(c)
     unsigned c;
{
    if(IC == EUC)
      return (c & 0x80);
    else
      return ((unsigned int)c>=0x81 && (unsigned int)c<=0x9f)
        || ((unsigned int)c>=0xe0 && (unsigned int)c<=0xfc);
}

void
setkcode(c)
     int c;
{
    if(c == 'E' || c == 'e' )
      output_kcode = EUC;
    else if(c == 'S' || c == 's')
      output_kcode = SJIS;
    else if(c == 'U' || c == 'u')
      output_kcode = UTF8;
    else if(c == 'I' || c == 'i')
#ifdef MSDOS
      output_kcode = SJIS;
#else
      output_kcode = IC;
#endif
    else{
        output_kcode = IC;
    }
    input_kcode = output_kcode;

#ifdef POSIX_ICONV
    if (output_dsc)
        iconv_close(output_dsc);
    output_dsc = iconv_open(ccode[output_kcode], ccode[IC]);
    if (output_dsc == (iconv_t)-1)
        output_dsc = iconv_open(ccode_alt[output_kcode], ccode[IC]);
    if (output_dsc == (iconv_t)-1)
        output_dsc = iconv_open(ccode[output_kcode], ccode_alt[IC]);
    if (output_dsc == (iconv_t)-1)
        output_dsc = iconv_open(ccode_alt[output_kcode], ccode_alt[IC]);
    if(input_dsc)
        iconv_close(input_dsc);
    input_dsc = iconv_open(ccode[IC] ,ccode[input_kcode]);
    if (input_dsc == (iconv_t)-1)
        input_dsc = iconv_open(ccode_alt[IC] ,ccode[input_kcode]);
    if (input_dsc == (iconv_t)-1)
        input_dsc = iconv_open(ccode[IC] ,ccode_alt[input_kcode]);
    if (input_dsc == (iconv_t)-1)
        input_dsc = iconv_open(ccode_alt[IC] ,ccode_alt[input_kcode]);
#endif
}
/*
**      EUC->SJIS
*/

unsigned char *
e2sj(s)
     unsigned char *s;
{
    unsigned char h,l;
    static unsigned char sw[2];

    h = s[0] & 0x7f;
    l = s[1] & 0x7f;

    sw[0] = ((h - 1) >> 1)+ ((h <= 0x5e) ? 0x71 : 0xb1);
    sw[1] = l + ((h & 1) ? ((l < 0x60) ? 0x1f : 0x20) : 0x7e);

    return sw;
}
/*
**      SJIS->EUC
*/
unsigned char *
sj2e(s)
     unsigned char *s;
{
    unsigned int h,l;
    static unsigned char sw[2];

    h = s[0];
    l = s[1];

    h = h + h - ((h <=0x9f) ? 0x00e1 : 0x0161);
    if( l<0x9f )
      l = l - ((l > 0x7f) ? 0x20 : 0x1f);
    else{
        l = l-0x7e;
        ++h;
    }
    sw[0] = h | 0x80;
    sw[1] = l | 0x80;
    return sw;
}
/*
**      translate string to internal kcode
*/
const char *
str2ic(s)
     const char *s;
{
    static unsigned char buf[1024];
    const unsigned char *up;
    unsigned char *p;
#ifndef POSIX_ICONV
    unsigned char *pp;
#endif

    if(!s)
      return s;

    buf[0] = '\0';

    if( IC==input_kcode ){
        strcpy((char *)buf, s);
        return (char *)buf;
    }

    p = buf;
#ifdef POSIX_ICONV
    if (input_dsc) {
        size_t src_len, dst_len;
        up = (unsigned char *)s;
        src_len = strlen(s);
        dst_len = sizeof(buf);
        if (iconv(input_dsc, (char**)&up, &src_len,
                (char**)&p, &dst_len) == (size_t)-1)
            goto noconvert;
    }
#else
    if( IC==EUC && input_kcode == SJIS ){
        while(*s){
            up = (unsigned char *)s;
            if(is_kanji(*up)){
                pp = sj2e((unsigned char *)s);
                *(p++) = pp[0];
                *(p++) = pp[1];
                s += 2;
            }
            else
              *(p++) = (unsigned char)*(s++);
        }
    }
#endif
    else{
#ifdef POSIX_ICONV
noconvert:
#endif
        strcpy((char *)buf, s);
        return (char *)buf;
    }

    *(p++) = '\0';
    return (char *)buf;
}

/*
**      translate string to output kcode
*/
const char *
ic2str(s)
     const char *s;
{
    static unsigned char buf[1024];
    const unsigned char *up;
    unsigned char *p;
#ifndef POSIX_ICONV
    unsigned char *pp;
#endif

    if(!s)
      return s;

    buf[0] = '\0';

    p = buf;
#ifdef POSIX_ICONV
    if(output_dsc){
        size_t src_len, dst_len;
        up = (unsigned char *)s;
        src_len = strlen(s);
        dst_len = sizeof(buf);
        if(iconv(output_dsc, (char**)&up, &src_len,
                (char**)&p, &dst_len) == (size_t)-1)
            goto noconvert;
    }
#else
    if( IC==EUC && output_kcode == SJIS ){
        while(*s){
            up = (unsigned char *)s;
            if( *up & 0x80 ){
                pp = e2sj((unsigned char *)s);
                *(p++) = pp[0];
                *(p++) = pp[1];
                s += 2;
            }
            else
              *(p++) = (unsigned char)*(s++);
        }
    }
#endif
    else{
#ifdef POSIX_ICONV
noconvert:
#endif
        strcpy((char *)buf, s);
        return (char *)buf;
    }

    *(p++) = '\0';
    return (char *)buf;
}

/*
**      primitive function
*/

/* print out 1 byte character to tty (no conversion) */
static void
tty_cputc(unsigned int c)
{
#if defined(NO_TERMS) && (defined(MSDOS) || defined(WIN32CON))
    xputc(c);
#else
    putchar(c);
#endif
} 

/* print out 2 bytes character to tty (no conversion) */
static void
tty_cputc2(unsigned int c, unsigned int c2)
{
#if defined(NO_TERMS) && (defined(MSDOS) || defined(WIN32CON))
    xputc2(c, c2);
#else
    putchar(c);
    putchar(c2);
#endif
} 

/* print out 1 byte character to tty (IC->output_kcode) */
static void
tty_jputc(unsigned int c)
{
#if defined(NO_TERMS) && (defined(MSDOS) || defined(WIN32CON))
    xputc(c);
#else
    putchar(c);
#endif
}

/* print out 2 bytes character to tty (IC->output_kcode) */
static void
tty_jputc2(unsigned int c, unsigned int c2)
{
#if defined(NO_TERMS) && (defined(MSDOS) || defined(WIN32CON))
    xputc2(c, c2);
#else
    putchar(c);
    putchar(c2);
#endif
}

/*
 *  2oCgobt@OȂo͂
 *  R[hϊs
 */
int
jbuffer(
     unsigned int c,
     unsigned int *buf,
     void (*f1)(unsigned int),
     void (*f2)(unsigned int, unsigned int))
{
    static unsigned int ibuf[2];
    unsigned int c1, c2;
#ifndef POSIX_ICONV
    unsigned char uc[2];
    unsigned char *p;
#endif

    if(!buf) buf = ibuf;
    if(!f1) f1 = tty_jputc;
    if(!f2) f2 = tty_jputc2;

    c = c & 0xff;

    if(!(buf[0]) && (is_kanji(c))){
        buf[1] = c;
        ++buf[0];
        return 0;
    }
    else if(buf[0]){
        c1 = buf[1];
        c2 = c;

        if(IC == output_kcode)
#ifdef POSIX_ICONV
            f2(c1, c2);
        else if (output_dsc) {
            char buf_in[2], buf_out[16];
            char *src = buf_in, *dst=buf_out;
            size_t src_len = 2, dst_len = sizeof(buf_out);
            *buf_in = c1;
            *(buf_in + 1) = c2;
            if (iconv(output_dsc, &src,
                      &src_len, &dst, &dst_len) == (size_t)-1) {
                f2(c1, c2);
            } else {
                *dst = '\0';
                dst = buf_out;
                while(*dst) f1(*(dst++));
            }
        }
#else
          ;
        else if(IC == EUC){
            switch(output_kcode){
              case SJIS:
                uc[0] = c1;
                uc[1] = c2;
                p = e2sj(uc);
                c1 = p[0];
                c2 = p[1];
                break;
              default:
                impossible("Unknown kcode!");
                break;
            }
        }
        else if(IC == SJIS){
            uc[0] = c1;
            uc[1] = c2;
            p = sj2e(uc);
            switch(output_kcode){
              case EUC:
                break;
              default:
                impossible("Unknown kcode!");
                break;
            }
        }
        f2(c1, c2);
#endif
        buf[0] = 0;
        return 2;
    }
    else if(c){
        f1(c);
        return 1;
    }
    return -1;
}

/*
 *  2oCgobt@OȂo͂
 *  R[hϊ͍sȂ
 */
int
cbuffer(
     unsigned int c,
     unsigned int *buf,
     void (*f1)(unsigned int),
     void (*f2)(unsigned int, unsigned int))
{
    static unsigned int ibuf[2];

    if(!buf) buf = ibuf;
    if(!f1) f1 = tty_cputc;
    if(!f2) f2 = tty_cputc2;

#ifdef POSIX_ICONV
    if (output_kcode == UTF8) {
        if (c) {
            f1(c);
            return 1;
        }
    }
    else
#endif
    if(!(buf[0]) && is_kanji(c)){
        buf[1] = c;
        ++buf[0];
        return 0;
    }
    else if(buf[0]){
        f2(buf[1], c);
        buf[0] = 0;
        return 2;
    }
    else if(c){
        f1(c);
        return 1;
    }
    return -1;
}

void
jputchar(int c)
{
    jbuffer((unsigned int)(c & 0xff), NULL, NULL, NULL);
}
void
cputchar(int c)
{
    cbuffer((unsigned int)(c & 0xff), NULL, NULL, NULL);
}

void
jputs(s)
     const char *s;
{
    while(*s)
      jputchar((unsigned char)*s++);
    jputchar('\n');
}

int
is_kanji2(s, pos)
     const char *s;
     int pos;
{
    unsigned char *str;

    str = (unsigned char *)s;
    while(*str && pos>0){
        if(is_kanji(*str)){
            str+=2;
            pos-=2;
        }
        else{
            ++str;
            --pos;
        }
    }
    if(pos<0)
      return 1;
    else
      return 0;
}

int
is_kanji1(s, pos)
     const char *s;
     int pos;
{
    unsigned char *str;

    str = (unsigned char *)s;
    while(*str && pos>0){
        if(is_kanji(*str)){
            str+=2;
            pos-=2;
        }
        else{
            ++str;
            --pos;
        }
    }
    if(!pos && is_kanji(*str))
      return 1;
    else
      return 0;
}

/*
 * ̐擪ʒu܂ŉoCg߂Kv邩vZ
 */
int
offset_in_kanji(s, pos)
     const unsigned char *s;
     int pos;
{
    static int mask[7] = {
        0,
        0xc0,
        0xe0,
        0xf0,
        0xf8,
        0xfc,
        0xfe,
    };
    if (output_kcode == UTF8) {
        int c = 1;
        int i;

        /* 擪Ȃ0 */
        if (pos == 0) {
            return 0;
        }
        
        pos--;
        /* O̕ASCII */
        if ((s[pos] & 0x80) == 0x00) {
            return 0;
        }

        for (i = pos; i >= 0; i--) {
            if ((s[i] & 0xc0) == 0xc0)
                break;
            c++;
        }

        if (s[i] < mask[c]) {
            return 0;
        } else {
            return c;
        }
    } else {
        return is_kanji2((char *)s, pos);
    }
}

/*
** 8rbgX[isspace()
*/
int
isspace_8(c)
     int c;
{
    unsigned int *up;

    up = (unsigned int *)&c;
    return *up<0x80 ? isspace(*up) : 0;
}
/*
** split string(str) including japanese before pos and return to
** str1, str2.
*/
void
split_japanese( str, str1, str2, pos )
     char *str;
     char *str1;
     char *str2;
     int pos;
{
    int len, i, j, k, mlen;
    char *pstr;
    char *pnstr;
    int sq_brac;

retry:
    len = strlen((char *)str);

    if( len < pos ){
        strcpy(str1,str);
        *str2 = '\0';
        return;
    }

    if(pos > 20)
      mlen = 20;
    else
      mlen = pos;

    i = pos;
    if(is_kanji2(str, i))
      --i;

/* 1:
** search space character
*/
    j = 0;
    while( j<mlen ){
        if(isspace_8(str[i-j])){
            --j;
            goto found;
        }
        else if(is_kanji1(str,i-j)){
            if(!strncmp(str+i-j,"@",2)){
                j -= 2;
                goto found;
            }
        }
        ++j;
    }

/* 3:
** search second bytes of japanese
*/
    j = 0;
    sq_brac = 0;
    while( j<mlen ){
        if(str[i-j] == '['){
            sq_brac = 0;
            goto found;
        }
        else if(str[i-j] == ']')
          sq_brac = 1;
        
        if(!sq_brac && is_kanji1(str,i-j)){
            goto found;
        }

        ++j;
    }
    if(pos > 2){
        pos -= 2;
        goto retry;
    }
found:

    if(pos > 2){
        if((str[i-j] == ']' ||
            str[i-j] == ')' ||
            str[i-j] == '}')){
            pos -= 2;
            goto retry;
        }
        else if(!strncmp(str+i-j, "n", 2) ||
                !strncmp(str+i-j, "j", 2) ||
                !strncmp(str+i-j, "p", 2)){
            pos -= 2;
            goto retry;
        }
        else if(!strncmp(str+i-j,"H",2) ||
                !strncmp(str+i-j,"A",2) ||
                !strncmp(str+i-j,"B",2) ||
                !strncmp(str+i-j,"C",2) ||
                !strncmp(str+i-j,"D",2)){
            pos -= 2;
            goto retry;
        }
    }

    if(i == j){
        /* _Ȃ */
        strcpy(str1, str);
        str2[0] = '\0';
        return;
    }

    pstr = str;

    pnstr = str1;
    for( k=0 ; k<i-j ; ++k )
      *(pnstr++) = *(pstr++);
    *(pnstr++) = '\0';

    pnstr = str2;
    for( ; str[k] ; ++k )
      *(pnstr++) = *(pstr++);
    *(pnstr++) = '\0';
}

void 
jrndm_replace(c)
     char *c;
{
    unsigned char cc[3];

    if(IC==SJIS)
      memcpy(cc, (char *)sj2e((unsigned char *)c), 2);
    else
      memcpy(cc, c, 2);

    cc[0] &= 0x7f;
    cc[1] &= 0x7f;

    switch(cc[0]){
      case 0x21:
        cc[1] = rn2(94) + 0x21;
        break;
      case 0x23:
        if(cc[1] <= 0x39) /* O`X */
          cc[1] = rn2(10) + 0x30;
        else if(cc[1] <= 0x5A) /* ``y */
          cc[1] = rn2(26) + 0x41;
        else if(cc[2] <= 0x7A) /* ` */
          cc[1] = rn2(26) + 0x61;
        break;
      case 0x24:
      case 0x25:
        cc[1] = rn2(83) + 0x21; /* ` or A` */
        break;
      case 0x26:
        if(cc[1] <= 0x30)
          cc[1] = rn2(24) + 0x21; /* ` MV */
        else
          cc[1] = rn2(24) + 0x41; /* ` MV */
        break;
      case 0x27:
        if(cc[1] <= 0x40)
          cc[1] = rn2(33) + 0x21; /* @`` VA */
        else
          cc[1] = rn2(33) + 0x51; /* p` VA */
        break;
      case 0x4f:
        cc[1] = rn2(51) + 0x21; /* @` r */
        break;
      case 0x74:
        cc[1] = rn2(4) + 0x21; /*     4*/
        break;
      default:
        if(cc[0] >= 0x30 && cc[1] <= 0x74)
          cc[1] = rn2(94) + 0x21;
        break;
    }

    cc[0] |= 0x80;
    cc[1] |= 0x80;

    if(IC==SJIS)
      memcpy(c, (char *)e2sj(cc), 2);
    else
      memcpy(c, cc, 2);
}

/*
 * "put off"ΏۂɂēK؂ɘa󂷂
 */
const char *
joffmsg(otmp, joshi)
register struct obj *otmp;
const char **joshi;
{
    static char buf[BUFSZ];

    *joshi = "";

    if(otmp->oclass == RING_CLASS){
        Sprintf(buf, "%s͂", body_part(FINGER));
        return buf;
    }
    if( otmp->oclass == AMULET_CLASS){
        return "͂";
    }
    else if(is_helmet(otmp))
      return "";
    else if(is_gloves(otmp))
      return "͂";
    else if(otmp->oclass == WEAPON_CLASS||is_shield(otmp)){
        *joshi = "";
        return "";
    }
    else if(is_suit(otmp))
      return "E";
    else
      return "͂";
}

/*
 * "put on"ΏۂɂēK؂ɘa󂷂
 */
const char *
jonmsg(otmp, joshi)
register struct obj *otmp;
const char **joshi;
{
    static char buf[BUFSZ];

    *joshi = "";

    if(otmp->oclass == RING_CLASS){
        Sprintf(buf, "%sɂ͂߂", body_part(FINGER));
        return buf;
    }
    else if(otmp->oclass == AMULET_CLASS)
      return "gɂ";
    else if(is_gloves(otmp))
      return "gɂ";
    else if(is_shield(otmp)){
        *joshi = "";
        return "g";
    }
    else if(is_helmet(otmp))
      return "Ԃ";
    else if(otmp->oclass == WEAPON_CLASS){
        Sprintf(buf, "%sɂ", body_part(HAND));
        return buf;
    }
    else if(is_boots(otmp))
      return "";
    else if(is_suit(otmp))
      return "";
    else
      return "gɂ";
}

/*
 * Ԃ
 */
const char *
numeral(obj)
register struct obj *obj;
{
    switch(obj->oclass){
      case WEAPON_CLASS:
      case WAND_CLASS:
        if(obj->otyp==SHURIKEN)
          return "";
        else
          return "{";
      case ARMOR_CLASS:
        return "";
      case GEM_CLASS:
      case ROCK_CLASS:
      case BALL_CLASS:
        return "";
      case SCROLL_CLASS:
        return "";
      case SPBOOK_CLASS:
        return "";
      case POTION_CLASS:
      case RING_CLASS:
      case AMULET_CLASS:
      case FOOD_CLASS:
        if(obj->quan < 10L)
          return "";
        else
          return "";
      case COIN_CLASS:
        return "";
      default:
        switch(obj->otyp){
          case CREDIT_CARD:
          case TOWEL:
          case BLINDFOLD:
            return "";
          case SKELETON_KEY:
          case TALLOW_CANDLE:
          case WAX_CANDLE:
          case PICK_AXE:
          case UNICORN_HORN:
          case LEASH:
          case STETHOSCOPE:
          case MAGIC_MARKER:
            return "{";
          case CRYSTAL_BALL:
            return "";
            break;
          default:
            if(obj->quan < 10L)
              return "";
            else
              return "";
        }
    }
}

/*-------------------------------------------------------------------------
        Sp̂p^[
  -------------------------------------------------------------------------*/
static const char *ro0 =
        "                        E  ECD  D                E      "
        "            RJ    TJ    OG}  ^b()|]|]            "
        "]  b      dEE          e  f   (  )            [  ]    {  "
        "}           q  r          u  v  imjn|    {|^_|F"
        "|  ^<    >  `EdB  {          Kbxrb"
        "bb    K^//    ~  b          ()b    <>    R"
        "  ȁځ  ɁP  ~F|->|<-bȁbɁ                  "
        "                            ߁߁(    )  i||    "
        "                            O/  /|  <>ɁR        "
        "                                    ^QbQ            "
        "<<>>/   ``dE                                  K`"
        "K^||bI|)s{a                  ()b                "
        "                                            ()bLb܁ځ܁ځ{"
        "u܂P /R                            ȂdR( ܂k )"
        "e߁ub  ||    |<  |_  mu||ub()eno  ܁Pb"
        "i /_VVm^_ɁbP                              "
        "  |  bEE|<    ||||    R  {|"
        "||A  vv^><m|                    ߂ǁRRMM"
        "]  񃉂  ̂JR  L  <     |+  j[  {  "
        "l     iX  t+  Ăy  i  T      [  [ ("
        "      i  b߂̂J)m悯    ĉ    n  "
        "  ^_    ͂܂    L7+݂()L{{    "
        "Bイm  IVHI̂ƃi/v`            "
        "                                C|tmmm  ~ۂs"
        "{    iJ  L  mtN  mNP  j ]R  i++T  JV  "
        "tlX  Z  Rm\  N^  i`  JmJmc  e  bR"
        "g  {m|Qts    mRn  n  kq  q  Pmt  t  ^_"
        "w  w  z  z  RtS߁ځRmR{    ۃRۃR߃R"
        "tmmmRtutLGAGtRmE  iNm        "
        "                        ȂdRbPځȂekV||n|b  |<  "
        "/ _um||ȃj߂b()ec>_ځPbɁb^_b{          "
        "                            |3B                                 "
        "                                                                "
        "                                                                "
        "                                                                "
        "                ȄCdRbPL  ekE  K  ܂||uI  |<  "
        "J|Pmu||b()e( ܁PbmO^_kۂb||ۄWY]  "
        "]||b-)  |O                                            "
        "                  ( ]s|      |u  {                  "
        "              s                                        "
        "                                                              "
        "        |                                  "
        "                                "
        "    ";

static const char *ro1 =
        "  \B ߎS          T      "
        "    Qؐhʓyly  ~ǏSH䎅{    I"
        "l~    剹v،l    C    |"
        "я        ܊      c  c  谎c  "
        "      M  #       L    |  mqv    ы"
        "    ダXdň  L  受|bHp  X  A    ]"
        "}    ~H  qʘ  H    풹K    谂dΚD  "
        "  L    ֏Vʊ  ܃剤  񃀌R  J]C  H"
        "  {CLicR  v    i  g  p  TMqŃqŉ  s"
        "i[  tMnڏZՑZ    c  ͙\"
        "~MǐS        ΋ΐk  泖LF    "
        "  ǒM    Vj  vđeSb    指"
        "tHyH    撹Rc            T  y"
        "Ăy  ňSM}  sgCq      \d͌"
        "gS珗  SѓlRc؉ؚFl  D  ؐщ"
        "              ʉ  L  D  J  䔰"
        "  I  Rbl𒎉L谐Hn/|։]p    S  "
        "  M񏬚햔y  D    OȝSl  Z  ʉΓc"
        "䔒      F    毙{͗[g  g  ݍ  "
        "ΐ|Ms\母嗢  \yj؎sb`  q  "
        "sL  o  eA  t  谐    p    Ԍ  e  "
        "  uR؋qřŌT      cj  HPQ    "
        "            ԊQ  \    v؎锒XT  "
        "Y    bؕS    |ă  H  Ck  "
        "r  谗͌    r    ~  s\슱        "
        "      j    谌  v    y      bM    "
        "      RHMĊ  谌cL          "
        "  Hq  ʊ  iڍRΏK  谌"
        "FŌŎ~  x    g  呴      RxzȐSR  "
        "ؙ{ڑؑ      泟tVl҉ї  "
        "H  vȊL  ȎԋRH{nZ\  `  x  "
        "x  ^  `  q}lr`  X`      vm  _  "
        "؋g  gΐ،߉  Ŋe  p  gd    NXCCؔT  "
        "y  C  RS؂T  y    vΉg|y  y  \"
        "b߃LyÙd      g  ߑ  䎦"
        "Nc    v    YZ  ́K\ی"
        "  v  ||j  }Sv  F  Z    v    ]́"
        "      틽HhnCŋ^  YĖ؁c  "
        "  Đ  俗͓yc\ʂsҌюl]  b  "
        "  l  俋  Sʌi L      vÓ"
        "gnnunڒU[  gH  H[  [  [  [  L  "
        "ћo  vԌ\\    vĐ؎i؋H  "
        "Ed  Nr  N  \g߃l  n    Y˝y\"
        "Y    OcevR  }  ㎩  谔T坼U\  v  c\ގ|"
        "  ĎԎT  yY    \|hԖŕv]    "
        "g泑Ԏ蔒s  CؐlPr_  ~gM    "
        "Cg            byMS  n    "
        "        쌢  ΌڎˌڌZbLԊ  "
        "  ŔnbYeʔʊt|    􉤌MO"
        "F  畽  mÏÎqZR  ԋ|ZPÝ،ß"
        "Z  SӚ  tÌ  gX𑫚Ì谌ٕœx  _    "
        "P܌牵]،  ӑΌጾǓьӉ  "
        "Z  K  YnH͌̓ʎqRcʐΌ"
        "yy@qqqy~sۃGȓyh讋|jr"
        "      G      X  i  VHVVVJVi"
        "cې΍XэčNHΈyT~󍂌M"
        "䊐esiHLLiB эFL΍L升烐Hŉѓ"
        "]l|͌֌  l  V  ԁÌZ  gʉюE"
        "э  Vv]ܐS  D    ~qŃw"
        "y\涍涍؍؍  Óq]SjC"
        "    RiG㳍؊VΏLߓy    "
        "  f    gh  ќcI  ؜~ʐ  ݉Ήуc"
        "ΙΗ[Ďc  ߜԜB Ր  y  ؍˔  Lˉ    "
        "  E  _  LVR  ΊHC      "
        "R    ̖؏\  qM  Ѝ  ݗ[  @  ؂џt"
        "Y  谔ʉ||ߘ  cێQ  U  ̝"
        "؜ΕĉFڜOZV^  vLшH  ԋҎaeYmCq"
        "i  CR  =]  䒆iC\ۏ䏗sꛙ  ЎmS"
        "cS|  mqҕq؎x  eq[+EEt  l~у􏬎"
        "x|pjiuLLՍ谐Hi~Ę"
        "    X~qyS      J  ʛΎ"
        "c  Slڎ  hV[HT}Ɠc|ԗRZJ{"
        "Kۖv  ѐSVo  ҊLv      ώvؔV    "
        "c  ^  gɏEԝ]l  lyyE˘j쏎  "
        "  C̙  ؎ىΎٍюىюڋՉE  f  |HS  "
        "  pL  e  яdd  S    Z  O  "
        "؋jJl  g    ސccHS  B  єTщΎ~"
        "l  HO  S      谌AԎ    яBؓыSC\"
        "d  {eD  ؙq\~><V]痢[ze    f  "
        "lZh  x  x  yRs{{    Rvw  C  n  "
        "  参  e{  y[  @  ԛ    ыŚ{lˋ"
        "ҏ    Ҏ        Ҋ͗]m\]  @S"
        "]        ҃`O  g  |嗧  "
        "  n    y؛Ə͜c    葖    "
        "i،؏і؏͖؏ŏ      ΋    ڐΏѐΏ"
        "lrћ  ď      sdߌr"
        "  ыd  vфۃi  k{      "
        "  ݐ  J  ؖ؏䑈    c  葮  "
        "  HЎ    e冎ƎƃNbp  HC㒇\  "
        "CCZ    n^  C  hғ  с`  ؗRbc"
        "l\  \  S  V  hcI  \JCmփC"
        "    mC  b  \    vni  cœѓᚘ"
        "ʐ  \    tΌڐęH    љ谋"
        "L  LR@Rĝ؋摖谋  ؜c؏  ŏ谋  "
        "o  K  {  cvD  s  ۗ͏      "
        "؍Ȗؐ~  MĐmߎl  ܌"
        "  ѐ  ьZ谖Џ̐{    ؋Ҍ  "
        "  |̎ӕvLyӐΕŎo    ><  t"
        "|ZJFpÒPCRY\Cj  Rml|PˉH"
        "F  ؑSؒO        O    D|O򎅜r"
        "[M  SLF    lrl|rga  "
        "^lPPPėЏ]yR  cѕD  D[D"
        "Α^lъĊyьˊ    P  ]  q    Ǝ捇"
        "  m        ]ЕSSR  \a  iؒUȓ"
        "Pʑq      ΕiNRؖڌSđ@  s  O"
        "q  iؑsߓyDV    qJna  ]  ]      L]  "
        "    L  ŎS  R  D  J  ZL"
        "lRÊO  iqqnU  ؐ    [c      L  "
        "    ؛M؍  nnPC{yyRЎ  "
        "S\Fٜvv      ߑL      ㍕"
        "CTli|ѐŊ咹    葁  ~        "
        "y泌V  Ꭸ    ьnŐmUK    谐Z"
        "ȋR  ؕw  ؑ谜O  VvyURUؓ  "
        "v  r  RD|P莨  UDi  |P"
        "ċғ৘  dtc͌  y|玨Sm  m谒  "
        "mrn}،cPTT|}{    юzrڃL\"
        "CR  SR  ڒU؎    ߌѐn؉J  "
        "    L  ꘫXk    АВ||ce  "
        "  \쌎  АcڒS䒎"
        "d  ŉG͕    VV񉤜cCL^      谖ؒ"
        ";g    TT    ؕꍑJ؋Kc    ؐΏ\爒  ؏t"
        "M  y⏗RCЋ谒        L"
        "  D  ЛMp          VΒlђ"
        "M쌾鑫h    ё        "
        "|R    ܌U  U  ԎU  ȎMy^    "
        "o  ÊLԉ]^Ő]ta  oJbZ  y  ]  "
        "ґ  S  ؓyx  ᠓  Lҗ]    xΎΖz͛\"
        "zS  Z    mt      咷    R  t  "
        "          Mʙ    ΒΓo        "
        "  ē[  d      _  OB  G  o  "
        "n      d͙c  y        e    "
        "㉺  eS    fډљXn      ؖ  []"
        "i  RT      Ó  R        ԕ"
        "E_  ԑ厦NclP~          DU  n쎅T"
        "cc  Ԍv谏    qT  qC    H++    "
        "tm_  ppnSE      lS`c  "
        "߃SO  R  R  ĐA  Myѓy߃c_  qc"
        "_  ȒCi  b      b      bΔgn  b  "
        "dT    OI  LؕssMДږksԓьȈʌ^"
        "  ^  ^  L  `mL  TѕH  m  "
        "      P  ĔM    \    \    UVn੗F  "
        "    ΒJ          {Γccm_{    є"
        "镗Fi  F    唰㒹V          Lʖ"
        "Ж}ʉ  :e  }  Д)e  cq  Mt  La  ѕ"
        "ΕŕŐH    ΍ʎMʐΔ  c  ț  "
        "S˔    񕶏qK  xΔڕK    \  b    "
        "L  k]E  ʉp  ќe  c䛙ڗrv  ؓ~"
        "  Xd镔c    |SS      c  ؉b"
        "N  Hc\  ՜c[    [  [Z  涎ٛc  "
        "яc  c  Vѐ    pn  Ŗ"
        "󊢃nC  Vy  ciЛtz  v    ҕt  "
        "t  {gݕv  L  g  t      TC"
        "  \    H  ㌢ceГclcl畜  "
        "  䃀    C܋t          щΚS  ĕ"
        "ĈَJ厨    u    ǋЊ  ǜO  aH  "
        "y؁SL      ߐΖǖږΜG  G  G  "
        "    G  X  Ɨ͏ƃOv֋xɕᚘ  ~p\  "
        "ԕb  yT  O  䞃    ș؍Kv"
        "  RR      i        Ε  "
        "          毁    H  V    "
        "yЖ`SSS  ˕  SÕڎd  ^"
        "L    KŃRq      ؃g  ќc"
        "t  e  W    ԉH{  M薃ΖSї  "
        "؝x    ^  z  ؛LؐC  Vk    "
        "    s  CN            S    Rb"
        "~K~Kt    O  h  Ėю  ږ[  \IJ"
        "؋DǙkU[@M  qDNY"
        "؛厅  Sʔ      SqM  Sڎ㦖щ  "
        "M\ҍ؍H  H  ˑĐnL働SF]  n"
        "Z    뎨  \  V    et  y    谗K  "
        "    O  O  R    OԑO调CEJ  j  iL  R  "
        "S  EؗRE  T  U  Uc  lElJG  b    L"
        "魒^N}z  Tӌ  Z  \ŗf  ͏J  p      "
        "B    r  r  e  e  b㴉谐ؗe  T    "
        "  H  ~    J  J  HHْ            "
        "Jce  e  ъeサK    茞      bуj䛙"
        "؎q؉כ  lb谐y  j    ce"
        "  [    [  c[  ė琶b  C  gvTKgjt  "
        "  cR瘵  ēlS              U"
        "  i  ʗ  ѕđC  biԘӕđCđCC  "
        "    cđ卡        ͗    ߃l  ߊ"
        "J߉J  ߙʗљʗџe͗  ߛSđCA  Γ  "
        "  A      D  ΌˊLeeJH    YO"
        "ǌďY    J      yqꋗ    l"
        "    уsLĹ  ؙAݖU    "
        "    ^      ";

static const char *ro2 =
        "T  s  w胋          m_k          ɗ\"
        "  G][  Hz  {    񊎐l^^T  ʐlCgCJCC"
        "CnY              ~g            "
        "S  L  r  ؘ      le    T        "
        "      Z      wf          {    _    "
        "R\F  F          `  S              "
        "  K                jc@    _          "
        "  i              X  PYZ        J  O  T"
        "kك ]c  RccRf  ʙk\k  Pٖ      "
        "        Z        h  X  gؙ{C{c{kv"
        "n  `탊    ナ    󙃛          O"
        "        [  @  @@      h  h  ͏͋g"
        "v͛  ͏  Jΐ        c  M        mtܐnc"
        "ᙯ  qkbyM  Pki\OO\\  "
        "y|]  ^ŒJŊ  mPʜcʑʉę  ʎzʝƃ  Q  "
        "Z    Px    働҃E"
        "ڃi烍Zb{\zo"
        "t\ڑV  s"
        "        X  F      M          调::  "
        "  P  O    b    q  e        H    @  P    "
        "  `  h    G  L      V                "
        "  A      c  pz              i؍  J  "
        "@              d            G      Y        "
        "  RߚLᚘK蚘Ꚙ  ޚ"
        "yyyRyyғy~ysyØ  yyytyyyys"
        "yy㉺yypyyyyyԓyyyyDyy"
        "yGyΓyCPtyyyayyԓyyy  yy  "
        "i  }  b    A  i  n        s    D  q  "
        "v^      eї[ʑv    \T  l\"
        "s  U        ُ  C  䏗UꏗΗr"
        "OC        jp        я"
        "  M  i    Pxn          [        d    "
        "jj  @            Tqq    zq  ۛq"
        "  w  q    ~q~bC                      Q  "
        "  HcPو  L  L      ƐPr  㛗c  "
        "䛙  ěL      RiRRRy    RR"
        "RRRRRRRَR荇  R      RXRђ  RJ{  "
        "                R    Ìi  [    S  "
        "      a      o  `  @      ^  o        ^  "
        "      <<  H  @bЃЋЎzД    谋ЉВ"
        "ЖК  TГƋЕ++TTO  ʛr"
        "ucznᢛgḛ谛      "
        "MRi\O  O                |  z||  F  e    |"
        "|    _ʃm  gcC  eeeeeeeݜe{"
        "e  k    e  eceʜe        N      S  "
        "        SU          i  \uCSe  xS"
        "  ́    S    {      rS          "
        "        W      SG  b              "
        "  M      c  tSH              u  "
        "C    S    S    q    O  I    N  Q  a  a  "
        "]  K            [  f        [  i    d    "
        "n  {    K            i        g      "
        "                T              r"
        "谜    Y    ˍLLLLHL          "
        "\    l      s  c    \  z荇      q    "
        "t      ͗  e  g    H  \            "
        "Pxy          虯              F  "
        "      _  P  `  [ʊi      q      H  "
        "        Hco  z                  K  "
        "B      ~  ooL        o      A  A    "
        "  y    ^        P    Cc𝼕  "
        "]]Н  @ǎplrl΋Ғf  Os"
        "ѕ[ԕԃ`    Vؓi  "
        "K܈  W    C      R  "
        "p  R        U  i      Ǔ        A  L  "
        "`    d\oǑ      Jz  sTؖؓؗ"
        "ؓ؊،Ȗ؍H؜T؎Rؖ؉  ،ؗ\؏ؕؕ،˖؏"
        "؜OU؉؉  ؙؑ؋؊ؓؐ˖؎؏oؕؕtؗ"
        "؉Zؑ\؊؋؉HؑCؚ؍l؎؏[ؑؖؓ؊p؝ۖؒ"
        "؍؈؛؝ؓߖؚؒؕ؛tі}ؘCщؘ؎،N؋"
        "ؑ؎ٞR  ؍،؉؋  ؍؜؏@؈؏f؏؞R  "
        "ؒmؑ؏ؔOؕ{ؕؖؓT؎ؘӖؔT؊F،Ӗ؏H؎،_"
        "ؐ؎Җؐrؒ泗іؙؖ`ؕ؞j؎ؐ؎Mh  ؋S"
        "؍؍v؍؍؊؍ؓ؉Hؔʖٖؔ؉Ζؐؖ؝Ӗؗeؗ"
        "ؗ؊sy  ؜c،\  ؎Hpl؎a؏]ؙؐؐʖؑؐё"
        "ؖؖ؉iؘK؊؊і؊؍Ŗ؏؟}@ؓoؓꟖؔؒ"
        "،؈Ӗ،hؙؕ@ؚ焖؟@؊ϖؙ谖ؔJؕo֖ؖ؉ؕi"
        "؞ٖ؊y胖ؗ؉d؝  ؉J،ؗʜc؊ጇ󌇓  "
        "    PH@o谌t[eeece"
        "eteeeeeedeoeTeT{t  ttN["
        "іы    щјіSčHR  y    "
        "    ~  S  s                O    l    "
        "        c  R          s          "
        "{  @        G      ܟ      y  V"
        "Đ씟                  ͜  @    yX"
        "  y    y      c  R  UH        q  "
        "  u    `  qh    @MW    J    q    "
        "P  ד  YꋃӖ        ܗW      Q    "
        "    j  b      z~  ~  q        U    "
        "  d            iK    L                "
        "J  o  l    o  R    \  {                  "
        "  ]\    펅[ΉΏΙcΙc΋ΓΕΕΈx  "
        "ΏΊe  Κ΍ΗΚ􋐖      ΍c৉ΝΏd"
        "Α΍vΈѝ  Ζ  ΜTꟉΓ։ΔԉΘΉΚʉΐΒB"
        "泏K谉Ξىḉ茕  ܔbF    ੊=|    ૚М"
        "ДውїS  `    N      b  "
        "  e            Z                "
        "        \    J  b  l              "
        "      e      Y        @  RR  K  ݖ`  "
        "`    S  Ln  ʋ      焋ʉ        ZhZ"
        "\犢ԊъSpHЙꊢ    "
        "ÍX}pccccccvcccccߓccecc]"
        "cc  cc  c    vRޛÛ"
        "U~ΛSrݛuۛϛś"
        "ږ    `Msқqʛ͛ݛț"
        "ᢛaݛ^{谛  ϛN谛^    ᠓V᠟t"
        "qX唒𔒊͔比RR䍔瘰MM"
        "rMMMMMgMMMڙaڛڏ  ڌړ|"
        "ڍږvږږڒڈڙZڑڐڔږ  ڍKڎҖڐ    "
        "    ڊڋMڈ谖ڙ@ڌۖڌږؐǒi冖"
        "΍HΐؐΔΜAΊݐΌ\ΑΉԔgΐΕΗΜ_ΓΒΐr"
        "Λݐ毐΍΍ΑCΓWΔnΝӐΕiΐΐӐΐꟐΓoΉ΋`"
        "c΋^ΗѐΞَioᎦKT__򎦊"
        "ĎdL  b烈єщы  іѝۉящёђ"
        "ющ񎦉љfщPэшїeьbэ҉њэΉы}ыT|"
        "䌊c^ND  ʌȌľ"
        "\痧ї藧  Sᗧ՗Њ      b  "
        "          {        O{    S      C  "
        "            Y                    "
        "    c    Ì    i  t        I        "
        "  L          M          @      "
        "                  ď\ĔČˉčĕĎ"
        "ēĕSčX^ĕėėǕđď@ĉԕď_ĐrĉƕėpĖĔ  "
        "ĎݕĎG  ]䎅cLD䎅"
        "s  ^@g󎅛ݎGÎCE"
        "ˎюV쎅ʛnӎߎ"
        "iv  ЎvnɎc  JWǎ  c"
        "֎[KݎA_厅U꟎ԎK\  "
        "pĎĎohČŎ^̎niŎdƎ"
        "ŎʍHʚ  ʊL  ʓ܊谙c  cSs        "
        "        F    vnvrYr  NrrߗrNr  r  "
        "rrrr  rxHHHHrHHOHGHHꟉHԔ"
        "V|VјVіΉ]ΔbΘCΏΐJΎ~Kߎ㎨R  "
        "m莨O]    J  Lq䒷쌎HSy"
        "ތbDጎRi݌C"
        "ጎ錎Tڌ񌎕{󌎌v݌kM魌"
        "ӌ݌ጎ䌎vԌ\a"
        "匎@ČlḌb莺  `PlPOPt"
        "PjPZP谍㎁qMMMMRMtMqMՏMMM"
        "M`MM֏MDMḏMˊvLF  ++  S    t      "
        "      f    ~        t      K        "
        "    e      {  @                  "
        "              ]e    E  G          "
        "  f              g    䙖A    Ok  "
        "    dR        R      i    b    [    "
        "{  D    M    䞋  V  C  J  `        "
        "  h      Q  `        Hc  w      q  S  "
        "~  P    e\  jc      h  K    a    "
        "  ΐ̐      e  [|  t  y  D  ыp    "
        "    N  {    ㎵Ҏsgg  Rl"
        "LÒuÒRߒݒ񒎐㒎}𒎎钎V"
        "C[t]TՒ"
        "񒎓cݒDȒՒӒci璎G`q䒎"
        "u  Y~풎谒͒K"
        "iԒ噌cz  hǒt峎M  "
        "    Nnsss  s调lc  ߈  llpєC"
        "lllclUllll_߃l\l㉺lsߙg"
        "lNߗ  ݌TlՔ߃lڃlЃlȃlRlGlۈ    lJl"
        "lՃl\  ߃lKlKldlPlc焈߃llălgl̃lel"
        "lk  GGGn  茩ތҌ`毌i  o    "
        "谊pZppprpp册gH匾"
        "䌾  D댾@{錾ΌT"
        "ьތौP茾jF茾s錾RG"
        "kFvacKӌʌv֌"
        "਌􌾖o֌֌  i焌Kǒ谒N"
        "ƌ谌^JQJJ  RHs泌泎Ҝc  "
        "涍涏涊e涋x涕S涗涙Z涔涔涔LLLLLL"
        "LLR  L  L  LҊLLLLLL^LLӊL"
        "LᶊLĊLLLiL̐ԔԎҔV  Sёx~v"
        "Α瑫둫}摫ǑǑʑ"
        "m_d`E  qґPʑ]@"
        "LΑi冑焑đ摫Ao呫W"
        "g|g{gLgSgggԂԖ  ԎԉԜcԎԎԊe"
        "ԛݎԖԎԖƎVԖԗԗԕvԑtԓcԚ͎ԍԓWԐ"
        "ԐԎԋԊԎԞَԗḐh  hhh  h    c  "
        "  R  O      @                  Y    "
        "  yXh      Rc  U      `    i        "
        "  @      h                u      J  "
        "˔bt        d  đCђћџtъÓщѓішђѐ"
        "ёїѓюMшюMќc泓іL  їRэK  "
        "͋nًҋڋNË"
        "΋cΓBsSKGы䊋"
        "ŋVोFԋc`rv"
        "eah֋cAK"
        "Ɤԋ貋c  Aًl"
        "ˋꞋ谋^^ǋ  剺R啶吅b"
        "sʖ\勤C剗P剂刽劈LҖj  "
        "n  dB焐      ~    S  L    P    y"
        "          c  v    S  @    B  n      "
        "  db  谛调lW谉J谎谉JJщJJJZJ"
        "JJJщJpJJJUJ焉JĉJ漉Jy_߉JމJ_"
        "ʕʌʉ}v͊vvnvvvUvvvv\v}v"
        "vBvvHv_v䧊vpvBvJṽP  ꉹ"
        "٘ŌśݕŐbŋgŊܕœÕŉʕŕFŎvŘœŕpḕ谕"
        "ŉ䕗[[HԐHHHHÐHH]HP"
        "HFHHHHӐHeHSHHMHH֐HHMHHꟐHF"
        "Hj񈽍nqnnnlnjnnnznnnenB"
        "n۔nZnRnn谔nnGnǔnannQnn[nݔnn"
        "nKn@nnḔnbnn谔nn퍜teڍK再"
        "Lic镖镕镖镙f镏镚镕镕镋镋g镏镙镐{"
        "镈镕o镑l||++馎s馋香Z馐  qcSSϋSS㦋S"
        "S  S}ዛ΋~닛|xnы"
        "TՋZV񋛎QՋOcv"
        "Ut狛jЋMM  t_N͋["
        "싛\L{Ḓ{{咹ʒʒ훗"
        "Òߒ𒹍s䒹䒹pޒL"
        "̒ڒ钹R  H  ~㒹Ғ"
        "^ʒŒԒ  d谒}bbĎ  "
        "厭Ď    ˘Қmvvm  m  іt"
        "t׍o荕cg}̋    ꃕ  "
        "ꆝꆓۓ~VplR@    ꏃqꏋꏊꏏꏗꏍꏌ_"
        "ꏌꏑꏉZꏙT  Ӌ    ^ʃ^g  b";

/*
 * 蕶̕
 */
static int
kanji2index(c1, c2)
unsigned char c1, c2;
{
    if (IC == SJIS) {
        /* SJIS */
        if (c1 >= 0xe0) c1 -= 0x40;
        c1 -= 0x81;
        if (c2 >= 0x80) c2--;
        c2 -= 0x40;
        return ((int)c1 * (0xfc - 0x40 + 1 - 1)) + (int)c2;
    } else {
        /* EUC */
        return (((int)c1 & 0x7f) - 0x21) * (0x7e - 0x21 + 1) +
               (((int)c2 & 0x7f) - 0x21);
    }
}

int
jrubout(engr, nxt, use_rubout, select_rnd)
     char *engr;
     int nxt;
     int use_rubout;
     int select_rnd;
{
    int j;
    unsigned char *s;
    const unsigned char *p;

    if(is_kanji2(engr, nxt)){
        return 1;
      }

    s = (unsigned char *)&engr[nxt];
    if (*s == ' ') return 1;

    if(!is_kanji1(engr, nxt)) return 0;

    j = kanji2index(engr[nxt], engr[nxt + 1]);
    if (j >= 0x0000 && j <= 0x02B1)
        p = (unsigned char *)&ro0[j << 2];
    else if (j >= 0x0582 && j <= 0x1116)
        p = (unsigned char *)&ro1[(j - 0x0582) << 2];
    else if (j >= 0x1142 && j <= 0x1E7F)
        p = (unsigned char *)&ro2[(j - 0x1142) << 2];
    else p = (unsigned char *)"H";
    
    if (p[2] != ' ' || p[3] != ' ') p += select_rnd * 2;
    engr[nxt] = *p++;
    engr[nxt + 1] = *p;

    return 1;
}

static struct trans_verb trans_verb_list[] = {
    {"adjust",  "ǂ", "", ""},
    {"call",    "ǂ", "", "Ă"},
    {"charge",  "ǂ", "", "[U"},
    {"dip into", "ǂ", "", "Z"},
    {"dip",     "ǂ", "", "Z"},
    {"drink",   "ǂ", "", ""},
    {"drop",    "ǂ", "", "u"},
    {"eat",     "ǂ", "", "Hׂ"},
    {"grease",  "ǂ", "", "h"},
    {"identify", "ǂ", "", "ʂ"},
    {"invoke",  "", "", "͂g"},
    {"name",    "ǂ", "", "Â"},
    {"put on",  "ǂ", "", "gɂ"},
    {"read",    "ǂ", "", "ǂ"},
    {"ready",   "ǂ", "", "U"},
    {"remove",  "ǂ", "", "͂"},
    {"rub",     "ǂ", "", ""},
    {"rub on the stone", "ǂ", "", "΂ł"},
    {"sacrifice", "ǂ", "", ""},
    {"take off", "", "", ""},
    {"throw",   "ǂ", "", ""},
    {"tin",     "ǂ", "", "ʋl߂ɂ"},
    {"untrap with", "ǂ", "", "gĉ"},
    {"use or apply", "ǂ", "", "g"},
    {"wear",    "ǂ", "", "gɂ"},
    {"write on", "ǂ", "", ""},
    {"write with", "ǂ", "", "gď"},
    {"zap",     "ǂ", "", "U肩"},
    {(void *)0, (void *)0},
};

static struct trans_verb wield = 
    {"wield",   "ǂ", "", (void *)0}
;

static struct trans_verb dummyverb = 
    {(void *)0, "ǂ", "", (void *)0}
;

struct trans_verb
*trans_verb(const char *en){
    struct trans_verb *list = trans_verb_list;

    if(!strcmp(en, wield.en)){
        wield.jp = body_part(HANDED);
        return &wield;
    }
    

    while(list->en){
        if(!strcmp(en, list->en)){
            return list;
        }
        list++;
    }

    dummyverb.en = en;
    dummyverb.jp = en;

    return &dummyverb;;
}
