%-*- Prolog -*-
% Baxter-Sagart Transcription
%
% mc_concat(+MCI, +MCF, +MCT, -MC)
% mc_initial(+MC, -MCI).
% mc_final(+MC, -MCF).
% mc_tone(+MC, -MCT).
% trad_cat(+MC, -Cat) Traditional Category
% ph_cat(+Zi, -Cat)   Phonological Category
% rhyme_name(+Final, +ABCD, -Rhyme, -Division)

:- module(emclib, [mc_concat/4, mc_initial/2, mc_final/2, mc_tone/2,
		   trad_cat/2, ph_cat/2, char/2, char/6, rhyme_name/4]).
:- use_module(bsocr, [char/6]).
:- use_module(initials, [initial/3, initial/1, palatal/1]).
:- use_module(finals, [she/2, rhyme/5, final/4]).

% mc_concat(+MCI, +MCF, +MCT, -MC)
mc_concat(MCI, MCF, MCT, MC) :-
    atom_length_sub(MCI, ILengthSub),
    atom_length_sub(MCF, FLengthSub),
    sub_atom(MCI, 0, ILengthSub, _, Initial),
    sub_atom(MCF, 1, FLengthSub, _, Final),
    tone_transcription(MCT, ToneTrans),
    mc_triple_concat(Initial, Final, ToneTrans, MC).

% mc_triple_concat(+Initial, +Final, +ToneTrans, -MC).
%   Initials -y- + Final -j- => -y-
%   Initials -yh- + Final -j- => -yh- ?
mc_triple_concat(Initial, Final, ToneTrans, MC) :-
    (atom_suffix(Initial, 'y') ; atom_suffix(Initial, 'yh')),
    atom_prefix(Final, 'j'), !,
    atom_split(Final, 'j', Final1), !,
    atomic_list_concat([Initial, Final1, ToneTrans], MC).
mc_triple_concat(Initial, Final, ToneTrans, MC) :-
    atomic_list_concat([Initial, Final, ToneTrans], MC).

% tone_transcription(?ABCD, ?Trancription).
tone_transcription('A', '').
tone_transcription('B', 'X').
tone_transcription('C', 'H').
tone_transcription('D', '').

% tone_name(?ABCD, ?Name).
tone_name('A', '平').
tone_name('B', '上').
tone_name('C', '去').
tone_name('D', '入').

% mc_initial(+MC, -MCI).
mc_initial(MC, MCI) :-
    mc_initial_sub(MC, I),
    atom_concat(I, '-', MCI).
mc_initial_sub(MC, MCI) :-
    atom_length_sub(MC, MCLengthSub),
    sub_atom(MC, 0, MCLengthSub, _, MCSub),
    mc_initial_match(MCSub, MCI).
mc_initial_match('', _) :- !, fail.
mc_initial_match(MC, MC) :- initial(MC), !.
mc_initial_match(MC, MCI) :-
    mc_initial_sub(MC, MCI).

% mc_final(+MC, -MCF).
mc_final(MC, MCF) :-
    mc_final_sub(MC, Final),
    atom_concat('-', Final, MCF).    
mc_final_sub(MC, Final) :-
    mc_initial_sub(MC, MCI), !,
    mc_core_syllable(MC, Core, _),
    atom_split(Core, MCI, F),
    palatalized(MCI, F, Final).

% palatalized(+Initial, +F, -Final)
%   TSyF => TSy + jF
%     TSyiF => TSy + iF
%     TSywiF => TSy + iF
palatalized(_, Final, Final) :-
    (atom_prefix(Final, 'i'); atom_prefix(Final, 'wi')), !.
palatalized(Initial, F, Final) :-
    palatal(Initial), !,
    atom_concat('j', F, Final).
palatalized(_, Final, Final).

% mc_tone(+MC, -ABCD).
mc_tone(MC, T) :-
    mc_core_syllable(MC, _, T).
mc_core_syllable(MC, Core, 'B') :-
    atom_suffix(MC, 'X'), !,
    atom_length_sub(MC, Lsub),
    sub_atom(MC, 0, Lsub, _, Core).
mc_core_syllable(MC, Core, 'C') :-
    atom_suffix(MC, 'H'), !,
    atom_length_sub(MC, Lsub),
    sub_atom(MC, 0, Lsub, _, Core).
mc_core_syllable(MC, MC, 'D') :-
    (atom_suffix(MC, 'p') ; atom_suffix(MC, 't') ; atom_suffix(MC, 'k')), !.
mc_core_syllable(MC, MC, 'A') :- !.

% Phonological Category
% ph_cat(+Zi, -Cat)
ph_cat(Zi, Cat) :-
    char(Zi, MC),
    trad_cat(MC, Cat).
char(Zi, MC) :-
    char(Zi, _, MC, _, _, _).

% Traditional Category
% trad_cat(+MC, -Cat)
%   TODO: Constraints for divisions
trad_cat(MC, Cat) :-
    mc_initial_sub(MC, I),
    mc_final_sub(MC, F),
    mc_tone(MC, T),
    initial(I, IName, _),
    rhyme_name(F, T, FName, Div),
    tone_name(T, TName),
    kaihe(F, Kaihe),
    she_name(FName, She),
    atomic_list_concat([IName, FName, Kaihe, Div, TName, She], Cat).

% rhyme_name(+Final, +ABCD, -Rhyme, -Division)
rhyme_name(Final, 'A', Rhyme, Div) :-
    final_shu(Name, Div, Final),
    rhyme(Name, Rhyme, _, _, _).
rhyme_name(Final, 'B', Rhyme, Div) :-
    final_shu(Name, Div, Final),
    rhyme(Name, _, Rhyme, _, _).
rhyme_name(Final, 'C', Rhyme, Div) :-
    final_shu(Name, Div, Final),
    rhyme(Name, _, _, Rhyme, _).
rhyme_name(Final, 'D', Rhyme, Div) :-
    final_ru(Name, Div, Final),
    rhyme(Name, _, _, _, Rhyme).

% she_name(+FName, -She) :-
she_name(FName, She) :-
    she(She, FList),
    member(FName, FList).

% kaihe(+Final, -Kaihe)
kaihe(F, '閉') :-
    (atom_prefix(F, 'w') ; atom_prefix(F, 'jw')), !.
kaihe(_, '開').

% final_shu(+Name, ?Division, ?Final)
% final_ru(+Name, ?Division, ?Final)
final_shu(Name, Division, Final) :- /* 舒*/
    final(Name, Division, FinalList, _),
    member(Final, FinalList).
final_ru(Name, Division, Final) :- /* 入*/
    final(Name, Division, _, FinalList),
    member(Final, FinalList).

%--------------------
% Utility Predicates
%--------------------

% atom_length_sub(+Atom, -LP)
atom_length_sub(Atom, LP) :-
    atom_length(Atom, Length),
    LP is Length - 1.

% atom_prefix(+Atom, -Prefix) Duplicated
% atom_prefix(Atom, Prefix) :-
%    sub_atom(Atom, 0, _, _, Prefix).

% atom_split(+Atom, +Prefix, -Suffix)
atom_split(Atom, Prefix, Suffix) :-
    atom_length(Prefix, PLength),
    sub_atom(Atom, PLength, _, 0, Suffix).

% atom_suffix(+Atom, -Suffix)
atom_suffix(Atom, Suffix) :-
    sub_atom(Atom, _, _, 0, Suffix).