/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javax.sql;

import java.util.EventListener;

/**
 * An interface used to receive events generated by a {@link PooledConnection}.
 * <p>
 * This interface would typically be implemented by a component which manages a
 * connection pool (a connection pool manager). A connection triggers an event
 * to a {@code ConnectionEventListener} either when the application closes a
 * connection it has been using or when a significant error occurs while the
 * connection is being used.
 * <p>
 * The connection pool manager can return closed connections to the pool for
 * later reuse. Connections experiencing an error should be discarded.
 */
public interface ConnectionEventListener extends EventListener {

    /**
     * Notifies the {@code ConnectionEventListener} that an application has
     * called the {@code close} method on a pooled connection.
     * 
     * @param theEvent
     *            a {@code ConnectionEvent} containing details about the source
     *            of the event.
     */
    public void connectionClosed(ConnectionEvent theEvent);

    /**
     * Notifies the {@code ConnectionEventListener} that an error has occurred
     * on a {@code PooledConnection}. This notification is triggered <i>before</i> the
     * {@code SQLException}, which is available through the event argument, is
     * thrown.
     * 
     * @param theEvent
     *            a {@code ConnectionEvent} containing details about the source
     *            of the event and the {@code SQLException} that has occurred.
     */
    public void connectionErrorOccurred(ConnectionEvent theEvent);
}
