/*
 * samma
 *
 * Copyright (C) 2006,2007,2008 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: zipconv.c,v $
 * $Revision: 1.25 $
 * $Date: 2014/05/09 04:43:07 $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <gmime/gmime.h>
#include <dirent.h>
#include <pthread.h>
#include <libdgstr.h>
#include <fcntl.h>
#include <fcntl.h>

#include "mailzip_config.h"
#include "mailsave.h"
#include "zipconv.h"
#include "log.h"
#include "mailzip_db.h"
#include "maildrop.h"
#include "sendmail.h"
#include "global.h"
#include "mailzip_tmpl.h"

pthread_mutex_t gmime_lock = PTHREAD_MUTEX_INITIALIZER;

static void zip_child(char **, char *, char *, char *);

void
free_name_list(struct name_list *head)
{
    int i;
    for (i = 0; (head + i)->attach_name != NULL; i++) {
        if ((head + i)->attach_name != NULL) {
            free((head + i)->attach_name);
        }
    }
    if (head != NULL) {
        free(head);
    }
}

void
free_rcptinfo(struct rcptinfo *info)
{
    struct rcptinfo *p, *tmpp;

    if (info != NULL) {
        for (p = info; p != NULL; p = tmpp) {
	    tmpp = p->Next;
	    free(p->passwd);
	    free(p->keyword);
	    if (p->rcptlist != NULL) {
	        free_rcptlist(p->rcptlist);
	    }
	    free(p);
        }
    }
}

void
mailzip_clean(struct mailzip mz)
{
    /* free mailzip structure */
    if (mz.zipdir != NULL) {
        /* remove zip directory */
        remove_file_with_dir_recursive(mz.zipdir);
	free(mz.zipdir);
    }

    /* free attach name list */
    if (mz.namelist != NULL) {
	free_name_list(mz.namelist);
    }

    /* free gmime memory field */
    if (mz.message != NULL) {
        pthread_mutex_lock(&gmime_lock);
        g_object_unref (mz.message);
        pthread_mutex_unlock(&gmime_lock);
    }

    if (mz.date != NULL) {
//        free(mz.date);
        /* Repairing The display of the date 2014/05/08 (Thu) */
        g_free(mz.date);
    }

    if (mz.encfilepath != NULL) {
        free(mz.encfilepath);
    }
    if (mz.attachfilename != NULL) {
        free(mz.attachfilename);
    }
}
/*
 * delete_attachmetns_mail
 *
 * Functions:
 *     - Parse mail and get a list of the names of the all files 
 *       attached on the original mail, then delete the files.
 *     - Make a path string of the DeleteListName file.
 *     - Creat and write the DeleteListName file.
 *     - Attach the DeleteListName file to the copied mail.
 *     - Send the copied mail to the rcipients who are the target
 *       of the attachments-deletion.
 *     - Unlink the DeleteListName file.
 *     - Delete recipients of the original mail who are 
 *       the target of the attachments-deletion.
 * Args:
 *     See blow.
 * Returns:
 *     ZIP_CONVERT_ACCEPT
 *     ZIP_CONVERT_FAILED
 *     ZIP_CONVERT_INVALID_FILENAME
 *     ZIP_CONVERT_SUCCESS
 */
extern int
delete_attachments_mail(SMFICTX *ctx, struct mailinfo *minfo, 
                 struct config *cfg, char *from, struct rcptinfo *rdmpasslist)
{
    struct mailzip mz;
    struct rcptinfo *p;
    int i, ret;
    char *temp = NULL, *rcptaddr;

    memset(&mz, 0, sizeof(struct mailzip));

    /* parse mail */
    ret = parse_mail(minfo, cfg, &mz);
    if ((ret == PM_NO_MULTIPART) || (ret == PM_NO_ATTACHFILE)) {
	mailzip_clean(mz);
	return ZIP_CONVERT_ACCEPT;
    } else if (ret == PM_FAILED) {
	mailzip_clean(mz);
	return ZIP_CONVERT_FAILED;
    } else if (ret == PM_INVALID_FILENAME) {
	mailzip_clean(mz);
	return ZIP_CONVERT_INVALID_FILENAME;
    }

    /* Make a path of the DeleteListName file */
    if (mk_encpath(cfg, &mz) == -1) {
	mailzip_clean(mz);
	return ZIP_CONVERT_FAILED;
    }

    if (rdmpasslist != NULL) {
	/* deletion */
	if (mk_deletelistfile(cfg, &mz) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}

	/* add attach file */
	if (add_attach_file(&mz, rdmpasslist) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}
    	/* send mail */
	if (sendmail(&mz, cfg, rdmpasslist, from) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
        }
        /* unlink the temporary file */
        if (unlink(mz.encfilepath) == -1) {
            log(ERR_FILE_REMOVE, "delete_attachments_mail", unlink(mz.encfilepath));
            mailzip_clean(mz);
            return ZIP_CONVERT_FAILED;
        }

        for (i = 0; (rdmpasslist->rcptlist + i)->rcpt_addr != NULL; i++) {
            rcptaddr = (rdmpasslist->rcptlist + i)->rcpt_addr;
            /* delete from envelope to */
            if (smfi_delrcpt(ctx, rcptaddr) == MI_FAILURE) {
                log(ERR_DELETE_ADDRESS, "delete_attachment_mail", rcptaddr);
	        mailzip_clean(mz);
                return ZIP_CONVERT_FAILED;
            }
        }
    } 

    mailzip_clean(mz);
    return ZIP_CONVERT_SUCCESS;
}

extern int
zip_convert_mail(SMFICTX *ctx, struct mailinfo *minfo, struct config *cfg,
                 char *from, struct rcptinfo *passlist, struct rcptinfo *rdmpasslist)
{
    struct mailzip mz;
    struct rcptinfo *p;
    int i, ret, chk_replace, chk_replace_noconv;
    char *temp = NULL, *notify_message, *rcptaddr, *confirmation_message;
    char *org_p;

    memset(&mz, 0, sizeof(struct mailzip));

    /* parse mail */
    ret = parse_mail(minfo, cfg, &mz);
    if ((ret == PM_NO_MULTIPART) || (ret == PM_NO_ATTACHFILE)) {
	mailzip_clean(mz);
	return ZIP_CONVERT_ACCEPT;
    } else if (ret == PM_FAILED) {
	mailzip_clean(mz);
	return ZIP_CONVERT_FAILED;
    } else if (ret == PM_INVALID_FILENAME) {
	mailzip_clean(mz);
	return ZIP_CONVERT_INVALID_FILENAME;
    }

    if (mk_encpath(cfg, &mz) == -1) {
	mailzip_clean(mz);
	return ZIP_CONVERT_FAILED;
    }

    if (passlist != NULL) {
	for (p = passlist; p != NULL; p = p->Next) {
	    /* encryption */
	    if (convert_zip(cfg, &mz, &p) != 0) {
		mailzip_clean(mz);
		return ZIP_CONVERT_FAILED;
	    }
	    /* add attach file */
	    if (add_attach_file(&mz, p) != 0) {
		mailzip_clean(mz);
		return ZIP_CONVERT_FAILED;
	    }
    	    /* send mail */
	    if (sendmail(&mz, cfg, p, from) != 0) {
		mailzip_clean(mz);
		return ZIP_CONVERT_FAILED;
	    }
            /* send confirmation mail */
            if ((cfg->cf_fixedpassnotify[0] == 'Y' || cfg->cf_fixedpassnotify[0] == 'y')) {
                org_p = p->passwd;	/* save original password */
                p->passwd = cfg->cf_fixedpassnotifypass;
	        temp = tmpl_init();


	        chk_replace = tmpl_tag_replace(temp, mz, p, from, &confirmation_message);
            switch(chk_replace) {
                case ERROR:
                    p->passwd = org_p;	/* restore original password */
                    tmpl_release(temp);
                    mailzip_clean(mz);
                    return ZIP_CONVERT_FAILED;
                case CONVERT_ERROR:
                    // replace not converted string
                    chk_replace_noconv = tmpl_tag_replace_noconv(mz, p, from,
                                                                 &confirmation_message,
                                                                 cfg->cf_errmsgtemplatepath);

                    if (chk_replace_noconv == ERROR) {
                        tmpl_release(temp);
                        mailzip_clean(mz);
                        return ZIP_CONVERT_FAILED;
                    }
	        }
            p->passwd = org_p;	/* restore original password */
	        tmpl_release(temp);

                // set header field "References"
                if (strcmp(cfg->cf_references, "yes") == 0) {
                    ret = set_references(minfo->ii_mbuf, &confirmation_message);
                    if (ret == ERROR) {
                        free(confirmation_message);
                        mailzip_clean(mz);
                        return ZIP_CONVERT_FAILED;
                    }
                }

	        if (passwd_sendmail(cfg, from, confirmation_message) != 0) {
	            free(confirmation_message);
	            mailzip_clean(mz);
	            return ZIP_CONVERT_FAILED;
	        }
	        free(confirmation_message);
            }

	    if (unlink(mz.encfilepath) == -1) {
		log(ERR_FILE_REMOVE, "convert_zip", unlink(mz.encfilepath));
		mailzip_clean(mz);
		return ZIP_CONVERT_FAILED;
	    }

            for (i = 0; (p->rcptlist + i)->rcpt_addr != NULL; i++) {
                rcptaddr = (p->rcptlist + i)->rcpt_addr;
                /* delete from envelope to */
                if (smfi_delrcpt(ctx, rcptaddr) == MI_FAILURE) {
                    log(ERR_DELETE_ADDRESS, "zip_convert_mail", rcptaddr);
		    mailzip_clean(mz);
                    return ZIP_CONVERT_FAILED;
                }
            }
	}
    } 

    if (rdmpasslist != NULL) {
	/* encryption */
	if (convert_zip(cfg, &mz, &rdmpasslist) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}
	/* add attach file */
	if (add_attach_file(&mz, rdmpasslist) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}

    	/* send mail */
	if (sendmail(&mz, cfg, rdmpasslist, from) != 0) {
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}

	temp = tmpl_init();

    chk_replace = tmpl_tag_replace(temp, mz, rdmpasslist, from, &notify_message);
    switch(chk_replace) {
        case ERROR:
            tmpl_release(temp);
            mailzip_clean(mz);
            return ZIP_CONVERT_FAILED;
        case CONVERT_ERROR:
            // replace before convert string
            chk_replace_noconv = tmpl_tag_replace_noconv(mz, rdmpasslist, from,
                                                         &notify_message, cfg->cf_errmsgtemplatepath);

            if (chk_replace_noconv == ERROR) {
                tmpl_release(temp);
                mailzip_clean(mz);
                return ZIP_CONVERT_FAILED;
            }
    }
	tmpl_release(temp);

        // set header field "references"
        if (strcmp(cfg->cf_references, "yes") == 0) {
            ret = set_references(minfo->ii_mbuf, &notify_message);
            if (ret == ERROR) {
                free(notify_message);
                mailzip_clean(mz);
                return ZIP_CONVERT_FAILED;
            }
        }

	if (passwd_sendmail(cfg, from, notify_message) != 0) {
	    free(notify_message);
	    mailzip_clean(mz);
	    return ZIP_CONVERT_FAILED;
	}
	free(notify_message);
        for (i = 0; (rdmpasslist->rcptlist + i)->rcpt_addr != NULL; i++) {
            rcptaddr = (rdmpasslist->rcptlist + i)->rcpt_addr;
            /* delete from envelope to */
            if (smfi_delrcpt(ctx, rcptaddr) == MI_FAILURE) {
                log(ERR_DELETE_ADDRESS, "zip_convert_mail", rcptaddr);
	        mailzip_clean(mz);
                return ZIP_CONVERT_FAILED;
            }
        }
    } 

#ifdef	DEBUG
    if (passlist != NULL) {
	log("------------- passlist list start -------------");
        for (p = passlist; p != NULL; p = p->Next) {
	    log("keyword: %s", p->keyword);
	    log("keyword_len: %d", p->keyword_len);
	    log("passwd: %s", p->passwd);
	    for (i = 0; (p->rcptlist + i)->rcpt_addr != NULL; i++) {
	        log("rcptaddr: %s", (p->rcptlist + i)->rcpt_addr);
	    }
	    log("encfilepath: %s", mz.encfilepath);
        }
	log("-------------- passlist list end --------------");
    }
    if (rdmpasslist != NULL) {
	log("------------- rdmpasslist list start -------------");
        for (p = rdmpasslist; p != NULL; p = p->Next) {
	    log("keyword: %s", p->keyword);
	    log("keyword_len: %d", p->keyword_len);
	    log("passwd: %s", p->passwd);
	    for (i = 0; (p->rcptlist + i)->rcpt_addr != NULL; i++) {
	        log("rcptaddr: %s", (p->rcptlist + i)->rcpt_addr);
	    }
	    log("encfilepath: %s", mz.encfilepath);
        }
	log("-------------- rdmpasslist list end --------------");
    }
#endif /* DEBUG */

    mailzip_clean(mz);
    return ZIP_CONVERT_SUCCESS;
}

#define CONTENTTYPE_HEADER	"Content-Type"
#define CONTENTTYPE_ZIP		"application/zip"
#define CONTENTDISPOSITION	"attachment"
int
add_attach_file(struct mailzip *mz, struct rcptinfo *rcpt)
{
    GMimeObject *object, *att;
    GMimePart *part;
    GMimeDataWrapper *wrapper;
    GMimeStream *r_stream, *f_stream;
    GMimeFilter *filter;
    FILE *fp;
    int index, i;
    const char *filename;

    pthread_mutex_lock(&gmime_lock);
    object = g_mime_message_get_mime_part(mz->message);

    if (GMIME_IS_MULTIPART (object)) {

#ifndef GMIME26
        index = g_mime_multipart_get_number((GMimeMultipart *)object);
#else
        index = g_mime_multipart_get_count((GMimeMultipart *)object);
#endif

        for (i = index - 1; i > 0; i--) {
            att = g_mime_multipart_get_part((GMimeMultipart *)object, i);

	    if (GMIME_IS_MULTIPART (att)) {
#ifndef GMIME24
                g_object_unref (att);
#endif
                att = NULL;
                continue;
	    }

            filename = g_mime_part_get_filename((GMimePart *)att);

            if (filename == NULL) {
#ifndef GMIME24
                g_object_unref (att);
#endif
                att = NULL;
                continue;
            }

#ifndef GMIME26
            g_mime_multipart_remove_part((GMimeMultipart *)object, att);
#else
            g_mime_multipart_remove((GMimeMultipart *)object, att);
#endif

#ifndef GMIME24
            g_object_unref (att);
#endif
        }
    }

    fp = fopen(mz->encfilepath, "r");

    if (fp == NULL) {
	log(ERR_FILE_OPEN, "add_attach_file", mz->encfilepath);
#ifndef GMIME24
        g_object_unref (object);
#endif
        pthread_mutex_unlock(&gmime_lock);
	return -1;
    }

    /* create part object */
    part = g_mime_part_new();

    /* set part object */
#ifndef GMIME26
    g_mime_part_set_content_header(part, CONTENTTYPE_HEADER, CONTENTTYPE_ZIP);
#else
    g_mime_object_set_header((GMimeObject *)part, CONTENTTYPE_HEADER, CONTENTTYPE_ZIP);
#endif


#ifndef GMIME26
    g_mime_part_set_encoding(part, GMIME_PART_ENCODING_BASE64);
    g_mime_part_set_content_disposition(part, CONTENTDISPOSITION);
#else
    g_mime_part_set_content_encoding(part, GMIME_CONTENT_ENCODING_BASE64);
    g_mime_object_set_disposition((GMimeObject *)part, CONTENTDISPOSITION);
#endif

    g_mime_part_set_filename(part, mz->attachfilename);
    // もしここでメモリリークが見つかった場合、修正例
    // (文字コードのテストをしっかりすべき）
    //g_mime_object_set_content_type_parameter((GMimeObject *)part, "filename", mz->attachfilename);
    //g_mime_object_set_content_disposition_parameter((GMimeObject *)part, "filename", mz->attachfilename);

    /* open stream */
    r_stream = g_mime_stream_file_new(fp); 

    /* create filter object */
#ifndef GMIME26
    filter = g_mime_filter_basic_new_type(GMIME_FILTER_BASIC_BASE64_ENC);
#else
    filter = g_mime_filter_basic_new(GMIME_CONTENT_ENCODING_BASE64, TRUE);
#endif

    /* set filter */
#ifndef GMIME26
    f_stream = g_mime_stream_filter_new_with_stream(r_stream);
#else
    f_stream = g_mime_stream_filter_new(r_stream);
#endif

    g_object_unref (r_stream);
    g_mime_stream_filter_add((GMimeStreamFilter *)f_stream, filter);

    /* create wrapper object */
#ifndef GMIME26
    wrapper = g_mime_data_wrapper_new_with_stream(f_stream, GMIME_PART_ENCODING_BASE64);
#else
    wrapper = g_mime_data_wrapper_new_with_stream(f_stream, GMIME_CONTENT_ENCODING_BASE64);
#endif

    /* set part */
    g_mime_part_set_content_object(part, wrapper);

#ifndef GMIME26
    g_mime_multipart_add_part((GMimeMultipart *)object, (GMimeObject *)part);
#else
    g_mime_multipart_add((GMimeMultipart *)object, (GMimeObject *)part);
#endif

    g_object_unref (filter);
    g_object_unref (part);
    g_object_unref (wrapper);
    g_object_unref (f_stream);

#ifndef GMIME24 
    g_object_unref (object);
#endif

    pthread_mutex_unlock(&gmime_lock);
    return 0;
}

/*
 * mk_new_filename
 *
 * Args:
 *	char      *newfilename	new filename
 *	constchar *filename	attach file name
 *	int	   count	same name count 
 *
 * Returns:
 *	NEW_SUCCESS		Success
 */
int
mk_new_filename(char *newfilename, const char *filename, char *dir, int count)
{
    char *ext, *p;
    char data[PATH_MAX + 1];
    char fullpath[PATH_MAX + 1];
    char tmpfilename[FILE_NAME_MAX + 1];
    int name_len = 0, ext_len = 0;
    int len, tmp_num = 0;
    int filename_len = strlen(filename);

    len = filename_len + FILE_RENAME_LEN;
    memset(tmpfilename, 0, sizeof(tmpfilename));
    
    if (filename_len > FILE_NAME_MAX) {
        ext = strrchr(filename, DOT);
        if (ext == NULL) {
	    strncpy(tmpfilename, filename, FILE_NAME_MAX);
        } else {
            tmp_num = ext - filename + DOTLEN + EXTMAXLEN;
            if (tmp_num <= FILE_NAME_MAX) {
	        strncpy(tmpfilename, filename, FILE_NAME_MAX);
            } else {
		ext_len = strlen(ext + 1);
		if (ext_len > EXTMAXLEN) {
		    *(ext + EXTMAXLEN + 1) = '\0';
		    ext_len = EXTMAXLEN;
		}
                name_len = FILE_NAME_MAX - DOTLEN - ext_len;
                p = strncpy(data, filename, name_len);
                *(p + name_len) = '\0';

                /* create new filename */
                snprintf(tmpfilename, FILE_NAME_MAX + 1, "%s%s", data, ext);
            }
        }
    } else {
	strncpy(tmpfilename, filename, FILE_NAME_MAX);
    }

    if (count != 0) {
	/* extension check */
	ext = strrchr(tmpfilename, DOT);
	if (ext == NULL) {
            snprintf(newfilename, len, FILE_RENAME_NOEXT, tmpfilename, count);
        } else {

            name_len = ext - tmpfilename;

            p = strncpy(data, tmpfilename, name_len);
            *(p + name_len) = '\0';

            /* create new filename */
            snprintf(newfilename, len, FILE_RENAME, data, count, ext);
        }
    } else {
	strncpy(newfilename, tmpfilename, len);
    }

    /* create full pathname */
    snprintf(fullpath, PATH_MAX, FILE_PATH, dir, newfilename);

    if (access(fullpath, F_OK) == 0) {
	mk_new_filename(newfilename, tmpfilename, dir, count + 1);
    }

    return(NEW_SUCCESS);
}

/*
 * remove_file_with_dir
 * 
 * Args:
 * 	char *target		remove target file/directory name
 *
 * Returns:
 * 	RM_SUCCESS		Success
 * 	RM_FAILED		Failed
 */
int
remove_file_with_dir_recursive(char *target)
{
    DIR           *dp;
    struct stat    buf;
    struct dirent *p;
    char new_target[PATH_MAX + 1];

    if (stat(target, &buf) != 0) {

	/* stat failed */
        if (errno != ENOENT) {
            log("%s: Stat failed.(%s)","remove_file_with_dir_recursive", target);
            return(RM_FAILED);
        }
        return(RM_SUCCESS);
    }
    if ((buf.st_mode & S_IFMT) != S_IFDIR) {

	/* remove file */
        if (unlink(target) != 0) {
            if (errno != ENOENT) {
		log(ERR_FILE_REMOVE, "remove_file_with_dir_recursive", target);
                return(RM_FAILED);
            }
        }
        return(RM_SUCCESS);

    } else {

        dp = opendir(target);
        if (dp == NULL) {
            return(RM_FAILED);
        }
        while ((p = readdir(dp))) {

            if ((strcmp(p->d_name, ".") == 0) ||
                (strcmp(p->d_name, "..") == 0)) {
                continue;
            }

	    snprintf(new_target, PATH_MAX + 1, FILE_PATH, target, p->d_name);

            if (remove_file_with_dir_recursive(new_target) != RM_SUCCESS) {
		closedir(dp);
                return(RM_FAILED);
            }
        }
	closedir(dp);

	/* remove directory */
        if (rmdir(target) != 0) {
            if (errno != ENOENT) {
		log(ERR_DIRECTORY_REMOVE, "remove_file_with_dir_recursive", target);
                return(RM_FAILED);
            }
        }
    }

    return(RM_SUCCESS);
}

/*
 * push_attnamelist
 *
 * Args:
 *	struct name_list **head		attach namelist structure
 *	char		  *str		attach name
 *
 * Returns:
 *	0			Success
 *	-1			Failed
 */
int
push_attnamelist(struct name_list **head, char *str)
{
    int i;
    struct name_list *tmphead;

    /* check current size */
    if (*head == NULL) {
        i = 0;
    } else {
        for (i = 0; (*head + i)->attach_name != NULL; i++);
    }

    /* (re)allocate memory */
    tmphead = realloc(*head, sizeof(struct name_list) * (i + 2));
    if (tmphead == NULL) {
        log(ERR_MEMORY_ALLOCATE, "push_list", "head", strerror(errno));
        return -1;
    }
    *head = tmphead;

    /* copy string */
    (*head + i)->attach_name = strdup(str);
    if ((*head + i)->attach_name == NULL) {
        log(ERR_MEMORY_ALLOCATE, "push_list",
            "(*head + i)->attach_name", strerror(errno));
        return -1;
    }
    (*head + i)->attach_name_len = strlen(str);

    /* end with NULL */
    (*head + i + 1)->attach_name = NULL;

    return 0;
}


#define TEMPDIRNAME	"tmp"
#define TEMPDIRLEN	3
/*
 * parse_mail
 *
 * Args:
 *	struct mailinfo *minfo	mailinfo structure
 *	struct config  *cfg	config structure
 *	struct mailzip *mz	mailzip structure
 *
 * Returns:
 *	PM_SUCCESS		Success
 *	PM_NO_MULTIPART		Multi-part Nothing
 *	PM_FAILED		Failed
 */
int
parse_mail(struct mailinfo *minfo, struct config *cfg, struct mailzip *mz)
{
    GMimeStream *r_stream;
    GMimeParser *parser;
    GMimeObject *object;
    const GMimeContentType *ct_object = NULL;
    FILE *rfp;
    char *tmpdir, *tmpstr;
    int rfd, len;
    int ret;
    char *cfg_encdir = NULL;

    cfg_encdir = ismode_delete ? cfg->cf_tmpdir : cfg->cf_encryptiontmpdir;
    pthread_mutex_lock(&gmime_lock);

    /* create object */
    if(minfo->ii_status & MS_STATUS_FILE) {
	DEBUGLOG("Create message object from file");
        /* stream */
	if ((rfd = dup(minfo->ii_fd)) < 0) {
	    log(ERR_FILE_OPEN, "parse_mail", minfo->ii_fd);
	    pthread_mutex_unlock(&gmime_lock);
	    return(PM_FAILED);
	}
        rfp = fdopen(rfd, "r");
        if (rfp == NULL) {
	    log(ERR_FILE_OPEN, "parse_mail", minfo->ii_fd);
	    pthread_mutex_unlock(&gmime_lock);
	    return(PM_FAILED);
        }
        fseek(rfp, 0L, SEEK_SET);

        r_stream = g_mime_stream_file_new (rfp);
    } else {
	DEBUGLOG("Create message object from memory");
	r_stream = g_mime_stream_mem_new_with_buffer (minfo->ii_mbuf, minfo->ii_len);
    }

    /* parser */
    parser = g_mime_parser_new_with_stream(r_stream);
    g_object_unref (r_stream);

    /* gmime object */
    mz->message = g_mime_parser_construct_message(parser);
    if (mz->message == NULL) {
	log(ERR_GMIME, "parse_mail", "g_mime_parser_construct_message");
        g_object_unref (parser);
	pthread_mutex_unlock(&gmime_lock);
	return(PM_FAILED);
    }
    g_object_unref (parser);

    /* Add header */
#ifndef GMIME26
    g_mime_message_add_header(mz->message, XHEADERNAME, XHEADERVALUE);
#else
#if 0   /* [2016.09.27] Add a header when sending mail. */
    g_mime_object_append_header((GMimeObject *)mz->message, XHEADERNAME, XHEADERVALUE);
#endif
#endif

    /* Get header subject */
    mz->subject = g_mime_message_get_subject(mz->message);

    /* Get header date */
    if (cfg->cf_settimezone == 0) {
#ifndef GMIME26
        mz->date = g_mime_message_get_date_string(mz->message);
#else
        mz->date = g_mime_message_get_date_as_string(mz->message);
#endif
    } else {
        /* Repairing The display of the date 2014/05/08 (Thu) */
        mz->date =  g_mime_utils_header_format_date (mz->message->date,
                                                     cfg->cf_settimezone);
    }

    /* gmime part object */
    object = g_mime_message_get_mime_part(mz->message);

    if (!GMIME_IS_MULTIPART (object)) {
#ifndef GMIME24
        g_object_unref (object);
#endif
	pthread_mutex_unlock(&gmime_lock);
	return(PM_NO_MULTIPART);
    }

    ct_object = g_mime_object_get_content_type(object);

    if ((strncmp(ct_object->type, CONTENT_T_MULTIPART, LEN_CONTENT_T_MULTIPART) == 0) && 
        ((strncmp(ct_object->subtype, CONTENT_SUBT_RELATED, LEN_CONTENT_SUBT_RELATED) == 0) ||
         (strncmp(ct_object->subtype, CONTENT_SUBT_ALTERNATIVE, LEN_CONTENT_SUBT_ALTERNATIVE) == 0))) {
#ifndef GMIME24
        g_object_unref (object);
#endif
	pthread_mutex_unlock(&gmime_lock);
	return(PM_NO_ATTACHFILE);
    }

    /* make zip directory */
    len = strlen(cfg_encdir) + strlen(TMPDIR_TMPL) + 2;
    tmpstr = (char *)malloc(len);

    if (tmpstr == NULL) {
	log(ERR_MEMORY_ALLOCATE, "parse_mail", "tmpstr", strerror(errno));
#ifndef GMIME24
        g_object_unref (object);
#endif
	pthread_mutex_unlock(&gmime_lock);
	return(PM_FAILED);
    }
    snprintf(tmpstr, len, FILE_PATH, cfg_encdir, TMPDIR_TMPL);

    tmpdir = mkdtemp(tmpstr);
    if (tmpdir == NULL) {
        log(ERR_DIRECTORY_MAKE, "parse_mail", tmpstr, strerror(errno));
#ifndef GMIME24
        g_object_unref (object);
#endif
	free(tmpstr);
	pthread_mutex_unlock(&gmime_lock);
	return(PM_FAILED);
    }
    mz->zipdir = strdup(tmpdir);
    if (mz->zipdir == NULL) {
	log(ERR_MEMORY_ALLOCATE, "parse_mail", "mz->zipdir", strerror(errno));
#ifndef GMIME24
        g_object_unref (object);
#endif
	free(tmpstr);
	pthread_mutex_unlock(&gmime_lock);
	return -1;
    }
    free(tmpstr);

    if ((ret = drop_file(object, cfg, mz)) != 0) {
#ifndef GMIME24
        g_object_unref (object);
#endif
	pthread_mutex_unlock(&gmime_lock);
        if (ret == -2) {
/* if cannot convert character code then set filename is unknown file */
            /* could not detect/convert attached filename's encoding
             * returning specific value */
            return(PM_INVALID_FILENAME);
        }
	return -1;
    }

#ifndef GMIME24
    g_object_unref (object);
#endif

    /* Not attach file */
    if (mz->namelist == NULL) {
	pthread_mutex_unlock(&gmime_lock);
        return(PM_NO_MULTIPART);
    }

    pthread_mutex_unlock(&gmime_lock);
    return(PM_SUCCESS);
}

int
drop_file(GMimeObject *object, struct config *cfg, struct mailzip *mz)
{
    int ret, i, index;
    GMimeObject *att;
    GMimeStream *w_stream;
    GMimeMessage *message;
    GMimeDataWrapper *wrapper;
    const char *filename;
    char *p_filename, *tmpname;
    char *tmpfilename;
    char newfilename[PATH_MAX + 1];
    char tmppath[PATH_MAX + 1];
    int count = 0;
    FILE *wfp;


    /* get attach file number */
#ifndef GMIME26
    index = g_mime_multipart_get_number((GMimeMultipart *)object);
#else
    index = g_mime_multipart_get_count((GMimeMultipart *)object);
#endif

    for (i = index - 1; i >= 0; i--) {
	att = g_mime_multipart_get_part((GMimeMultipart *)object, i);

	if (GMIME_IS_MESSAGE_PART(att)) {
	    filename = MESSAGE_FILENAME;
	} else if (GMIME_IS_MULTIPART (att)) {
	    ret = drop_file(att, cfg, mz);

#ifndef GMIME24
	    g_object_unref (att);
#endif

	    if (ret < 0) {
		return -1;
	    } 
	    continue;

	} else {
DEBUGLOG("getting filename...");

	    /* get filename */
	    filename = g_mime_part_get_filename((GMimePart *)att);

	    if (filename == NULL) {
DEBUGLOG("Failed to get filename when index = %d", i);
#ifndef GMIME24 
	        g_object_unref (att);
#endif
	        continue;
	    }
	}

	/* replace slash charactor */
	for (p_filename = (char *)filename; *p_filename != '\0'; p_filename++) {
	    if (*p_filename == SLASH) {
		*p_filename = SLASH_REPLACE_CHAR;
	    }
	}

	tmpname = strdup(filename);

	if (tmpname == NULL) {
	    log(ERR_MEMORY_ALLOCATE, "drop_file", "tmpname", strerror(errno));
#ifndef GMIME24
	    g_object_unref (att);
#endif
	    return -1;
        }

	/* convert str code */
	if (dg_str2code_replace(tmpname, &tmpfilename, STR_UTF8, cfg->cf_strcode) != 0) {
            /* START ADD 20150323 */
            /* If the conversion of the str2code is failed, the name of 
             * the attaching file is set by cf_attachmentfilealias.
             */
            tmpfilename = strdup(cfg->cf_attachmentfilealias);
            if (tmpfilename == NULL) {
	        log(ERR_MEMORY_ALLOCATE, "drop_file", "tmpfilename", strerror(errno));

#ifndef GMIME24
	        g_object_unref (att);
#endif

	        return -1;
            }

	    log(ERR_CHARCODE_CONVERT, "drop_file");
            /* delete old code */
            //g_object_unref (att);
	    //free(tmpname);
            /* could not detect/convert encoding
             * returning specific value */
	    //return -2;

            /* END ADD 20150323 */
	}
	free(tmpname);

        if (ismode_delete) {
            strcpy(newfilename, tmpfilename);
        } else {
            mk_new_filename(newfilename, tmpfilename, mz->zipdir, count);
        } 

        if (push_attnamelist(&(mz->namelist), newfilename) != 0) {
#ifndef GMIME24
            g_object_unref (att);
#endif
            free(tmpfilename);
            return -1;
        }

        free(tmpfilename);

        snprintf(tmppath, PATH_MAX, FILE_PATH, mz->zipdir, newfilename);
        DEBUGLOG("attach_file_name :%s\n", tmppath);

        if (GMIME_IS_MESSAGE_PART(att)) {
            wfp = fopen(tmppath, "w");
            if (wfp == NULL) {
                log(ERR_FILE_OPEN, "drop_file", tmppath);
#ifndef GMIME24
                g_object_unref (att);
#endif
                return -1;
            }
            w_stream = g_mime_stream_file_new (wfp);

            message = g_mime_message_part_get_message((GMimeMessagePart *)att);

#ifndef GMIME26
            ret = g_mime_message_write_to_stream (message, w_stream);
#else
            ret = g_mime_object_write_to_stream ((GMimeObject *)message, w_stream);
#endif

#ifndef GMIME24
            g_object_unref (message);
#endif
            g_object_unref (w_stream);

        } else if (!ismode_delete) {
            wfp = fopen(tmppath, "w");
            if (wfp == NULL) {
                log(ERR_FILE_OPEN, "drop_file", tmppath);
#ifndef GMIME24
                g_object_unref (att);
#endif
                return -1;
            }
            w_stream = g_mime_stream_file_new (wfp);

            wrapper = g_mime_part_get_content_object((GMimePart *)att);
	    if (wrapper == NULL) {
	        log(ERR_GMIME, "drop_file", "g_mime_part_get_content_object");

#ifndef GMIME24
	        g_object_unref (att);
#endif

                g_object_unref (w_stream);
	        return -1;
	    }

	    ret = g_mime_data_wrapper_write_to_stream(wrapper, w_stream);

#ifndef GMIME24
            g_object_unref (wrapper);
#endif
            g_object_unref (w_stream);

	}
	if (ret < 0) {
	    log(ERR_GMIME, "drop_file", "g_mime_data_wrapper_write_to_stream");

#ifndef GMIME24
	    g_object_unref (att);
#endif

	    return -1;
	}

#ifndef GMIME26
        g_mime_multipart_remove_part((GMimeMultipart *)object, att);
#else
        g_mime_multipart_remove((GMimeMultipart *)object, (GMimeObject *)att);
#endif

#ifndef GMIME24
	g_object_unref (att);
#endif
    }

    return 0;
}

/*
 * mk_deletelistfile
 *
 * Function
 *      Creat a temporary file to save namelist data in.
 *
 * Argument
 *      struct config  *cfg	config structure
 *      char *tmpdir		temporary directory name
 *
 * Return value
 *      0       Normal end.
 *      -1      Abormal end.
 */
int
mk_deletelistfile(struct config *cfg, struct mailzip *mz)
{
    FILE *fp;
    int i, retw;
    
    /* open a deletelistfile named by mz->encfilepath. */
    fp = fopen(mz->encfilepath, "w");
    if (fp == NULL) {
        log(ERR_FILE_OPEN, "mk_deletelistfile", mz->encfilepath);
        return -1;
    }
    
    for (i = 0; (mz->namelist + i)->attach_name != NULL; i++) {
        retw = fwrite((mz->namelist + i)->attach_name, 
                      (mz->namelist + i)->attach_name_len, 1, fp);
        if(retw < 0) {
            log(ERR_FILE_WRITE, "mk_deletelistfile");
            fclose(fp);
            return -1;
        }
        retw = fprintf(fp, "\r\n");
        if(retw < 0) {
            log(ERR_FILE_WRITE, "mk_deletelistfile");
            fclose(fp);
            return -1;
        }

    }

    fclose(fp);

    return (0);
}
#define CDCOMMAND	"cd"
#define QOPTION		"-q"
#define ANDCOMMAND	"&&"
#define CURRENTDIR	"./"
/*
 * convert_zip
 *
 * Function
 *      ZIP conversion
 *
 * Argument
 *      struct config  *cfg	config structure
 *      char *tmpdir		temporary directory name
 *      char *passwd		encryption password
 *
 * Return value
 *      0       Normal end.
 *      -1      Abormal end.
 */
int
convert_zip(struct config *cfg, struct mailzip *mz, struct rcptinfo **rcpt)
{
    int    ret;
    pid_t  pid;
    char *list[5];
    int     sts = 0;

    list[0] = cfg->cf_zipcommand;
    list[1] = QOPTION;
    list[2] = mz->encfilepath;
    list[3] = CURRENTDIR;
    list[4] = NULL;

    /*
     * make child process
     */
    if ((pid = fork()) == 0) {
	/* child process */
	zip_child(list, cfg->cf_zipcommandopt, mz->zipdir, (*rcpt)->passwd);
    } else if (pid > 0) {
        /* parent process */
        ret = waitpid(pid, &sts, WUNTRACED);
	if (ret == -1) {
            log(ERR_WAIT_CHILD, "convert_zip", strerror(errno));
            return -1;
	}
        if (WIFEXITED(sts)) {       /* return or exit */
            ret = WEXITSTATUS(sts); /* exit code */
            if (ret != 0) {
                log(ERR_ZIP_CONVERT, "convert_zip", mz->encfilepath);
                return -1;
            }
        } else {
            log(ERR_WAIT_CHILD, "convert_zip", strerror(errno));
            return -1;
        }
    } else {
	/* make process failed */
        log(ERR_FORK_CREATE, "convert_zip", strerror(errno));
	return -1;
    }

    return (0);
}

/*--------------------------------------------------------*
 * child process
 *--------------------------------------------------------*/
static void
zip_child(char **list, char *commandopt, char *dir, char *passwd)
{
    char *envstr;
    /*
     * set environment variable
     */
    if ((commandopt == NULL) || (strncmp(commandopt, OPTIONEND, OPTIONENDLEN) == 0)) {
        envstr = malloc(strlen(OPTIONRP) + strlen(passwd) + 3);
        sprintf(envstr, "%s %s", OPTIONRP, passwd);
    } else {
        envstr = malloc(strlen(OPTIONRP) + strlen(passwd) + strlen(commandopt) + 3);
        sprintf(envstr, "%s %s %s", OPTIONRP, passwd, commandopt);
    }
    if (setenv(ENV_ZIPOPT, envstr, OVERWRITE) < 0) {
        log(ERR_SET_ENV, "zip_child", ENV_ZIPOPT);
        free(envstr);
	exit(1);
    }
    free(envstr);

    /* chage encryption directory */
    if (chdir(dir) != 0) {
        log(ERR_DIRECTORY_CHANGE, "zip_child", dir);
        exit(1); 
    }

    /*
     * Execution
     */
    if (execvp(list[0], list) < 0) {
        log(ERR_EXEC_FILE, "zip_child", list[0], strerror(errno));
        exit(1);
    }
    
    unsetenv(ENV_ZIPOPT);

    exit(0);
}

/*
 * check_ascii
 *
 * args:
 *  char *str           mail address
 * return:
 *  0           is ascii
 *  -1          not ascii
 */
int
check_ascii(char *str)
{
    int ret;
    int i;

    for (i = 0; str[i] != '\0'; i++) {

        /* Ascii Check */
        ret = isascii(str[i]);

        /* Not Ascii */
        if (ret == 0) {
            log(ERR_MAILADDR_UNKNOWN_TYPE, "check_ascii", str);
            return -1;
        }
    }

    return 0;
}

int
mk_encpath(struct config *cfg, struct mailzip *mz)
{
    int strdatenum;
    struct tm *ti; 
    time_t now;
    char *filename, *path;
    char *tmpdir = mz->zipdir;
    char *cfg_attachname = NULL;

    /* Switch cfg_attachname according to the run modes */
    if (ismode_delete) {
        cfg_attachname = cfg->cf_deletelistname;
    } else if (ismode_harmless) {
        cfg_attachname = cfg->cf_zipfilename;
    } else {
        cfg_attachname = cfg->cf_zipfilename;
    }

    /* get now time */
    time(&now);
    ti = localtime(&now);

    /* allocate date string */
    strdatenum = strlen(cfg_attachname) + STRTIMENUM + 1;
    filename = malloc(strdatenum);
    if (filename == NULL) {
	log(ERR_MEMORY_ALLOCATE, "add_enclist", "rcptinfo", strerror(errno));
	return -1;
    }

    strftime(filename, strdatenum - 1, cfg_attachname, ti);
    mz->attachfilename = filename;

    /* create zip (delete-report) file path in Encryption (Delete) mode */
    path = malloc(strlen(tmpdir) + strdatenum);
    if (path == NULL) {
	log(ERR_MEMORY_ALLOCATE, "add_enclist", "rcptinfo", strerror(errno));
        free(filename);
	return -1;
    }
    sprintf(path, FILE_PATH, tmpdir, filename);

    mz->encfilepath = path;

    return 0;
}

/* get_msgid
 * 
 * get the message id from file attached mail
 *
 * Args:
 *    char **buf        buffer of set the message id
 *    char *mail        mail data of file attached mail
 *
 * Return:
 *    ERROR      -1      error
 *    SUCCESS     0      success
 */
int get_msgid(char **buf, char *mail) {

    //variables
    int id_len = 0;
    char *p = NULL;
    char *head = NULL;
    char *tail = NULL;      
    char *tmp = NULL;

    // search Message-ID field
    p = strstr(mail, MSG_ID);   
    if (p == NULL) {
        return ERROR;
    }

    // search "<"
    head = strchr(p, '<');
    if (head == NULL) {
        return ERROR;
    }
    // search ">"
    tail = strchr(head, '>');
    if (tail == NULL) {
        return ERROR;
    }
    // check the length of Message-ID
    id_len = strlen(head) - strlen(tail) + 1;

    // allocate memory for Message-ID(2 is '\n' and '\0')
    tmp = (char *)calloc(REF_LEN + id_len + 2, sizeof(char));
    if (tmp == NULL) {
        return ERROR;
    }
    *buf = tmp;

    // set References field
    strcat(*buf, REF);
    memcpy(*buf + strlen(REF), head, id_len);
    strcat(*buf, "\n\0");

    return SUCCESS;
}

/* set_references
 * 
 * add header field "References" to password confirming mail
 *
 * Args:
 *    char *mail_data	strings of mail data include header of file 
 *                      attached mail
 *    char *passwd_mail mail data of password confirming mail
 *
 * Return:
 *    ERROR      -1      error
 *    SUCCESS     0      success
 */
int set_references(char *mail_data, char **passwd_mail) {

    // variables
    char *tmp = NULL;
    char *ref = NULL;
    int ret = 0;

    //get message id
    ret = get_msgid(&ref, mail_data);
    if (ret == ERROR) {
        return ERROR;
    }

    // allocate memory for mail 
    tmp = realloc(*passwd_mail, strlen(ref) + strlen(*passwd_mail) + 1);
    if (tmp == NULL) {
        free(ref);
        return ERROR;
    }
    *passwd_mail = tmp;

    // set references
    memmove(*passwd_mail + strlen(ref), *passwd_mail, strlen(*passwd_mail) + 1);
    memcpy(*passwd_mail, ref, strlen(ref));
    free(ref);

    return SUCCESS;
}
