/*
 * samma
 *
 * Copyright (C) 2006,2007,2008,2011 DesigNET, INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * $RCSfile: mailzip_config.c.in,v $
 * $Revision: 1.14 $
 * $Date: 2014/05/09 04:43:07 $
 */

#include <stdio.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <syslog.h>
#include <pthread.h>
#include <libdgconfig.h>
#include <gmime/gmime.h>
#include <ldap.h>
#include <lber.h>

#define _MAILZIP_CONFIG_C_
#include "mailzip_config.h"

#include "log.h"
#include "mailzip_db.h"
#include "samma_policy.h"
#include "global.h"
#include "mailzip_tmpl.h"
#include "netlist.h"

static pthread_mutex_t config_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t config_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int config_ref = 0;
static int config_ref_old = 0;

struct config *cur_cfg = NULL;
struct config *new_cfg = NULL;
struct config *old_cfg = NULL;

static pthread_mutex_t tmpl_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t tmpl_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int tmpl_ref = 0;
static int tmpl_ref_old = 0;

/* ADD(20150316)*/
static pthread_mutex_t whitelist_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t whitelist_ref_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t whitelist_ref_old_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t whitelist_reload_lock = PTHREAD_MUTEX_INITIALIZER;
static int whitelist_ref = 0;
static int whitelist_ref_old = 0;


struct cfentry cfe[] = {
    {
        "CommandPort", CF_INT_PLUS, "17777",
        OFFSET(struct config, cf_commandport), is_port
    },
    {
        "CommandPass", CF_STRING, NULL,
        OFFSET(struct config, cf_commandpass), is_notnull
    },
    {
        "SyslogFacility", CF_STRING, "local1",
        OFFSET(struct config, cf_syslogfacility), is_syslog_facility,
    },
    {
        "ListenIP", CF_STRING, "127.0.0.1",
        OFFSET(struct config, cf_listenip), is_ipaddr
    },
    {
        "ListenPort", CF_INT_PLUS, "20026",
        OFFSET(struct config, cf_listenport), is_port
    },
    {
        "EncryptionTmpDir", CF_STRING, NULL,
        OFFSET(struct config, cf_encryptiontmpdir), is_writable_directory
    },
    {
        "ZipCommand", CF_STRING, NULL,
        OFFSET(struct config, cf_zipcommand), is_readable_file
    },
    {
        "ZipCommandOpt", CF_STRING, "--",
        OFFSET(struct config, cf_zipcommandopt), NULL
    },
    {
        "SenderDB", CF_STRING, "hash:/usr/local/etc/samma/sender.db",
        OFFSET(struct config, cf_senderdb), is_dbpath
    },
    {
        "RcptDB", CF_STRING, "hash:/usr/local/etc/samma/rcpt.db",
        OFFSET(struct config, cf_rcptdb), is_dbpath
    },
    {
        "TemplatePath", CF_STRING, "/usr/local/etc/samma/samma.tmpl",
        OFFSET(struct config, cf_templatepath), is_readable_file
    },
    {
        "WhitelistPath", CF_STRING, "/usr/local/etc/samma/whitelist",
        OFFSET(struct config, cf_whitelistpath), NULL
    },
    {
        "SendmailCommand", CF_STRING, NULL,
        OFFSET(struct config, cf_sendmailcommand), is_notnull
    },
    {
        "SendPasswordCommand", CF_STRING, "--",
        OFFSET(struct config, cf_sendpasswordcommand), is_notnull
    },
    {
        "ZipfileName", CF_STRING, NULL,
        OFFSET(struct config, cf_zipfilename), is_notnull
    },
    {
        "AttachmentFileAlias", CF_STRING, "UnknownFileName",
        OFFSET(struct config, cf_attachmentfilealias), is_notnull
    },
    {
        "MailsaveTmpDir", CF_STRING, NULL,
        OFFSET(struct config, cf_mailsavetmpdir), is_writable_directory
    },
    {
        "PasswordLength", CF_INT_PLUS, "8",
        OFFSET(struct config, cf_passwordlength), is_passwd_length
    },
    {
        "StrCode", CF_STRING, NULL,
        OFFSET(struct config, cf_strcode), is_strcode
    },
    {
        "DefaultEncryption", CF_STRING, NULL,
        OFFSET(struct config, cf_defaultencryption), is_yesno
    },
    {
        "DefaultPassword", CF_STRING, "",
        OFFSET(struct config, cf_defaultpassword), is_notnull
    },
    {
        "UserPolicy", CF_STRING, NULL,
        OFFSET(struct config, cf_userpolicy), is_yesno
    },
    {
        "LoopCheck", CF_STRING, "no",
        OFFSET(struct config, cf_loopcheck), is_yesno
    },
    {
        "LDAPUri", CF_STRING, "",
        OFFSET(struct config, cf_ldapuri), is_notnull
    },
    {
        "LDAPBaseDN", CF_STRING, "",
        OFFSET(struct config, cf_ldapbasedn), is_notnull
    },
    {
        "LDAPBindDN", CF_STRING, "",
        OFFSET(struct config, cf_ldapbinddn), is_notnull
    },
    {
        "LDAPBindPassword", CF_STRING, "",
        OFFSET(struct config, cf_ldapbindpassword), is_notnull
    },
    {
        "LDAPFilter", CF_STRING, "",
        OFFSET(struct config, cf_ldapfilter), is_notnull
    },
    {
        "FixedPassNotify", CF_STRING, "no",
        OFFSET(struct config, cf_fixedpassnotify), is_yesno
    },
    {
        "FixedPassNotifyPass", CF_STRING, "[Default Password]",
        OFFSET(struct config, cf_fixedpassnotifypass), is_notnull
    },
    {
	"AutoBccOption", CF_STRING, "no",
	OFFSET(struct config, cf_autobccoption), is_yesno
    },
    {
	"DefaultAutoBccConditionString", CF_STRING, "",
	OFFSET(struct config, cf_defaultautobccconditionstring), is_notnull
    },
    {
	"DefaultAutoBccMailAddr", CF_STRING, "",
	OFFSET(struct config, cf_defaultautobccmailaddr), is_notnull
    },
    {   
	"References", CF_STRING, "no",
	OFFSET(struct config, cf_references), is_yesno
    },
    {
        "SetTimeZone", CF_INTEGER, "0",
        OFFSET(struct config, cf_settimezone), is_timezone
    }
};

#define NCONFIG (sizeof(cfe) / sizeof(struct cfentry))

struct dbset db_kind[] = {
    {
        HASH, HASH_SIZE, DB_HASH
    },
    {
        BTREE, BTREE_SIZE, DB_BTREE
    }
};

struct str_code_set str_code[] = {
    {STR_UTF8, STR_UTF8_LEN},
    {STR_JIS, STR_JIS_LEN},
    {STR_EUC, STR_EUC_LEN},
    {STR_SJIS, STR_SJIS_LEN}
};

#define STRCODENUM (sizeof(str_code) / sizeof(struct str_code_set))

char *cur_tmpl = NULL;
char *new_tmpl = NULL;
char *old_tmpl = NULL;

/* ADD(20150316) */
/* struct of whitelist */
struct whitelist *cur_whitelist = NULL;
struct whitelist *new_whitelist = NULL;
struct whitelist *old_whitelist = NULL;

/*
 * setup_config()
 *
 * Setup config structure.
 *
 * args: (void)
 *
 * Return value:
 *  struct config *	Success
 *  NULL		error
 */
static struct config *
setup_config()
{
    struct config *cf = NULL;

    // allocate memory
    cf = (struct config *)malloc(sizeof(struct config));
    if (cf == NULL) {
	log(ERR_MEMORY_ALLOCATE, "setup_config", "cf", strerror(errno));
	return NULL;
    }

    // set all members to NULL
    memset(cf, 0, sizeof(struct config));
    return cf;
}

static void
free_config(struct config *cfg)
{
    if (cfg == NULL) {
	return;
    }
    if (cfg->cf_syslogfacility != NULL) {
	free(cfg->cf_syslogfacility);
    }
    if (cfg->cf_listenip != NULL) {
	free(cfg->cf_listenip);
    }
    if (cfg->cf_encryptiontmpdir != NULL) {
	free(cfg->cf_encryptiontmpdir);
    }
    if (cfg->cf_zipcommand != NULL) {
	free(cfg->cf_zipcommand);
    }
    if (cfg->cf_zipcommandopt != NULL) {
	free(cfg->cf_zipcommandopt);
    }
    if (cfg->cf_senderdb != NULL) {
	free(cfg->cf_senderdb);
    }
    if (cfg->cf_rcptdb != NULL) {
	free(cfg->cf_rcptdb);
    }
    if (cfg->cf_templatepath != NULL) {
	free(cfg->cf_templatepath);
    }
    if (cfg->cf_whitelistpath != NULL) {
        free(cfg->cf_whitelistpath);
    }
    if (cfg->cf_senderdbpath != NULL) {
	free(cfg->cf_senderdbpath);
    }
    if (cfg->cf_rcptdbpath != NULL) {
	free(cfg->cf_rcptdbpath);
    }
    if (cfg->cf_defaultencryption != NULL) {
	free(cfg->cf_defaultencryption);
    }
    if (cfg->cf_defaultpassword != NULL) {
	free(cfg->cf_defaultpassword);
    }
    if (cfg->cf_userpolicy != NULL) {
	free(cfg->cf_userpolicy);
    }

    /* START ADD 2015 */
    if (cfg->cf_loopcheck != NULL) {
        free(cfg->cf_loopcheck);
    }
    if (cfg->cf_attachmentfilealias != NULL) {
        free(cfg->cf_attachmentfilealias);
    }
    /* END ADD 2015 */

    if (cfg->cf_ldapuri != NULL) {
	free(cfg->cf_ldapuri);
    }
    if (cfg->cf_ldapbasedn != NULL) {
	free(cfg->cf_ldapbasedn);
    }
    if (cfg->cf_ldapbinddn != NULL) {
	free(cfg->cf_ldapbinddn);
    }
    if (cfg->cf_ldapbindpassword != NULL) {
	free(cfg->cf_ldapbindpassword);
    }
    if (cfg->cf_ldapfilter != NULL) {
	free(cfg->cf_ldapfilter);
    }
    if (cfg->cf_fixedpassnotify != NULL) {
        free(cfg->cf_fixedpassnotify);
    }
    if (cfg->cf_fixedpassnotifypass != NULL) {
        free(cfg->cf_fixedpassnotifypass);
    }
    if (cfg->cf_autobccoption != NULL) {
	free(cfg->cf_autobccoption);
    }
    if (cfg->cf_defaultautobccconditionstring != NULL) {
	free(cfg->cf_defaultautobccconditionstring);
    }
    if (cfg->cf_defaultautobccmailaddr != NULL) {
	free(cfg->cf_defaultautobccmailaddr);
    }
    if (cfg->cf_references != NULL) {
	free(cfg->cf_references);
    }

    /* START ADD */
    if (cfg->cf_commandpass != NULL) {
        free(cfg->cf_commandpass);
        cfg->cf_commandpass = NULL;
    }
    if (cfg->cf_sendmailcommand != NULL) {
        free(cfg->cf_sendmailcommand);
        cfg->cf_sendmailcommand = NULL;
    }
    if (cfg->cf_sendpasswordcommand != NULL) {
        free(cfg->cf_sendpasswordcommand);
        cfg->cf_sendpasswordcommand = NULL;
    }
    if (cfg->cf_zipfilename != NULL) {
        free(cfg->cf_zipfilename);
        cfg->cf_zipfilename = NULL;
    }
    if (cfg->cf_mailsavetmpdir != NULL) {
        free(cfg->cf_mailsavetmpdir);
        cfg->cf_mailsavetmpdir = NULL;
    }
    if (cfg->cf_strcode != NULL) {
        free(cfg->cf_strcode);
        cfg->cf_strcode = NULL;
    }
    /* END ADD */

    free(cfg);
    return;
}

static void *
cfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&config_reload_lock);
    free_config(old_cfg);
    old_cfg = NULL;
    pthread_mutex_unlock(&config_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

/*
 * config_init()
 *
 * Get current config structure pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  struct config *		config structure pointer
 */
struct config *
config_init()
{
    struct config *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    config_ref ++;

    // get config structure pointer
    ret_ptr = cur_cfg;

    pthread_mutex_unlock(&config_ref_lock);
    pthread_mutex_unlock(&config_lock);
    return ret_ptr;
}

/*
 * config_release()
 * 
 * Countdown config reference counter.
 *
 * Args:
 *  struct config *cfg		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
config_release(struct config *cfg)
{
    pthread_mutex_lock(&config_lock);
    if (old_cfg == cfg) {
	// case to release old config
	pthread_mutex_lock(&config_ref_old_lock);
	config_ref_old --;
	if (config_ref_old == 0) {
	    pthread_mutex_unlock(&config_reload_lock);
	}
	pthread_mutex_unlock(&config_ref_old_lock);
    } else {
	// case to release cur config
	pthread_mutex_lock(&config_ref_lock);
	config_ref --;
	pthread_mutex_unlock(&config_ref_lock);
    }
    pthread_mutex_unlock(&config_lock);
}

/*
 * reload_config()
 *
 * Reload configuration file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of config file)
 */
int
reload_config(char *file)
{
    struct config *cfg = NULL;
    int ret;
    pthread_t cfree;
    int i;

    pthread_mutex_lock(&config_ref_old_lock);
    ret = config_ref_old;
    pthread_mutex_unlock(&config_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_CONFIG_RELOADING, "reload_config", file);
	return 1;
    }

    // setup config structure
    cfg = setup_config();
    if (cfg == NULL) {
	return CFG_NG;
    }

    // read config file
    ret = read_config(file, cfe, NCONFIG, cfg);

    if (ret > 0) {
	// case config file error
	free_config(cfg);
	return 2;
    }
    if (ret < 0) {
	// case system error
	free_config(cfg);
	return CFG_NG;
    }

    // set db infomation
    for (i = 0; i < DBNUM; i++) {
        ret = strncmp(cfg->cf_senderdb, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            cfg->cf_senderdbpath = strdup(cfg->cf_senderdb + db_kind[i].db_len);
	    cfg->cf_senderdbtype = db_kind[i].db_type;
        }
        ret = strncmp(cfg->cf_rcptdb, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            cfg->cf_rcptdbpath = strdup(cfg->cf_rcptdb + db_kind[i].db_len);
	    cfg->cf_rcptdbtype = db_kind[i].db_type;
        }
	if ((cfg->cf_senderdbpath != NULL) && (cfg->cf_rcptdbpath != NULL)) {
	    break;
	}
    }

    // Check LDAP and AutoBCC configuration
    if ((cfg->cf_userpolicy[0] == 'Y') || (cfg->cf_userpolicy[0] == 'y')) {
        ret = check_ldap_config(cfg);
        if (ret != 0) {
            free_config(cfg);
            return CFG_NG;
        }
    }
    else
    {
	if ((cfg->cf_autobccoption[0] == 'Y') || (cfg->cf_autobccoption[0] == 'y')) {
	    log(ERR_AUTOBCC_OPTION, "AutoBccOption");
	    free_config(cfg);
	    return CFG_NG;
	}
    }

    if ((cfg->cf_autobccoption[0] == 'Y' || cfg->cf_autobccoption[0] == 'y')) {
	ret = check_autobcc_config(cfg);
	if (ret != 0) {
	    free_config(cfg);
	    return CFG_NG;
	}
    }

    // template file reload 
    if (reload_tmpl(cfg->cf_templatepath) != 0) {
	free_config(cfg);
	return CFG_NG;
    }

    // whitelist file reload
    if (reload_whitelist(cfg->cf_whitelistpath) != 0) {
        free_config(cfg);
        return CFG_NG;
    }

    // change config structure
    pthread_mutex_lock(&config_lock);
    pthread_mutex_lock(&config_ref_lock);
    pthread_mutex_lock(&config_ref_old_lock);
    config_ref_old = config_ref;
    config_ref = 0;
    if (config_ref_old > 0) {
	pthread_mutex_lock(&config_reload_lock);
    }
    pthread_mutex_unlock(&config_ref_old_lock);
    pthread_mutex_unlock(&config_ref_lock);

    // create config free thread
    ret = pthread_create(&cfree, NULL, cfree_handler, NULL);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_CREATE, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(cfree);
    if (ret != 0) {
	free_config(cfg);
	pthread_mutex_unlock(&config_lock);
	log(ERR_THREAD_DETACH, "reload_config",
	    "cfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_cfg = cur_cfg;
    cur_cfg = cfg;
    switch_log(cfg->cf_syslogfacility);
    pthread_mutex_unlock(&config_lock);

    return 0;
}

static void *
tfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&tmpl_reload_lock);
    free(old_tmpl);
    old_tmpl = NULL;
    pthread_mutex_unlock(&tmpl_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

static void *
wfree_handler(void *arg)
{
    int ret = 0;
    pthread_mutex_lock(&whitelist_reload_lock);
    whilelist_free(old_whitelist);
    old_whitelist = NULL;
    pthread_mutex_unlock(&whitelist_reload_lock);
    pthread_exit(&ret);
    return NULL;
}

/*
 * tmpl_init()
 *
 * Get current template data pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  char *		template data pointer
 */
char *
tmpl_init()
{
    char *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&tmpl_lock);
    pthread_mutex_lock(&tmpl_ref_lock);
    tmpl_ref ++;

    // get template data pointer
    ret_ptr = cur_tmpl;

    pthread_mutex_unlock(&tmpl_ref_lock);
    pthread_mutex_unlock(&tmpl_lock);
    return ret_ptr;
}

/*
 * whitelist_init()
 *
 * Get current whitelist data pointer,
 * and countup the reference counter.
 *
 * Args: (void)
 *
 * Return value:
 *  struct whitelist *        whitelist data pointer
 */
struct whitelist *
whitelist_init()
{
    struct whitelist *ret_ptr;

    // countup reference counter
    pthread_mutex_lock(&whitelist_lock);
    pthread_mutex_lock(&whitelist_ref_lock);
    whitelist_ref ++;

    // get template data pointer
    ret_ptr = cur_whitelist;

    pthread_mutex_unlock(&whitelist_ref_lock);
    pthread_mutex_unlock(&whitelist_lock);
    return ret_ptr;
}

/*
 * tmpl_release()
 * 
 * Countdown template reference counter.
 *
 * Args:
 *  char *tmpdata		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
tmpl_release(char *tmpdata)
{
    pthread_mutex_lock(&tmpl_lock);
    if (old_tmpl == tmpdata) {
	// case to release old template data
	pthread_mutex_lock(&tmpl_ref_old_lock);
	tmpl_ref_old --;
	if (tmpl_ref_old == 0) {
	    pthread_mutex_unlock(&tmpl_reload_lock);
	}
	pthread_mutex_unlock(&tmpl_ref_old_lock);
    } else {
	// case to release cur template data
	pthread_mutex_lock(&tmpl_ref_lock);
	tmpl_ref --;
	pthread_mutex_unlock(&tmpl_ref_lock);
    }
    pthread_mutex_unlock(&tmpl_lock);
}

/*
 * whitelist_release()
 * 
 * Countdown whitelist reference counter.
 *
 * Args:
 *  char *tmpdata		To release pointer.
 *
 * Return value:
 *  (void)
 */
void
whitelist_release(struct whitelist *whitelistdata)
{
    pthread_mutex_lock(&whitelist_lock);
    if ((old_whitelist == whitelistdata) && (old_whitelist != NULL)) {
	// case to release old whitelist data
	pthread_mutex_lock(&whitelist_ref_old_lock);
	whitelist_ref_old --;
	if (whitelist_ref_old == 0) {
	    pthread_mutex_unlock(&whitelist_reload_lock);
	}
	pthread_mutex_unlock(&whitelist_ref_old_lock);
    } else {
	// case to release cur whitelist data
	pthread_mutex_lock(&whitelist_ref_lock);
	whitelist_ref --;
	pthread_mutex_unlock(&whitelist_ref_lock);
    }
    pthread_mutex_unlock(&whitelist_lock);
}


/*
 * reload_tmpl()
 *
 * Reload template file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of template file)
 */
int
reload_tmpl(char *file)
{
    int ret;
    pthread_t tfree;
    char *tmpl_data = NULL;

    pthread_mutex_lock(&tmpl_ref_old_lock);
    ret = tmpl_ref_old;
    pthread_mutex_unlock(&tmpl_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_TEMPLATE_RELOADING, "reload_tmpl", file);
	return 1;
    }

    // read tmplate file
    if (tmpl_read(&(tmpl_data), file) != 0) {
	return 2;
    }

    // change template data
    pthread_mutex_lock(&tmpl_lock);
    pthread_mutex_lock(&tmpl_ref_lock);
    pthread_mutex_lock(&tmpl_ref_old_lock);
    tmpl_ref_old = tmpl_ref;
    tmpl_ref = 0;
    if (tmpl_ref_old > 0) {
	pthread_mutex_lock(&tmpl_reload_lock);
    }
    pthread_mutex_unlock(&tmpl_ref_old_lock);
    pthread_mutex_unlock(&tmpl_ref_lock);

    // create template free thread
    ret = pthread_create(&tfree, NULL, tfree_handler, NULL);
    if (ret != 0) {
	free(tmpl_data);
	pthread_mutex_unlock(&tmpl_lock);
	log(ERR_THREAD_CREATE, "reload_tmpl",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(tfree);
    if (ret != 0) {
	free(tmpl_data);
	pthread_mutex_unlock(&tmpl_lock);
	log(ERR_THREAD_DETACH, "reload_tmpl",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_tmpl = cur_tmpl;
    cur_tmpl = tmpl_data;
    pthread_mutex_unlock(&tmpl_lock);

    return 0;
}


/*
 * reload_whitelist()
 *
 * Reload whitelist file
 *
 * Args:
 *  char *file		Configuration file name
 *
 * Return value:
 *  0			Success
 *  CFG_NG		System error
 *  1			Temporaly error (during reloading)
 *  2			Temporaly error (because of whitelist file)
 */
int
reload_whitelist(char *file)
{
    int ret;
    pthread_t tfree;
    struct whitelist *whitelist_data = NULL;

    pthread_mutex_lock(&whitelist_ref_old_lock);
    ret = whitelist_ref_old;
    pthread_mutex_unlock(&whitelist_ref_old_lock);
    if (ret > 0) {
	// case reloading
	log(ERR_WHITELIST_RELOADING, "reload_whitelist", file);
	return 1;
    }

    // read whitelist file
    if (whitelist_read(&(whitelist_data), file) != 0) {
	return 1;
    }

    // change template data
    pthread_mutex_lock(&whitelist_lock);
    pthread_mutex_lock(&whitelist_ref_lock);
    pthread_mutex_lock(&whitelist_ref_old_lock);
    whitelist_ref_old = whitelist_ref;
    whitelist_ref = 0;
    if (whitelist_ref_old > 0) {
	pthread_mutex_lock(&whitelist_reload_lock);
    }
    pthread_mutex_unlock(&whitelist_ref_old_lock);
    pthread_mutex_unlock(&whitelist_ref_lock);

    // create whitelist free thread
    ret = pthread_create(&tfree, NULL, wfree_handler, NULL);
    if (ret != 0) {
	whilelist_free(whitelist_data);
	pthread_mutex_unlock(&tmpl_lock);
	log(ERR_THREAD_CREATE, "reload_whitelist",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }
    ret = pthread_detach(tfree);
    if (ret != 0) {
	whilelist_free(whitelist_data);
	pthread_mutex_unlock(&whitelist_lock);
	log(ERR_THREAD_DETACH, "reload_whitelist",
	    "tfree_handler", strerror(errno));
	return CFG_NG;
    }

    // switch pointer
    old_whitelist = cur_whitelist;
    cur_whitelist = whitelist_data;

    pthread_mutex_unlock(&whitelist_lock);

    return 0;
}
/*
 * is_dbpath()
 *
 * Check if the string is valid DB type.
 * Check if the string is valid path.
 *
 * Args:
 *  char *path			string
 *
 * Return value:
 *  NULL			Success
 *  Invalid argument		Error
 */
static char *
is_dbpath(char *path)
{
    static char errbuf[MAX_CONFIG_LINE];
    char *retp, *file = NULL;
    int   ret, i, n;

    for (i = 0; i < DBNUM; i++) {
        ret = strncmp(path, db_kind[i].db_name, db_kind[i].db_len);
        if (ret == 0) {
            file = path + db_kind[i].db_len;
            break;
        }
    }
    if (file == NULL || *file == '\0') {
        n = sizeof(ERR_CONF_DBPATH) + strlen(path);
        snprintf(errbuf, n, ERR_CONF_DBPATH, path);
        return(errbuf);
    }
    retp = is_readable_file(file);
    if (retp != NULL) {
        n = sizeof(ERR_CONF_DBPATH) + strlen(path);
        snprintf(errbuf, n, ERR_CONF_DBPATH, path);
        return(errbuf);
    }
    return (NULL);
}

/*
 * is_executable_file()
 *
 * Check file permission.
 *
 * Args:
 *  char *			str
 *
 * Return value:
 *  NULL			Success
 *  Invalid argument		Error
 */
char *
is_executable_file(char *str)
{
    static char errbuf[MAX_CONFIG_LINE];
    struct stat st;
    int n = 0;

    if (stat(str, &st) < 0) {
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }

    if (S_ISDIR(st.st_mode)) {
        errno = EISDIR;
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }

    if (access(str, X_OK) != 0) {
        n = sizeof(ERR_CONF_FILEDIR) + strlen(str) + strlen(strerror(errno));
        snprintf(errbuf, n, ERR_CONF_FILEDIR, str, strerror(errno));
        return (errbuf);
    }
    return (NULL);
}

/*
 * is_passwd_length()
 *
 * Check Password Length.
 *
 * Args:
 *  int				len
 *
 * Return value:
 *  NULL			Success
 *  ErrorMessage		Error
 */
char *
is_passwd_length(int len)
{
    if ((len < PASSMIN) || (PASSMAX < len)) {
        return (ERR_CONF_PASSLEN);
    }
    return (NULL);
}

/*
 * is_strcode()
 *
 * Check StrCode Value.
 *
 * Args:
 *  char *			str
 *
 * Return value:
 *  NULL			Success
 *  ErrorMessage		Error
 */
char *
is_strcode(char *str)
{
    int i;

    for (i = 0; i < STRCODENUM; i++) {
        if (strncmp(str, str_code[i].code_name, str_code[i].code_len) == 0) {
	    return (NULL);
        }
    }
    return (ERR_STR_CODE);
}

/*
 * is_notnull()
 *
 * Check Value.
 *
 * Args:
 *  char *			str
 *
 * Return value:
 *  NULL			Success
 *  ErrorMessage		Error
 */
char *
is_notnull(char *str)
{
    if (str != NULL) {
	if (strlen(str) > 0) {
	    return (NULL);
	}
    }
    return (ERR_NULL_VALUE);
}

/*
 * is_yesno()
 *
 * Check Yes or No Value.
 *
 * Args:
 *  char *			str
 *
 * Return value:
 *  NULL			Success
 *  ErrorMessage		Error
 */
char *
is_yesno(char *str)
{
    if (str == NULL) {
        return (ERR_YESNO);
    }

    if ((strcasecmp(str, STR_YES) != 0) &&
        (strcasecmp(str, STR_NO) != 0)){
        return (ERR_YESNO);
    }

    return (NULL);
}

/*
 * check_ldap_config()
 *
 * Check LDAP configuration value.
 *
 * Args:
 *  struct config               *cfg
 *
 * Return value:
 *  0                           Success
 *  -1                          Error
 */
static int
check_ldap_config(struct config *cfg)
{
    LDAP *ld;

    if (cfg->cf_ldapuri[0] == '\0') {
        log(ERR_LDAP_CONFIG, "LdapUri");
        return (-1);
    }

    if (cfg->cf_ldapbasedn[0] == '\0') {
        log(ERR_LDAP_CONFIG, "LdapBaseDN");
        return (-1);
    }

    if (cfg->cf_ldapbinddn[0] == '\0') {
        log(ERR_LDAP_CONFIG, "LdapBindDN");
        return (-1);
    }

    if (cfg->cf_ldapbindpassword[0] == '\0') {
        log(ERR_LDAP_CONFIG, "LdapBindPassword");
        return (-1);
    }

    if (cfg->cf_ldapfilter[0] == '\0') {
        log(ERR_LDAP_CONFIG, "LdapFilter");
        return (-1);
    }

    ld = ldap_connect(cfg->cf_ldapuri, cfg->cf_ldapbinddn, cfg->cf_ldapbindpassword);
    if (ld == NULL) {
        return (-1);
    }
    ldap_unbind_ext_s(ld, NULL, NULL);

    return (0);
}

/*
 * check_autobcc_config()
 *
 * Check AutoBCC configuration value.
 *
 * Args:
 *  struct config               *cfg
 *
 * Return value:
 *  0                           Success
 *  -1                          Error
 */
static int
check_autobcc_config(struct config *cfg)
{
    if (cfg->cf_defaultautobccconditionstring[0] == '\0') {
	log(ERR_AUTOBCC_PARAM, "DefaultAutoBccConditionString");
	return (-1);
    }

    if (cfg->cf_defaultautobccmailaddr[0] == '\0') {
	log(ERR_AUTOBCC_PARAM, "DefaultAutoBccMailAddr");
	return (-1);
    }

    return (0);
}

/*
 * is_ldapfilter()
 *
 * Check LDAP Filter.
 *
 * Args:
 *  char *                      str
 *
 * Return value:
 *  NULL                        Success
 *  ErrorMessage                Error
 */
char *
is_ldapfilter(char *str)
{
    char *tmp;
    int count = 0;

    if ((str == NULL) || (*str == '\0')) {
        return (ERR_LDAP_FILTER);
    }

    for (tmp = str; *tmp != '\0'; tmp++) {
        if (*tmp == '%') {
            count++;
        }
    }

    if (count != 1) {
        return (ERR_LDAP_FILTER);
    }

    return (NULL);
}

/*
 * is_timezone()
 *
 * Check Timezone Value.
 * (Repairing The display of the date 2014/05/08 (Thu))
 *
 * Args:
 *  int                         timezone
 *
 * Return value:
 *  NULL                        Success
 *  ErrorMessage                Error
 */
char *
is_timezone(int timezone)
{
    int minutes = 0;

    /* check MAX value and minimum value */
    if ((timezone < TIMEZONEMIN) || (timezone > TIMEZONEMAX)) {
        return (ERR_CONF_TIMEZONE);
    }
 
    minutes = timezone % 100;
    if (timezone < 0) {
        minutes = minutes * (-1);
    }

    /* check the two lower digits 
     * OK: 00(0), 15, 30, 45
     * NG: the others
     */
    if ((minutes > 45) || ((minutes % 15) != 0)) {
        return (ERR_CONF_TIMEZONE);
    }
    return (NULL);
}
