/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : sgy.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 2002/09/10
 *    Last                 : 2007/10/31
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <unistd.h>
//#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "intl.h"

#include "MemoryLeak.h"
#include "List_cad.h"
#include "List_Dimension.h"
#include "List_PolyLine.h"
#include "List_Ellipse.h"
#include "List_insert.h"
#include "List_Vertex.h"
#include "List_Block.h"
#include "List_Undo.h"
#include "List_Select.h"
#include "global.h"
#include "etc.h"
#include "Draw.h"
#include "culcfunc.h"
#include "font.h"
#include "Dimension.h"
#include "MsgBox.h"
#include "sagcad.h"

#define _SGY_
#include "sgy.h"





/* -------------------------------------------------------------------
 *	
 *	
 * Ver 0 : Visual Basic 版
 * Ver 1 : Visual C 版の安定板
 * Ver 2 : Visual C 版の開発板
 * Ver 3 : Visual C 版の開発板
 */
int sgyFileSave(char *SaveFileName)
{
	FILE *stream;
	char FileName[256];
//	char str[256];
//	int Ret;

	/* ファイルを一時的に「.tmp」で作成 */
	sprintf(FileName, "%s", SaveFileName);
	getFileName(FileName);
	cutExtName(FileName);
	strcat(FileName, ".sgy");

	/* File Open */
	if ((stream = fopen( FileName, "w")) == NULL) {
		g_print( _("The file could not be opened. [%s]\n") , FileName);
		return 0;
	}


	/* WaitBox 作成 */
//	hWaitBox = CreateDialog( (HINSTANCE)GetWindowLong(MainWindow, GWL_HINSTANCE), 
//							MAKEINTRESOURCE(IDD_WAIT_BOX_SAVE), 
//							MainWindow, 
//							(DLGPROC) WaitBoxProc);
//	SetWinCenter(hWaitBox);
//	EnableWindow(MainWindow, FALSE);
//	ShowWindow(hWaitBox, SW_SHOW);


	/* TABLES SECTION */
	tables_write(stream);

	/* TRACE SECTION */
	trace_write(stream);

	/* DIMENSION SECTION */
	dimension_write(stream);

	/* ELLIPSE SECTION */
	ellipse_write(stream);

	/* POLYLINE SECTION */
	PolyLine_write(stream);

	/* EOF SECTION */
	eof_write(stream);

	/* File Close */
	fclose(stream);


//	Sleep(500);
//	DestroyWindow(hWaitBox);
//	hWaitBox = NULL;
//	EnableWindow(MainWindow, TRUE);
//	SetFocus(MainWindow);

#ifdef ONE_SHOT_DEBUG
//	OneShotLog("sgyFileSave out\n");
#endif


	return 1;
}



/* -------------------------------------------------------------------
 * tables_write
 *	
 *	
 *	
 */
int tables_write(FILE *stream)
{
	char str[256], c = '\0', str_m[20][256], str_layer[256];
	int i;

	sprintf(str, "TABLES\n");
	write_locale_from_utf8(str, stream, data_version);


	/* Data Version 5 */
	sprintf(str, "Ver\n");
	write_locale_from_utf8(str, stream, data_version);
	sprintf(str, " %d.0\n", data_version);
	write_locale_from_utf8(str, stream, data_version);

	/* < CompanyName > */
	sprintf(str, "CompanyName\n");
	write_locale_from_utf8(str, stream, data_version);
	sprintf(str, " %s\n", sagcad_user.CompanyName);
	write_locale_from_utf8(str, stream, data_version);
	
	/* < UserName > */
	sprintf(str, "UserName\n");
	write_locale_from_utf8(str, stream, data_version);
	sprintf(str, " %s\n", sagcad_user.UserName);
	write_locale_from_utf8(str, stream, data_version);

	/* Mag,MagX,MagY */
	sprintf(str, "Mag,MagX,MagY\n");
	write_locale_from_utf8(str, stream, data_version);
	FloatOut6(str_m[0], Mag, 0);
	FloatOut6(str_m[1], MagX, 0);
	FloatOut6(str_m[2], MagY, 0);
	sprintf(str, " %s,%s,%s\n", str_m[0], str_m[1], str_m[2]);
	write_locale_from_utf8(str, stream, data_version);

	/* View */
	sprintf(str, "View\n");
	write_locale_from_utf8(str, stream, data_version);
	FloatOut6(str_m[0],  GamenMag[0], 0);
	FloatOut6(str_m[1],  GamenMagX[0], 0);
	FloatOut6(str_m[2],  GamenMagY[0], 0);
	FloatOut6(str_m[3],  GamenMag[1], 0);
	FloatOut6(str_m[4],  GamenMagX[1], 0);
	FloatOut6(str_m[5],  GamenMagY[1], 0);
	FloatOut6(str_m[6],  GamenMag[2], 0);
	FloatOut6(str_m[7],  GamenMagX[2], 0);
	FloatOut6(str_m[8],  GamenMagY[2], 0);
	FloatOut6(str_m[9],  GamenMag[3], 0);
	FloatOut6(str_m[10], GamenMagX[3], 0);
	FloatOut6(str_m[11], GamenMagY[3], 0);
	FloatOut6(str_m[12], GamenMag[4], 0);
	FloatOut6(str_m[13], GamenMagX[4], 0);
	FloatOut6(str_m[14], GamenMagY[4], 0);
	FloatOut6(str_m[15], GamenMag[5], 0);
	FloatOut6(str_m[16], GamenMagX[5], 0);
	FloatOut6(str_m[17], GamenMagY[5], 0);
	sprintf(str,	" %s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n", 
					str_m[0], str_m[1], str_m[2], str_m[3], str_m[4], str_m[5], str_m[6], str_m[7], str_m[8], 
					str_m[9], str_m[10], str_m[11], str_m[12], str_m[13], str_m[14], str_m[15], str_m[16], str_m[17]);
	write_locale_from_utf8(str, stream, data_version);

	/* Now Color,Style,Layer */
	sprintf(str, "Now Color,Style,Layer\n");
	write_locale_from_utf8(str, stream, data_version);
	sprintf(str, " 0x%x,%d,%d\n", (int)NowColor, NowStyle, NowLayer);
	write_locale_from_utf8(str, stream, data_version);

	/* PRINT INFO */
	sprintf(str, "PRINT INFO\n");
	write_locale_from_utf8(str, stream, data_version);
	sprintf(str,	" %d,%d,%f,%f,%d,%d\n", 
					printer.Paper, 
					printer.Orientation, 
					printer.up_scale, 
					printer.low_scale, 
					printer.Draw_View, 
					printer.WakuLayer);
	write_locale_from_utf8(str, stream, data_version);

	/* Layer 0 -> 63 */
	sprintf(str, "LAYER\n");
	write_locale_from_utf8(str, stream, data_version);
	for (i = 0 ; i < 64 ; i++) {
		if (Layer[i].draw == 0)	c = '0';
		else if (Layer[i].draw == 1)	{
			if (NowLayer == i)	c = '5';
			else	c = '1';
		}
		str_layer[i] = c;
	}
	str_layer[64] = '\n';
	str_layer[65] = '\0';
	write_locale_from_utf8(str_layer, stream, data_version);

	/* Layer 64 -> 127 */
	for (i = 64 ; i < 128 ; i++) {
		if (Layer[i].draw == 0) c = '0';
		else if (Layer[i].draw == 1) {
			if (NowLayer == i)	c = '5';
			else	c = '1';
		}
		str_layer[i-64] = c;
	}
	str_layer[64] = '\n';
	str_layer[65] = '\0';
	write_locale_from_utf8(str_layer, stream, data_version);

	/* Layer 128 -> 191 */
	for (i = 128 ; i < 192 ; i++) {
		if (Layer[i].draw == 0)	c = '0';
		else if (Layer[i].draw == 1) {
			if (NowLayer == i)	c = '5';
			else	c = '1';
		}
		str_layer[i-128] = c;
	}
	str_layer[64] = '\n';
	str_layer[65] = '\0';
	write_locale_from_utf8(str_layer, stream, data_version);

	/* Layer 192 -> 255 */
	for (i = 192 ; i < 256 ; i++) {
		if (Layer[i].draw == 0)	c = '0';
		else if (Layer[i].draw == 1) {
			if (NowLayer == i)	c = '5';
			else	c = '1';
		}
		str_layer[i-192] = c;
	}
	str_layer[64] = '\n';
	str_layer[65] = '\0';
	write_locale_from_utf8(str_layer, stream, data_version);

	sprintf(str, "TABLES END\n");
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}





/* -------------------------------------------------------------------
 * trace_write
 *	
 */
int trace_write(FILE *stream)
{
	CAD_LIST *p;
	char str[256];


	sprintf(str, "TRACE (Layer, Code, Style, Color, SX, SY, EX, EY, CX, CY, R, stock)\n");
	write_locale_from_utf8(str, stream, data_version);

	/* 各図形要素 */
	p = cad_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		if (trace_data_write(stream, p) == 0) return 0;
		p = p->next;	/* ポインタを次のデータに移す */
	}

	sprintf(str, "TRACE END\n");
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}





/* -------------------------------------------------------------------
 * trace_data_write
 *	
 */
int trace_data_write(FILE *stream, CAD_LIST *p)
{
	char str[256];
	char SX[20], SY[20], EX[20], EY[20], CX[20], CY[20], R[20];


	if (p == NULL || p->cad == NULL) return 0;

	FloatOut_n(SX, p->cad->sx, sagcad_digits.data_digits, 0);
	FloatOut_n(SY, p->cad->sy, sagcad_digits.data_digits, 0);
	FloatOut_n(EX, p->cad->ex, sagcad_digits.data_digits, 0);
	FloatOut_n(EY, p->cad->ey, sagcad_digits.data_digits, 0);
	FloatOut_n(CX, p->cad->cx, sagcad_digits.data_digits, 0);
	FloatOut_n(CY, p->cad->cy, sagcad_digits.data_digits, 0);
	FloatOut_n( R, p->cad->r , sagcad_digits.data_digits, 0);

	sprintf(str, "%d,%d,%d,0x%x,%s,%s,%s,%s,%s,%s,%s,0\n",
				p->cad->layer, p->cad->code, p->cad->style, (int)p->cad->color,
				SX, SY, EX, EY, CX, CY, R);
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}





/* -------------------------------------------------------------------
 * dimension_write
 *	
 */
int dimension_write(FILE *stream)
{
	DIMENSION_LIST *p;
	char str[256];


	sprintf(str, "DIMENSION\n");
	write_locale_from_utf8(str, stream, data_version);

	/* Version 4 Type */
	/* 各図形要素 */
	p = dimension_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		if (dimension_data_write(stream, p) == 0) return 0;
		p = p->next;	/* ポインタを次のデータに移す */
	}
	sprintf(str, "DIMENSION END\n");
	write_locale_from_utf8(str, stream, data_version);
	return 1;
}





/* -------------------------------------------------------------------
 * dimension_data_write
 *	
 */
int dimension_data_write(FILE *stream, DIMENSION_LIST *p)
{
	char str[256];
	char FontHeight[20], FontSpace[20], Angle[20], SearchPointX[20], SearchPointY[20];
	char SX[20], SY[20], EX[20], EY[20], CX[20], CY[20], R[20];
	int i;


	if (p == NULL || p->dimension == NULL) return 0;

	FloatOut_n(FontHeight, p->dimension->FontHeight, sagcad_digits.data_digits, 0);
	FloatOut_n(FontSpace, p->dimension->FontSpace, sagcad_digits.data_digits, 0);
	FloatOut_n(Angle, p->dimension->Angle, sagcad_digits.data_digits, 0);
	FloatOut_n(SearchPointX, p->dimension->SearchPointX, sagcad_digits.data_digits, 0);
	FloatOut_n(SearchPointY, p->dimension->SearchPointY, sagcad_digits.data_digits, 0);

	/*			  L  DT   T 	 T		T	   T	FT FH FS A	X  Y  P  I*/
	sprintf(str, "%d,%d,%d,\"%s\",\"%s\",\"%s\",%d,%s,%s,%s,%s,%s,%d,%d\n",
			p->dimension->Layer, p->dimension->DrawType,  p->dimension->Type, 
			p->dimension->Text, p->dimension->UpperText, p->dimension->LowerText, 
			p->dimension->FontType, FontHeight, FontSpace, 
			Angle, SearchPointX, SearchPointY, p->dimension->StartPoint, p->dimension->index);
	write_locale_from_utf8(str, stream, data_version);


	if (p->dimension->index != 0 && p->dimension->AssistLine != NULL) {
		for ( i = 0 ; i < p->dimension->index ; i++ ) {
			FloatOut_n(SX, p->dimension->AssistLine[i].sx, sagcad_digits.data_digits, 0);
			FloatOut_n(SY, p->dimension->AssistLine[i].sy, sagcad_digits.data_digits, 0);
			FloatOut_n(EX, p->dimension->AssistLine[i].ex, sagcad_digits.data_digits, 0);
			FloatOut_n(EY, p->dimension->AssistLine[i].ey, sagcad_digits.data_digits, 0);
			FloatOut_n(CX, p->dimension->AssistLine[i].cx, sagcad_digits.data_digits, 0);
			FloatOut_n(CY, p->dimension->AssistLine[i].cy, sagcad_digits.data_digits, 0);
			FloatOut_n( R, p->dimension->AssistLine[i].r , sagcad_digits.data_digits, 0);
			sprintf (str, "  %d,%s,%s,%s,%s,%s,%s,%s\n", p->dimension->AssistLine[i].defin, SX, SY, EX, EY, CX, CY, R);
			write_locale_from_utf8(str, stream, data_version);
		}
	}

	return 1;
}





//#ifdef TEST
/* -------------------------------------------------------------------
 * ellipse_write
 *	
 */
int ellipse_write(FILE *stream)
{
	ELLIPSE_LIST *p;
	char str[256];


	sprintf(str, "ELLIPSE (Layer, Code, Style, Color, CX, CY, DX, DY, K, SA, EA)\n");
	write_locale_from_utf8(str, stream, data_version);

	/* 各図形要素 */
	p = ellipse_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while (p != NULL) {
		if (ellipse_data_write(stream, p) == 0) return 0;
		p = p->next;	/* ポインタを次のデータに移す */
	}

	sprintf(str, "ELLIPSE END\n");
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}





/* -------------------------------------------------------------------
 * ellipse_data_write
 *	
 */
int ellipse_data_write(FILE *stream, ELLIPSE_LIST *p)
{
	char str[256];
	char CX[20], CY[20], DX[20], DY[20], K[20], SA[20], EA[20];


	if (p == NULL || p->ellipse == NULL) return 0;

	FloatOut_n(CX, p->ellipse->cx, sagcad_digits.data_digits, 0);
	FloatOut_n(CY, p->ellipse->cy, sagcad_digits.data_digits, 0);
	FloatOut_n(DX, p->ellipse->dx, sagcad_digits.data_digits, 0);
	FloatOut_n(DY, p->ellipse->dy, sagcad_digits.data_digits, 0);
	FloatOut_n(K,  p->ellipse->k,  sagcad_digits.data_digits, 0);
	FloatOut_n(SA, p->ellipse->sa, sagcad_digits.data_digits, 0);
	FloatOut_n(EA, p->ellipse->ea , sagcad_digits.data_digits, 0);

	sprintf(str, "%d,%d,%d,0x%x,%s,%s,%s,%s,%s,%s,%s\n",
				p->ellipse->layer, p->ellipse->code, p->ellipse->style, (int)p->ellipse->color,
				CX, CY, DX, DY, K, SA, EA);
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}
//#endif





/* -------------------------------------------------------------------
 * PolyLine_write
 *	
 */
int PolyLine_write(FILE *stream)
{
	POLYLINE_LIST *p;
	char str[256];


	sprintf(str, "POLYLINE\n");
	write_locale_from_utf8(str, stream, data_version);

	/* 各図形要素 */
	p = polyline_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		if (PolyLine_data_write(stream, p) == 0) return 0;
		p = p->next;	/* ポインタを次のデータに移す */
	}

	sprintf(str, "POLYLINE END\n");
	write_locale_from_utf8(str, stream, data_version);

	return 1;
}





/* -------------------------------------------------------------------
 * PolyLine_data_write
 *	
 */
int PolyLine_data_write(FILE *stream, POLYLINE_LIST *p)
{
	char PITCH[20], X[20], Y[20];
	char str[256];
	VERTEX_LIST *pv;


	if (p == NULL || p->polyline->vertex_list_info.head == NULL) return 0;


	FloatOut_n(PITCH, p->polyline->pitch, sagcad_digits.data_digits, 0);
	sprintf(str, "%d,%d,%d,0x%x,%d,%s,%d,%d\n",
				p->polyline->layer, p->polyline->code, p->polyline->style, (int)(p->polyline->color),
				p->polyline->split, PITCH, p->polyline->frag, (int)vertex_list_num(&p->polyline->vertex_list_info));
	write_locale_from_utf8(str, stream, data_version);



	pv = p->polyline->vertex_list_info.head;
	while (pv != NULL) {
		FloatOut_n(X, pv->vertex->x, sagcad_digits.data_digits, 0);
		FloatOut_n(Y, pv->vertex->y, sagcad_digits.data_digits, 0);
		sprintf(str, "  %s,%s\n", X, Y);
		write_locale_from_utf8(str, stream, data_version);

		pv = pv->next;
	}


	return 1;
}





/* -------------------------------------------------------------------
 * eof_write
 *	
 */
int eof_write(FILE *stream)
{
	char str[256];

	/* EOF SECTION */
	sprintf(str, "EOF\n");
	write_locale_from_utf8(str, stream, data_version);
	return 1;
}










/* -------------------------------------------------------------------
 * データ読み込み
 *	
 */
int sgyFileOpen(char *OpenFileName)
{
	int Version = 0;
	char str[256], msg[256];
	FILE *stream;

	//g_print("open sgy file [%s] !\n", g_locale_to_utf8(OpenFileName, -1, NULL, NULL, NULL));

	/* 図面ディレクトリ名 */
	sprintf(sagcad_user.CadPath, "%s", OpenFileName);
	cutFileName(sagcad_user.CadPath);

	/* ---------------------------------
	 * ファイルが壊れていないかを確認
	 */ 
//	if (FileCheck(OpenFileName) == 0) {
//		g_print("sgy file broken !\n");
//		return 0;
//	}

	/* ---------------------------------
	 * ファイルをオープン
	 */ 
	if ( (stream = fopen( OpenFileName , "r" )) == NULL ){
		sprintf(str,  _("The file could not be opened. [%s]\n") , OpenFileName);
		g_print("%s", str);
	}

	cad_list_all_free(&cad_list_info);
	dimension_list_all_free(&dimension_list_info);
	polyline_list_all_free(&polyline_list_info);

	while(1){
		if (read_utf8_from_locale(stream, str, CHARSET_LOCALE) == 0) {
			fclose(stream);
			OneShotLog("FILE Data Error\n");
			g_print("FILE Data Error\n");
			return 0;
		}

		/* -------------------------------------------------
		 * TABLES SECTION
		 */
		if (strstr(str, "TABLES") != NULL ) {
			sprintf(msg, "Reading SGY       ■■□□□□□    [TABLES SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			Version = sgyTableSectionRead(stream);
			if (Version < 0 || Version > 5) {
				fclose(stream);
				OneShotLog("TABLES Data Error\n");
				return 0;
			}
			g_print("Data Version = %d.0 !\n", Version);
		}


		/* -------------------------------------------------
		 * TRACE SECTION
		 */
		else if (strstr(str, "TRACE") != NULL ) {
			sprintf(msg, "Reading SGY       ■■■□□□□    [TRACE SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			if ( sgyTraceSectionRead(stream, Version) == 0 ) {
				fclose(stream);
				OneShotLog("CAD Data Error\n");
				return 0;
			}
		}


		/* -------------------------------------------------
		 * DEMENSION SECTION
		 */
		else if (strstr(str, "DEMENSION") != NULL || strstr(str,"DIMENSION") != NULL) {
			sprintf(msg, "Reading SGY       ■■■■□□□    [DIMENSION SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			if ( sgyDimensionSectionRead(stream, Version) == 0 ) {
				fclose(stream);
				OneShotLog("DIMENSION Data Error\n");
				return 0;
			}
		}


		/* -------------------------------------------------
		 * ELLIPSE SECTION
		 */
		else if (strstr(str, "ELLIPSE") == str) {
			sprintf(msg, "Reading SGY       ■■■■■□□    [ELLIPSE SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			if ( sgyEllipseSectionRead(stream, Version) == 0 ) {
				fclose(stream);
				OneShotLog("ELLIPSE Data Error\n");
				return 0;
			}
		}


		/* -------------------------------------------------
		 * POLYLINE SECTION
		 */
		else if (strstr(str, "POLYLINE") == str) {
			sprintf(msg, "Reading SGY       ■■■■■□□    [POLYLINE SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			if ( sgyPolyLineSectionRead(stream, Version) == 0 ) {
				fclose(stream);
				OneShotLog("POLYLINE Data Error\n");
				return 0;
			}
		}


		/* -------------------------------------------------
		 * EOF SECTION
		 */
		else if (strstr(str, "EOF") != NULL ) {
			sprintf(msg, "Reading SGY       ■■■■■■□    [EOF SECTION]");
			StatusMsg(msg);
			UpDate_widget(MainWindow);
			sprintf(msg, "Reading Font      ■■■■■■■");
			StatusMsg(msg);
			UpDate_widget(MainWindow);

			fclose(stream);
			ModifiedFlag = 0;

			if (Mag == 0) Mag = 1;
			if (GamenMag[0] == 0) GamenMag[0] = 1;
			if (GamenMag[1] == 0) GamenMag[1] = 1;
			if (GamenMag[2] == 0) GamenMag[2] = 1;
			if (GamenMag[3] == 0) GamenMag[3] = 1;
			if (GamenMag[4] == 0) GamenMag[4] = 1;
			if (GamenMag[5] == 0) GamenMag[5] = 1;
			return 1;
		}
	}

//	dimension_list_print();

	return 0;
}





/* -------------------------------------------------------------------
 * sgy Table Section Read
 *	
 * Version 情報を返す
 * エラーの時はマイナス値を返す
 */
int sgyTableSectionRead(FILE *stream)
{
	char str[256], ret[256], *ptr;
	int Ver = 5, j;

	while(1) {
		if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;

		/* TABLES SECTION END */
		if (strstr(str, "TABLES END") != NULL ) return Ver;

		/* Data Version get */
		else if (strstr(str,"Ver") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			if (strstr(str,"1.0") != NULL ) 
				Ver = 1;
			else if (strstr(str,"2.0") != NULL ) 
				Ver = 2;
			else if (strstr(str,"3.0") != NULL ) 
				Ver = 3;
			else if (strstr(str,"4.0") != NULL ) 
				Ver = 4;
			else if (strstr(str,"5.0") != NULL ) 
				Ver = 5;
			else Ver = 0;
		}

		/* Mag,MagX,MagY */
		else if (strstr(str,"Mag,MagX,MagY") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			LineDataSplit(str, ',', 1, ret);
			SetMag((float) atof(ret));
			LineDataSplit(str, ',', 2, ret);
			MagX = (float) atof(ret);
			LineDataSplit(str, ',', 3, ret);
			MagY = (float) atof(ret);
		}

		/* View */
		else if (strstr(str,"View") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			/* View 0 */
			LineDataSplit(str, ',', 1, ret);
			GamenMag[0] = (float) atof(ret);
			LineDataSplit(str, ',', 2, ret);
			GamenMagX[0] = (float) atof(ret);
			LineDataSplit(str, ',', 3, ret);
			GamenMagY[0] = (float) atof(ret);
			/* View 1 */
			LineDataSplit(str, ',', 4, ret);
			GamenMag[1] = (float) atof(ret);
			LineDataSplit(str, ',', 5, ret);
			GamenMagX[1] = (float) atof(ret);
			LineDataSplit(str, ',', 6, ret);
			GamenMagY[1] = (float) atof(ret);
			/* View 2 */
			LineDataSplit(str, ',', 7, ret);
			GamenMag[2] = (float) atof(ret);
			LineDataSplit(str, ',', 8, ret);
			GamenMagX[2] = (float) atof(ret);
			LineDataSplit(str, ',', 9, ret);
			GamenMagY[2] = (float) atof(ret);
			/* View 3 */
			LineDataSplit(str, ',', 10, ret);
			GamenMag[3] = (float) atof(ret);
			LineDataSplit(str, ',', 11, ret);
			GamenMagX[3] = (float) atof(ret);
			LineDataSplit(str, ',', 12, ret);
			GamenMagY[3] = (float) atof(ret);
			/* View 4 */
			LineDataSplit(str, ',', 13, ret);
			GamenMag[4] = (float) atof(ret);
			LineDataSplit(str, ',', 14, ret);
			GamenMagX[4] = (float) atof(ret);
			LineDataSplit(str, ',', 15, ret);
			GamenMagY[4] = (float) atof(ret);
			/* View 5 */
			LineDataSplit(str, ',', 16, ret);
			GamenMag[5] = (float) atof(ret);
			LineDataSplit(str, ',', 17, ret);
			GamenMagX[5] = (float) atof(ret);
			LineDataSplit(str, ',', 18, ret);
			GamenMagY[5] = (float) atof(ret);
		}

		/* Now Color,Style,Layer */
		else if (strstr(str,"Now Color,Style,Layer") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			LineDataSplit(str, ',', 1, ret);
			NowColor = atoi16(ret);
			LineDataSplit(str, ',', 2, ret);
			NowStyle = atoi(ret);
			LineDataSplit(str, ',', 3, ret);
			NowLayer = atoi(ret);

			sprintf(str,"Layer %d",NowLayer);
			//SetWindowText(layer, str);
			set_label_text(layer, str);
		}



		/* View0_M */
		else if (strstr(str,"View0_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[0] = (float) atof(str);
		}
		else if (strstr(str,"View0_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[0] = (float) atof(str);
		}
		else if (strstr(str,"View0_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[0] = (float) atof(str);
		}
		/* View1_M */
		else if (strstr(str,"View1_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[1] = (float) atof(str);
		}
		else if (strstr(str,"View1_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[1] = (float) atof(str);
		}
		else if (strstr(str,"View1_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[1] = (float) atof(str);
		}
		/* View2_M */
		else if (strstr(str,"View2_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[2] = (float) atof(str);
		}
		else if (strstr(str,"View2_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[2] = (float) atof(str);
		}
		else if (strstr(str,"View2_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[2] = (float) atof(str);
		}
		/* View3_M */
		else if (strstr(str,"View3_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[3] = (float) atof(str);
		}
		else if (strstr(str,"View3_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[3] = (float) atof(str);
		}
		else if (strstr(str,"View3_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[3] = (float) atof(str);
		}
		/* View4_M */
		else if (strstr(str,"View4_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[4] = (float) atof(str);
		}
		else if (strstr(str,"View4_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[4] = (float) atof(str);
		}
		else if (strstr(str,"View4_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[4] = (float) atof(str);
		}
		/* View5_M */
		else if (strstr(str,"View5_M") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMag[5] = (float) atof(str);
		}
		else if (strstr(str,"View5_X") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagX[5] = (float) atof(str);
		}
		else if (strstr(str,"View5_Y") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			GamenMagY[5] = (float) atof(str);
		}

		/* MAG */
		else if (strstr(str,"MAG") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			SetMag((float) atof(str));
		}
		else if (strstr(str,"MagX") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			MagX = (float) atof(str);
		}
		else if (strstr(str,"MagY") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			MagY = (float) atof(str);
		}

		/* カレントカラー取得 */
		else if (strstr(str,"NOW COLOR") != NULL ) {
			/* Ver 1 or 2 */
			if (Ver == 1 || Ver == 2) {
				if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
				NowColor = (long) atoi16(str);
			}
			/* Ver 3 */
			else if (Ver == 3) {
				if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
				NowColor = (long) atoi16(str);
			}
			/* Ver 0 */
			else {
				if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
				NowColor = QBtoRGB(atoi(str)); /* QB Color */
			}
		}

		/* カレント線種取得 */
		else if (strstr(str,"NOW STYLE") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			NowStyle = atoi(str);
		}

		/* カレントレイヤー取得 */
		else if (strstr(str,"NOW LAYER") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			NowLayer = atoi(str);
			sprintf(str,"Layer %d",NowLayer);
			set_label_text(layer, str);
		}



		/* PRINT INFO */
		else if (strstr(str,"PRINT INFO") != NULL ) {
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			LineDataSplit(str, ',', 1, ret);
			printer.Paper = atoi(ret);
			if (printer.Paper == 9) {
				printer.Paper = A4_PAPER;
			}
			LineDataSplit(str, ',', 2, ret);
			printer.Orientation = atoi(ret);
			LineDataSplit(str, ',', 3, ret);
			printer.up_scale = (float) atof(ret);
			LineDataSplit(str, ',', 4, ret);
			printer.low_scale = (float) atof(ret);
			printer.scale = printer.up_scale / printer.low_scale;
			LineDataSplit(str, ',', 5, ret);
			printer.Draw_View = atoi(ret);
			LineDataSplit(str, ',', 6, ret);
			printer.WakuLayer = atoi(ret);
		}



		/* Layer Setting get */
		else if (strstr(str,"LAYER") != NULL ) {
			/* 0-63 */
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			ptr = str;
			for (j = 0 ; j < 64 ; j++) {
				if (*ptr == '0') Layer[j].draw = 0;
				if (*ptr == '1') Layer[j].draw = 1;
				if (*ptr == '5') {
					Layer[j].draw = 1;
				}
				ptr++;
			}
			/* 64-127 */
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			ptr = str;
			for (j = 64 ; j < 128 ; j++) {
				if (*ptr == '0') Layer[j].draw = 0;
				if (*ptr == '1') Layer[j].draw = 1;
				if (*ptr == '5') {
					Layer[j].draw = 1;
				}
				ptr++;
			}
			/* 128-191 */
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			ptr = str;
			for (j = 128 ; j < 192 ; j++) {
				if (*ptr == '0') Layer[j].draw = 0;
				if (*ptr == '1') Layer[j].draw = 1;
				if (*ptr == '5') {
					Layer[j].draw = 1;
				}
				ptr++;
			}
			/* 192-255 */
			if (read_utf8_from_locale(stream, str, Ver) == 0) return -1;
			ptr = str;
			for (j = 192 ; j < 256 ; j++) {
				if (*ptr == '0') Layer[j].draw = 0;
				if (*ptr == '1') Layer[j].draw = 1;
				if (*ptr == '5') {
					Layer[j].draw = 1;
				}
				ptr++;
			}
		}
	}
	return -1;
}




/* -------------------------------------------------------------------
 * sgy Trace Section Read
 *	
 */
int sgyTraceSectionRead(FILE *stream, int Version)
{
	char str[256];
	CAD cad;

	while(1) {
		if (read_utf8_from_locale(stream, str, Version) == 0) return 0;

		if (strstr(str,"TRACE END") != NULL ) return 1;

		else {
			if (TraceLook(str, &cad, Version) == 0) {
				//MsgBox("ERROR", "File Read Error.  (Trace Data)");
				return 0;
			}
			/* CAD Data をリストの最初に追加 */
			cad_list_add_first(&cad, &cad_list_info);
		}
	}
	return 0;
}




/* -------------------------------------------------------------------
 * sgy Dimension Section Read
 *	
 */
int sgyDimensionSectionRead(FILE *stream, int Version)
{
	double scale;
	char str[256];
	DIMENSION dimension;
	DIMENSION_LIST *p;

	while(1) {
		if (read_utf8_from_locale(stream, str, Version) == 0) return 0;

		if (strstr(str, "DEMENSION END") != NULL || strstr(str, "DIMENSION END") != NULL) return 1;

		else {
			if (DimensionLook(str, &dimension, stream, Version) == 0) {
				//MsgBox("ERROR", "File Read Error.  (Dimension Data)");
				return 0;
			}
			if (dimension.StartPoint > 10) {
				dimension.FitPointY = 4;
				dimension.FitPointX = dimension.StartPoint - 10;
			}
			else {
				dimension.FitPointY = 1;
				dimension.FitPointX = dimension.StartPoint;
			}
			dimension.text_color = sagcad_color.Dimension;
			dimension.Assist_color = sagcad_color.Dimension;
			scale = dimension.FontHeight / 9;
			dimension.FontSpace = sagcad_dimension.Font_space * scale;
			dimension.WordSpace = sagcad_dimension.Word_space * scale;
			dimension.FontOverLineSpace = sagcad_dimension.Font_space_line * scale;
			
			p = dimension_list_add_first(&dimension, &dimension_list_info);
			//CulcTextVector (drawing_area, p->dimension);
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * sgy Ellipse Section Read
 *	
 */
int sgyEllipseSectionRead(FILE *stream, int Version)
{
	char str[256];
	ELLIPSE ellipse;

	while(1) {
		if (read_utf8_from_locale(stream, str, Version) == 0) return 0;

		if (strstr(str,"ELLIPSE END") != NULL ) return 1;

		else {
			if (EllipseLook(str, &ellipse, Version) == 0) {
				//MsgBox("ERROR", "File Read Error.  (Trace Data)");
				return 0;
			}
			/* CAD Data をリストの最初に追加 */
			ellipse_list_add_first(&ellipse, &ellipse_list_info);
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * ELLIPSE データ詳細読み込み
 * 
 */
int EllipseLook(char *str, ELLIPSE *a, int Version)
{
	char ret[255];

	LineDataSplit(str, ',', 1, ret);
	a->layer = atoi(ret);
	LineDataSplit(str, ',', 2, ret);
	a->code = atoi(ret);
	LineDataSplit(str, ',', 3, ret);
	a->style = atoi(ret);

	LineDataSplit(str, ',', 4, ret);
	if (Version == 1 || Version == 2) a->color = (long)atoi16(ret);
	else if (Version == 0) a->color = QBtoRGB(atoi16(ret));
	else if (Version == 3) a->color = (long) atoi16(ret);
	else if (Version == 4) a->color = (long) atoi16(ret);

	LineDataSplit(str, ',', 5, ret);
	a->cx = atof(ret);
	LineDataSplit(str, ',', 6, ret);
	a->cy = atof(ret);
	LineDataSplit(str, ',', 7, ret);
	a->dx = atof(ret);
	LineDataSplit(str, ',', 8, ret);
	a->dy = atof(ret);
	LineDataSplit(str, ',', 9, ret);
	a->k  = atof(ret);
	LineDataSplit(str, ',', 10, ret);
	a->sa = atof(ret);
	LineDataSplit(str, ',', 11, ret);
	a->ea = atof(ret);

	return 1;
}





/* -------------------------------------------------------------------
 * sgy PolyLine Section Read
 *	
 */
int sgyPolyLineSectionRead(FILE *stream, int Version)
{
	char str[256];
	POLYLINE PolyLine;

	while(1) {
		if (read_utf8_from_locale(stream, str, Version) == 0) return 0;

		if (strstr(str,"POLYLINE END") != NULL ) return 1;

		else {
			init_polyline(&PolyLine);
			if (PolyLineLook(str, &PolyLine, stream, Version) == 0) {
				//MsgBox("ERROR", "File Read Error.  (PolyLine Data)");
				return 0;
			}
			/* POLYLINE Data をリストの最初に追加 */
			polyline_list_add_first(&PolyLine, &polyline_list_info);
		}
	}
	return 0;
}




/* -------------------------------------------------------------------
 * PolyLine Data Version 4
 *	
 */
int PolyLineLook(char *str, POLYLINE *a, FILE *stream, int Version)
{
	char ret[256];
	int i;
	VERTEX vertex;


	LineDataSplit(str, ',', 1, ret);
	a->layer = atoi(ret);

	LineDataSplit(str, ',', 2, ret);
	a->code = atoi(ret);

	LineDataSplit(str, ',', 3, ret);
	a->style = atoi(ret);

	LineDataSplit(str, ',', 4, ret);
	a->color = (long) atoi16(ret);

	LineDataSplit(str, ',', 5, ret);
	a->split = atoi(ret);

	LineDataSplit(str, ',', 6, ret);
	a->pitch = atof(ret);

	LineDataSplit(str, ',', 7, ret);
	a->frag = atof(ret);

	LineDataSplit(str, ',', 8, ret);
	a->index = atoi(ret);


	a->vertex_list_info.head = NULL;
	a->vertex_list_info.tail = NULL;
	/* VERTEX Read */
	if (a->index == 0) return 1;
	for ( i= 0 ; i < a->index ; i++ ) {
		if (read_utf8_from_locale(stream, str, Version) == 0) return 0;

		LineDataSplit(str, ',', 1, ret);
		vertex.x = atof(ret);

		LineDataSplit(str, ',', 2, ret);
		vertex.y = atof(ret);

		vertex_list_add_last(&vertex, &a->vertex_list_info);
	}
	return 1;
}





/* -------------------------------------------------------------------
 * 図形データ詳細読み込み
 *	
 * Data Version 0 : VisualBasic 版 SagCAD カラーデータが、QB Color を使っていた。
 * Data Version 1 & 2 は、共通
 *	
 */
int TraceLook(char *str, CAD *a, int Version)
{
	char ret[255];

	LineDataSplit(str, ',', 1, ret);
	a->layer = atoi(ret);
	LineDataSplit(str, ',', 2, ret);
	a->code = atoi(ret);
	LineDataSplit(str, ',', 3, ret);
	a->style = atoi(ret);

	LineDataSplit(str, ',', 4, ret);
	if (Version == 1 || Version == 2) a->color = (long)atoi16(ret);
	else if (Version == 0) a->color = QBtoRGB(atoi16(ret));
	else if (Version == 3) a->color = (long) atoi16(ret);
	else if (Version == 4) a->color = (long) atoi16(ret);
	else if (Version == 5) a->color = (long) atoi16(ret);

	LineDataSplit(str, ',', 5, ret);
	a->sx = atof(ret);
	LineDataSplit(str, ',', 6, ret);
	a->sy = atof(ret);
	LineDataSplit(str, ',', 7, ret);
	a->ex = atof(ret);
	LineDataSplit(str, ',', 8, ret);
	a->ey = atof(ret);
	LineDataSplit(str, ',', 9, ret);
	a->cx = atof(ret);
	LineDataSplit(str, ',', 10, ret);
	a->cy = atof(ret);
	LineDataSplit(str, ',', 11, ret);
	a->r = atof(ret);
	LineDataSplit(str, ',', 12, ret);
	a->OnScreen = atoi(ret);


	if (a->code == 0) {
		a->ex = 0; a->ey = 0; a->cx = 0; a->cy = 0; a->r  = 0;
	}
	else if (a->code == 1) {
		a->cx = 0; a->cy = 0; a->r	= 0;
	}
	else if (a->code == 4) {
		a->sx = 0; a->sy = 0; a->ex = 0; a->ey = 0;
	}
	return 1;
}





/* -------------------------------------------------------------------
 * Version 1 用のカラーデータ変換関数
 *	
 */
long QBtoRGB(int QB_color)
{
	switch (QB_color) {
		case 0: return 0; break;
		case 1: return 0x800000; break;
		case 2: return 0x008000; break;
		case 3: return 0x808000; break;
		case 4: return 0x0000c0; break;
		case 5: return 0xff00ff; break;
		case 6: return 0x00c0c0; break;
		case 7: return 0xe0e0e0; break;
		case 8: return 0x808080; break;
		case 9: return 0xff0000; break;
		case 10: return 0x00ff00; break;
		case 11: return 0xffff00; break;
		case 12: return 0x0000ff; break;
		case 13: return 0xff00ff; break;
		case 14: return 0x00ffff; break;
		case 15: return 0xffffff; break;
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 寸法データ詳細読み込み
 *	
 * Data Version 0 & 1 は、共通
 * Data Version 2 は、必要なデータだけを記録し、読みこみ時に計算する。
 *	
 */
int DimensionLook(char *str, DIMENSION *a, FILE *stream, int Version)
{
	/* Dimension Data Version 1 */
	if (Version == 1 || Version == 0)
		DimensionVer1(str, a);

	/* Dimension Data Version 2 */
	else if (Version == 2)		
		DimensionVer2(str, a);

	/* Dimension Data Version 3 */
	else if (Version == 3)
		DimensionVer3(str, a);

	/* Dimension Data Version 4 */
	else if (Version == 4)
		DimensionVer4(str, a, stream);

	/* Dimension Data Version 5 */
	else if (Version == 5)
		DimensionVer4(str, a, stream);

	else return 0;


	DimensionAngleCH(&a->Angle, &a->StartPoint);
	return 1;
}





/* -------------------------------------------------------------------
 * ファイルが壊れていないかを確認
 *	
 */ 
int FileCheck(char *FileName)
{
	char str[256];
	FILE *stream;
	int Ret, check;


	/* -----------------------------------------------------
	 * ファイルが壊れていないかを確認
	 */ 
	if ( (stream = fopen( FileName , "r" )) == NULL ) {
		sprintf(str,  _("The file could not be opened. [%s]\n") , FileName);
		g_print("%s", str);
		return 0;
	}
	Ret = 1;
	check = 0;
	while (Ret == 1) {
		Ret = read_utf8_from_locale(stream, str, CHARSET_LOCALE);
		if (strstr(str, "EOF") != NULL) check = 1;
	}
	if (check != 1) {
		sprintf(str, _("It thinks that a file is broken. [%s]\n"), FileName);
		MsgBox("SagCAD ERROR", str, " ", NULL, NULL, "OK");
		return 0;
	}
	fclose(stream);
	return 1;
}





/* -------------------------------------------------------------------
 * Dimension Data Version 0 & 1
 *	
 * type = 10 : 水平Ｘ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 20 : 垂直Ｙ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 30 : 距離
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 40 : 半径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[3], SA = X[4], EA = Y[4]
 * type = 50 : 直径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[6]
 * type = 60 : 角度
 *	   中心(X[1],Y[1]), 点1(X[2],Y[2]), 点2(X[3],Y[3]), クリック点(X[4],Y[4])
 * type = 70 : 座標
 *	   点(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 80 : 引出線
 *	   矢印の先端(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 90 : 注釈
 *	   クリック点(X[1],Y[1])
 */
int DimensionVer1(char *str, DIMENSION *a)
{
	char ret[255], str2[256];
	DIMENSION_CULC pic;


	a->Layer = 0;
	a->DrawType = sagcad_dimension.Draw_Type;
	a->FontType = sagcad_dimension.font_Type;
	a->FontHeight = sagcad_dimension.Font_height;
	a->FontSpace = sagcad_dimension.Font_space;
	a->Type = 0;
	strcpy(a->Text, "");
	strcpy(a->UpperText, "");
	strcpy(a->LowerText, "");
	a->Angle = 0;
	a->SearchPointX = 0;
	a->SearchPointY = 0;
	a->StartPoint = 0;
	a->diagram_of_char_index = 0;
	a->diagram_of_char = NULL;
	a->index = 0;
	a->AssistLine = NULL;


	LineDataSplit(str, ',', 1, ret);
	a->Layer = atoi(ret);
	LineDataSplit(str, ',', 2, ret);
	a->Type = atoi(ret);
	LineDataSplit(str, ',', 3, ret);
	strcpy(a->Text, ret);
	LineDataSplit(str, ',', 6, ret);
	if (a->Type == 90) a->SearchPointX = atof(ret);
	else pic.X[1] = atof(ret);
	LineDataSplit(str, ',', 7, ret);
	if (a->Type == 90) a->SearchPointY = atof(ret);
	else pic.Y[1] = atof(ret);

	/* 注釈以外 */
	if (a->Type != 90) {
		LineDataSplit(str, ',', 8, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 9, ret);
		pic.Y[2] = atof(ret);
	}

	if (a->Type == 10 || a->Type == 20 || a->Type == 30 || a->Type == 40 || a->Type == 60) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[3] = atof(ret);
		if (a->Type != 40) {
			LineDataSplit(str, ',', 11, ret);
			pic.Y[3] = atof(ret);
		}
	}


	if (a->Type == 40 || a->Type == 60) {
		LineDataSplit(str, ',', 12, ret);
		pic.X[4] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[4] = atof(ret);
	}


	LineDataSplit(str, ',', 14, ret);
	pic.X[5] = atof(ret);
	LineDataSplit(str, ',', 15, ret);
	pic.Y[5] = atof(ret);

	if (a->Type == 50) {
		LineDataSplit(str, ',', 17, ret);
		pic.X[6] = atof(ret);
	}

	strcpy(str2, a->Text);
	switch(a->Type) {
		case 10:
			Dimension_X_Culc(a, pic); break;
		case 20:
			Dimension_Y_Culc(a, pic); break;
		case 30:
			Dimension_P_Culc(a, pic); break;
		case 40:
			Dimension_R_Culc(a, pic); break;
		case 50:
			Dimension_D_Culc(a, pic); break;
		case 60:
			Dimension_A_Culc(a, pic); break;
		case 70:
			Dimension_PT_Culc(a, pic); break;
		case 80:
			pic.X[3] = 0;
			pic.Y[3] = 0;
			Dimension_HD_Culc(a, pic); break;
		case 90:
			break;
	}
	strcpy(a->Text, str2);

	return 1;
}





/* -------------------------------------------------------------------
 * Dimension Data Version 2
 *	
 * type = 10 : 水平Ｘ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 20 : 垂直Ｙ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 30 : 距離
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 40 : 半径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[3], SA = X[4], EA = Y[4]
 * type = 50 : 直径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[6]
 * type = 60 : 角度
 *	   中心(X[1],Y[1]), 点1(X[2],Y[2]), 点2(X[3],Y[3]), クリック点(X[4],Y[4])
 * type = 70 : 座標
 *	   点(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 80 : 引出線
 *	   矢印の先端(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 90 : 注釈
 *	   クリック点(X[1],Y[1])
 */
int DimensionVer2(char *str, DIMENSION *a)
{
	char ret[255], str2[256];
	DIMENSION_CULC pic;


	a->Layer = 0;
	a->DrawType = sagcad_dimension.Draw_Type;
	a->FontType = sagcad_dimension.font_Type;
	a->FontHeight = sagcad_dimension.Font_height;
	a->FontSpace = sagcad_dimension.Font_space;
	a->Type = 0;
	strcpy(a->Text, "");
	strcpy(a->UpperText, "");
	strcpy(a->LowerText, "");
	a->Angle = 0;
	a->SearchPointX = 0;
	a->SearchPointY = 0;
	a->StartPoint = 0;
	a->diagram_of_char_index = 0;
	a->diagram_of_char = NULL;
	a->index = 0;
	a->AssistLine = NULL;


	LineDataSplit(str, ',', 1, ret);
	a->Layer = atoi(ret);
	LineDataSplit(str, ',', 2, ret);
	a->Type = atoi(ret);
	LineDataSplit(str, ',', 5, ret);
	strcpy(a->Text, ret);
	LineDataSplit(str, ',', 6, ret);
	strcpy(a->UpperText, ret);
	LineDataSplit(str, ',', 7, ret);
	strcpy(a->LowerText, ret);

	if (a->Type == 10 || a->Type == 20 || a->Type == 30) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[1] = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		pic.Y[1] = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[2] = atof(ret);
		LineDataSplit(str, ',', 14, ret);
		pic.X[3] = atof(ret);
		LineDataSplit(str, ',', 15, ret);
		pic.Y[3] = atof(ret);
	}

	else if (a->Type == 40) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[1] = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		pic.Y[1] = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[2] = atof(ret);
		LineDataSplit(str, ',', 14, ret);
		pic.X[3] = atof(ret);
		LineDataSplit(str, ',', 15, ret);
		pic.X[4] = atof(ret);
		LineDataSplit(str, ',', 16, ret);
		pic.Y[4] = atof(ret);
	}

	else if (a->Type == 50) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[1] = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		pic.Y[1] = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[2] = atof(ret);
		LineDataSplit(str, ',', 14, ret);
		pic.X[6] = atof(ret);
	}

	else if (a->Type == 60) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[1] = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		pic.Y[1] = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[2] = atof(ret);
		LineDataSplit(str, ',', 14, ret);
		pic.X[3] = atof(ret);
		LineDataSplit(str, ',', 15, ret);
		pic.Y[3] = atof(ret);
		LineDataSplit(str, ',', 16, ret);
		pic.X[4] = atof(ret);
		LineDataSplit(str, ',', 17, ret);
		pic.Y[4] = atof(ret);
	}

	else if (a->Type == 70 || a->Type == 80) {
		LineDataSplit(str, ',', 10, ret);
		pic.X[1] = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		pic.Y[1] = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		pic.X[2] = atof(ret);
		LineDataSplit(str, ',', 13, ret);
		pic.Y[2] = atof(ret);
		pic.X[3] = 0;
		pic.Y[3] = 0;
	}

	else if (a->Type == 90) {
		LineDataSplit(str, ',', 10, ret);
		a->SearchPointX = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		a->SearchPointY = atof(ret);
	}

	strcpy(str2, a->Text);
	switch(a->Type) {
		case 10:
			Dimension_X_Culc(a, pic); break;
		case 20:
			Dimension_Y_Culc(a, pic); break;
		case 30:
			Dimension_P_Culc(a, pic); break;
		case 40:
			Dimension_R_Culc(a, pic); break;
		case 50:
			Dimension_D_Culc(a, pic); break;
		case 60:
			Dimension_A_Culc(a, pic); break;
		case 70:
			Dimension_PT_Culc(a, pic); break;
		case 80:
			Dimension_HD_Culc(a, pic); break;
		case 90:
			break;
	}
	strcpy(a->Text, str2);

	return 1;
}





#ifdef DOC
/* dimension */
sagcad_dimension.Draw_Type;						/* 1  1:実寸  2:尺度実寸  3:文字サイズ固定 */
sagcad_dimension.font_Type;						/* 1 */
sagcad_dimension.Font_height;					/* 2.5 */
sagcad_dimension.Font_space;					/* 0.5 */
sagcad_dimension.Font_space_line;				/* 0.6 */
sagcad_dimension.arrow_length;					/* 2 */
sagcad_dimension.arrow_angle;					/* 15 */
sagcad_dimension.arrow_line_extension;			/* 3 */
sagcad_dimension.assistance_line_extension;	/* 1 */
sagcad_dimension.assistance_line_space;		/* 1 */
sagcad_dimension.dimension_figure;				/* 3 */
sagcad_dimension.font_file[256];				/* sagfont.sft */
sagcad_dimension.font_file2[256];				/* sagfont2.sft */


a->Layer;			
a->DrawType;		/* 0:縮尺実寸	1:固定サイズ	2:実寸表示 */
a->Text[256];		
a->UpperText[20];	
a->LowerText[20];	
a->FontType;
a->FontHeight;		
a->FontSpace;		
a->Angle;
a->SearchPointX;
a->SearchPointY;
a->StartPoint;		/* 文字の書き出し位置 */
a->index;
a->AssistLine;	
#endif





/* -------------------------------------------------------------------
 * Dimension Data Version 3
 *	
 * type = 10 : 水平Ｘ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 20 : 垂直Ｙ
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 30 : 距離
 *	   (X[1],Y[1]),(X[2],Y[2]),(X[3],Y[3])
 * type = 40 : 半径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[3], SA = X[4], EA = Y[4]
 * type = 50 : 直径
 *	   円の中心(X[1],Y[1]), クリック点(X[2],Y[2]), R = X[6]
 * type = 60 : 角度
 *	   中心(X[1],Y[1]), 点1(X[2],Y[2]), 点2(X[3],Y[3]), クリック点(X[4],Y[4])
 * type = 70 : 座標
 *	   点(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 80 : 引出線
 *	   矢印の先端(X[1],Y[1]), クリック点(X[2],Y[2])
 * type = 90 : 注釈
 *	   クリック点(X[1],Y[1])
 * type = 100 : テキスト
 *	   クリック点(X[1],Y[1])
 */
int DimensionVer3(char *str, DIMENSION *a)
{
	char ret[255], str2[256];
	DIMENSION_CULC pic;


	a->Layer = 0;
	a->DrawType = sagcad_dimension.Draw_Type;
	a->FontType = sagcad_dimension.font_Type;
	a->FontHeight = sagcad_dimension.Font_height;
	a->FontSpace = sagcad_dimension.Font_space;
	a->Type = 0;
	strcpy(a->Text, "");
	strcpy(a->UpperText, "");
	strcpy(a->LowerText, "");
	a->Angle = 0;
	a->SearchPointX = 0;
	a->SearchPointY = 0;
	a->StartPoint = 0;
	a->diagram_of_char_index = 0;
	a->diagram_of_char = NULL;
	a->index = 0;
	a->AssistLine = NULL;


	LineDataSplit(str, ',', 1, ret);
	a->Layer = atoi(ret);
	LineDataSplit(str, ',', 2, ret);
	a->Type = atoi(ret);
	if (a->Type == 100) {
		LineDataSplit(str, ',', 3, ret);
		a->FontType = atoi(ret);
		LineDataSplit(str, ',', 4, ret);
		a->DrawType = atoi(ret);
		LineDataSplit(str, ',', 5, ret);
		strcpy(a->Text, ret);
		LineDataSplit(str, ',', 6, ret);
		strcpy(a->UpperText, ret);
		LineDataSplit(str, ',', 7, ret);
		strcpy(a->LowerText, ret);
		LineDataSplit(str, ',', 8, ret);
		a->FontHeight = (float) atof(ret);
		LineDataSplit(str, ',', 9, ret);
		a->FontSpace = (float) atof(ret);

		LineDataSplit(str, ',', 10, ret);
		a->SearchPointX = atof(ret);
		LineDataSplit(str, ',', 11, ret);
		a->SearchPointY = atof(ret);
		LineDataSplit(str, ',', 12, ret);
		a->Angle = (float) atof(ret);

		a->index = 0;
		a->AssistLine = NULL;
		return 2;
	}
	else {
		LineDataSplit(str, ',', 3, ret);
		strcpy(a->Text, ret);
//MsgBox("test", a->Text);
		LineDataSplit(str, ',', 4, ret);
		strcpy(a->UpperText, ret);
		LineDataSplit(str, ',', 5, ret);
		strcpy(a->LowerText, ret);

		if (a->Type == 10 || a->Type == 20 || a->Type == 30) {
			LineDataSplit(str, ',', 6, ret);
			pic.X[1] = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			pic.Y[1] = atof(ret);
			LineDataSplit(str, ',', 8, ret);
			pic.X[2] = atof(ret);
			LineDataSplit(str, ',', 9, ret);
			pic.Y[2] = atof(ret);
			LineDataSplit(str, ',', 10, ret);
			pic.X[3] = atof(ret);
			LineDataSplit(str, ',', 11, ret);
			pic.Y[3] = atof(ret);
		}

		else if (a->Type == 40) {
			LineDataSplit(str, ',', 6, ret);
			pic.X[1] = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			pic.Y[1] = atof(ret);
			LineDataSplit(str, ',', 8, ret);
			pic.X[2] = atof(ret);
			LineDataSplit(str, ',', 9, ret);
			pic.Y[2] = atof(ret);
			LineDataSplit(str, ',', 10, ret);
			pic.X[3] = atof(ret);
			LineDataSplit(str, ',', 11, ret);
			pic.X[4] = atof(ret);
			LineDataSplit(str, ',', 12, ret);
			pic.Y[4] = atof(ret);
		}

		else if (a->Type == 50) {
			LineDataSplit(str, ',', 6, ret);
			pic.X[1] = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			pic.Y[1] = atof(ret);
			LineDataSplit(str, ',', 8, ret);
			pic.X[2] = atof(ret);
			LineDataSplit(str, ',', 9, ret);
			pic.Y[2] = atof(ret);
			LineDataSplit(str, ',', 10, ret);
			pic.X[6] = atof(ret);
		}

		else if (a->Type == 60) {
			LineDataSplit(str, ',', 6, ret);
			pic.X[1] = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			pic.Y[1] = atof(ret);
			LineDataSplit(str, ',', 8, ret);
			pic.X[2] = atof(ret);
			LineDataSplit(str, ',', 9, ret);
			pic.Y[2] = atof(ret);
			LineDataSplit(str, ',', 10, ret);
			pic.X[3] = atof(ret);
			LineDataSplit(str, ',', 11, ret);
			pic.Y[3] = atof(ret);
			LineDataSplit(str, ',', 12, ret);
			pic.X[4] = atof(ret);
			LineDataSplit(str, ',', 13, ret);
			pic.Y[4] = atof(ret);
		}

		else if (a->Type == 70 || a->Type == 80) {
			LineDataSplit(str, ',', 6, ret);
			pic.X[1] = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			pic.Y[1] = atof(ret);
			LineDataSplit(str, ',', 8, ret);
			pic.X[2] = atof(ret);
			LineDataSplit(str, ',', 9, ret);
			pic.Y[2] = atof(ret);
			pic.X[3] = 0;
			pic.Y[3] = 0;
		}

		else if (a->Type == 90) {
			LineDataSplit(str, ',', 6, ret);
			a->SearchPointX = atof(ret);
			LineDataSplit(str, ',', 7, ret);
			a->SearchPointY = atof(ret);
		}

//MsgBox("test before", a->Text);
		strcpy(str2, a->Text);
		switch(a->Type) {
			case 10:
				Dimension_X_Culc(a, pic); break;
			case 20:
				Dimension_Y_Culc(a, pic); break;
			case 30:
				Dimension_P_Culc(a, pic); break;
			case 40:
				Dimension_R_Culc(a, pic); break;
			case 50:
				Dimension_D_Culc(a, pic); break;
			case 60:
				Dimension_A_Culc(a, pic); break;
			case 70:
				Dimension_PT_Culc(a, pic); break;
			case 80:
				Dimension_HD_Culc(a, pic); break;
			case 90:
				break;
		}
		strcpy(a->Text, str2);
	}
//MsgBox("test after", a->Text);
	return 1;
}





/* -------------------------------------------------------------------
 * Dimension Data Version 4
 *	
 */
int DimensionVer4(char *str, DIMENSION *a, FILE *stream)
{
	char ret[256];
	int i;


	a->diagram_of_char_index = 0;
	a->diagram_of_char = NULL;


	LineDataSplit(str, ',', 1, ret);
	a->Layer = atoi(ret);

	LineDataSplit(str, ',', 2, ret);
	a->DrawType = atoi(ret);

	LineDataSplit(str, ',', 3, ret);
	a->Type = atoi(ret);

	LineDataSplit(str, ',', 4, ret);
	strcpy (a->Text, ret);

	LineDataSplit(str, ',', 5, ret);
	strcpy (a->UpperText, ret);

	LineDataSplit(str, ',', 6, ret);
	strcpy (a->LowerText, ret);

	LineDataSplit(str, ',', 7, ret);
	a->FontType = atoi(ret);

	LineDataSplit(str, ',', 8, ret);
	a->FontHeight = (float) atof(ret);

	LineDataSplit(str, ',', 9, ret);
	a->FontSpace = (float) atof(ret);

	LineDataSplit(str, ',', 10, ret);
	a->Angle = (float) atof(ret);

	LineDataSplit(str, ',', 11, ret);
	a->SearchPointX = atof(ret);

	LineDataSplit(str, ',', 12, ret);
	a->SearchPointY = atof(ret);

	LineDataSplit(str, ',', 13, ret);
	a->StartPoint = atoi(ret);

	LineDataSplit(str, ',', 14, ret);
	a->index = atoi(ret);

	if (a->index == 0) return 1;

	/* データ (AssistLine) を書き込むための領域を確保する */
	a->AssistLine = (ASSISTANCE *)xmalloc(a->index * sizeof(ASSISTANCE));

	for ( i= 0 ; i < a->index ; i++ ) {
		if (read_utf8_from_locale(stream, str, 4) == 0) return 0;

		LineDataSplit(str, ',', 1, ret);
		a->AssistLine[i].defin = atoi(ret);

		LineDataSplit(str, ',', 2, ret);
		a->AssistLine[i].sx = atof(ret);

		LineDataSplit(str, ',', 3, ret);
		a->AssistLine[i].sy = atof(ret);

		LineDataSplit(str, ',', 4, ret);
		a->AssistLine[i].ex = atof(ret);

		LineDataSplit(str, ',', 5, ret);
		a->AssistLine[i].ey = atof(ret);

		LineDataSplit(str, ',', 6, ret);
		a->AssistLine[i].cx = atof(ret);

		LineDataSplit(str, ',', 7, ret);
		a->AssistLine[i].cy = atof(ret);

		LineDataSplit(str, ',', 8, ret);
		a->AssistLine[i].r	= atof(ret);
	}
	return 1;
}





/* ====================================================================
 * ===	Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *	  Project			   : sagcad
 *	  Source			   : sgy.c
 * ====================================================================
 */





/* -------------------------------------------------------------------
 * クリップファイルに文字列をコピー
 * 
 * /home/user/.sagcad/SagCAD_Clip
 * 
 */
int WriteClipFile(void)
{
	char Path[256], str[256];
	FILE *stream;
	SELECT_LIST *sp;


	/* -----------------------------------------------------
	 * Open File
	 */
	GetConfigPath(str);
	sprintf(Path, "%s/SagCAD_Clip", str);
	if ((stream = fopen( Path, "w")) == NULL) {
		sprintf(str,  _("The file could not be opened. [%s]\n") , Path);
		MsgBox("SagCAD ERROR", str, " ", NULL, NULL, "OK");
		return 0;
	}


	/* -----------------------------------------------------
	 * TRACE SECTION
	 */
	sprintf(str, "TRACE (Layer, Code, Style, Color, SX, SY, EX, EY, CX, CY, R, 予備)\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);
	/* 各図形要素 */
	sp = select_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while (sp != NULL) {
		if (sp->select->index == 2 || sp->select->index == 4) {
			if (trace_data_write(stream, cad_search_cad(sp->select->diagram_point.cad_point, &cad_list_info)) == 0) {
				fclose(stream);
				return 0;
			}
		}
		else if (sp->select->index == 3) {
			/*  */
		}
		sp = sp->next;	/* ポインタを次のデータに移す */
	}
	sprintf(str, "TRACE END\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);



	/* -----------------------------------------------------
	 * DIMENSION SECTION
	 */
	sprintf(str, "DIMENSION\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);
	/* 各寸法要素 */
	sp = select_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while (sp != NULL) {
		if (sp->select->index == 2 || sp->select->index == 4) {
			/*  */
		}
		else if (sp->select->index == 3) {
			if (dimension_data_write(stream, dimension_search_dimension(sp->select->diagram_point.dimension_point, &dimension_list_info) ) == 0) {
				fclose(stream);
				return 0;
			}
		}
		sp = sp->next;	/* ポインタを次のデータに移す */
	}
	sprintf(str, "DIMENSION END\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);



	/* -----------------------------------------------------
	 * POLYLINE SECTION
	 */
	sprintf(str, "POLYLINE\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);
	/* 各図形要素 */
	sp = select_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while (sp != NULL) {
		if (sp->select->index == 8) {
			if (PolyLine_data_write(stream, 
									polyline_search_polyline(sp->select->diagram_point.polyline_point, 
															 &polyline_list_info)) == 0) 
			{
				fclose(stream);
				return 0;
			}
		}
		sp = sp->next;	/* ポインタを次のデータに移す */
	}
	sprintf(str, "POLYLINE END\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);



	/* -----------------------------------------------------
	 * ELLIPSE SECTION
	 */
	sprintf(str, "ELLIPSE (Layer, Code, Style, Color, CX, CY, DX, DY, K, SA, EA)\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);
	/* 各図形要素 */
	sp = select_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while (sp != NULL) {
		if (sp->select->index == 16) {
			if (ellipse_data_write(stream, 
								   ellipse_search_ellipse(sp->select->diagram_point.ellipse_point, 
														  &ellipse_list_info)) == 0) 
			{
				fclose(stream);
				return 0;
			}
		}
		sp = sp->next;	/* ポインタを次のデータに移す */
	}
	sprintf(str, "ELLIPSE END\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);



	/* -----------------------------------------------------
	 * INSERT SECTION
	 */



	/* -----------------------------------------------------
	 * EOF SECTION
	 */
	sprintf(str, "EOF\n");
	write_locale_from_utf8(str, stream, CHARSET_UTF8);



	/* -----------------------------------------------------
	 * Close File
	 */
	fclose(stream);
	return 1;
}





/* -------------------------------------------------------------------
 * クリップファイルから文字列の取得
 * 
 * /home/user/.sagcad/SagCAD_Clip
 * 
 */
int ReadClipFile(void)
{
	int Undo_frag = 0;
	FILE *stream;
	CAD cad;
	DIMENSION dimension;
	ELLIPSE ellipse;
	POLYLINE polyline;
	char Path[256], str[256];


	/* -----------------------------------------------------
	 * Open File
	 */
	GetConfigPath(str);
	sprintf(Path, "%s/SagCAD_Clip", str);
	if ((stream = fopen( Path, "r")) == NULL) {
		sprintf(str, _("The file could not be opened. [%s]\n"), Path);
		MsgBox("SagCAD ERROR", str, " ", NULL, NULL, "OK");
		return 0;
	}


	while(1){
		if (read_utf8_from_locale(stream, str, CHARSET_UTF8) == 0) {
			fclose(stream);
			return 0;
		}

		/* -----------------------------------------------------
		 * TRACE SECTION
		 */
		if(strstr(str, "TRACE") != NULL) {
			while(1) {
				if (read_utf8_from_locale(stream, str, CHARSET_UTF8) == 0) {
					fclose(stream);
					return 0;
				}

				if (strstr (str, "TRACE END") != NULL ) break;

				else {
					if (TraceLook(str, &cad, 4) == 0) {
						MsgBox("SagCAD ERROR", "Clip File Read Error.  (Trace Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}
					cad.layer = NowLayer;

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* CAD Data をリストの最初に追加 */
					cad_list_add_first_with_undo(&cad, &cad_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * DIMENSION SECTION
		 */
		else if(strstr(str, "DIMENSION") != NULL) {
			/* Undo_index をインクリメントする */
			undo_index_plus();

			while(1) {
				if (read_utf8_from_locale(stream, str, CHARSET_UTF8) == 0) {
					fclose(stream);
					return 0;
				}

				if(strstr(str, "DEMENSION END") != NULL || strstr(str, "DIMENSION END") != NULL) break;

				else {
					if (DimensionLook(str, &dimension, stream, 4) == 0) {
						MsgBox("SagCAD ERROR", "Clip File Read Error.  (Dimension Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}
					dimension.Layer = NowLayer;

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* DIMENSION Data をリストの最初に追加 */
					dimension_list_add_first_with_undo(&dimension, &dimension_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * POLYLINE SECTION
		 */
		if (strstr(str, "POLYLINE") != NULL) {
			while(1) {
				if (read_utf8_from_locale(stream, str, CHARSET_UTF8) == 0) {
					fclose(stream);
					return 0;
				}

				if (strstr (str, "POLYLINE END") != NULL ) break;

				else {
					if (PolyLineLook(str, &polyline, stream, 4) == 0) {
						MsgBox("SagCAD ERROR", "Clip File Read Error.  (PolyLine Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}
					polyline.layer = NowLayer;

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* POLYLINE Data をリストの最初に追加 */
					polyline_list_add_first_with_undo(&polyline, &polyline_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * ELLIPSE SECTION
		 */
		if (strstr(str, "ELLIPSE") != NULL) {
			while(1) {
				if (read_utf8_from_locale(stream, str, CHARSET_UTF8) == 0) {
					fclose(stream);
					return 0;
				}

				if (strstr (str, "ELLIPSE END") != NULL ) break;

				else {
					if (EllipseLook(str, &ellipse, 4) == 0) {
						MsgBox("SagCAD ERROR", "Clip File Read Error.  (Ellipse Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}
					ellipse.layer = NowLayer;

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* ELLIPSE Data をリストの最初に追加 */
					ellipse_list_add_first_with_undo(&ellipse, &ellipse_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * INSERT SECTION
		 */



		/* -----------------------------------------------------
		 * EOF SECTION
		 */
		else if(strstr(str,"EOF") != NULL ) break;

	}
	fclose(stream);
	return 1;
}





/* -------------------------------------------------------------------
 * 部品図読み込み File名 取得
 * 
 */
int ReadPartsFileNameGet(char *OpenFileName, double x, double y)
{
	int Version = 5, i, Undo_frag = 0;
	char str[256];
	FILE *stream;
	CAD cad;
	DIMENSION dimension;


	/* ---------------------------------
	 * ファイルが壊れていないかを確認
	 */ 
	if (FileCheck(OpenFileName) == 0) return 0;

	/* ---------------------------------
	 * ファイルをオープン
	 */ 
	if( (stream	 = fopen( OpenFileName , "r" )) == NULL ){
		printf( _("The file could not be opened. [%s]\n"), OpenFileName);
	}


	while(1){
		if (read_utf8_from_locale(stream, str, Version) == 0) {
			fclose(stream);
			return 0;
		}

		/* -------------------------------------------------
		 * TABLES SECTION
		 */
		if ( strstr(str,"TABLES") != NULL ) {
			while(1) {
				if (read_utf8_from_locale(stream, str, Version) == 0) {
					fclose(stream);
					return 0;
				}

				if ( strstr(str, "TABLES END") != NULL ) break;

				else if ( strstr(str, "Ver") != NULL ) {
					if (read_utf8_from_locale(stream, str, Version) == 0) {
						fclose(stream);
						return 0;
					}
					if (strstr(str, "1.0") != NULL ) Version = 1;
					else if (strstr(str, "2.0") != NULL ) Version = 2;
					else if (strstr(str, "3.0") != NULL ) Version = 3;
					else if (strstr(str, "4.0") != NULL ) Version = 4;
					else if (strstr(str, "5.0") != NULL ) Version = 5;
					else Version = 0;
				}
			}
		}


		/* -----------------------------------------------------
		 * TRACE SECTION
		 */
		if ( strstr(str, "TRACE") != NULL ) {
			while(1) {
				if (read_utf8_from_locale(stream, str, Version) == 0) {
					fclose(stream);
					return 0;
				}

				if (strstr (str, "TRACE END") != NULL ) break;

				else {
					if (TraceLook(str, &cad, Version) == 0) {
						MsgBox("ERROR", "File Read Error.  (Trace Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}
					cad.sx	= cad.sx + x;
					cad.sy	= cad.sy + y;
					cad.ex	= cad.ex + x;
					cad.ey	= cad.ey + y;
					cad.cx	= cad.cx + x;
					cad.cy	= cad.cy + y;
					cad.layer = NowLayer;

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* CAD Data をリストの最初に追加 */
					cad_list_add_first_with_undo(&cad, &cad_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * DIMENSION SECTION
		 */
		else if ( strstr(str, "DIMENSION") != NULL ) {
			while(1) {
				if (read_utf8_from_locale(stream, str, Version) == 0) {
					fclose(stream);
					return 0;
				}

				if(strstr(str, "DEMENSION END") != NULL || strstr(str, "DIMENSION END") != NULL) break;

				else {
					if (DimensionLook(str, &dimension, stream, Version) == 0) {
						MsgBox("ERROR", "File Read Error.  (Dimension Data)", " ", NULL, NULL, "OK");
						fclose(stream);
						return 0;
					}

					dimension.Layer = NowLayer;
					dimension.SearchPointX = dimension.SearchPointX + x;
					dimension.SearchPointY = dimension.SearchPointY + y;
					
					for ( i=0 ; i < dimension.index ; i++) {
						dimension.AssistLine[i].sx = dimension.AssistLine[i].sx + x;
						dimension.AssistLine[i].sy = dimension.AssistLine[i].sy + y;
						dimension.AssistLine[i].ex = dimension.AssistLine[i].ex + x;
						dimension.AssistLine[i].ey = dimension.AssistLine[i].ey + y;
						dimension.AssistLine[i].cx = dimension.AssistLine[i].cx + x;
						dimension.AssistLine[i].cy = dimension.AssistLine[i].cy + y;
					}

					if (Undo_frag == 0) {
						Undo_frag = 1;
						undo_index_plus();
					}
					/* DIMENSION Data をリストの最初に追加 */
					dimension_list_add_first_with_undo(&dimension, &dimension_list_info);
				}
			}
		}



		/* -----------------------------------------------------
		 * EOF SECTION
		 */
		else if(strstr(str,"EOF") != NULL ) break;

	}
	fclose(stream);
	return 1;
}





#ifdef AFTER
/* -------------------------------------------------------------------
 * sagfont.sft 作成ユーティリティ
 */
int MakeFont(void)
{
	FILE *stream;
	char str[256], mstr[256], moji[256];
	char SX[256], SY[256], EX[256], EY[256];
	char CX[256], CY[256], R[256];
	double sx, sy, ex, ey, cx, cy, r;
	int i;


	str[0] = '\0';
	InputBox_s(hMainWnd, "文字 ?", str, moji);
	if (moji[0] == 'C' )	return 0;


	sprintf(str, "%s%s.sft", sagcad_user.CadPath, moji);
	if((stream = fopen( str, "w")) == NULL) {
		sprintf(mstr,  _("The file could not be opened. [%s]\n") , str);
		MessageBox(NULL, mstr, (LPCSTR)"動作確認", MB_OK);
		return 0;
	}


	sprintf(str, "s_%s\n", moji);
	write_locale_from_utf8(str, stream, CHARSET_UTF8);


	FloatOut(str, 100, 0);
	InputBox_s(hMainWnd, "width ?", str, moji);
	if (moji[0] == 'C' )	return 0;
	sprintf(str, "%s\n", moji);
	write_locale_from_utf8(str, stream, CHARSET_UTF8);

	FloatOut(str, 100, 0);
	InputBox_s(hMainWnd, "height ?", str, moji);
	if (moji[0] == 'C' )	return 0;
	sprintf(str, "%s\n", moji);
	write_locale_from_utf8(str, stream, CHARSET_UTF8);


	sprintf(str, "%d\n", SelectData-1);
	write_locale_from_utf8(str, stream, CHARSET_UTF8);


	if(cad[Select[SelectData].Data].Code != 0) return 0;

	for (i = 1 ; i < SelectData ; i++) {
		sx = cad[Select[i].Data].SX - cad[Select[SelectData].Data].SX;
		sy = cad[Select[i].Data].SY - cad[Select[SelectData].Data].SY;
		ex = cad[Select[i].Data].EX - cad[Select[SelectData].Data].SX;
		ey = cad[Select[i].Data].EY - cad[Select[SelectData].Data].SY;
		cx = cad[Select[i].Data].CX - cad[Select[SelectData].Data].SX;
		cy = cad[Select[i].Data].CY - cad[Select[SelectData].Data].SY;
		 r = cad[Select[i].Data].R;

		if (cad[Select[i].Data].Code == 0) {
			ex = 0;
			ey = 0;
			cx = 0;
			cy = 0;
			r  = 0;
		}
		else if (cad[Select[i].Data].Code == 1) {
			cx = 0;
			cy = 0;
			r  = 0;
		}
		else if (cad[Select[i].Data].Code == 4) {
			sx = 0;
			sy = 0;
			ex = 0;
			ey = 0;
		}

		FloatOut6(SX, sx, 0);
		FloatOut6(SY, sy, 0);
		FloatOut6(EX, ex, 0);
		FloatOut6(EY, ey, 0);
		FloatOut6(CX, cx, 0);
		FloatOut6(CY, cy, 0);
		FloatOut6( R,  r, 0);

		sprintf(str, "%d,%s,%s,%s,%s,%s,%s,%s\n",
					cad[Select[i].Data].Code,
					SX, SY, EX, EY, CX, CY, R);
		write_locale_from_utf8(str, stream, CHARSET_UTF8);
	}

	fclose(stream);
	return 1;
}
#endif





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : sgy.c
 * ====================================================================
 */
