/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : DXF_write.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 2002/12/24
 *    Last                 : 2007/10/14
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "intl.h"

#include "List_cad.h"
#include "List_Dimension.h"
#include "List_PolyLine.h"
#include "List_Ellipse.h"
#include "List_Vertex.h"
#include "List_Block.h"
#include "List_Undo.h"
#include "List_Select.h"
#include "global.h"
#include "culcfunc.h"
#include "etc.h"
#include "sagcad.h"
#include "Assistance.h"
#include "MemoryLeak.h"

#define _DXF_WRITE_
#include "DXF_write.h"



/* -------------------------------------------------------------------
 * DxfFileSave
 *	
 */
int DxfFileSave(char *SaveFileName)
{
	FILE *stream;
	char FileName[256];


	sprintf(FileName, "%s", SaveFileName);
	/* File Open */
	if ((stream = fopen( FileName, "w")) == NULL) {
		g_print(_("The file could not be opened. [%s]\n"), FileName);
	}

	/* HEADER SECTION */
	header_section_write(stream);
	/* classes_section_write */
	classes_section_write(stream);
	/* TABLES SECTION */
	tables_section_write(stream);
	/* BLOCKS SECTION */
	blocks_section_write(stream);
	/* ENTITIES SECTION */
	entities_section_write(stream);
	/* objects_section_write */
	objects_section_write(stream);
	/* EOF SECTION */
	eof_section_write(stream);

	/* File Close */
	fclose(stream);
	printf(_("The file saved. [%s]\n"), FileName);
	return 1;
}





/* -------------------------------------------------------------------
 * ヘッダ・セクション
 * header_section_write
 *	
 */
int header_section_write(FILE *stream)
{
	char str[256];
//	double left, top, right, bottom;


	/* HEADER SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "HEADER\n");
	fputs(str, stream );

	/* DXF のバージョン */
	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$ACADVER\n");
	fputs(str, stream );
	sprintf(str, "  1\n");
	fputs(str, stream );
	sprintf(str, "AC1012\n");//AC1009
	fputs(str, stream );

#ifdef TEST
	/* 基点設定コマンドで設定される挿入基点 */
	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$INSBASE\n");
	fputs(str, stream );
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );
	sprintf(str, " 30\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );

	/* 図形範囲の左下 */
	left   =	MagX - ((sagcad_system.ViewX/Mag)/2);
	top	   =	MagY + (((sagcad_system.ViewX/Mag)/2)*sagcad_system.Aspect);
	right  =	MagX + ((sagcad_system.ViewX/Mag)/2);
	bottom =	MagY - (((sagcad_system.ViewX/Mag)/2)*sagcad_system.Aspect);
	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$EXTMIN\n");
	fputs(str, stream );
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", left);
	fputs(str, stream );
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", bottom);
	fputs(str, stream );

	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$EXTMAX\n");
	fputs(str, stream );
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", right);
	fputs(str, stream );
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", top);
	fputs(str, stream );

	/* XY 図面範囲左下 */
	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$LIMMIN\n");
	fputs(str, stream );
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", left);
	fputs(str, stream );
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", bottom);
	fputs(str, stream );

	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$LIMMAX\n");
	fputs(str, stream );
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", right);
	fputs(str, stream );
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", top);
	fputs(str, stream );

	/* 線種のグローバルなスケール */
	sprintf(str, "  9\n");
	fputs(str, stream );
	sprintf(str, "$LTSCALE\n");
	fputs(str, stream );
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "1.0\n");
	fputs(str, stream );
#endif


	/* HEADER SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * tables_section_write
 *	
 */
int tables_section_write(FILE *stream)
{
	char str[256];


	/* TABLES_SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "TABLES\n");
	fputs(str, stream );


	/* テーブルセクション　線種設定 */
	tables_section_LTYPE_write(stream);

	/* テーブルセクション　レイヤー設定 */
	tables_section_LAYER_write(stream);

	/* テーブルセクション　スタイル設定 */
	tables_section_STYLE_write(stream);

	/* < TABLES STYLE start > */


	/* TABLES_SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * tables_section_LTYPE_write
 *	
 * テーブルセクション　線種設定
 */
int tables_section_LTYPE_write(FILE *stream)
{
	char str[256];

	/* TABLES start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "TABLE\n");
	fputs(str, stream );


	/* LTYPE start */
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "LTYPE\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "     9\n");
	fputs(str, stream );


	/* CONTINUOUS (実線) */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "LTYPE\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "CONTINUOUS\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "    64\n");
	fputs(str, stream );
	sprintf(str, "  3\n");
	fputs(str, stream );
	sprintf(str, "実線\n");
	fputs(str, stream );
	sprintf(str, " 72\n");
	fputs(str, stream );
	sprintf(str, "    65\n");
	fputs(str, stream );
	sprintf(str, " 73\n");
	fputs(str, stream );
	sprintf(str, "     0\n");
	fputs(str, stream );
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );


	/* HIDDEN (破線) */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "LTYPE\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "HIDDEN\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "    64\n");
	fputs(str, stream );
	sprintf(str, "  3\n");
	fputs(str, stream );
	sprintf(str, "破線\n");
	fputs(str, stream );
	sprintf(str, " 72\n");
	fputs(str, stream );
	sprintf(str, "    65\n");
	fputs(str, stream );
	sprintf(str, " 73\n");
	fputs(str, stream );
	sprintf(str, "     2\n");
	fputs(str, stream );
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[2].max + sagcad_line_style[2].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[2].max);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[2].min);
	fputs(str, stream );


	/* DASHDOT (１点鎖線) */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "LTYPE\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "DASHDOT\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "    64\n");
	fputs(str, stream );
	sprintf(str, "  3\n");
	fputs(str, stream );
	sprintf(str, "１点鎖線\n");
	fputs(str, stream );
	sprintf(str, " 72\n");
	fputs(str, stream );
	sprintf(str, "    65\n");
	fputs(str, stream );
	sprintf(str, " 73\n");
	fputs(str, stream );
	sprintf(str, "     4\n");
	fputs(str, stream );
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[3].max + (sagcad_line_style[3].min * 3));
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[3].max);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[3].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[3].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[3].min);
	fputs(str, stream );


	/* DIVIDE (２点鎖線) */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "LTYPE\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "DIVIDE\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "    64\n");
	fputs(str, stream );
	sprintf(str, "  3\n");
	fputs(str, stream );
	sprintf(str, "２点鎖線\n");
	fputs(str, stream );
	sprintf(str, " 72\n");
	fputs(str, stream );
	sprintf(str, "    65\n");
	fputs(str, stream );
	sprintf(str, " 73\n");
	fputs(str, stream );
	sprintf(str, "     6\n");
	fputs(str, stream );
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[4].max + (sagcad_line_style[4].min * 6));
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[4].max);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[4].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[4].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[4].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "%d\n", sagcad_line_style[4].min);
	fputs(str, stream );
	sprintf(str, " 49\n");
	fputs(str, stream );
	sprintf(str, "-%d\n", sagcad_line_style[4].min);
	fputs(str, stream );


	/* TABLES end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDTAB\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * tables_section_LAYER_write
 *	
 * テーブルセクション　レイヤー設定
 */
int tables_section_LAYER_write(FILE *stream)
{
	char str[256], dummy[256];
	int i, n, dxf_color;
	int Layer_Data[256];
	CAD_LIST *p;
	DIMENSION_LIST *pd;


	/* TABLES start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "TABLE\n");
	fputs(str, stream );


	/* LAYER start */
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "LAYER\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );


	/* Layer_Data 配列の初期化 */
	/* レイヤー名が変更されていれば、出力する */
	for (i=0 ; i < 256 ; i++) {
		sprintf(dummy, "%d", i);
		if (search(Layer[i].name, dummy) != NULL && strlen(Layer[i].name) == strlen(dummy)) {
			Layer_Data[i] = 0;
		}
		else {
			Layer_Data[i] = 1;
		}
	}

	/* 図形データがあれば 1 */
	/* 各図形要素 */
	p = cad_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		/* 表示 */
		if (Layer[p->cad->layer].draw != 0 || NowLayer == p->cad->layer) {
			Layer_Data[p->cad->layer] = 1;
		}
		/* 非表示 */
		else if (Layer[p->cad->layer].draw == 0 && NowLayer != p->cad->layer) {
			Layer_Data[p->cad->layer] = -1;
		}
		p = p->next;	/* ポインタを次のデータに移す */
	}

	/* 寸法データがあれば 1 */
	/* 各寸法要素 */
	pd = dimension_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(pd != NULL) {
		if (Layer[pd->dimension->Layer].draw != 0 || NowLayer == pd->dimension->Layer) {
			Layer_Data[pd->dimension->Layer] = 1;
		}
		else if (Layer[pd->dimension->Layer].draw == 0 && NowLayer != pd->dimension->Layer) {
			Layer_Data[pd->dimension->Layer] = -1;
		}
		pd = pd->next;	/* ポインタを次のデータに移す */
	}


	/* 使用しているレイヤーの個数を数える */
	n = 0;
	for ( i = 0 ; i < 256 ; i++ ) {
		if (Layer_Data[i] == 1 || Layer_Data[i] == -1) {
			n++;
		}
	}
	/* 使用しているレイヤーの個数を書き込む */
	sprintf(str, "  %d\n", n);
	fputs(str, stream );


	/* 各レイヤー情報を記述 */
	for ( i = 0 ; i < 256 ; i++ ) {
		if (Layer_Data[i] == 1 || Layer_Data[i] == -1) {
			sprintf(str, "  0\n");
			fputs(str, stream );
			sprintf(str, "LAYER\n");
			fputs(str, stream );
			sprintf(str, "  2\n");
			fputs(str, stream );
			sprintf(str, "%s\n", Layer[i].name);
			fputs(str, stream );
			sprintf(str, " 70\n");
			fputs(str, stream );
			sprintf(str, "   0\n");
			fputs(str, stream );
			
			sprintf(str, " 62\n");
			fputs(str, stream );
			/* 表示 */
			dxf_color = SgyToDxf_color(Layer[i].color);
			if (Layer_Data[i] == 1) {
				if (dxf_color < 10) {
					sprintf(str, "     %d\n", dxf_color);
				}
				else if (dxf_color < 100) {
					sprintf(str, "    %d\n", dxf_color);
				}
				else if (dxf_color < 256) {
					sprintf(str, "   %d\n", dxf_color);
				}
			}
			/* 非表示 */
			else if (Layer_Data[i] == -1) {
				if (dxf_color < 10) {
					sprintf(str, "    -%d\n", dxf_color);
				}
				else if (dxf_color < 100) {
					sprintf(str, "   -%d\n", dxf_color);
				}
				else if (dxf_color < 256) {
					sprintf(str, "  -%d\n", dxf_color);
				}
			}
			fputs(str, stream );
			sprintf(str, "  6\n");
			fputs(str, stream );
			sprintf(str, "CONTINUOUS\n");
			fputs(str, stream );
		}
	}


	/* TABLES end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDTAB\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * tables_section_STYLE_write
 *	
 * テーブルセクション　スタイル設定
 */
int tables_section_STYLE_write(FILE *stream)
{
	char str[256];


	/* TABLE start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "TABLE\n");
	fputs(str, stream );


	/* STYLE start */
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "STYLE\n");
	fputs(str, stream );
	sprintf(str, " 70\n");
	fputs(str, stream );
	sprintf(str, "   1\n");
	fputs(str, stream );


	/* STYLE １個目 start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "STYLE\n");
	fputs(str, stream );
	sprintf(str, "  2\n");			// ┓字体の名前
	fputs(str, stream );
	sprintf(str, "STANDARD\n");		// ┛　"STANDARD"
	fputs(str, stream );
	sprintf(str, " 70\n");			// ┓字体のフラグ
	fputs(str, stream );
	sprintf(str, "     0\n");		// ┛　通常は 0 or 64
	fputs(str, stream );
	sprintf(str, " 40\n");			// ┓固定された文字高さ　指定されていない場合は 0.0
	fputs(str, stream );
	sprintf(str, "0.0\n");			// ┛　
	fputs(str, stream );
	sprintf(str, " 41\n");			// ┓高さに対する幅の係数　デフォルト値は 1.0
	fputs(str, stream );
	sprintf(str, "1.0\n");			// ┛　
	fputs(str, stream );
	sprintf(str, " 50\n");			// ┓傾斜角度　デフォルト値は 0.0
	fputs(str, stream );
	sprintf(str, "0.0\n");			// ┛　
	fputs(str, stream );
	sprintf(str, " 71\n");			// ┓文字作成フラグ　デフォルト値は 0.0
	fputs(str, stream );
	sprintf(str, "     0\n");		// ┛　
	fputs(str, stream );
	sprintf(str, " 42\n");			// ┓最後に使用された文字高さ
	fputs(str, stream );
	sprintf(str, "2.5\n");			// ┛　
	fputs(str, stream );
	sprintf(str, "  3\n");			// ┓１バイトコードに対するフォント名
	fputs(str, stream );
	sprintf(str, "txt\n");			// ┛　デフォルト値は txt
	fputs(str, stream );
	sprintf(str, "  4\n");			// ┓２バイトコードに対するフォント名
	fputs(str, stream );
	sprintf(str, "bigfont\n");		// ┛　デフォルト値は なし　行の先頭にリターンコード
	fputs(str, stream );


	/* TABLE end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDTAB\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * classes_section_write
 *	
 */
int classes_section_write(FILE *stream)
{
	char str[256];


	/* CLASSES_SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "CLASSES\n");
	fputs(str, stream );




	/* CLASSES_SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * objects_section_write
 *	
 */
int objects_section_write(FILE *stream)
{
	char str[256];


	/* OBJECTS_SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "OBJECTS\n");
	fputs(str, stream );




	/* OBJECTS_SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * blocks_section_write
 *	
 */
int blocks_section_write(FILE *stream)
{
	char str[256];


	/* BLOCKS_SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "BLOCKS\n");
	fputs(str, stream );




	/* BLOCKS_SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_write
 *	
 */
int entities_section_write(FILE *stream)
{
	CAD_LIST *p;
	DIMENSION_LIST *dp;
	POLYLINE_LIST *pp;
	ELLIPSE_LIST *ep;
	char str[256];


	/* ENTITIES SECTION start */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SECTION\n");
	fputs(str, stream );
	sprintf(str, "  2\n");
	fputs(str, stream );
	sprintf(str, "ENTITIES\n");
	fputs(str, stream );


	/* 各図形要素 */
	p = cad_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		/* POINT */
		if (p->cad->code == 0) {
			entities_section_point_write(p->cad, stream);
		}
		/* LINE */
		else if (p->cad->code == 1) {
			entities_section_line_write(p->cad, stream);
		}
		/* ARC */
		else if (p->cad->code == 2) {
			entities_section_arc_write(p->cad, stream);
		}
		/* CIRCLE */
		else if (p->cad->code == 4) {
			entities_section_circle_write(p->cad, stream);
		}
		p = p->next;	/* ポインタを次のデータに移す */
	}


	/* 寸法要素 */
	dp = dimension_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(dp != NULL) {
		/* Assistant */
		entities_section_dimension_AssistLine_write(dp->dimension, stream);
		/* Text */
		entities_section_dimension_Text_write(dp->dimension, stream);

		dp = dp->next;	/* ポインタを次のデータに移す */
	}


//#ifdef TEST
	/* ポリライン */
	pp = polyline_list_info.head;
	while(pp != NULL) {
		entities_section_polyline_write(pp->polyline, stream);
		pp = pp->next;
	}


	/* 楕円 */
	ep = ellipse_list_info.head;
	while(ep != NULL) {
		entities_section_ellipse_write(ep->ellipse, stream);
		ep = ep->next;
	}
//#endif


	/* ENTITIES SECTION end */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "ENDSEC\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_point_write
 *	
 */
int entities_section_point_write(CAD *cad, FILE *stream)
{
	char str[256];

	sprintf(str, "  0\n");
	fputs(str, stream );

	/* POINT */
	sprintf(str, "POINT\n");
	fputs(str, stream );
	/* Layer */
	sprintf(str, "  8\n");
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[cad->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(cad->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(cad->color));
	fputs(str, stream );

	/* SX */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->sx);
	fputs(str, stream );
	/* SY */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->sy);
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_line_write
 *	
 */
int entities_section_line_write(CAD *cad, FILE *stream)
{
	char str[256];

	sprintf(str, "  0\n");
	fputs(str, stream );

	/* LINE */
	sprintf(str, "LINE\n");
	fputs(str, stream );
	/* Layer */
	sprintf(str, "  8\n");
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[cad->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(cad->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(cad->color));
	fputs(str, stream );

	/* SX */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->sx);
	fputs(str, stream );
	/* SY */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->sy);
	fputs(str, stream );
	/* EX */
	sprintf(str, " 11\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->ex);
	fputs(str, stream );
	/* EY */
	sprintf(str, " 21\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->ey);
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_arc_write
 *	
 */
int entities_section_arc_write(CAD *cad, FILE *stream)
{
	char str[256];
	struct RtnDat rtn;

	sprintf(str, "  0\n");
	fputs(str, stream );

	/* ARC */
	sprintf(str, "ARC\n");
	fputs(str, stream );
	/* Layer */
	sprintf(str, "  8\n");
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[cad->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(cad->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(cad->color));
	fputs(str, stream );

	/* CX */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->cx);
	fputs(str, stream );
	/* CY */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->cy);
	fputs(str, stream );
	/* R */
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->r);
	fputs(str, stream );

	/* SA */
	rtn.sx[1] = cad->cx;
	rtn.sy[1] = cad->cy;
	rtn.ex[1] = cad->sx;
	rtn.ey[1] = cad->sy;
	la(&rtn);
	sprintf(str, " 50\n");
	fputs(str, stream );
	sprintf(str, "%f\n", rtn.angle);
	fputs(str, stream );

	/* EA */
	rtn.sx[1] = cad->cx;
	rtn.sy[1] = cad->cy;
	rtn.ex[1] = cad->ex;
	rtn.ey[1] = cad->ey;
	la(&rtn);
	sprintf(str, " 51\n");
	fputs(str, stream );
	sprintf(str, "%f\n", rtn.angle);
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_circle_write
 * 
 */
int entities_section_circle_write(CAD *cad, FILE *stream)
{
	char str[256];

	sprintf(str, "  0\n");
	fputs(str, stream );

	/* CIRCLE */
	sprintf(str, "CIRCLE\n");
	fputs(str, stream );
	sprintf(str, "  8\n");
	/* Layer */
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[cad->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(cad->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(cad->color));
	fputs(str, stream );

	/* CX */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->cx);
	fputs(str, stream );
	/* CY */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->cy);
	fputs(str, stream );
	/* R */
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%f\n", cad->r);
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_polyline_write
 * 
 */
int entities_section_polyline_write(POLYLINE *polyline, FILE *stream)
{
	char str[256];
	VERTEX_LIST *pv;


	sprintf(str, "  0\n");
	fputs(str, stream );

	/* POLYLINE */
	sprintf(str, "POLYLINE\n");
	fputs(str, stream );
	sprintf(str, "  8\n");
	/* Layer */
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[polyline->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(polyline->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(polyline->color));
	fputs(str, stream );

	/* frag */
	sprintf(str, " 70\n");
	fputs(str, stream );
	if (polyline->code == 8) {
		if ((polyline->frag & 1) == 1) {
			sprintf(str, "     1\n");
		}
		else {
			sprintf(str, "     0\n");
		}
	}
	else if (polyline->code == 16) {
		if ((polyline->frag & 1) == 1) {
			sprintf(str, "     9\n");
		}
		else {
			sprintf(str, "     8\n");
		}
	}


	sprintf(str, "     %d\n", polyline->frag);
	fputs(str, stream );

	/* スプライン */
	sprintf(str, " 75\n");
	fputs(str, stream );
	if (polyline->code == 8) {
		sprintf(str, "     0\n");
	}
	else if (polyline->code == 16) {
		sprintf(str, "     8\n");
	}
	fputs(str, stream );

	/*  */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );
	/*  */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );
	/*  */
	sprintf(str, " 30\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );

	pv = polyline->vertex_list_info.head;
	while (pv != NULL) {
		/*  */
		sprintf(str, "  0\n");
		fputs(str, stream );
		sprintf(str, "VERTEX\n");
		fputs(str, stream );
		
		/* Layer */
		sprintf(str, "  8\n");
		fputs(str, stream );
		sprintf(str, "%s\n", Layer[polyline->layer].name);
		fputs(str, stream );
		
		/* 10 */
		sprintf(str, " 10\n");
		fputs(str, stream );
		sprintf(str, "%f\n", pv->vertex->x);
		fputs(str, stream );
		
		/* 20 */
		sprintf(str, " 20\n");
		fputs(str, stream );
		sprintf(str, "%f\n", pv->vertex->y);
		fputs(str, stream );
		
		/* 30 */
		sprintf(str, " 30\n");
		fputs(str, stream );
		sprintf(str, "0.0\n");
		fputs(str, stream );
		
		pv = pv->next;
	}

	/* VERTEX END */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "SEQEND\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_ellipse_write
 * 
 */
int entities_section_ellipse_write(ELLIPSE *ellipse, FILE *stream)
{
	char str[256];


	sprintf(str, "  0\n");
	fputs(str, stream );

	/* ELLIPSE */
	sprintf(str, "ELLIPSE\n");
	fputs(str, stream );
	sprintf(str, "  8\n");
	/* Layer */
	fputs(str, stream );
	sprintf(str, "%s\n", Layer[ellipse->layer].name);
	fputs(str, stream );

	/* LineStyle */
	sprintf(str, "  6\n");
	fputs(str, stream );
	SagToDXF_Style(ellipse->style, str);
	fputs(str, stream );

	/* Color */
	sprintf(str, " 62\n");
	fputs(str, stream );
	sprintf(str, "    %d\n", SgyToDxf_color(ellipse->color));
	fputs(str, stream );

	/* 中心 X */
	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", ellipse->cx);
	fputs(str, stream );
	/* 中心 Y */
	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", ellipse->cy);
	fputs(str, stream );
	/* 中心 Z */
	sprintf(str, " 30\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );

	/* 長軸の一方の端点 X (中心を原点とみなした相対座標) X */
	sprintf(str, " 11\n");
	fputs(str, stream );
	sprintf(str, "%f\n", ellipse->dx);
	fputs(str, stream );
	/* 長軸の一方の端点 X (中心を原点とみなした相対座標) Y */
	sprintf(str, " 21\n");
	fputs(str, stream );
	sprintf(str, "%f\n", ellipse->dy);
	fputs(str, stream );
	/* 長軸の一方の端点 X (中心を原点とみなした相対座標) Z */
	sprintf(str, " 31\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );

	/* 短軸の長さ(長軸の長さに対する比率) */
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%f\n", ellipse->k);
	fputs(str, stream );

	/* 開始角度 (ラジアン) */
	sprintf(str, " 41\n");
	fputs(str, stream );
	sprintf(str, "%f\n", degrad(ellipse->sa));
	fputs(str, stream );

	/* 終了角度 (ラジアン) */
	sprintf(str, " 42\n");
	fputs(str, stream );
	sprintf(str, "%f\n", degrad(ellipse->ea));
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_dimension_AssistLine_write
 *	
 */
int entities_section_dimension_AssistLine_write(DIMENSION *a, FILE *stream)
{
//	char str[256];
	int i;
	CAD cad;


	if (a->index > 0 && a->AssistLine != NULL) {
		for (i = 0 ; i < a->index ; i++) {
			switch (a->AssistLine[i].defin) {
				/* 点 */
				case 0:
					//PointBasic(hDC, a->AssistLine[i].sx, a->AssistLine[i].sy, color);
					break;
				/* 線 */
				case 1:
					//LineBasic(hDC,	a->AssistLine[i].sx, a->AssistLine[i].sy, 
					//				a->AssistLine[i].ex, a->AssistLine[i].ey, 5);
					cad.layer = a->Layer;
					cad.code = 1;
					cad.style = 5;
					cad.color = 0xffffff;
					cad.sx = a->AssistLine[i].sx;
					cad.sy = a->AssistLine[i].sy;
					cad.ex = a->AssistLine[i].ex;
					cad.ey = a->AssistLine[i].ey;
					cad.cx = 0;
					cad.cy = 0;
					cad.r = 0;
					cad.OnScreen = 0;
					entities_section_line_write(&cad, stream);
					break;
				/* 円弧 */
				case 2:
					//ArcBasic(hDC,	a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
					//				a->AssistLine[i].sx, a->AssistLine[i].sy, 
					//				a->AssistLine[i].ex, a->AssistLine[i].ey, 5);
					break;
				/* 円 */
				case 4:
					//CircleBasic(hDC, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 5);
					break;
				/* 終点に矢印(線) */
				case 10:
					DXF_LineEndArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType);
					break;
				/* 終点に矢印(線) */
				case 11:
					DXF_LineEndArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType);
					break;
				/* 両端に矢印(線) */
				case 20:
					DXF_LineBothArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType);
					break;
				/* 両端に矢印(線) */
				case 21:
					DXF_LineBothArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType);
					break;
				/* 両端に逆矢印(線) */
				case 30:
					DXF_LineBothConverseArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
												a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType);
					break;
				/* 両端に逆矢印(線) */
				case 31:
					DXF_LineBothConverseArrow(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
												a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType);
					break;
				/* 始点に矢印(円弧) */
	//			case 40:
				/* 終点に矢印(円弧) */
	//			case 50:
				/* 両端に矢印(円弧) */
				case 60:
					DXF_ArcBothArrow(	stream, a->Layer, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
										a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType);
					break;
				/* 両端に矢印(円弧) */
				case 61:
					DXF_ArcBothArrow(	stream, a->Layer, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
										a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType);
					break;
				/* 補助線 */
				case 70:
					DXF_AssistanceLine(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										a->AssistLine[i].ex, a->AssistLine[i].ey, a->DrawType);
					break;
				/* (矢印)補助線の延長 */
				case 80:
					DXF_LineEndExtension(	stream, a->Layer, a->AssistLine[i].sx, a->AssistLine[i].sy, 
											a->AssistLine[i].ex, a->AssistLine[i].ey, a->DrawType);
					break;
			}
		}
	}
	else if (a->index > 0 && a->AssistLine == NULL) {
		OneShotLog("寸法線で index > 0 なのに AssistLine が NULL です。\n");
	}

	return 1;
}





/* -------------------------------------------------------------------
 * entities_section_dimension_Text_write
 *	
 */
int entities_section_dimension_Text_write(DIMENSION *a, FILE *stream)
{
	char str[256];
	int point;
	double height;


	/* Text */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "TEXT\n");
	fputs(str, stream );

	/* Layer */
	sprintf(str, "  8\n");
	fputs(str, stream );
	sprintf(str, "Lay%d\n", a->Layer);
	fputs(str, stream );

	sprintf(str, " 10\n");
	fputs(str, stream );
	sprintf(str, "%f\n", a->SearchPointX);
	fputs(str, stream );

	sprintf(str, " 20\n");
	fputs(str, stream );
	sprintf(str, "%f\n", a->SearchPointY);
	fputs(str, stream );

	sprintf(str, " 30\n");
	fputs(str, stream );
	sprintf(str, "0.0\n");
	fputs(str, stream );



	/* 文字高さ */
	/* -----------------------------------------------------
	 * 表示設定による調整
	 * 
	 */
	/* 実寸表示 */
	if (a->DrawType == DRAW_REAL){
		height = a->FontHeight;
	}
	/* 図面の縮尺表示 */
	else if (a->DrawType == DRAW_DISP) {
		height = a->FontHeight / printer.scale;
	}
	/* 文字サイズ固定表示 */
	else if (a->DrawType == DRAW_CONST){
		height = a->FontHeight / Mag;
	}
	else {
		height = a->FontHeight;
	}

	/* 文字高さ */
	sprintf(str, " 40\n");
	fputs(str, stream );
	sprintf(str, "%f\n", a->FontHeight);
	fputs(str, stream );



	/* 回転角度 */
	sprintf(str, " 50\n");
	fputs(str, stream );
	sprintf(str, "%f\n", a->Angle);
	fputs(str, stream );

	/* 文字列 */
	sprintf(str, "  1\n");
	fputs(str, stream );
	sprintf(str, "%s\n", a->Text);
	fputs(str, stream );

	/* 位置合わせＸ */
	if (a->StartPoint > 9) {
		point = a->StartPoint - 10;
	}
	else {
		point = a->StartPoint;
	}
	if (point > 0) {
		sprintf(str, " 72\n");
		fputs(str, stream );
		sprintf(str, "   %d\n", point);
		fputs(str, stream );
		
		sprintf(str, " 11\n");
		fputs(str, stream );
		sprintf(str, "%f\n", a->SearchPointX);
		fputs(str, stream );

		sprintf(str, " 21\n");
		fputs(str, stream );
		sprintf(str, "%f\n", a->SearchPointY);
		fputs(str, stream );

		sprintf(str, " 31\n");
		fputs(str, stream );
		sprintf(str, "0.0\n");
		fputs(str, stream );
	}

	/* 位置合わせＹ */
	if (a->StartPoint > 9) {
		sprintf(str, " 73\n");
		fputs(str, stream );
		sprintf(str, "   3\n");
		fputs(str, stream );
	}
	else {
		//
	}

	return 1;
}





/* -------------------------------------------------------------------
 * eof_section_write
 *	
 */
int eof_section_write(FILE *stream)
{
	char str[256];

	/* < EOF SECTION > */
	sprintf(str, "  0\n");
	fputs(str, stream );
	sprintf(str, "EOF\n");
	fputs(str, stream );

	return 1;
}





/* -------------------------------------------------------------------
 * SagToDXF_Style
 *	
 */
int SagToDXF_Style(int style, char *str)
{
	switch (style) {
		case 0:
			sprintf(str, "CONTINUOUS\n");
			break;
		case 1:
			sprintf(str, "CONTINUOUS\n");
			break;
		case 2:
			sprintf(str, "HIDDEN\n");
			break;
		case 3:
			sprintf(str, "DASHDOT\n");
			break;
		case 4:
			sprintf(str, "PHANTOM\n");
			break;
		case 5:
			sprintf(str, "CONTINUOUS\n");
			break;
		case 6:
			sprintf(str, "CONTINUOUS\n");
			break;
	}
	return 1;
}





/* -------------------------------------------------------------------
 * 終点に矢印 (defin = 10)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 *	
 */
int DXF_LineEndArrow(FILE *stream, int LAYER, double SX, double SY, double EX, double EY, int LineFrag, int DrawType)
{
	struct RtnDat LAH, PAPH;
	double EX1, EY1, EX2, EY2;
	CAD cad;


	/* -----------------------------------------------------
	 * LA	直線の角度
	 *		 (A.sx[1],A.SY[1]),(A.ex[1],A.EY[1])
	 *				 Ans A.ANGLE
	 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	/* -----------------------------------------------------
	 * PAP	始点と角度と距離で直線の終点を求める
	 *		 (A.sx[1] , A.sy[1]) , A.ANGLE , A.l
	 *				 Ans  (A.ex[1] , A.EY[1])
	 */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	/* (EX1 , EY1) */
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	/* (EX2 , EY2) */
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	if (LineFrag != 0) {
//		LineBasic(hDC, SX, SY, EX, EY, 1);
		cad.layer = LAYER;
		cad.code = 1;
		cad.style = 5;
		cad.color = 0xffffff;
		cad.sx = SX;
		cad.sy = SY;
		cad.ex = EX;
		cad.ey = EY;
		cad.cx = 0;
		cad.cy = 0;
		cad.r = 0;
		cad.OnScreen = 0;
		entities_section_line_write(&cad, stream);
	}

//	LineBasic(hDC, EX, EY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, EX, EY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	return 0;
}





/* -------------------------------------------------------------------
 * 両端に矢印 (defin = 20)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int DXF_LineBothArrow(FILE *stream, int LAYER, double SX, double SY, double EX, double EY, int LineFrag, int DrawType)
{
	struct RtnDat LAH, PAPH;
	double EX1, EY1, EX2, EY2;
	CAD cad;


	/* LA	直線の角度 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

//	LineBasic(hDC, EX, EY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, EX, EY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	/* LA	直線の角度 */
	LAH.sx[1] = SX;
	LAH.sy[1] = SY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);

	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

//	LineBasic(hDC, SX, SY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, SX, SY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	if (LineFrag != 0) {
//		LineBasic(hDC, SX, SY, EX, EY, 1);
		cad.layer = LAYER;
		cad.code = 1;
		cad.style = 5;
		cad.color = 0xffffff;
		cad.sx = SX;
		cad.sy = SY;
		cad.ex = EX;
		cad.ey = EY;
		cad.cx = 0;
		cad.cy = 0;
		cad.r = 0;
		cad.OnScreen = 0;
		entities_section_line_write(&cad, stream);
	}

	return 0;
}





/* -------------------------------------------------------------------
 * 両端に逆矢印 (defin = 30)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int DXF_LineBothConverseArrow(FILE *stream, int LAYER, double SX, double SY, double EX, double EY, int LineFrag, int DrawType)
{
	struct RtnDat LAH,PAPH;
	double EX1,EY1,EX2,EY2;
	CAD cad;


	/* LA	直線の角度 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);

	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

//	LineBasic(hDC, SX, SY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, SX, SY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	/* LA	直線の角度 */
	LAH.sx[1] = SX;
	LAH.sy[1] = SY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);
	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

//	LineBasic(hDC, EX, EY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, EX, EY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	if (LineFrag != 0) {
//		LineBasic(hDC, SX, SY, EX, EY, 1);
		cad.layer = LAYER;
		cad.code = 1;
		cad.style = 5;
		cad.color = 0xffffff;
		cad.sx = SX;
		cad.sy = SY;
		cad.ex = EX;
		cad.ey = EY;
		cad.cx = 0;
		cad.cy = 0;
		cad.r = 0;
		cad.OnScreen = 0;
		entities_section_line_write(&cad, stream);
	}

	return 0;
}





/* -------------------------------------------------------------------
 * 両端に矢印(円弧) (defin = 60)
 *	
 * ArcFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int DXF_ArcBothArrow(FILE *stream, int LAYER, double CX, double CY, double R, double SX, double SY, double EX, double EY, int ArcFrag, int DrawType)
{
	/* Dumy = ArcBow(CX, CY, R, SA,EA, INDEX) */

	struct RtnDat PAPH, LAH;
	double SA, EA;
	double SX1, SY1, SX2, SY2;
	double EX1, EY1, EX2, EY2;
	CAD cad;


	/* 中心点から始点の線の角度 SA を求める。 */
	/* LA	  直線の角度 */
	LAH.sx[1] = CX;
	LAH.sy[1] = CY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	SA = LAH.angle;
	if(SA > 360) SA = SA - 360;

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = SA + 90 - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	SX1 = PAPH.ex[1];
	SY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = SA + 90 + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	SX2 = PAPH.ex[1];
	SY2 = PAPH.ey[1];

//	LineBasic(hDC, SX, SY, SX1, SY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = SX1;
	cad.ey = SY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, SX, SY, SX2, SY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = SX2;
	cad.ey = SY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);



	/* 中心点から終点の線の角度 EA を求める。 */
	/* LA	  直線の角度 */
	LAH.sx[1] = CX;
	LAH.sy[1] = CY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);
	EA = LAH.angle;
	if(EA > 360) EA = EA - 360;

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = EA - 90 - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = EA - 90 + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

//	LineBasic(hDC, EX, EY, EX1, EY1, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX1;
	cad.ey = EY1;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

//	LineBasic(hDC, EX, EY, EX2, EY2, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = EX;
	cad.sy = EY;
	cad.ex = EX2;
	cad.ey = EY2;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	if (ArcFrag != 0) {
//		ArcBasic (hDC, CX, CY, R, SX, SY, EX, EY, 5);
		cad.layer = LAYER;
		cad.code = 2;
		cad.style = 5;
		cad.color = 0xffffff;
		cad.sx = SX;
		cad.sy = SY;
		cad.ex = EX;
		cad.ey = EY;
		cad.cx = CX;
		cad.cy = CY;
		cad.r = R;
		cad.OnScreen = 0;
		entities_section_arc_write(&cad, stream);
	}

	return 0;
}





/* -------------------------------------------------------------------
 * 補助線を描画   始点側:引出隙間	終点側:引出延長 (defin = 70)
 *	
 * 1 線の角度(Angle) を求める。
 * 2 線の長さ(LineLen) を求める。
 * 3 start_point を求める。
 * 4 end_point を求める。
 * 5 補助線を描画
 *	
 */
int DXF_AssistanceLine(FILE *stream, int LAYER, double SX, double SY, double EX, double EY, int DrawType)
{
	double Angle, LineLen, start_pointX, start_pointY, end_pointX, end_pointY;
	struct RtnDat a;
	CAD cad;
	int Ret;

	/* 1 線の角度(Angle) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = la(&a);
	Angle = a.angle;

	/* 2 線の長さ(LineLen) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = pp(&a);
	LineLen = a.l;

	/* 3 start_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = sagcad_dimension.assistance_line_space / printer.scale;
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = (sagcad_dimension.assistance_line_space / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = sagcad_dimension.assistance_line_space;
	else 
		a.l = sagcad_dimension.assistance_line_space;
	Ret = pap(&a);
	start_pointX = a.ex[1];
	start_pointY = a.ey[1];

	/* 4 end_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = LineLen + sagcad_dimension.assistance_line_extension / printer.scale;
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = LineLen + (sagcad_dimension.assistance_line_extension / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = LineLen + sagcad_dimension.assistance_line_extension;
	else 
		a.l = LineLen + sagcad_dimension.assistance_line_extension;
	Ret = pap(&a);
	end_pointX = a.ex[1];
	end_pointY = a.ey[1];

	/* 5 補助線を描画*/
//	LineBasic(hDC, start_pointX, start_pointY, end_pointX, end_pointY, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = start_pointX;
	cad.sy = start_pointY;
	cad.ex = end_pointX;
	cad.ey = end_pointY;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);

	return 1;
}





/* -------------------------------------------------------------------
 * (矢印)補助線の延長を描画 (defin = 80)
 *	
 * (SX,SY)-(EX,EY) で指定された直線を sagcad_dimension.arrow_line_extension 
 * で指定された長さだけ終点を延長する。
 *	
 * 1 線の角度(Angle) を求める。
 * 2 線の長さ(LineLen) を求める。
 * 3 end_point を求める。
 * 4 補助線を描画
 *	
 */
int DXF_LineEndExtension(FILE *stream, int LAYER, double SX, double SY, double EX, double EY, int DrawType)
{
	double Angle, LineLen, end_pointX, end_pointY;
	struct RtnDat a;
	CAD cad;
	int Ret;

	/* 1 線の角度(Angle) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = la(&a);
	Angle = a.angle;

	/* 2 線の長さ(LineLen) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = pp(&a);
	LineLen = a.l;

	/* 3 end_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = LineLen + (sagcad_dimension.arrow_line_extension / printer.scale);
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = LineLen + (sagcad_dimension.arrow_line_extension / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = LineLen + sagcad_dimension.arrow_line_extension;
	else 
		a.l = LineLen + sagcad_dimension.arrow_line_extension;

	Ret = pap(&a);
	end_pointX = a.ex[1];
	end_pointY = a.ey[1];

	/* 4 補助線を描画*/
//	LineBasic(hDC, SX, SY, end_pointX, end_pointY, 1);
	cad.layer = LAYER;
	cad.code = 1;
	cad.style = 5;
	cad.color = 0xffffff;
	cad.sx = SX;
	cad.sy = SY;
	cad.ex = end_pointX;
	cad.ey = end_pointY;
	cad.cx = 0;
	cad.cy = 0;
	cad.r = 0;
	cad.OnScreen = 0;
	entities_section_line_write(&cad, stream);
	return 1;
}





/* ====================================================================
 * ===	Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *	  Project			   : SagCAD
 *	  Source			   : DXF_write.cpp
 * ====================================================================
 */





/* -------------------------------------------------------------------
 * SgyToDxf_color
 *	
 *	SGY のカラーを DXF のカラーに変更
 *	
 */
int SgyToDxf_color(long SgyColor)
{
	int i;
	static const long colormap[257]=
	{
		0x000000,	// color 0	Black, also By Block
		0xff0000,	// color 1	Red
		0xffff00,	// color 2	Yellow
		0x00ff00,	// color 3	Green
		0x00ffff,	// color 4	Cyan
		0x0000ff,	// color 5	Blue
		0xff00ff,	// color 6	Magenta
		0xffffff,	// color 7	White
		0x989898,	// color 8	Grey
		0xC0C0C0,	// color 9
		0xff0000,	// color 10
		0xff8080,	// color 11
		0xa60000,	// color 12
		0xa65353,	// color 13
		0x800000,	// color 14
		0x804040,	// color 15
		0x4c0000,	// color 16
		0x4c2626,	// color 17
		0x260000,	// color 18
		0x261313,	// color 19
		0xff4000,	// color 20
		0xff9f80,	// color 21
		0xa62900,	// color 22
		0xa66853,	// color 23
		0x802000,	// color 24
		0x805040,	// color 25
		0x4c1300,	// color 26
		0x4c3026,	// color 27
		0x260a00,	// color 28
		0x261813,	// color 29
		0xff8000,	// color 30
		0xffbf80,	// color 31
		0xa65300,	// color 32
		0xa67c53,	// color 33
		0x804000,	// color 34
		0x806040,	// color 35
		0x4c2600,	// color 36
		0x4c3926,	// color 37
		0x261300,	// color 38
		0x261d13,	// color 39
		0xffbf00,	// color 40
		0xffdf80,	// color 41
		0xa67c00,	// color 42
		0xa69153,	// color 43
		0x806000,	// color 44
		0x807040,	// color 45
		0x4c3900,	// color 46
		0x4c4326,	// color 47
		0x261d00,	// color 48
		0x262113,	// color 49
		0xffff00,	// color 50
		0xffff80,	// color 51
		0xa6a600,	// color 52
		0xa6a653,	// color 53
		0x808000,	// color 54
		0x808040,	// color 55
		0x4c4c00,	// color 56
		0x4c4c26,	// color 57
		0x262600,	// color 58
		0x262613,	// color 59
		0xbfff00,	// color 60
		0xdfff80,	// color 61
		0x7ca600,	// color 62
		0x91a653,	// color 63
		0x608000,	// color 64
		0x708040,	// color 65
		0x394c00,	// color 66
		0x434c26,	// color 67
		0x1d2600,	// color 68
		0x212613,	// color 69
		0x80ff00,	// color 70
		0xbfff80,	// color 71
		0x53a600,	// color 72
		0x7ca653,	// color 73
		0x408000,	// color 74
		0x608040,	// color 75
		0x264c00,	// color 76
		0x394c26,	// color 77
		0x132600,	// color 78
		0x1d2613,	// color 79
		0x40ff00,	// color 80
		0x9fff80,	// color 81
		0x29a600,	// color 82
		0x68a653,	// color 83
		0x208000,	// color 84
		0x508040,	// color 85
		0x134c00,	// color 86
		0x304c26,	// color 87
		0x0a2600,	// color 88
		0x182613,	// color 89
		0x00ff00,	// color 90
		0x80ff80,	// color 91
		0x00a600,	// color 92
		0x53a653,	// color 93
		0x008000,	// color 94
		0x408040,	// color 95
		0x004c00,	// color 96
		0x264c26,	// color 97
		0x002600,	// color 98
		0x132613,	// color 99
		0x00ff40,	// color 100
		0x80ff9f,	// color 101
		0x00a629,	// color 102
		0x53a668,	// color 103
		0x008020,	// color 104
		0x408050,	// color 105
		0x004c13,	// color 106
		0x264c30,	// color 107
		0x00260a,	// color 108
		0x132618,	// color 109
		0x00ff80,	// color 110
		0x80ffbf,	// color 111
		0x00a653,	// color 112
		0x53a67c,	// color 113
		0x008040,	// color 114
		0x408060,	// color 115
		0x004c26,	// color 116
		0x264c39,	// color 117
		0x002613,	// color 118
		0x13261d,	// color 119
		0x00ffbf,	// color 120
		0x80ffdf,	// color 121
		0x00a67c,	// color 122
		0x53a691,	// color 123
		0x008060,	// color 124
		0x408070,	// color 125
		0x004c39,	// color 126
		0x264c43,	// color 127
		0x00261d,	// color 128
		0x132621,	// color 129
		0x00ffff,	// color 130
		0x80ffff,	// color 131
		0x00a6a6,	// color 132
		0x53a6a6,	// color 133
		0x008080,	// color 134
		0x408080,	// color 135
		0x004c4c,	// color 136
		0x264c4c,	// color 137
		0x002626,	// color 138
		0x132626,	// color 139
		0x00bfff,	// color 140
		0x80dfff,	// color 141
		0x007ca6,	// color 142
		0x5391a6,	// color 143
		0x006080,	// color 144
		0x407080,	// color 145
		0x00394c,	// color 146
		0x26434c,	// color 147
		0x001d26,	// color 148
		0x132126,	// color 149
		0x0080ff,	// color 150
		0x80bfff,	// color 151
		0x0053a6,	// color 152
		0x537ca6,	// color 153
		0x004080,	// color 154
		0x406080,	// color 155
		0x00264c,	// color 156
		0x26394c,	// color 157
		0x001326,	// color 158
		0x131d26,	// color 159
		0x0040ff,	// color 160
		0x809fff,	// color 161
		0x0029a6,	// color 162
		0x5368a6,	// color 163
		0x002080,	// color 164
		0x405080,	// color 165
		0x00134c,	// color 166
		0x26304c,	// color 167
		0x000a26,	// color 168
		0x131826,	// color 169
		0x0000ff,	// color 170
		0x8080ff,	// color 171
		0x0000a6,	// color 172
		0x5353a6,	// color 173
		0x000080,	// color 174
		0x404080,	// color 175
		0x00004c,	// color 176
		0x26264c,	// color 177
		0x000026,	// color 178
		0x131326,	// color 179
		0x4000ff,	// color 180
		0x9f80ff,	// color 181
		0x2900a6,	// color 182
		0x6853a6,	// color 183
		0x200080,	// color 184
		0x504080,	// color 185
		0x13004c,	// color 186
		0x30264c,	// color 187
		0x0a0026,	// color 188
		0x181326,	// color 189
		0x8000ff,	// color 190
		0xbf80ff,	// color 191
		0x5300a6,	// color 192
		0x7c53a6,	// color 193
		0x400080,	// color 194
		0x604080,	// color 195
		0x26004c,	// color 196
		0x39264c,	// color 197
		0x130026,	// color 198
		0x1d1326,	// color 199
		0xbf00ff,	// color 200
		0xdf80ff,	// color 201
		0x7c00a6,	// color 202
		0x9153a6,	// color 203
		0x600080,	// color 204
		0x704080,	// color 205
		0x39004c,	// color 206
		0x43264c,	// color 207
		0x1d0026,	// color 208
		0x211326,	// color 209
		0xff00ff,	// color 210
		0xff80ff,	// color 211
		0xa600a6,	// color 212
		0xa653a6,	// color 213
		0x800080,	// color 214
		0x804080,	// color 215
		0x4c004c,	// color 216
		0x4c264c,	// color 217
		0x260026,	// color 218
		0x261326,	// color 219
		0xff00bf,	// color 220
		0xff80df,	// color 221
		0xa6007c,	// color 222
		0xa65391,	// color 223
		0x800060,	// color 224
		0x804070,	// color 225
		0x4c0039,	// color 226
		0x4c2643,	// color 227
		0x26001d,	// color 228
		0x261321,	// color 229
		0xff0080,	// color 230
		0xff80bf,	// color 231
		0xa60053,	// color 232
		0xa6537c,	// color 233
		0x800040,	// color 234
		0x804060,	// color 235
		0x4c0026,	// color 236
		0x4c2639,	// color 237
		0x260013,	// color 238
		0x26131d,	// color 239
		0xff0040,	// color 240
		0xff809f,	// color 241
		0xa60029,	// color 242
		0xa65368,	// color 243
		0x800020,	// color 244
		0x804050,	// color 245
		0x4c0013,	// color 246
		0x4c2630,	// color 247
		0x26000a,	// color 248
		0x261318,	// color 249
		0x545454,	// color 250
		0x767676,	// color 251
		0x989898,	// color 252
		0xbbbbbb,	// color 253
		0xdddddd,	// color 254
		0xffffff,	// color 255
		0x000000 	// color 256 Black, also by By Layer
	};

	for (i = 0 ; i < 257 ; i++) {
		if (SgyColor == colormap[i]) {
			return i;
		}
	}
	return 3;
}





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : DXF_write.c
 * ====================================================================
 */

