# $Id: preferences.rb,v 1.2 2005/02/20 16:50:49 kentarow Exp $

require "rexml/document"
require "iconv"
require "ostruct"  
require "gtk2"

##
#
#
class Preferences
  class Error < StandardError; end

  CODING = 'EUC-JP'

  # 
  def initialize(source, userfile)
    @filename = userfile
    @widget = nil
    @widget_hash = Hash.new

    File.open(source) { |fd|
      @xml = REXML::Document.new(decode(fd.read))
    }
    @root = @xml.elements['preferences']
    @elem_hash = set_hash(Hash.new, @root)

    if File.exist?(userfile)
      xml = nil
      File.open(userfile) { |fd|
        xml = REXML::Document.new(decode(fd.read))
      }
      root = xml.elements['preferences']
      user_hash = set_hash(Hash.new, root)
      @elem_hash.each_key { |key|
        if user_hash[key]
          self[key] = user_hash[key].text
        end
      }
      set_size(root.attributes['width'].to_i, 
               root.attributes['height'].to_i)
    end
  end

  #
  def [](field)
    if @elem_hash[field]
      @elem_hash[field].text.to_s
    else
      raise(Error, "no such field, '#{field}'")
    end
  end

  #
  def []=(field, text)
    if @elem_hash[field]
      if @elem_hash[field].text != text.to_s
        @elem_hash[field].text = text.to_s
      end
    else
      raise(Error, "no such field, '#{field}'")
    end
  end

  # ɥ
  def window(title = 'Preferences')
    unless @window
      @widget_hash.clear
      vbox = Gtk::VBox.new
      vbox.set_border_width(4)
      vbox.add(make_table(@root))
      bbox = Gtk::HButtonBox.new

      button = Gtk::Button.new(Gtk::Stock::OK)
      button.signal_connect('clicked') {
        apply
        @window.destroy
      }
      bbox.add(button)

      button = Gtk::Button.new(Gtk::Stock::APPLY)
      button.signal_connect('clicked') { apply }
      bbox.add(button)

      button = Gtk::Button.new(Gtk::Stock::CANCEL)
      button.signal_connect('clicked') { @window.destroy }
      bbox.add(button)

      vbox.pack_start(bbox, false)

      @window = Gtk::Window.new
      @window.set_title(title)
      @window.set_default_size(@root.attributes['width'].to_i, 
                               @root.attributes['height'].to_i)
      @window.add(vbox)
      @window.signal_connect('destroy') { @widget = nil }
      @window.show_all
    end
    @window
  end

  #
  def set_size(width, height)
    @root.attributes['width'] = width.to_s
    @root.attributes['height'] = height.to_s
  end

  #
  def save
    File.open(@filename, "w") { |fd|
      fd.write(encode(@xml.to_s))
    }
  end

  private

  #
  def set_hash(hash, xml)
    xml.each { |elem|
      if elem.is_a?(REXML::Element)
        case elem.name
        when 'frame', 'hidden'
          set_hash(hash, elem)
        else
          if hash[elem.name]
            raise(Error, "duplicate field, '#{elem.name}'")
          end
          hash[elem.name] = elem
        end
      end
    }
    hash
  end

  #
  def decode(buff)
    Iconv.conv('UTF-8', CODING, buff)
  end

  #
  def encode(buff)
    Iconv.conv(CODING, 'UTF-8', buff)
  end

  #
  def make_table(xml)
    table = Gtk::Table.new(0, 2)
    table.set_border_width(4)
    row = 0
    xml.each { |elem|
      if elem.is_a?(REXML::Element)
        case elem.name
        when 'frame'
          frame = Gtk::Frame.new(elem.attributes['label'])
          frame.add(make_table(elem))
          table.n_rows = row + 1
          table.attach(frame, 0, 2, row, row + 1)
          row += 1
        when 'hidden'
          # nothing
        else
          case elem.attributes['class'].upcase.intern
          when :STRING
            table.n_rows = row + 1
            table.attach(label(elem), 0, 1, row, row + 1, Gtk::FILL)
            table.attach(entry(elem), 1, 2, row, row + 1)
            row += 1
          when :MENU
            table.n_rows = row + 1
            table.attach(label(elem), 0, 1, row, row + 1, Gtk::FILL)
            table.attach(menu(elem), 1, 2, row, row + 1)
            row += 1
          end
        end
      end
    }
    table
  end

  #
  def label(elem)
    label = Gtk::Label.new(elem.attributes['label'])
    label.set_xalign(1.0)
    label.set_xpad(8)
    label
  end

  #
  def entry(elem)
    entry = Gtk::Entry.new
    entry.set_text(elem.text.to_s)
    @widget_hash[elem.name] = OpenStruct.new(:widget => entry)
    entry
  end

  #
  def menu(elem)
    menu = Gtk::Menu.new
    group = nil
    list = Array.new
    elem.elements['items'].each { |item|
      if item.is_a?(REXML::Element)
        menuitem = Gtk::RadioMenuItem.new(group, item.attributes['label'])
        group = menuitem
        menu.append(menuitem)
        list << item.name
      end
    }
    optionmenu = Gtk::OptionMenu.new
    optionmenu.set_menu(menu)
    optionmenu.set_history(list.index(elem.text.to_s).to_i)
    @widget_hash[elem.name] = OpenStruct.new(:widget => optionmenu, 
                                             :list => list)
    optionmenu
  end

  #
  def apply
    @widget_hash.each { |field, elem|
      case elem.widget
      when Gtk::Entry
        self[field] = elem.widget.text
      when Gtk::OptionMenu
        self[field] = elem.list[elem.widget.history]
      end
    }
  end
end

if $0 == __FILE__
  pref = Preferences.new("rucd.xml", "user.xml")

  Gtk.init

  window = pref.window
  window.signal_connect('size-allocate') {
    pref.set_size(*window.size)
  }
  window.signal_connect('destroy') {
    pref.save
    exit
  }

  Gtk.main
end
