// -*-c++-*-

/*!
  \file view_config_control.cpp
  \brief field canvas configuration control class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <cassert>

#include "app_config.h"
#include "main_data.h"
#include "main_frame.h"
#include "score_board_painter.h"

#include "view_config.h"
#include "view_config_dialog.h"
#include "debug_message_frame.h"

#include "view_config_control.h"

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigControl::ViewConfigControl( MainFrame * main_frame,
                                      const MainData & main_data )
    : M_main_frame( main_frame )
    , M_main_data( main_data )
    , M_config_data( static_cast< ViewConfig * >( 0 ) )
    , M_config_dlg( static_cast< ViewConfigDialog * >( 0 ) )
    , M_debug_frame( static_cast< DebugMessageFrame * >( 0 ) )
{
    assert( main_frame );

    M_main_frame->connect( SWID_CANVAS_GRASS_LINES,
                           this, &ViewConfigControl::recvGrassLines );
    M_main_frame->connect( SWID_CANVAS_GRASS_CHECKER,
                           this, &ViewConfigControl::recvGrassChecker );
    M_main_frame->connect( SWID_CANVAS_GRASS_NORMAL,
                           this, &ViewConfigControl::recvGrassNormal );

    M_main_frame->connect( SWID_CANVAS_ZOOMIN,
                           this, &ViewConfigControl::recvZoomIn );
    M_main_frame->connect( SWID_CANVAS_ZOOMOUT,
                           this, &ViewConfigControl::recvZoomOut );
    M_main_frame->connect( SWID_CANVAS_UNZOOM,
                           this, &ViewConfigControl::recvUnzoom );
    M_main_frame->connect( SWID_CANVAS_ENLARGE,
                           this, &ViewConfigControl::recvEnlarge );

    M_main_frame->connect( SWID_CANVAS_DRAW_SCORE_BOARD,
                           this, &ViewConfigControl::recvShowScoreBoard );
    M_main_frame->connect( SWID_CANVAS_DRAW_BALL,
                           this, &ViewConfigControl::recvShowBall );
    M_main_frame->connect( SWID_CANVAS_DRAW_PLAYER,
                           this, &ViewConfigControl::recvShowPlayers );

    M_main_frame->connect( SWID_CANVAS_DRAW_PLAYER_NUMBER,
                           this, &ViewConfigControl::recvShowPlayerNumber );
    M_main_frame->connect( SWID_CANVAS_DRAW_STAMINA,
                           this, &ViewConfigControl::recvShowStamina );
    M_main_frame->connect( SWID_CANVAS_DRAW_HETERO_NUMBER,
                           this, &ViewConfigControl::recvShowHeteroNumber );
    M_main_frame->connect( SWID_CANVAS_DRAW_VIEW_CONE,
                           this, &ViewConfigControl::recvShowViewCone );
    M_main_frame->connect( SWID_CANVAS_DRAW_CONTROL_AREA,
                           this, &ViewConfigControl::recvShowControlArea );
    M_main_frame->connect( SWID_CANVAS_DRAW_FLAGS,
                           this, &ViewConfigControl::recvShowFlags );
    M_main_frame->connect( SWID_CANVAS_DRAW_OFFSIDE_LINE,
                           this, &ViewConfigControl::recvShowOffsideLine );
    M_main_frame->connect( SWID_CANVAS_DRAW_VORONOI_DIAGRAM,
                           this, &ViewConfigControl::recvShowVoronoiDiagram );
    M_main_frame->connect( SWID_CANVAS_DRAW_DELAUNAY_TRIANGLE,
                           this, &ViewConfigControl::recvShowDelaunayTriangle );
    M_main_frame->connect( SWID_CHOICE_VORONOI_TARGET,
                           this, &ViewConfigControl::recvChoiceVoronoiTarget );

    M_main_frame->connect( SWID_CANVAS_SELECT_AUTO_ALL,
                           this, &ViewConfigControl::recvSelectAutoAll );
    M_main_frame->connect( SWID_CANVAS_SELECT_AUTO_LEFT,
                           this, &ViewConfigControl::recvSelectAutoLeft );
    M_main_frame->connect( SWID_CANVAS_SELECT_AUTO_RIGHT,
                           this, &ViewConfigControl::recvSelectAutoRight );
    M_main_frame->connect( SWID_CANVAS_SELECT_FIX,
                           this, &ViewConfigControl::recvSelectFix );
    M_main_frame->connect( SWID_CANVAS_SELECT_PLAYER,
                           this, &ViewConfigControl::recvSelectPlayer );
    M_main_frame->connect( SWID_CANVAS_UNSELECT_PLAYER,
                           this, &ViewConfigControl::recvUnselectPlayer );

    M_main_frame->connect( SWID_CANVAS_FOCUS_POINT,
                           this, &ViewConfigControl::recvFocusPoint );
    M_main_frame->connect( SWID_CANVAS_FOCUS_BALL,
                           this, &ViewConfigControl::recvFocusBall );
    M_main_frame->connect( SWID_CANVAS_FOCUS_PLAYER,
                           this, &ViewConfigControl::recvFocusPlayer );
    M_main_frame->connect( SWID_CANVAS_FOCUS_FIX,
                           this, &ViewConfigControl::recvFocusFix );


    M_main_frame->connect( SWID_CANVAS_TOGGLE_BALL_AUTO_TRACE,
                           this, &ViewConfigControl::recvToggleBallAutoTrace );
    M_main_frame->connect( SWID_CANVAS_TOGGLE_PLAYER_AUTO_TRACE,
                           this, &ViewConfigControl::recvTogglePlayerAutoTrace );
    M_main_frame->connect( SWID_SET_AUTO_TRACE_START,
                           this, &ViewConfigControl::recvSetAutoTraceStart );
    M_main_frame->connect( SWID_SET_AUTO_TRACE_PERIOD,
                           this, &ViewConfigControl::recvSetAutoTracePeriod );

    M_main_frame->connect( SWID_CANVAS_TOGGLE_TRACE_MODE,
                           this, &ViewConfigControl::recvToggleLineTrace );
    M_main_frame->connect( SWID_SET_BALL_TRACE_START,
                           this, &ViewConfigControl::recvSetBallTraceStart );
    M_main_frame->connect( SWID_SET_BALL_TRACE_END,
                           this, &ViewConfigControl::recvSetBallTraceEnd );
    M_main_frame->connect( SWID_SET_PLAYER_TRACE_START,
                           this, &ViewConfigControl::recvSetPlayerTraceStart );
    M_main_frame->connect( SWID_SET_PLAYER_TRACE_END,
                           this, &ViewConfigControl::recvSetPlayerTraceEnd );

    M_main_frame->connect( SWID_CANVAS_BALL_TRACE_SELECT_ALL,
                           this, &ViewConfigControl::recvBallTraceSelectAll );
    M_main_frame->connect( SWID_CANVAS_PLAYER_TRACE_SELECT_ALL,
                           this, &ViewConfigControl::recvPlayerTraceSelectAll );

    M_main_frame->connect( SWID_SET_BALL_FUTURE_CYCLE,
                           this, &ViewConfigControl::recvSetBallFutureCycle );
    M_main_frame->connect( SWID_SET_PLAYER_FUTURE_CYCLE,
                           this, &ViewConfigControl::recvSetPlayerFutureCycle );

    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW,
                           this, &ViewConfigControl::recvShowDebugView );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_BALL,
                           this, &ViewConfigControl::recvShowDebugViewBall );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_SELF,
                           this, &ViewConfigControl::recvShowDebugViewSelf );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_PLAYERS,
                           this, &ViewConfigControl::recvShowDebugViewPlayers );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_COMMENT,
                           this, &ViewConfigControl::recvShowDebugViewComment );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_FIGURE,
                           this, &ViewConfigControl::recvShowDebugViewFigure );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_TARGET,
                           this, &ViewConfigControl::recvShowDebugViewTarget );
    M_main_frame->connect( SWID_CANVAS_DRAW_DEBUG_VIEW_MESSAGE,
                           this, &ViewConfigControl::recvShowDebugViewMessage );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::setData( ViewConfig * data )
{
    assert( data );
    M_config_data = data;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::setDialog( ViewConfigDialog * dlg )
{
    assert( dlg );
    M_config_dlg = dlg;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::setDebugMessageFrame( DebugMessageFrame * debug_frame )
{
    assert( debug_frame );
    M_debug_frame = debug_frame;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvGrassLines( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setGrassType( ViewConfig::GRASS_LINES );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setGrassType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvGrassChecker( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setGrassType( ViewConfig::GRASS_CHECKER );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setGrassType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvGrassNormal( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setGrassType( ViewConfig::GRASS_NORMAL );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setGrassType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvZoomIn( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->zoomIn();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setScale();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvZoomOut( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->zoomOut();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setScale();
    }

    M_main_frame->updateView();
}


/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::setScale( const double & scale )
{
    if ( M_config_data )
    {
        M_config_data->setScale( scale );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setScale();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvUnzoom( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->unzoom();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setScale();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvEnlarge( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleEnlarge();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setEnlarge();
    }

    M_main_frame->handle( EventMessage( SWID_CHANGED_VIEW_CONFIG ) );
    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowScoreBoard( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowScoreBoard();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowScoreBoard();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowBall( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowBall();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowBall();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowPlayers( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowPlayers();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowPlayers();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowPlayerNumber( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowPlayerNumber();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowPlayerNumber();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowStamina( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowStamina();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowStamina();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowHeteroNumber( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowHeteroNumber();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowHeteroNumber();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowViewCone( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowViewCone();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowViewCone();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowControlArea( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowControlArea();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowControlArea();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowFlags( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowFlags();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowFlags();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowOffsideLine( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowOffsideLine();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowOffsideLine();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowVoronoiDiagram( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowVoronoiDiagram();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowVoronoiDiagram();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDelaunayTriangle( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDelaunayTriangle();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setShowDelaunayTriangle();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvChoiceVoronoiTarget( const boost::any * data )
{
    const int * id = boost::any_cast< int >( data );
    if ( ! id )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }


    rcsc::SideID side = static_cast< rcsc::SideID >( *id );

    if ( M_config_data )
    {
        M_config_data->setVoronoiTarget( side );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setVoronoiTarget();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvFocusPoint( const boost::any * data )
{
    const wxPoint * point = boost::any_cast< wxPoint >( data );
    if ( ! point )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setFocusPoint( point->x, point->y );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setFocusType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvFocusBall( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setFocusType( ViewConfig::FOCUS_BALL );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setFocusType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvFocusPlayer( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setFocusType( ViewConfig::FOCUS_PLAYER );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setFocusType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvFocusFix( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setFocusType( ViewConfig::FOCUS_POINT );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setFocusType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSelectAutoAll( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerSelectType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSelectAutoLeft( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setPlayerSelectType( ViewConfig::SELECT_AUTO_LEFT );
    }
    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerSelectType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSelectAutoRight( const boost::any * )
{
    if ( M_config_data )
    {
        // toggle
        M_config_data->setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        if ( M_config_dlg )
        {
            M_config_dlg->setPlayerSelectType();
        }
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSelectFix( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->setPlayerSelectType( ViewConfig::SELECT_FIX );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerSelectType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSelectPlayer( const boost::any * data )
{
    const int * unum = boost::any_cast< int >( data );
    if ( ! unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    // if right player, *unum is negative value.
    rcsc::SideID side = rcsc::NEUTRAL;
    int set_unum = *unum;

    if ( 1 <= *unum && *unum <= 11 )
    {
        side = rcsc::LEFT;
    }
    else if ( -11 <= *unum && *unum <= -1 )
    {
        side = rcsc::RIGHT;
        set_unum = -set_unum;
    }
    else
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << "unexpected number " << *unum << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setPlayerSelectType( ViewConfig::SELECT_FIX );
        M_config_data->setSelectedNumber( side, set_unum );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerSelectType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvUnselectPlayer( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->unselectPlayer();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerSelectType();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvToggleBallAutoTrace( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleBallAutoTrace();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setBallAutoTrace();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvTogglePlayerAutoTrace( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->togglePlayerAutoTrace();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerAutoTrace();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetAutoTraceStart( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setAutoTraceStart( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setAutoTraceStart();
    }

    if ( ! M_config_data
         && ( M_config_data->isBallAutoTrace()
              || M_config_data->isPlayerAutoTrace() )
         )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetAutoTracePeriod( const boost::any * data )
{
    const std::size_t * period = boost::any_cast< std::size_t >( data );
    if ( ! period )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setAutoTracePeriod( *period );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setAutoTraceStart();
    }

    if ( ! M_config_data
         && ( M_config_data->isBallAutoTrace()
              || M_config_data->isPlayerAutoTrace() )
         )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvToggleLineTrace( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleLineTrace();
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setLineTrace();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetBallTraceStart( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setBallTraceStart( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setBallTraceStart();
    }

    if ( ! M_config_data
         && M_config_data->isShownBallTrace() )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetBallTraceEnd( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setBallTraceEnd( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setBallTraceEnd();
    }

    if ( ! M_config_data
         && M_config_data->isShownBallTrace() )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetPlayerTraceStart( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setPlayerTraceStart( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerTraceStart();
    }

    if ( ! M_config_data
         && M_config_data->isShownPlayerTrace() )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetPlayerTraceEnd( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setPlayerTraceEnd( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerTraceEnd();
    }

    if ( ! M_config_data
         && M_config_data->isShownPlayerTrace() )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvBallTraceSelectAll( const boost::any * )
{
    if ( M_config_data )
    {
        const std::vector< MonitorViewPtr > & view
            = M_main_data.getViewHolder().getMonitorViewCont();
        if ( ! view.empty() )
        {
            M_config_data->setBallTraceStart( view.front()->cycle() );
            M_config_data->setBallTraceEnd( view.back()->cycle() );
            if ( M_config_dlg )
            {
                M_config_dlg->setBallTraceStart();
                M_config_dlg->setBallTraceEnd();
            }
            M_main_frame->updateView();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvPlayerTraceSelectAll( const boost::any * )
{
    if ( M_config_data )
    {
        const std::vector< MonitorViewPtr > & view
            = M_main_data.getViewHolder().getMonitorViewCont();
        if ( ! view.empty() )
        {
            M_config_data->setPlayerTraceStart( view.front()->cycle() );
            M_config_data->setPlayerTraceEnd( view.back()->cycle() );
            if ( M_config_dlg )
            {
                M_config_dlg->setPlayerTraceStart();
                M_config_dlg->setPlayerTraceEnd();
            }
            M_main_frame->updateView();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetBallFutureCycle( const boost::any * data )
{
    const int * cycle = boost::any_cast< int >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setBallFutureCycle( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setBallFutureCycle();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvSetPlayerFutureCycle( const boost::any * data )
{
    const int * cycle = boost::any_cast< int >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( M_config_data )
    {
        M_config_data->setPlayerFutureCycle( *cycle );
    }

    if ( M_config_dlg )
    {
        M_config_dlg->setPlayerFutureCycle();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugView( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugView();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugView();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewSelf( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewSelf();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewSelf();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewBall( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewBall();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewBall();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewPlayers( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewPlayers();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewPlayers();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewComment( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewComment();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewComment();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewFigure( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewFigure();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewFigure();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewTarget( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewTarget();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewTarget();
    }

    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigControl::recvShowDebugViewMessage( const boost::any * )
{
    if ( M_config_data )
    {
        M_config_data->toggleShowDebugViewMessage();
    }

    if ( M_debug_frame )
    {
        M_debug_frame->setShowDebugViewMessage();
    }

    M_main_frame->updateView();
}
