// -*-c++-*-

/*!
  \file main_frame.cpp
  \brief main application frame class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/config.h>   // wxFileConfig
#include <wx/fileconf.h> // wxFileConfig
#include <wx/filename.h>
#include <wx/grid.h>
#include <wx/image.h> // wxInitAllImageHandler()
#include <wx/progdlg.h>
#endif

#include <iostream>

#include <rcsc/geom/vector_2d.h>

#include "id.h"
#include "soccerwindow_app.h"
#include "app_config.h"

#include "debug_message_frame.h"
#include "debug_server.h"
#include "detail_dialog.h"
#include "field_canvas.h"
#include "image_save_dialog.h"
#include "logplayer.h"
#include "main_data.h"
#include "main_toolbar.h"
#include "monitor_client.h"
//#include "monitor_toolbar.h"
#include "player_type_grid.h"
#include "view_config_control.h"
#include "view_config_dialog.h"

#include "main_frame.h"

#if defined(__WXGTK__) || defined(__WXX11__) || defined(__WXMOTIF__) || defined(__WXMAC__)
#  include "xpm/soccerwindow2-nostr.xpm"
#endif

/*-------------------------------------------------------------------*/
/*!
  constructing main frame.
*/
MainFrame::MainFrame( const wxString & title )
    : wxFrame( static_cast< wxWindow * >( 0 ), -1, title,
               wxDefaultPosition, //wxPoint( 64, 32 ),
               wxSize( 620, 480 ) )
    , M_data( this )
    , M_main_tbar( static_cast< MainToolBar * >( 0 ) )
    , M_field_canvas( static_cast< FieldCanvas * >( 0 ) )
    , M_view_config_ctrl( new ViewConfigControl( this, M_data ) )
    , M_view_config_dlg( static_cast< ViewConfigDialog * >( 0 ) )
    , M_detail_dlg( static_cast< DetailDialog * >( 0 ) )
    , M_image_save_dlg( static_cast< ImageSaveDialog * >( 0 ) )
    , M_debug_message( static_cast< DebugMessageFrame * >( 0 ) )
    , M_logplayer( new LogPlayer( this, M_data ) )
{
    // set event handlers
    connectEvent();

    // set minimal window size
    this->SetSizeHints( 280, 220 );
    // set loaded frame size (or keep default size)
    this->SetSize( AppConfig::instance().framePosX(),
                   AppConfig::instance().framePosY(),
                   AppConfig::instance().frameWidth(),
                   AppConfig::instance().frameHeight() );
    //this->Center();

    // load options from config file
    loadConfig();

    // create attached tools.
    createMenuBar();
    createToolBar();
    createStatusBar();

    // create internal windows
    createWindows();

    //this->SetAutoLayout( true );

    // apply initial window options.
    if ( AppConfig::instance().hideToolBar() )
    {
        if ( this->GetToolBar() )
        {
            this->GetToolBar()->Hide();
            this->Fit();
        }
    }

    if ( AppConfig::instance().hideStatusBar() )
    {
        if ( this->GetStatusBar() )
        {
            this->GetStatusBar()->Hide();
            this->Fit();
        }
    }

    if ( AppConfig::instance().maximize() )
    {
        this->Maximize( true );
    }

    if ( AppConfig::instance().fullScreen() )
    {
        this->ShowFullScreen( true );
    }

    M_view_config_ctrl->setData( & M_data.getViewConfig() );

    SetIcon( wxICON( soccerwindow2_nostr ) );
}

/*-------------------------------------------------------------------*/
/*!
  destructing main frame.
*/
MainFrame::~MainFrame()
{
    if ( isMonitorConnected()
         && AppConfig::instance().killServer() )
    {
        M_monitor_client->killServer();
    }
    saveConfig();
}

/*-------------------------------------------------------------------*/
/*!
  creates dynamic event signal connect
*/
void
MainFrame::connectEvent()
{
    // ---------------------------------------------------
    // register wx event hander
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( MainFrame::handleCloseEvent ) );

    // command event : menu
    Connect( wxID_ANY, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleMenuEvent ) );
    // command event : button
    Connect( wxID_ANY, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( MainFrame::handleCommandEvent ) );

    Connect( wxID_ANY, wxEVT_SIZE,
             wxSizeEventHandler( MainFrame::handleSizeEvent ) );

    // ---------------------------------------------------
    // register original event handler

    // frame related
    connect( SWID_QUIT, this, &MainFrame::recvQuit );
    connect( SWID_SHOW_ABOUT_DIALOG, this, &MainFrame::recvShowAboutDialog );

    // show/toggle switches
    connect( SWID_TOGGLE_MENUBAR, this, &MainFrame::recvToggleMenuBar );
    connect( SWID_TOGGLE_TOOLBAR, this, &MainFrame::recvToggleToolBar );
    connect( SWID_TOGGLE_STATUSBAR, this, &MainFrame::recvToggleStatusBar );
    connect( SWID_TOGGLE_FULLSCREEN, this, &MainFrame::recvToggleFullScreen );

    connect( SWID_SHOW_OPEN_RCG_DIALOG, this, &MainFrame::recvShowOpenRCGDialog );
    connect( SWID_SHOW_CONNECT_DIALOG, this, &MainFrame::recvShowConnectDialog );
    connect( SWID_SHOW_PLAYER_TYPE_DIALOG, this, &MainFrame::recvShowPlayerTypeDialog );
    connect( SWID_SHOW_DETAIL_DIALOG, this, &MainFrame::recvShowDetailDialog );
    connect( SWID_SHOW_IMAGE_SAVE_DIALOG, this, &MainFrame::recvShowImageSaveDialog );
    connect( SWID_SHOW_VIEW_CONFIG_DIALOG, this, &MainFrame::recvShowViewConfigDialog );
    connect( SWID_SHOW_DEBUG_MESSAGE_FRAME, this, &MainFrame::recvShowDebugMessageFrame );


    // monitor related
    connect( SWID_MONITOR_CONNECT, this, &MainFrame::recvMonitorConnect );
    connect( SWID_MONITOR_DISCONNECT, this, &MainFrame::recvMonitorDisconnect ) ;

    connect( SWID_REQUEST_LIVE_MODE, this, &MainFrame::recvRequestLiveMode );

    connect( SWID_REQUEST_RESTART_SERVER, this, &MainFrame::recvRequestRestartServer );

    connect( SWID_RECEIVE_MONITOR_PACKET, this, &MainFrame::recvReceiveMonitorPacket );

    // debug server related
    connect( SWID_DEBUG_SERVER_START, this, &MainFrame::recvDebugServerStart );
    connect( SWID_DEBUG_SERVER_STOP, this, &MainFrame::recvDebugServerStop );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize menu & menu bar.
  called from constructor.
*/
void
MainFrame::createMenuBar()
{
    // create menu
    wxMenuBar * menubar = new wxMenuBar(
#if 0
#ifdef __WXGTK__
                                        wxMB_DOCKABLE
#endif
#endif
                                        );

    // attach each menu tree
    menubar->Append( createMenuFile(), _( "&File" ) );
    menubar->Append( createMenuMonitor(), _( "&Monitor" ) );
    //menubar->Append( menu_logplayer, _( "&Logplayer" ) );
    menubar->Append( createMenuView(), _( "&View" ) );
    //menubar->Append( createMenuDialog(), _( "D&ialog" ) );
    //menubar->Append( createMenuDebug(), _( "&Debug" ) );
    menubar->Append( createMenuTool(), _( "&Tool" ) );
    menubar->Append( createMenuHelp(), _( "&Help" ) );

    // append menubar to MainFrame.
    this->SetMenuBar( menubar );
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuFile()
{
    wxMenu * menu = new wxMenu;
    menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                  _( "&Open rcg file...\tctrl-o" ),
                  _( "Open RoboCup Game Log file" ) );
    menu->AppendSeparator();

    menu->Append( SWID_QUIT,
                  _( "&Quit\tctrl-q" ),
                  _( "Quit application" ) );

    return menu;
}


/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuMonitor()
{
    wxMenu * menu = new wxMenu;
    menu->Append( SWID_MONITOR_KICKOFF,
                  _( "&KickOff\tctrl-k" ),
                  _( "Start the game" ) );
    menu->Append( SWID_REQUEST_LIVE_MODE,
                  _( "&Live Mode\tctrl-l" ),
                  _( "Live mode" ) );
    menu->AppendSeparator();
    menu->Append( SWID_MONITOR_CONNECT,
                  _( "&Connect\tctrl-c" ),
                  _( "Connect to the rcssserver on localhost" ) );
    menu->Append( SWID_SHOW_CONNECT_DIALOG,
                  _( "Connect &to ..." ),
                  _( "Connect to the rcssserver on other host" ) );
    menu->Append( SWID_MONITOR_DISCONNECT,
                  _( "&Disconnect" ),
                  _( "Disonnect from rcssserver" ) );
    menu->AppendSeparator();
    menu->Append( SWID_REQUEST_KILL_SERVER,
                  _( "&Kill server" ),
                  _( "Kill the rcssserver process" ) );
    menu->Append( SWID_REQUEST_RESTART_SERVER,
                  _( "(Re)&start server" ),
                  _( "(Re)start rcssserver" ) );
    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuView()
{
    wxMenu * menu = new wxMenu;

    ////////////////////////////////////////////////////////
#if 0
    menu->Append( SWID_TOGGLE_MENUBAR,
                  _( "&Menu Bar\tctrl-m" ),
                  _( "Show/Hide Menu Bar" ) );
#endif
    menu->Append( SWID_TOGGLE_TOOLBAR,
                  _( "&Tool Bar" ),
                  _( "Show/Hide Tool Bar" ) );
    menu->Append( SWID_TOGGLE_STATUSBAR,
                  _( "&Status Bar" ),
                  _( "Show/Hide Status Bar" ) );
    menu->AppendSeparator();
    menu->Append( SWID_TOGGLE_FULLSCREEN,
                  _( "&Full Screen\talt-enter" ),
                  _( "Toggle Full Screen" ) );
    ////////////////////////////////////////////////////////
    menu->AppendSeparator();

    menu->Append( SWID_SHOW_PLAYER_TYPE_DIALOG,
                  _( "&Player Type List\tctrl-t" ),
                  _( "Show player type parameters dialog" ) );
    menu->Append( SWID_SHOW_DETAIL_DIALOG,
                  _( "&Object Detail\tctrl-i" ),
                  _( "Show detail information dialog" ) );
    ////////////////////////////////////////////////////////
    menu->AppendSeparator();

    menu->Append( SWID_SHOW_VIEW_CONFIG_DIALOG,
                  _( "&Preference\tctrl-v" ),
                  _( "Show view preference dialog" ) );
    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuTool()
{
    wxMenu * menu = new wxMenu;

    menu->Append( SWID_SHOW_DEBUG_MESSAGE_FRAME,
                  _( "Debug &Message\tctrl-d" ),
                  _( "Show debug message window" ) );

    menu->AppendSeparator();


    menu->Append( SWID_DEBUG_SERVER_START,
                  _( "Start Debug Server" ),
                  _( "Start to wait debug connection." ) );
    menu->Enable( SWID_DEBUG_SERVER_START, false );
    menu->Append( SWID_DEBUG_SERVER_STOP,
                  _( "Stop Debug Server" ),
                  _( "Disconnect debug connection." ) );
    menu->Enable( SWID_DEBUG_SERVER_STOP, false );

    menu->AppendSeparator();

    menu->Append( SWID_SHOW_IMAGE_SAVE_DIALOG,
                  _( "Save &Image" ),
                  _( "Save game log data as image files" ) );

    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuHelp()
{
    wxMenu * menu = new wxMenu;

    menu->Append( SWID_SHOW_ABOUT_DIALOG,
                  _( "&About" ),
                  _( "About soccerwindow2" ) );

    return menu;
}

/*-------------------------------------------------------------------*/
/*!
  create main toolbar.
*/
wxToolBar *
MainFrame::OnCreateToolBar( long style,
                            wxWindowID id,
                            const wxString & name )
{
    M_main_tbar = new MainToolBar( this, style, this, M_data );
    return M_main_tbar;
}

/*-------------------------------------------------------------------*/
/*!
  create & realize toolbar.
  called from constructor.
*/
void
MainFrame::createToolBar()
{
    const long style = ( wxTB_FLAT
                         //#ifdef __WXGTK__
                         //| wxTB_DOCKABLE
                         //#endif
                         );
    this->CreateToolBar( style );

    //wxToolBar * tbar = new MonitorToolBar( this,
    //                                       wxTB_FLAT
    //                                       | wxVERTICAL );
    //tbar->SetSize(-1, this->GetClientSize().y );
    //tbar->Move( 0, 0 );
    //tbar->Realize();
}

/*-------------------------------------------------------------------*/
/*!
  create & realize status bar.
  called from constructor.
*/
void
MainFrame::createStatusBar()
{
    // already exist
    if ( this->GetStatusBar() )
    {
        return;
    }

    const int chw = this->GetCharWidth();
    const int widths[] = { -1, 20 * chw };
    this->CreateStatusBar( WXSIZEOF( widths ),
                           wxST_SIZEGRIP,
                           SWID_STATUSBAR );
    this->SetStatusText( wxT( "Ready" ), 0 );
    this->GetStatusBar()->SetStatusWidths( WXSIZEOF( widths ), widths );
}

/*-------------------------------------------------------------------*/
/*!
  \brief create all windows that has this mainframe as a parent
*/
void
MainFrame::createWindows()
{
    M_field_canvas = new FieldCanvas( this, this, M_data );
    M_field_canvas->SetSize( this->GetClientSize() );
    //M_field_canvas->SetSize( 20, 0, this->GetClientSize().x - 20, this->GetClientSize().y );

    M_field_canvas->SetFocus();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::loadConfig()
{
    //std::cerr << "MainFrame::loadConfig" << std::endl;
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();
    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/MainFrame" ) );

    config_file.SetPath( old_path );

    ///////////////////////////////////////////////////////////////

    old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/App" ) );

    wxString tmp_str;

    if ( AppConfig::instance().gameLogDir().empty() )
    {
        config_file.Read( wxT( "game-log-dir" ), &tmp_str, wxT( "" ) );
#ifdef UNICODE
        AppConfig::instance().setGameLogDir( (const char*)tmp_str.mbc_str() );
#else
        AppConfig::instance().setGameLogDir( (const char*)tmp_str.c_str() );
#endif
    }

    if ( AppConfig::instance().debugLogDir().empty() )
    {
        config_file.Read( wxT( "debug-log-dir" ), &tmp_str, wxT( "" ) );
#ifdef UNICODE
        AppConfig::instance().setDebugLogDir( (const char*)tmp_str.mbc_str() );
#else
        AppConfig::instance().setDebugLogDir( (const char*)tmp_str.c_str() );
#endif
    }

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!
  save configuration to file.
  called from destructor.
*/
void
MainFrame::saveConfig()
{
    //std::cerr << "MainFrame::saveConfig" << std::endl;
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();
    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/MainFrame" ) );

    config_file.SetPath( old_path );

    ///////////////////////////////////////////////////////////////

    old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/App" ) );

    config_file.Write( wxT( "server-command" ),
                       wxString( AppConfig::instance().serverPath().c_str(),
                                 *wxConvCurrent ) );

    config_file.Write( wxT( "game-log-dir" ),
                       wxString( AppConfig::instance().gameLogDir().c_str(),
                                 *wxConvCurrent ) );
    config_file.Write( wxT( "debug-log-dir" ),
                       wxString( AppConfig::instance().debugLogDir().c_str(),
                                 *wxConvCurrent ) );

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showAboutDialog()
{
    wxString msg( PACKAGE,
                  *wxConvCurrent );
    msg += wxT( "-" );
    msg += wxT( VERSION );
    msg += wxT( "\n\n"
                "soccerwindow2 is a viewer applicaton for\n"
                "the RoboCup Soccer Simulator.\n"
                "  http://sserver.sourceforge.net/\n"
                "\n"
                "soccerwindow2 Development Site:\n"
                "  http://rctools.sourceforge.jp/" );
    ::wxMessageBox( msg,
                    wxT( "About soccerwindow2" ),   // title
                    wxOK | wxICON_INFORMATION, // style
                    this ); // parent frame
}

/*-------------------------------------------------------------------*/
/*!
  \brief quit main frame
*/
void
MainFrame::quit()
{
    //std::cerr << "quit main frame" << std::endl;
    if ( isMonitorConnected()
         && AppConfig::instance().killServer() )
    {
        M_monitor_client->killServer();
    }
    this->Destroy();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleMenuBar()
{
    if ( this->GetMenuBar() )
    {
        if ( this->GetMenuBar()->IsShown() )
        {
            this->GetMenuBar()->Hide();
        }
        else
        {
            this->GetMenuBar()->Show();
        }
        this->Fit();
    }
    else
    {
        createMenuBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleToolBar()
{
    if ( this->GetToolBar() )
    {
        if ( this->GetToolBar()->IsShown() )
        {
            this->GetToolBar()->Hide();
        }
        else
        {
            this->GetToolBar()->Show();
        }
        this->Fit();
    }
    else
    {
        createToolBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleStatusBar()
{
    if ( this->GetStatusBar() )
    {
        if ( this->GetStatusBar()->IsShown() )
        {
            this->GetStatusBar()->Hide();
        }
        else
        {
            this->GetStatusBar()->Show();
            this->SetStatusText( wxT( "Ready" ), 0 );
        }
        this->Fit();
    }
    else
    {
        createStatusBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleFullScreen()
{
    this->ShowFullScreen( ! this->IsFullScreen() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleMaximize()
{
    this->Maximize( ! this->IsMaximized() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
MainFrame::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    //std::cerr << "MainFrame::handleCloseEvent()" << std::endl;
    this->quit();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
MainFrame::handleSizeEvent( wxSizeEvent & WXUNUSED( event ) )
{
    M_field_canvas->SetSize( this->GetClientSize() );

    //std::cerr << "sizeEvent width = " << this->GetClientSize().GetWidth()
    //<< " height = " << this->GetClientSize().GetHeight()
    //<< std::endl;
    M_data.getViewConfig().updateFieldSize( this->GetClientSize().GetWidth(),
                                            this->GetClientSize().GetHeight() );
    if ( M_view_config_dlg )
    {
        M_view_config_dlg->setScale();
    }

    //wxLayoutAlgorithm layout;
    //layout.LayoutFrame( this );

    //this->Layout();
}

/*-------------------------------------------------------------------*/
/*!
  menu event handler.
*/
void
MainFrame::handleMenuEvent( wxCommandEvent & event )
{
    //std::cerr << "MainFrame::handleMenuEvent" << std::endl;
    handle( EventMessage( event.GetId() ) );
}

/*-------------------------------------------------------------------*/
/*!
  command event handler.
*/
void
MainFrame::handleCommandEvent( wxCommandEvent & event )
{
    //std::cerr << "MainFrame::handleCommandEvent" << std::endl;
    handle( EventMessage( event.GetId() ) );
}

/*-------------------------------------------------------------------*/
/*!
  \brief show file open dialog, and load view data from .rcg[.gz] file
*/
void
MainFrame::showOpenRCGDialog()
{
#ifdef HAVE_LIBZ
    wxString wild_card( wxT( "Game Log files (*.rcg;*.rcg.gz)|*.rcg;*.rcg.gz"
                             "|All files (*.*)|*.*" ) );
#else
    wxString wild_card( wxT( "Game Log files (*.rcg)|*.rcg"
                             "|All files (*.*)|*.*" ) );
#endif

    wxString default_dir( AppConfig::instance().gameLogDir().c_str(),
                          *wxConvCurrent );

#if 1
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              default_dir, // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }
    wxString file_path = file_dialog.GetPath();
#else
    wxString file_path
        = ::wxFileSelector( wxT( "Open" ), // message
                            default_dir, // default dir
                            wxT( "" ), // default file name,
                            wxT( "" ), // default extention
                            wild_card, // wild card
                            wxOPEN, // flag,
                            this // parent window
                            );
    if ( file_path.IsEmpty() )
    {
        return;
    }
#endif

    // load rcg data from file
    std::string file_path_std;
#ifdef UNICODE
    file_path_std = (const char*)file_path.mbc_str();
#else
    file_path_std = (const char*)file_path.c_str();
#endif

    loadRCG( file_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainFrame::loadRCG( const std::string & filepath )
{
    wxString filepath_wx( filepath.c_str(),
                          *wxConvCurrent );

    if ( ! M_data.loadRCG( filepath ) )
    {
        // failed...
        wxString err_msg = wxT( "Failed to read [" );
        err_msg += filepath_wx;
        err_msg += wxT( "]!!!" );
        ::wxMessageBox( err_msg,
                        wxT( "RCG file read error!" ), // title
                        wxOK | wxICON_ERROR ); // style
        return false;
    }

    if ( M_data.getViewHolder().getMonitorViewCont().size() == 0 )
    {
        wxString err_msg = wxT( "Empty Log [" );
        err_msg += filepath_wx;
        err_msg += wxT( "]!!!" );
        ::wxMessageBox( err_msg,
                        wxT( "RCG file is empty!" ), // title
                        wxOK | wxICON_ERROR ); // style
        return false;
    }

    if ( ::wxIsAbsolutePath( filepath_wx ) )
    {
        std::string dir_str;
#ifdef UNICODE
        dir_str = (const char*)::wxPathOnly( filepath_wx ).mbc_str();
#else
        dir_str = (const char*)::wxPathOnly( filepath_wx ).c_str();
#endif
        AppConfig::instance().setGameLogDir( dir_str );
    }

    updateView();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::updateView()
{
    if ( M_main_tbar
         && M_main_tbar->IsShown() )
    {
        M_main_tbar->update();
    }

    if ( M_field_canvas
         && M_field_canvas->IsShown() )
    {
        M_field_canvas->draw();
    }

    if ( M_detail_dlg
         && M_detail_dlg->IsShown() )
    {
        M_detail_dlg->update();
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::updateStatusText( const double & x,
                             const double & y )
{
    if ( GetStatusBar()
         && GetStatusBar()->IsShown() )
    {
        wxString pos_str;
        pos_str.Printf( wxT( "(%.2f %.2f)" ), x, y );
        GetStatusBar()->SetStatusText( pos_str, 1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowPlayerTypeDialog( const boost::any * )
{
    static wxDialog * s_dialog = static_cast< wxDialog * >( 0 );
    static PlayerTypeGrid * s_grid = static_cast< PlayerTypeGrid * >( 0 );

    if ( s_dialog )
    {
        if ( s_dialog->IsShown() )
        {
            s_dialog->Hide();
            return;
        }

        s_grid->setValues( M_data.getViewHolder().getPlayerTypes() );
    }
    else
    {
        s_dialog = new wxDialog( this,
                                 SWID_DIALOG_PLAYER_TYPE, // id
                                 wxT( "Player Type Paramaters" )  // title
                                 , wxPoint( 200, 0 ) //wxDefaultPosition // position
                                 , wxDefaultSize
                                 , wxCAPTION | wxSYSTEM_MENU | wxCLOSE_BOX
                                 | wxRESIZE_BORDER
                                 );
        s_grid = new PlayerTypeGrid( s_dialog );
        s_grid->setValues( M_data.getViewHolder().getPlayerTypes() );
    }

    wxSize grid_size = s_grid->GetSize();
    /*
      std::cerr << "grid_size = "
      << grid_size.GetHeight() << " "
      << grid_size.GetWidth()
      << std::endl;
    */
    s_dialog->SetClientSize( std::min( 1000, grid_size.GetWidth() + 22 ),
                             grid_size.GetHeight() + 22 );

    s_dialog->Show();

    this->SetFocus();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowDetailDialog( const boost::any * data )
{
    if ( M_detail_dlg )
    {
        // toggle show/hide
        M_detail_dlg->Show( ! M_detail_dlg->IsShown() );
    }
    else
    {
        M_detail_dlg = new DetailDialog( this, this, M_data );
        M_detail_dlg->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowViewConfigDialog( const boost::any * data )
{
    if ( M_view_config_dlg )
    {
        // toggle show/hide
        M_view_config_dlg->Show( ! M_view_config_dlg->IsShown() );
    }
    else
    {
        M_view_config_dlg = new ViewConfigDialog( this, this,
                                                  M_view_config_ctrl,
                                                  M_data.getViewConfig() );
        M_view_config_ctrl->setDialog( M_view_config_dlg );
        M_view_config_dlg->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowImageSaveDialog( const boost::any * data )
{
    if ( M_image_save_dlg )
    {
        // toggle show/hide
        M_image_save_dlg->Show( ! M_image_save_dlg->IsShown() );
    }
    else
    {
        M_image_save_dlg = new ImageSaveDialog( this, this, M_data );
        M_image_save_dlg->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowDebugMessageFrame( const boost::any * data )
{
    if ( M_debug_message )
    {
        M_debug_message->Show( ! M_debug_message->IsShown() );
    }
    else
    {
        M_debug_message = new DebugMessageFrame( this, this, M_data );
        M_view_config_ctrl->setDebugMessageFrame( M_debug_message );
        M_debug_message->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvShowConnectDialog( const boost::any * )
{
    wxString hostname = wxGetTextFromUser( wxT( "Input server host" ),// message
                                           wxT( "Input text" ), // caption
                                           wxT( "" ), // default value
                                           this ); // parent
    if ( hostname.Length() > 0 )
    {
#ifdef UNICODE
        std::cerr << "host = " << (const char*)hostname.mbc_str()
                  << std::endl;
        connectMonitorClient( (const char*)(hostname.mbc_str()) );
#else
        std::cerr << "host = " << (const char*)hostname.c_str()
                  << std::endl;
        connectMonitorClient( (const char*)(hostname.c_str()) );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::connectMonitorClient( const char * hostname )
{
    if ( std::strlen( hostname ) == 0 )
    {
        std::cerr << "Empty host name! Connection failed!" << std::endl;
        return;
    }

    std::cerr << "Connect to rcssserver on [" << hostname << "]" << std::endl;

    M_monitor_client
        = boost::shared_ptr< MonitorClient >
        ( new MonitorClient( AppConfig::instance().clientVersion(),
                             hostname,
                             AppConfig::instance().port(),
                             this,
                             M_data.getViewHolder() ) );
    if ( ! M_monitor_client->isConnected() )
    {
        std::cerr << "Conenction failed." << std::endl;
        M_monitor_client.reset();
        return;
    }

    // reset all data
    M_data.clear();
    if ( M_debug_message )
    {
        M_debug_message->clear();
    }
    handle( EventMessage( SWID_CANVAS_UNZOOM ) );

    M_monitor_client->sendDispInit();
    M_logplayer->setLiveMode();

    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        menubar->Enable( SWID_DEBUG_SERVER_START, true );
        menubar->Enable( SWID_DEBUG_SERVER_STOP, false );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::restartServer()
{
    bool run_debug_server = false;

    if ( M_monitor_client )
    {
        if ( M_debug_server )
        {
            run_debug_server = true;
            std::cerr << "============debug server is runnning" << std::endl;
            M_debug_server.reset();
        }
        M_monitor_client->killServer();
        M_monitor_client.reset();

        ::wxSleep( 1 ); // sleep 1 sec
    }

    std::string rcssserver = AppConfig::instance().serverPath();

    if ( rcssserver.empty() )
    {
        ::wxMessageBox( wxT( "Empty command string to invoke the rcssserver." ),
                        wxT( "Error" ),   // title
                        wxOK | wxICON_ERROR, // style
                        this ); // parent frame
        return;
    }

    rcssserver += " > /dev/null 2>&1 &";

    if ( std::system( rcssserver.c_str() ) == -1 )
    {
        ::wxMessageBox( wxT( "Failed to start rcssserver." ),
                        wxT( "Error" ),   // title
                        wxOK | wxICON_ERROR, // style
                        this ); // parent frame
        return;
    }

    ::wxSleep( 1 ); // sleep 1 sec

    connectMonitorClient( "localhost" );

    if ( run_debug_server )
    {
        std::cerr << "=========== restart debug server" << std::endl;
        startDebugServer();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::startDebugServer()
{
    if ( M_debug_server )
    {
        // already created
        return;
    }

    std::cerr << "Start Debug Server" << std::endl;

    int port = static_cast< int >( AppConfig::instance().debugServerPort() );

    M_debug_server
        = boost::shared_ptr< DebugServer >
        ( new DebugServer( port, M_data.getViewHolder() ) );
    if ( ! M_debug_server->isConnected() )
    {
        std::cerr << "failed to create Debug Server" << std::endl;
        M_debug_server.reset();
        return;
    }

    M_debug_server->start();


    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        menubar->Enable( SWID_DEBUG_SERVER_START, false );
        menubar->Enable( SWID_DEBUG_SERVER_STOP, true );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvMonitorConnect( const boost::any * )
{
    connectMonitorClient( "localhost" );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvMonitorConnectTo( const boost::any * data )
{
    const std::string * hostname = boost::any_cast< std::string >( data );
    if ( ! hostname )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( hostname->empty() )
    {
        return;
    }

    connectMonitorClient( hostname->c_str() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::disconnect()
{
    if ( M_monitor_client )
    {
        std::cerr << "Disconnect" << std::endl;
        M_monitor_client.reset(); // delete current connection
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvRequestLiveMode( const boost::any * )
{
    if ( M_monitor_client )
    {
        M_logplayer->setLiveMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvReceiveMonitorPacket( const boost::any * )
{
    if ( M_logplayer->isLiveMode() )
    {
        handle( EventMessage( SWID_SET_VIEW_DATA_INDEX_LAST ) );

        if ( AppConfig::instance().autoQuitMode() )
        {
            static long s_game_end_time = 0;

            if ( M_data.getViewHolder().getLastPlayMode() != rcsc::PM_TimeOver )
            {
                s_game_end_time = 0;
            }
            else
            {
                if ( s_game_end_time == 0 )
                {
                    s_game_end_time = ::wxGetLocalTime();
                }
                else
                {
                    if ( ::wxGetLocalTime() - s_game_end_time
                         >= AppConfig::instance().waitSeconds() )
                    {
                        std::cerr << "Elapsed " << AppConfig::instance().waitSeconds()
                                  << " seconds after game end\n"
                                  << "Exit..."
                                  << std::endl;
                        ::wxExit();
                    }
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::recvDebugServerStop( const boost::any * )
{
    if ( M_debug_server )
    {
        std::cerr << "Stop Debug Server" << std::endl;
        M_debug_server.reset();
    }

    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        if ( isMonitorConnected() )
        {
            menubar->Enable( SWID_DEBUG_SERVER_START, true );
        }
        else
        {
            menubar->Enable( SWID_DEBUG_SERVER_START, false );
        }
        menubar->Enable( SWID_DEBUG_SERVER_STOP, false );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainFrame::isMonitorConnected() const
{
    return ( M_monitor_client
             && M_monitor_client->isConnected() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::saveImage( const int start_cycle,
                      const int end_cycle,
                      const std::string & saved_dir,
                      const std::string & name_prefix,
                      const int format_type )
{
    static bool s_first = true;
    if ( s_first )
    {
        ::wxInitAllImageHandlers();
        s_first = false;
    }

    if ( M_data.getViewHolder().getMonitorViewCont().empty() )
    {
        return;
    }

    if ( isMonitorConnected() )
    {
        ::wxMessageBox( wxT( "You cannot carete image fils\n"
                             " while the monitor client is running!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        this ); // parent frame
        return;
    }

    const std::size_t last_idx = M_data.getViewIndex();

    const std::size_t first = M_data.getViewHolder().getIndexOf( start_cycle );
    const std::size_t last = M_data.getViewHolder().getIndexOf( end_cycle );

    if ( first > last )
    {
        return;
    }

    handle( EventMessage( SWID_LOGPLAYER_GOTO_INDEX, first ) );

    wxBitmapType type = static_cast< wxBitmapType >( format_type );

    // create file path base
    std::string file_path_base_std = saved_dir;
    if ( file_path_base_std.empty() )
    {
        file_path_base_std = "./";
    }
    if ( *file_path_base_std.rbegin() != '/' )
    {
        file_path_base_std += '/';
    }

    // check directory
    {
        wxString dir( file_path_base_std.c_str(),
                      *wxConvCurrent );
        if ( ! ::wxDirExists( dir )
             && ! ::wxMkdir( dir ) )
        {
            ::wxMessageBox( wxT( "Failed to create image save directory!" ),
                            wxT( "Error" ),   // title
                            wxOK | wxICON_ERROR, // style
                            this ); // parent frame
            return;
        }
    }


    file_path_base_std += name_prefix;
    if ( ! file_path_base_std.empty()
         && *file_path_base_std.rbegin() == '/' )
    {
        file_path_base_std.erase( file_path_base_std.length() - 1 );
    }

    const wxString file_path_base( file_path_base_std.c_str(),
                                   *wxConvCurrent );

    wxString file_ext;
    switch ( type ) {
    case wxBITMAP_TYPE_BMP:
        file_ext = wxT( ".bmp" );
        break;
    case wxBITMAP_TYPE_PNG:
        file_ext = wxT( ".png" );
        break;
    case wxBITMAP_TYPE_JPEG:
        file_ext = wxT( ".jpg" );
        break;
    case wxBITMAP_TYPE_XPM:
        file_ext = wxT( ".xpm" );
        break;
    default:
        file_ext = wxT( ".bmp" );
        break;
    }

    wxString file_path = file_path_base;
    file_path += wxT( "00000" );
    file_path += file_ext;

    // show progress dialog
    wxProgressDialog  dlg( wxT( "Image Save Progress" ),
                           file_path,
                           static_cast< int >( last - first ),
                           this,
                           wxPD_CAN_ABORT |
                           wxPD_AUTO_HIDE |
                           wxPD_APP_MODAL |
                           wxPD_ELAPSED_TIME |
                           wxPD_ESTIMATED_TIME |
                           wxPD_REMAINING_TIME );

    // create bitmap
    const wxSize size = M_field_canvas->GetSize();
    wxBitmap bmp( size.GetWidth(), size.GetHeight() );

    wxMemoryDC mem_dc;
    mem_dc.SelectObject( bmp );

    // main loop
    for ( std::size_t i = first; i <= last; ++i )
    {
        M_data.setViewIndex( i );
        M_data.update( size.GetWidth(), size.GetHeight() );

        M_field_canvas->drawAll( mem_dc );

        // full file path
        wxString count;
        count.sprintf( wxT( "%05d" ), i );

        file_path = file_path_base;
        file_path += count;
        file_path += file_ext;

        if ( ! dlg.Update( static_cast< int >( i - first ),
                           file_path ) )
        {
            std::cerr << "canceled" << std::endl;
            break;
        }
        //std::cout << "save image " << file_path << std::endl;
        wxImage img = bmp.ConvertToImage();
        if ( ! img.SaveFile( file_path, type ) )
        {
            std::cerr << "Failed to save image file "
                      << file_path
                      << std::endl;
            break;
        }
    }

    //dlg.Destroy();

    handle( EventMessage( SWID_LOGPLAYER_GOTO_INDEX, last_idx ) );
}
