// -*-c++-*-

/*!
  \file main_data.cpp
  \brief main data supplier class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have receiveEd a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <iostream>
#include <cassert>
#include <cmath>
#include <string>

#include <rcsc/gz/gzfstream.h>
#include <rcsc/rcg.h>

#include "id.h"
#include "app_config.h"
#include "main_frame.h"
#include "monitor_view_data.h"
#include "view_config.h"
#include "view_holder.h"

#include "main_data.h"

/*-------------------------------------------------------------------*/
/*!

*/
MainData::MainData( MainFrame * main_frame )
    : M_main_frame( main_frame )
    , M_view_config()
    , M_view_holder( main_frame )
    , M_view_index( 0 )
{
    assert( main_frame );

    M_main_frame->connect( SWID_SET_VIEW_DATA_INDEX,
                           this, &MainData::recvSetViewDataIndex );
    M_main_frame->connect( SWID_SET_VIEW_DATA_INDEX_FIRST,
                           this, &MainData::recvSetViewDataIndexFirst );
    M_main_frame->connect( SWID_SET_VIEW_DATA_INDEX_LAST,
                           this, &MainData::recvSetViewDataIndexLast );
    M_main_frame->connect( SWID_SET_VIEW_DATA_CYCLE,
                           this, &MainData::recvSetViewDataCycle );
    M_main_frame->connect( SWID_STEP_VIEW_DATA_FORWARD,
                           this, &MainData::recvStepViewDataForward );
    M_main_frame->connect( SWID_STEP_VIEW_DATA_BACK,
                           this, &MainData::recvStepViewDataBack );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::clear()
{
    M_view_index = 0;
    M_view_holder.clear();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainData::loadRCG( const std::string & file_path )
{
    std::cerr << "open game log = [" << file_path << "]" << std::endl;
#ifdef HAVE_LIBZ
    rcsc::gzifstream fin( file_path.c_str() );
#else
    std::ifstream fin( file_path.c_str(),
                       std::ios_base::in | std::ios_base::binary );
#endif
    if ( ! fin.is_open() )
    {
        std::cerr << "failed to open rcg file. [" << file_path << "]" << std::endl;
        return false;
    }

    if ( ! fin.good() )
    {
        std::cerr << "input stream of rcg file. [" << file_path << "] is not good." << std::endl;
        return false;
    }

    // disconnect
    M_main_frame->handle( EventMessage( SWID_MONITOR_DISCONNECT ) );
    // stop logplayer
    M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STOP ) );

    // clear all data
    clear();

    // create rcg loader
    rcsc::rcg::Loader loader( M_view_holder );
    // create rcg parser
    rcsc::rcg::ParserPtr parser = rcsc::rcg::make_parser( fin );

    // do parse & handle data
    if ( ! parser
         || ! parser->parse( fin, loader ) )
    {
        std::cerr << "failed to parse [" << file_path << "]." << std::endl;
    }
    else
    {
        M_view_holder.pushBackLatestViewData();
    }

    fin.close();

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  \brief update player selection, focus point, field size, and so on.
*/
void
MainData::update( const int width,
                  const int height )
{
    const MonitorViewPtr view = getCurrentViewData();

    // update selected number
    if ( config().isPlayerAutoSelect() )
    {
        selectBallNearestPlayer( view );
    }

    // update focus point
    if ( view )
    {
        if ( config().getFocusType() == ViewConfig::FOCUS_BALL )
        {
            getViewConfig().updateFocusPoint( view->ball().x(), view->ball().y() );
        }
        else if ( config().getFocusType() == ViewConfig::FOCUS_PLAYER
                  && config().getSelectedNumber() != 0 )
        {
            int id = config().getSelectedNumber();
            if ( id < 0 )
            {
                id = -1*id + 11;
            }
            id -= 1;
            const Player & p = view->players()[id];
            if ( p.isAlive() )
            {
                getViewConfig().updateFocusPoint( p.x(), p.y() );
            }
        }
        else
        {
            // already set
        }
    }

    // update field scale and related things
    getViewConfig().updateFieldSize( width, height );
    if ( gdi().getScoreBoardFont().GetPointSize() != config().getScoreBoardFontSize() )
    {
        M_gdi_config.resizeScoreBoardFont( config().getScoreBoardFontSize() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::selectBallNearestPlayer( const MonitorViewPtr view )
{
    if ( ! view )
    {
        return;
    }

    ViewConfig::PlayerSelectType old_type = config().getPlayerSelectType();

    rcsc::Vector2D ball_pos( view->ball().x(), view->ball().y() );
    double min_dist2 = 40000.0;

    rcsc::SideID side = rcsc::NEUTRAL;
    int unum = 0;

    const std::size_t first = ( old_type == ViewConfig::SELECT_AUTO_RIGHT
                                ? 11
                                : 0 );
    const std::size_t last = ( old_type == ViewConfig::SELECT_AUTO_LEFT
                               ? 11
                               : 22 );
    const std::vector< Player > & players = view->players();
    for ( std::size_t i = first; i < last; ++i )
    {
        if ( players[i].isAlive() )
        {
            double d2 = ball_pos.dist2( rcsc::Vector2D( players[i].x(),
                                                        players[i].y() ) );
            if ( d2 < min_dist2 )
            {
                min_dist2 = d2;
                side = players[i].side();
                unum = players[i].unum();
            }
        }
    }

    if ( unum != 0 )
    {
        getViewConfig().setSelectedNumber( side, unum );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
/*
void
MainData::recvLoadRCG( const boost::any * data )
{
    const std::string * filepath = boost::any_cast< std::string >( data );
    if ( ! filepath )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( filepath->empty() )
    {
        return;
    }

    loadRCG( *filepath );
}
*/
/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvSetViewDataIndex( const boost::any * data )
{
    const std::size_t * idx = boost::any_cast< std::size_t >( data );
    if ( ! idx )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    if ( *idx >= getViewHolder().getMonitorViewCont().size() )
    {
        return;
    }

    M_view_index = *idx;
    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvSetViewDataIndexFirst( const boost::any * )
{
    M_view_index = 0;
    if ( getViewHolder().getMonitorViewCont().size() > 0 )
    {
        M_main_frame->updateView();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvSetViewDataIndexLast( const boost::any * )
{
    if ( getViewHolder().getMonitorViewCont().size() > 0 )
    {
        M_view_index = getViewHolder().getMonitorViewCont().size() - 1;
        M_main_frame->updateView();
    }
    else
    {
        M_view_index = 0;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvSetViewDataCycle( const boost::any * data )
{
    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    M_view_index = getViewHolder().getIndexOf( *cycle );
    M_main_frame->updateView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvStepViewDataForward( const boost::any * )
{
    if ( M_view_index < getViewHolder().getMonitorViewCont().size() - 1 )
    {
        ++M_view_index;
        M_main_frame->updateView();
    }
    else
    {
        if ( AppConfig::instance().autoLoopMode() )
        {
            M_view_index = 0;
            M_main_frame->updateView();
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STOP ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainData::recvStepViewDataBack( const boost::any * )
{
    if ( 0 < M_view_index )
    {
        --M_view_index;
        M_main_frame->updateView();
    }
    else
    {
        if ( AppConfig::instance().autoLoopMode() )
        {
            M_view_index = getViewHolder().getMonitorViewCont().size() - 1;
            M_main_frame->updateView();
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STOP ) );
        }
    }
}
