// -*-c++-*-

/*!
  \file field_painter.cpp
  \brief field painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include "app_config.h"
#include "view_config.h"
#include "gdi_config.h"

#include "field_painter.h"

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::draw( wxDC & dc ) const
{
    drawBackGround( dc );
    drawLines( dc );
    drawPenaltyAreaLines( dc );
    drawGoalAreaLines( dc );
    drawGoals( dc );
    if ( M_conf.isShownFlags() )
    {
        drawFlags( dc );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawBackGround( wxDC & dc ) const
{
    // set brush
    dc.SetBackground( M_gdi.getFieldBrush() );
    // crear with current brush
    dc.Clear();

    ////////////////////////////////////////////////////////////////
    if ( M_conf.getGrassType() == ViewConfig::GRASS_LINES )
    {
        drawLinesGrass( dc );
    }
    else if ( M_conf.getGrassType() == ViewConfig::GRASS_CHECKER )
    {
        drawCheckerGrass( dc );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawLinesGrass( wxDC & dc ) const
{
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( M_gdi.getFieldDarkBrush() );

    const int left_x = ( M_conf.getFieldCenter().x
                         - M_conf.scale( AppConfig::PITCH_LENGTH / 2 ) );
    const int top_y = ( M_conf.getFieldCenter().y
                        - M_conf.scale( AppConfig::PITCH_WIDTH / 2 ) );
    const int right_x = ( M_conf.getFieldCenter().x
                          + M_conf.scale( AppConfig::PITCH_LENGTH / 2 ) );
    const int bottom_y = ( M_conf.getFieldCenter().y
                           + M_conf.scale( AppConfig::PITCH_WIDTH / 2 ) );

    const int grid = M_conf.scale( 10.0 ); // grid length
    int i = M_conf.getFieldCenter().x - grid/2;
    int cnt = 0;
    while ( i > left_x )
    {
        if ( ! ( cnt % 2 ) )
        {
            int left = i - grid;
            dc.DrawRectangle( left, top_y - grid/2,
                              i - left, bottom_y - top_y + grid );
        }
        i -= grid;
        ++cnt;
    }
    i = M_conf.getFieldCenter().x + grid/2;
    cnt = 0;
    while ( i < right_x )
    {
        if ( ! ( cnt % 2 ) )
        {
            int right = i + grid;
            dc.DrawRectangle( i, top_y - grid/2,
                              right - i, bottom_y - top_y + grid );
        }
        i += grid;
        ++cnt;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawCheckerGrass( wxDC & dc )const
{
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( M_gdi.getFieldDarkBrush() );

    const int left_x = ( M_conf.getFieldCenter().x
                         - M_conf.scale( AppConfig::PITCH_LENGTH / 2 ) );
    const int top_y = ( M_conf.getFieldCenter().y
                        - M_conf.scale( AppConfig::PITCH_WIDTH / 2 ) );
    const int right_x = ( M_conf.getFieldCenter().x
                          + M_conf.scale( AppConfig::PITCH_LENGTH / 2 ) );
    const int bottom_y = ( M_conf.getFieldCenter().y
                           + M_conf.scale( AppConfig::PITCH_WIDTH / 2 ) );

    const int grid = M_conf.scale( 10.0 ); // grid length
    int flag = 0;
    for ( int i = M_conf.getFieldCenter().x; i > left_x; i -= grid )
    {
        int cnt = flag;
        int left = i - grid;
        for ( int j = M_conf.getFieldCenter().y; j > top_y; j -= grid )
        {
            if ( cnt % 2 )
            {
                int top = j - grid;
                dc.DrawRectangle( left, top, i - left, j - top );
            }
            ++cnt;
        }
        ++flag;
    }
    flag = 1;
    for ( int i = M_conf.getFieldCenter().x;  i > left_x; i -= grid )
    {
        int cnt = flag;
        int left = i - grid;
        for ( int j = M_conf.getFieldCenter().y; j < bottom_y; j += grid )
        {
            if ( cnt % 2 )
            {
                int bottom = j + grid;
                dc.DrawRectangle( left, j, i - left, bottom - j );
            }
            ++cnt;
        }
        ++flag;
    }
    flag = 0;
    for ( int i = M_conf.getFieldCenter().x;  i < right_x; i += grid )
    {
        int cnt = flag;
        int right = i + grid;
        for ( int j = M_conf.getFieldCenter().y; j < bottom_y; j += grid )
        {
            if ( cnt % 2 )
            {
                int bottom = j + grid;
                dc.DrawRectangle( i, j, right - i, bottom - j );
            }
            ++cnt;
        }
        ++flag;
    }
    flag = 1;
    for ( int i = M_conf.getFieldCenter().x;  i < right_x; i += grid )
    {
        int cnt = flag;
        int right = i + grid;
        for ( int j = M_conf.getFieldCenter().y; j > top_y; j -= grid )
        {
            if ( cnt % 2 )
            {
                int top = j - grid;
                dc.DrawRectangle( i, top, right - i, j - top );
            }
            ++cnt;
        }
        ++flag;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawLines( wxDC & dc ) const
{
    // set gdi objects
    dc.SetPen( M_gdi.getLinePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    // set screen coordinates of field
    wxCoord left_x   = M_conf.getScreenX( - AppConfig::PITCH_LENGTH * 0.5 );
    wxCoord right_x  = M_conf.getScreenX( + AppConfig::PITCH_LENGTH * 0.5 );
    wxCoord top_y    = M_conf.getScreenY( - AppConfig::PITCH_WIDTH * 0.5 );
    wxCoord bottom_y = M_conf.getScreenY( + AppConfig::PITCH_WIDTH * 0.5 );

    // side lines & goal lines
    dc.DrawLine( left_x, top_y, right_x, top_y );
    dc.DrawLine( right_x, top_y, right_x, bottom_y );
    dc.DrawLine( right_x, bottom_y, left_x, bottom_y );
    dc.DrawLine( left_x, bottom_y, left_x, top_y );

    if ( AppConfig::instance().keepaway() )
    {
        // keepaway area
        wxCoord ka_left = M_conf.getScreenX( - AppConfig::instance().keepawayLength() * 0.5 );
        wxCoord ka_right = M_conf.getScreenX( + AppConfig::instance().keepawayLength() * 0.5 );
        wxCoord ka_top = M_conf.getScreenY( - AppConfig::instance().keepawayWidth() * 0.5 );
        wxCoord ka_bottom = M_conf.getScreenY( + AppConfig::instance().keepawayWidth() * 0.5 );

        dc.DrawLine( ka_left, ka_top, ka_right, ka_top );
        dc.DrawLine( ka_right, ka_top, ka_right, ka_bottom );
        dc.DrawLine( ka_right, ka_bottom, ka_left, ka_bottom );
        dc.DrawLine( ka_left, ka_bottom, ka_left, ka_top );
    }
    else
    {
        // center line
        dc.DrawLine( M_conf.getFieldCenter().x, top_y,
                     M_conf.getFieldCenter().x, bottom_y );

        // center circle
        wxCoord center_radius = M_conf.scale( AppConfig::CENTER_CIRCLE_R );
        dc.DrawCircle( M_conf.getFieldCenter(), center_radius );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawPenaltyAreaLines( wxDC & dc ) const
{
    // set gdi objects
    dc.SetPen( M_gdi.getLinePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    // set screen coordinates of field
    wxCoord left_x   = M_conf.getScreenX( - AppConfig::PITCH_LENGTH * 0.5 );
    wxCoord right_x  = M_conf.getScreenX( + AppConfig::PITCH_LENGTH * 0.5 );

    // set penalty area params
    wxCoord pen_top_y    = M_conf.getScreenY( - AppConfig::PENALTY_AREA_WIDTH / 2 );
    wxCoord pen_bottom_y = M_conf.getScreenY( + AppConfig::PENALTY_AREA_WIDTH / 2 );
    double pen_circle_y_degree_abs =
        std::acos( ( AppConfig::PENALTY_AREA_LENGTH - AppConfig::PENALTY_SPOT_DIST )
                   / AppConfig::PENALTY_CIRCLE_R )
        * ( 180.0 / M_PI );
    wxCoord pen_circle_r = M_conf.scale( AppConfig::PENALTY_CIRCLE_R );
    wxCoord pen_circle_size = M_conf.scale( AppConfig::PENALTY_CIRCLE_R * 2.0 );

    // left penalty area X
    wxCoord pen_x = M_conf.getScreenX( -( AppConfig::PITCH_LENGTH*0.5
                                             - AppConfig::PENALTY_AREA_LENGTH ) );
    // left arc
    wxCoord pen_spot_x = M_conf.getScreenX( -( AppConfig::PITCH_LENGTH*0.5
                                                  - AppConfig::PENALTY_SPOT_DIST ) );
    dc.DrawEllipticArc( pen_spot_x - pen_circle_r + 1,
                        M_conf.getFieldCenter().y - pen_circle_r,
                        pen_circle_size, pen_circle_size,
                        pen_circle_y_degree_abs, -pen_circle_y_degree_abs );
    // left rectangle
    dc.DrawLine( left_x, pen_top_y, pen_x, pen_top_y );
    dc.DrawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    dc.DrawLine( pen_x, pen_bottom_y, left_x, pen_bottom_y );
    // left spot
    dc.DrawPoint( pen_spot_x, M_conf.getFieldCenter().y );

    // right penalty area X
    pen_x = M_conf.getScreenX( +( AppConfig::PITCH_LENGTH*0.5
                                     - AppConfig::PENALTY_AREA_LENGTH ) );
    // right arc
    pen_spot_x = M_conf.getScreenX( +( AppConfig::PITCH_LENGTH*0.5
                                          - AppConfig::PENALTY_SPOT_DIST ) );
    dc.DrawEllipticArc( pen_spot_x - pen_circle_r,
                        M_conf.getFieldCenter().y - pen_circle_r,
                        pen_circle_size, pen_circle_size,
                        180.0 + pen_circle_y_degree_abs - 0.5,
                        180.0 - pen_circle_y_degree_abs + 0.5 );
    // right rectangle
    dc.DrawLine( right_x, pen_top_y, pen_x, pen_top_y );
    dc.DrawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    dc.DrawLine( pen_x, pen_bottom_y, right_x, pen_bottom_y );
    // right spot
    dc.DrawPoint( pen_spot_x, M_conf.getFieldCenter().y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawGoalAreaLines( wxDC & dc ) const
{
    // set gdi objects
    dc.SetPen( M_gdi.getLinePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    // set screen coordinates of field
    wxCoord left_x   = M_conf.getScreenX( - AppConfig::PITCH_LENGTH * 0.5 );
    wxCoord right_x  = M_conf.getScreenX( + AppConfig::PITCH_LENGTH * 0.5 );

    // set coordinates M_confs
    wxCoord goal_area_y_abs = M_conf.scale( AppConfig::GOAL_AREA_WIDTH*0.5 );
    wxCoord goal_area_top_y = M_conf.getFieldCenter().y - goal_area_y_abs;
    wxCoord goal_area_bottom_y = M_conf.getFieldCenter().y + goal_area_y_abs;

    // left goal area
    wxCoord goal_area_x = M_conf.getScreenX( - AppConfig::PITCH_LENGTH*0.5
                                                + AppConfig::GOAL_AREA_LENGTH );
    dc.DrawLine( left_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    dc.DrawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    dc.DrawLine( goal_area_x, goal_area_bottom_y, left_x, goal_area_bottom_y );

    // right goal area
    goal_area_x = M_conf.getScreenX( AppConfig::PITCH_LENGTH*0.5
                                        - AppConfig::GOAL_AREA_LENGTH );
    dc.DrawLine( right_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    dc.DrawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    dc.DrawLine( goal_area_x, goal_area_bottom_y, right_x, goal_area_bottom_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawGoals( wxDC & dc ) const
{
    // set gdi objects
    dc.SetPen( *wxBLACK_PEN );
    dc.SetBrush( *wxBLACK_BRUSH );

    // set coordinates param
    wxCoord goal_top_y = M_conf.getScreenY( - AppConfig::GOAL_WIDTH*0.5 );
    wxCoord goal_size_x = M_conf.scale( AppConfig::GOAL_DEPTH );
    wxCoord goal_size_y = M_conf.scale( AppConfig::GOAL_WIDTH );
    wxCoord post_size = M_conf.scale( AppConfig::GOAL_POST_R );
    wxCoord post_x;

    // left goal
    dc.DrawRectangle( M_conf.getScreenX( - AppConfig::PITCH_LENGTH*0.5
                                            - AppConfig::GOAL_DEPTH ),
                      goal_top_y,
                      goal_size_x,
                      goal_size_y );
    if ( post_size >= 1 )
    {
        post_x = M_conf.getScreenX( - AppConfig::PITCH_LENGTH*0.5
                                       + AppConfig::GOAL_POST_R );
        dc.DrawCircle( post_x, goal_top_y + post_size, post_size );
        dc.DrawCircle( post_x, goal_top_y + goal_size_y - post_size, post_size );
    }
    // right goal
    dc.DrawRectangle( M_conf.getScreenX( AppConfig::PITCH_LENGTH*0.5 ) + 1,
                      goal_top_y,
                      goal_size_x + 1,
                      goal_size_y );
    if ( post_size >= 1 )
    {
        post_x = M_conf.getScreenX( AppConfig::PITCH_LENGTH*0.5
                                       - AppConfig::GOAL_POST_R );
        dc.DrawCircle( post_x, goal_top_y + post_size, post_size );
        dc.DrawCircle( post_x, goal_top_y + goal_size_y - post_size, post_size );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldPainter::drawFlags( wxDC & dc ) const
{
    // set gdi objects
    dc.SetPen( M_gdi.getLinePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    // set size or coordinates params
    wxCoord flag_radius = M_conf.scale( 0.5 );
    if ( flag_radius < 2 ) flag_radius = 2;
    if ( flag_radius > 5 ) flag_radius = 5;

    wxCoord x, y;
    wxCoord pitch_half_length = M_conf.scale( AppConfig::PITCH_LENGTH*0.5 );
    wxCoord pitch_half_width = M_conf.scale( AppConfig::PITCH_WIDTH*0.5 );
    wxCoord pitch_margin_x = M_conf.scale( AppConfig::PITCH_LENGTH*0.5
                                              + AppConfig::PITCH_MARGIN );
    wxCoord pitch_margin_y = M_conf.scale( AppConfig::PITCH_WIDTH*0.5
                                              + AppConfig::PITCH_MARGIN );
    wxCoord penalty_x = M_conf.scale( AppConfig::PITCH_LENGTH*0.5
                                         - AppConfig::PENALTY_AREA_LENGTH );
    wxCoord penalty_y = M_conf.scale( AppConfig::PENALTY_AREA_WIDTH*0.5 );
    wxCoord goal_y = M_conf.scale( AppConfig::GOAL_WIDTH*0.5 );
    wxCoord scale10 = M_conf.scale( 10.0 );
    wxCoord scale20 = M_conf.scale( 20.0 );
    wxCoord scale30 = M_conf.scale( 30.0 );
    wxCoord scale40 = M_conf.scale( 40.0 );
    wxCoord scale50 = M_conf.scale( 50.0 );

    // goal left
    x = M_conf.getFieldCenter().x - pitch_half_length;
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // goal right
    x = M_conf.getFieldCenter().x + pitch_half_length;
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag c
    x  = M_conf.getFieldCenter().x;
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag c t
    x = M_conf.getFieldCenter().x;
    y = M_conf.getFieldCenter().y - pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );

    // flag c b
    x = M_conf.getFieldCenter().x;
    y = M_conf.getFieldCenter().y + pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );
    // flag l t
    x = M_conf.getFieldCenter().x - pitch_half_length;
    y = M_conf.getFieldCenter().y - pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );
    // flag l b
    x = M_conf.getFieldCenter().x - pitch_half_length;
    y = M_conf.getFieldCenter().y + pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );
    // flag r t
    x = M_conf.getFieldCenter().x + pitch_half_length;
    y = M_conf.getFieldCenter().y - pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );
    // flag r b
    x = M_conf.getFieldCenter().x + pitch_half_length;
    y = M_conf.getFieldCenter().y + pitch_half_width;
    dc.DrawCircle( x, y, flag_radius );
    // flag p l t
    x = M_conf.getFieldCenter().x - penalty_x;
    y = M_conf.getFieldCenter().y - penalty_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag p l c
    x = M_conf.getFieldCenter().x - penalty_x;
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag p l b
    x = M_conf.getFieldCenter().x - penalty_x;
    y = M_conf.getFieldCenter().y + penalty_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag p r t
    x = M_conf.getFieldCenter().x + penalty_x;
    y = M_conf.getFieldCenter().y - penalty_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag p r c
    x = M_conf.getFieldCenter().x + penalty_x;
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag p r b
    x = M_conf.getFieldCenter().x + penalty_x;
    y = M_conf.getFieldCenter().y + penalty_y;
    dc.DrawCircle( x, y, flag_radius );

    // flag g l t
    x = M_conf.getFieldCenter().x - pitch_half_length;
    y = M_conf.getFieldCenter().y - goal_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag g l b
    x = M_conf.getFieldCenter().x - pitch_half_length;
    y = M_conf.getFieldCenter().y + goal_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag g r t
    x = M_conf.getFieldCenter().x + pitch_half_length;
    y = M_conf.getFieldCenter().y - goal_y;
    dc.DrawCircle( x, y, flag_radius );
    // flag g r b
    x = M_conf.getFieldCenter().x + pitch_half_length;
    y = M_conf.getFieldCenter().y + goal_y;
    dc.DrawCircle( x, y, flag_radius );

    // flag t ...

    y = M_conf.getFieldCenter().y - pitch_margin_y;
    // flag t l 50
    x = M_conf.getFieldCenter().x - scale50;
    dc.DrawCircle( x, y, flag_radius );
    // flag t l 40
    x = M_conf.getFieldCenter().x - scale40;
    dc.DrawCircle( x, y, flag_radius );
    // flag t l 30
    x = M_conf.getFieldCenter().x - scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag t l 20
    x = M_conf.getFieldCenter().x - scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag t l 10
    x = M_conf.getFieldCenter().x - scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag t 0
    x = M_conf.getFieldCenter().x;
    dc.DrawCircle( x, y, flag_radius );
    // flag t r 10
    x = M_conf.getFieldCenter().x + scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag t r 20
    x = M_conf.getFieldCenter().x + scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag t r 30
    x = M_conf.getFieldCenter().x + scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag t r 40
    x = M_conf.getFieldCenter().x + scale40;
    dc.DrawCircle( x, y, flag_radius );
    // flag t r 50
    x = M_conf.getFieldCenter().x + scale50;
    dc.DrawCircle( x, y, flag_radius );

    // flag b ...

    y = M_conf.getFieldCenter().y + pitch_margin_y;
    // flag b l 50
    x = M_conf.getFieldCenter().x - scale50;
    dc.DrawCircle( x, y, flag_radius );
    // flag b l 40
    x = M_conf.getFieldCenter().x - scale40;
    dc.DrawCircle( x, y, flag_radius );
    // flag b l 30
    x = M_conf.getFieldCenter().x - scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag b l 20
    x = M_conf.getFieldCenter().x - scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag b l 10
    x = M_conf.getFieldCenter().x - scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag b 0
    x = M_conf.getFieldCenter().x;
    dc.DrawCircle( x, y, flag_radius );
    // flag b r 10
    x = M_conf.getFieldCenter().x + scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag b r 20
    x = M_conf.getFieldCenter().x + scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag b r 30
    x = M_conf.getFieldCenter().x + scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag b r 40
    x = M_conf.getFieldCenter().x + scale40;
    dc.DrawCircle( x, y, flag_radius );
    // flag b r 50
    x = M_conf.getFieldCenter().x + scale50;
    dc.DrawCircle( x, y, flag_radius );

    // flag l ...

    x = M_conf.getFieldCenter().x - pitch_margin_x;
    // flag l t 30
    y = M_conf.getFieldCenter().y - scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag l t 20
    y = M_conf.getFieldCenter().y - scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag l t 10
    y = M_conf.getFieldCenter().y - scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag l 0
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag l b 10
    y = M_conf.getFieldCenter().y + scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag l b 20
    y = M_conf.getFieldCenter().y + scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag l b 30
    y = M_conf.getFieldCenter().y + scale30;
    dc.DrawCircle( x, y, flag_radius );

    // flag r ...

    x = M_conf.getFieldCenter().x + pitch_margin_x;
    // flag r t 30
    y = M_conf.getFieldCenter().y - scale30;
    dc.DrawCircle( x, y, flag_radius );
    // flag r t 20
    y = M_conf.getFieldCenter().y - scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag r t 10
    y = M_conf.getFieldCenter().y - scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag r 0
    y = M_conf.getFieldCenter().y;
    dc.DrawCircle( x, y, flag_radius );
    // flag r b 10
    y = M_conf.getFieldCenter().y + scale10;
    dc.DrawCircle( x, y, flag_radius );
    // flag r b 20
    y = M_conf.getFieldCenter().y + scale20;
    dc.DrawCircle( x, y, flag_radius );
    // flag r b 30
    y = M_conf.getFieldCenter().y + scale30;
    dc.DrawCircle( x, y, flag_radius );
}
