// -*-c++-*-

/*!
  \file event_handler.h
  \brief event handler class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifndef SOCCER_WINDOW2_EVENT_HANDLER_H
#define SOCCER_WINDOW2_EVENT_HANDLER_H

#include <vector>
#include <map>
#include <boost/any.hpp>
#include <boost/function.hpp>

/*!
  \class EventMessage
  \brief event message object
*/
class EventMessage {
private:
    int M_id; //!< event Id
    boost::any M_data; //!< event data

    //! Default constructor shouldn't be used!
    EventMessage()
        : M_id( 0 )
      { }

public:

    explicit
    EventMessage( const int id )
        : M_id( id )
      { }

    //! constructor that uses template data type
    template < typename Data >
    EventMessage( const int id,
                  const Data & data )
        : M_id( id )
        , M_data( data )
      { }

    //! returns event id
    int getId() const
      {
          return M_id;
      }

    //! return data
    const
    boost::any * getData() const
      {
          return ( & M_data );
      }
};

///////////////////////////////////////////////////////////////////////

/*!
  \class EventHandler
  \brief event handler function pool
*/
class EventHandler {
private:
    //typedef boost::function1< void, const boost::any * > EventFunc;
    typedef boost::function< void ( const boost::any * ) > EventFunc;
    typedef std::vector< EventFunc > EventFuncCont;
    typedef std::map< int, EventFuncCont > HandlerMap;

    //! key: event id,  value: function container
    HandlerMap M_handlers;

public:

    EventHandler()
      { }

    virtual
    ~EventHandler()
      { }

    /*!
      \brief set the specified function object associated with event_id
      \param event_id event Id to be removed
      \param handler pointer to the acceptor object
      \param memfun function pointer
    */
    template < typename Handler >
    void connect( const int event_id,
                  Handler * handler,
                  void (Handler::*memfun)( const boost::any * ) )
      {
          M_handlers[event_id].push_back( std::bind1st( std::mem_fun( memfun ),
                                                        handler ) );
      }

    /*!
      \brief remove the specified function object associated with event_id
      \param event_id event Id to be removed
      \param handler pointer to the acceptor object
      \param memfun function pointer
    */
    template < typename Handler >
    void disconnect( const int event_id,
                     Handler * handler,
                     void (Handler::*memfun)( const boost::any * ) )
      {
          HandlerMap::iterator it = M_handlers.find( event_id );
          if ( it == M_handlers.end() )
          {
              return;
          }

          EventFunc func = std::bind1st( std::mem_fun( memfun ),
                                         handler );
          it->second.erase( std::remove( it->second.begin(),
                                         it->second.end(),
                                         func ),
                            it->second.end() );
      }

    /*!
      \brief remove all event function associated with event_id
      \param event_id event Id to be removed.
     */
    void disconnect( const int event_id )
      {
          M_handlers.erase( event_id );
      }

    /*!
      \brief accept event message and handle it.
      \param ev event message object
    */
    void handle( const EventMessage & ev );
};

#endif
