// -*-c++-*-

/*!
  \file player_control_painter.cpp
  \brief player control painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <rcsc/param/player_type.h>
#include <rcsc/param/server_param.h>

#include "main_data.h"
#include "monitor_view_data.h"

#include "player_control_painter.h"

/*-------------------------------------------------------------------*/
/*!
  \param dc reference to the device context
*/
void
PlayerControlPainter::draw( wxDC & dc ) const
{
    const ViewHolder & holder = M_data.getViewHolder();
    int number = M_data.config().getSelectedNumber();
    if ( number == 0 )
    {
        return;
    }

    const MonitorViewPtr view = M_data.getCurrentViewData();
    const Ball & ball = view->ball();
    const Player & player = view->players()[( number > 0 ? number - 1 : -number - 1 + 11 )];
    const rcsc::ServerParam & sparam = rcsc::ServerParam::i();
    const rcsc::PlayerType & ptype = holder.getPlayerTypes()[player.type()];
    if ( M_data.config().isShownControlArea()
         && ball.hasDelta() )
    {
        rcsc::Vector2D ppos( player.x(), player.y() );
        rcsc::Vector2D bpos( ball.x(), ball.y() );
        rcsc::Vector2D player_to_ball = bpos - ppos;
        player_to_ball.rotate( - player.body() );

        rcsc::Vector2D bnext( bpos.x + ball.deltaX(),
                              bpos.y + ball.deltaY() );
        double ball_dist = player_to_ball.r();
        if ( ball_dist < ptype.kickableArea() )
        {
            double max_kick_accel
                = sparam.maxPower()
                * sparam.kickPowerRate()
                * ( 1.0 - 0.25 * player_to_ball.th().abs() / 180.0
                    - 0.25
                    * ( ball_dist - ptype.playerSize() - sparam.ballSize() )
                    / ptype.kickableMargin() );
            if ( max_kick_accel > sparam.ballAccelMax() )
            {
                max_kick_accel = sparam.ballAccelMax();
            }
            int max_kick_accel_screen = M_data.config().scale( max_kick_accel );
            wxPoint bnext_screen( M_data.config().getScreenX( bnext.x ),
                                  M_data.config().getScreenY( bnext.y ) );

            dc.SetPen( M_data.gdi().getKickAccelPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            // draw no noise next ball move only by inertia
            dc.DrawLine( M_data.config().getScreenX( bpos.x ),
                         M_data.config().getScreenY( bpos.y ),
                         bnext_screen.x,
                         bnext_screen.y );
            // draw next ball reachable area
            dc.DrawCircle( bnext_screen.x, bnext_screen.y,
                           max_kick_accel_screen );
            // draw kick info text
            dc.SetFont( M_data.gdi().getPlayerFont() );
            dc.SetTextForeground( M_data.gdi().getKickAccelPen().GetColour() );
            dc.SetBackgroundMode( wxTRANSPARENT );
            wxString msg;
            msg.Printf( wxT( "MaxAccel=%.3f" ), max_kick_accel );
            dc.DrawText( msg,
                         bnext_screen.x + 10,
                         bnext_screen.y );
        }
    }
}
