// -*-c++-*-

/*!
  \file player.cpp
  \brief player data class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include <cmath>

#include "player.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

//! accessible only from this file;
namespace {

const double DEG2RAD = M_PI / 180.0;
const double RAD2DEG = 180.0 / M_PI;

}

/*-------------------------------------------------------------------*/
/*!
  constructor
*/
Player::Player()
    : M_side( rcsc::NEUTRAL )
    , M_unum( 0 )
    , M_bits( 0x0000 )
{
    M_data.mode = 0;
    M_data.type = 0;

    M_data.x = 0;
    M_data.y = 0;
    M_data.deltax = 0;
    M_data.deltay = 0;
    M_data.body_angle = 0;
    M_data.head_angle = 0;

    M_data.view_width = 0;
    M_data.view_quality = 0;

    M_data.stamina = 0;
    M_data.effort = 0;
    M_data.recovery = 0;

    M_data.kick_count = 0;
    M_data.dash_count = 0;
    M_data.turn_count = 0;
    M_data.say_count = 0;
    M_data.turn_neck_count = 0;
    M_data.catch_count = 0;
    M_data.move_count = 0;
    M_data.change_view_count = 0;

    // set real value
    setBody();
    setHead();
    setViewWidth();
}

/*-------------------------------------------------------------------*/
/*!
  all data is converted to v3 RCG format.
  but some data(view info, command count...) are set to ZERO.
  \param s side ID
  \param unum uniform number
  \param p v2 RCG data.
*/
void
Player::assign( const rcsc::SideID side,
                const int unum,
                const rcsc::rcg::pos_t & p )
{
    // raw view data
    // convert necwork byte order to local byte order
    // and convert to the v3 log format.
    M_data.mode = ntohs( p.enable );
    M_data.type = 0;

    M_data.x = static_cast< long >
        ( static_cast< double >( static_cast< short >( ntohs( p.x ) ) )
          * ( rcsc::rcg::SHOWINFO_SCALE2 / rcsc::rcg::SHOWINFO_SCALE ) );
    M_data.y = static_cast< long >
        ( static_cast< double >( static_cast< short >( ntohs( p.y ) ) )
          * ( rcsc::rcg::SHOWINFO_SCALE2 / rcsc::rcg::SHOWINFO_SCALE ) );
    M_data.deltax = 0;
    M_data.deltay = 0;
    M_data.body_angle = static_cast< long >
        ( ::rint( static_cast< double >( static_cast< short >( ntohs( p.angle ) ) )
                  * DEG2RAD
                  * rcsc::rcg::SHOWINFO_SCALE2 ) );
    M_data.head_angle = 0;

    M_data.view_width = 0;
    M_data.view_quality = 0;

    M_data.stamina = 0;
    M_data.effort = static_cast< long >( rcsc::rcg::SHOWINFO_SCALE2 );
    M_data.recovery = static_cast< long >( rcsc::rcg::SHOWINFO_SCALE2 );

    M_data.kick_count = 0;
    M_data.dash_count = 0;
    M_data.turn_count = 0;
    M_data.say_count = 0;
    M_data.turn_neck_count = 0;
    M_data.catch_count = 0;
    M_data.move_count = 0;
    M_data.change_view_count = 0;

    // player id
    M_side = side;
    M_unum = unum;

    // set bit mask
    M_bits = 0x0000;

    // set real value
    setBody();
    setHead();
    setViewWidth();
}

/*-------------------------------------------------------------------*/
/*!
  assign new data
*/
void
Player::assign( const rcsc::SideID side,
                const int unum,
                const rcsc::rcg::player_t & p )
{
    // raw view data
    // convert necwork byte order to local byte order
    M_data.mode = ntohs( p.mode );
    M_data.type = ntohs( p.type );

    M_data.x = ntohl( p.x );
    M_data.y = ntohl( p.y );
    M_data.deltax = ntohl( p.deltax );
    M_data.deltay = ntohl( p.deltay );
    M_data.body_angle = ntohl( p.body_angle ); // radian
    M_data.head_angle = ntohl( p.head_angle ); // radian

    M_data.view_width = ntohl( p.view_width ); // radian
    M_data.view_quality = ntohs( p.view_quality );

    M_data.stamina = ntohl( p.stamina );
    M_data.effort = ntohl( p.effort );
    M_data.recovery = ntohl( p.recovery );

    M_data.kick_count = ntohs( p.kick_count );
    M_data.dash_count = ntohs( p.dash_count );
    M_data.turn_count = ntohs( p.turn_count );
    M_data.say_count = ntohs( p.say_count );
    M_data.turn_neck_count = ntohs( p.turn_neck_count );
    M_data.catch_count = ntohs( p.catch_count );
    M_data.move_count = ntohs( p.move_count );
    M_data.change_view_count = ntohs( p.change_view_count );

    // player id
    M_side = side;
    M_unum = unum;

    // set bit mask
    M_bits = 0x0000;
    M_bits |= HAVE_TYPE;
    M_bits |= HAVE_DELTA;
    M_bits |= HAVE_STAMINA;
    M_bits |= HAVE_VIEW;
    M_bits |= HAVE_COUNT;

    // set real value
    setBody();
    setHead();
    setViewWidth();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Player::setBody()
{
    M_body = ( static_cast< double >( M_data.body_angle )
               / rcsc::rcg::SHOWINFO_SCALE2 )
        * RAD2DEG;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Player::setHead()
{
    double neck
        = ( static_cast< double >( M_data.head_angle )
            / rcsc::rcg::SHOWINFO_SCALE2 )
        * RAD2DEG;
    neck += body();

    while ( neck > 180.0 )
    {
        neck -= 360.0;
    }
    while ( neck < -180.0 )
    {
        neck += 360.0;
    }

    M_head = neck;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Player::setViewWidth()
{
    M_view_width = ( static_cast< double >( M_data.view_width )
                     / rcsc::rcg::SHOWINFO_SCALE2 )
        * RAD2DEG;
}
