// -*-c++-*-

/*!
  \file feditor_dialog.cpp
  \brief Formation editor operation Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// For compilers that support precompilation, includes "wx/wx.h".
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <iostream>

#include "id.h"

#include "feditor_data.h"
#include "feditor_frame.h"

#include "feditor_dialog.h"

/*-------------------------------------------------------------------*/
/*!

*/
FEditorDialog::FEditorDialog( FEditorFrame * parent )
    : wxDialog( parent,
                SWID_DIALOG_EDITOR, wxT( "Edit Box" ),
                wxDefaultPosition,
                wxDefaultSize,
                wxCAPTION | wxSYSTEM_MENU )
{
    createWindows();
    setDefaultValues();
    connectEvents();

    wxRect cli_disp = ::wxGetClientDisplayRect();
    this->SetSize( 0, cli_disp.GetHeight() - this->GetSize().y - 32,
                   -1, -1 );
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditorDialog::~FEditorDialog()
{
    //std::cerr << "delete FEditorDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::connectEvents()
{
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( FEditorDialog::handleCloseEvent ) );

    Connect( SWID_EDITOR_MIRROR_BALL, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( FEditorDialog::handleMirrorBall ) );

    Connect( SWID_EDITOR_APPLY_TO_FIELD, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( FEditorDialog::handleApplyToField ) );

}


/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::createWindows()
{
    wxBoxSizer * main_sizer = new wxBoxSizer( wxVERTICAL );

    // formation index and name
    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );
        sizer->Add( new wxStaticText( this, -1, wxT( "Training Method Name" ),
                                      wxDefaultPosition, wxDefaultSize,
                                      wxALIGN_CENTRE ),
                    1, 0, 0 );
        M_training_type_name = new wxTextCtrl( this, SWID_EDITOR_FORMATION_NAME,
                                               wxT( "---" ),
                                               wxDefaultPosition, wxDefaultSize );
        M_training_type_name->SetEditable( false );
        M_training_type_name->SetBackgroundColour( *wxLIGHT_GREY );

        sizer->Add( M_training_type_name, 1, 0, 0 );

        main_sizer->Add( sizer, 0, wxEXPAND | wxLEFT | wxRIGHT | wxTOP, 2 );
    }
    // ball info
    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );
        sizer->Add( new wxStaticText( this, -1, wxT( "Ball" ),
                                      wxDefaultPosition, wxSize( 40, -1 ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( " X:" ),
                                      wxDefaultPosition, wxSize( 24, -1 ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );
        M_ball_pos_x = new wxTextCtrl( this, -1, wxT( "0" ) );
        sizer->Add( M_ball_pos_x, 0, wxALIGN_CENTER_VERTICAL, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( " Y:" ),
                                      wxDefaultPosition, wxSize( 24, -1 ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );
        M_ball_pos_y = new wxTextCtrl( this, -1, wxT( "0" ) );
        sizer->Add( M_ball_pos_y, 0, wxALIGN_CENTER_VERTICAL, 0 );

        // selection id
        int width, height;
        this->GetTextExtent( wxT( "ReverseY" ), &width, &height );
        sizer->Add( new wxButton( this, SWID_EDITOR_MIRROR_BALL,
                                  wxT( "ReverseY" ),
                                  wxDefaultPosition, wxSize( width + 12, -1 ) ),
                    0, wxALIGN_CENTER_VERTICAL, 2 );

        main_sizer->Add( sizer, 0, wxEXPAND | wxLEFT | wxRIGHT | wxTOP, 2 );

    }
    // label info
    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxStaticText( this, -1, wxT( "Unum" ),
                                      wxDefaultPosition, wxSize( 40, -1 ) ),
                    0, 0, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( "Mirror" ),
                                      wxDefaultPosition, wxSize( 40, -1 ) ),
                    0, 0, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( "Role" ),
                                      wxDefaultPosition, wxDefaultSize,
                                      wxALIGN_CENTER ),
                    1, 0, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( "BaseX" ),
                                      wxDefaultPosition, wxDefaultSize,
                                      wxALIGN_CENTER ),
                    1, 0, 0 );
        sizer->Add( new wxStaticText( this, -1, wxT( "BaseY" ),
                                      wxDefaultPosition, wxDefaultSize,
                                      wxALIGN_CENTER ),
                    1, 0, 0 );

        main_sizer->Add( sizer, 0, wxEXPAND | wxLEFT | wxRIGHT | wxTOP, 2 );
    }

    for ( int i = 0; i < 11; ++i )
    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );
        wxString unum_str;
        unum_str.Printf( wxT( "%2d" ), i + 1 );
        sizer->Add( new wxStaticText( this, -1, unum_str,
                                      wxDefaultPosition, wxSize( 40, -1 ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );
        M_mirror_unum[i] = new wxTextCtrl( this, -1, wxT( "0" ),
                                           wxDefaultPosition, wxSize( 40, -1 ) );
        sizer->Add( M_mirror_unum[i], 0, 0, 0 );
        M_role_name[i] = new wxTextCtrl( this, -1 );
        sizer->Add( M_role_name[i], 1, 0, 0 );
        M_pos_x[i] = new wxTextCtrl( this, -1 );
        sizer->Add( M_pos_x[i], 1, 0, 0 );
        M_pos_y[i] = new wxTextCtrl( this, -1 );
        sizer->Add( M_pos_y[i], 1, 0, 0 );

        main_sizer->Add( sizer, 0, wxEXPAND | wxLEFT | wxRIGHT | wxTOP, 2 );
    }

    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxButton( this, SWID_EDITOR_APPLY_TO_FIELD,
                                  wxT( "Apply" ) ),
                    0, wxRIGHT, 4 );

        main_sizer->Add( sizer, 0, wxEXPAND | wxLEFT | wxRIGHT | wxTOP, 2 );
    }

    main_sizer->Add( 0, 2 ); // bottom space
    main_sizer->SetSizeHints( this );
    this->SetAutoLayout( true );
    this->SetSizer( main_sizer );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::setDefaultValues()
{
    //std::cerr << "FEditorDialog::setDefaultValues" << std::endl;
    //M_behave_mode->SetSelection( 0 );

    wxString temp;
    for ( int i = 0; i < 11; ++i )
    {
        M_mirror_unum[i]->SetValue( wxT( "0" ) );
        M_role_name[i]->SetValue( wxT( "Role" ) );
        M_pos_x[i]->SetValue( wxT( "0.0" ) );
        M_pos_y[i]->SetValue( wxT( "0.0" ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::update()
{
    boost::shared_ptr< FEditorData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditorDialog::update  no data ptr" << std::endl;
        return;
    }

    const boost::shared_ptr< const rcsc::Formation > f = data_ptr->getFormation();
    if ( ! f )
    {
        std::cerr << "FEditorDialog::update  no formation data" << std::endl;
        return;
    }

    M_training_type_name->SetValue( wxString( f->methodName().c_str(),
                                              *wxConvCurrent ) );
    M_training_type_name->SetEditable( false );
    M_training_type_name->SetBackgroundColour( *wxLIGHT_GREY );

    wxString msg;

    // ball info
    msg.Printf( wxT( "%.2f" ), data_ptr->getBall().x );
    M_ball_pos_x->SetValue( msg );
    msg.Printf( wxT( "%.2f" ), data_ptr->getBall().y );
    M_ball_pos_y->SetValue( msg );

    // player info
    const std::vector< rcsc::Vector2D > & positions = data_ptr->getOurPlayers();
    for ( int unum = 1; unum <= 11; ++unum )
    {
        const bool mirror = f->isMirrorType( unum );
        msg.Printf( wxT( "%2d" ), f->getMirrorNumber( unum ) );
        M_mirror_unum[unum - 1]->SetValue( msg );

        M_role_name[unum - 1]->SetValue( wxString( f->getRoleName( unum ).c_str(),
                                                   *wxConvCurrent ) );

        msg.Printf( wxT( "%.2f" ), positions[unum - 1].x );
        M_pos_x[unum - 1]->SetValue( msg );
        msg.Printf( wxT( "%.2f" ), positions[unum - 1].y );
        M_pos_y[unum - 1]->SetValue( msg );

        const wxColor & col = ( mirror ? *wxLIGHT_GREY : *wxWHITE );

        M_role_name[unum - 1]->SetBackgroundColour( col );
        M_pos_x[unum - 1]->SetBackgroundColour( col );
        M_pos_y[unum - 1]->SetBackgroundColour( col );

        M_role_name[unum - 1]->SetEditable( ! mirror );
        //M_pos_x[unum - 1]->SetEditable( ! mirror );
        //M_pos_y[unum - 1]->SetEditable( ! mirror );
    }
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    this->Hide();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::handleMirrorBall( wxCommandEvent & WXUNUSED( event ) )
{
    boost::shared_ptr< FEditorData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditorDialog::handleMirrorBall  no data ptr" << std::endl;
        return;
    }

    data_ptr->reverseY();
    /*
    double x, y;
    if ( M_ball_pos_x->GetValue().ToDouble( &x )
         && M_ball_pos_y->GetValue().ToDouble( &y ) )
    {
        y *= -1.0;

        data_ptr->moveBallTo( x, y );
    }
    */
    data_ptr->notifyDraw();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorDialog::handleApplyToField( wxCommandEvent & WXUNUSED( event ) )
{
    //std::cerr << "FEditorDialog::applyToField" << std::endl;

    boost::shared_ptr< FEditorData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditorDialog::handleApplyToField  no data ptr" << std::endl;
        return;
    }

    // ball
    {
        double x, y;
        if ( M_ball_pos_x->GetValue().ToDouble( &x )
             && M_ball_pos_y->GetValue().ToDouble( &y ) )
        {
            data_ptr->setBallPosition( x, y );
        }
    }

    // players
    for ( int unum = 1; unum <= 11; ++unum )
    {
        long mirror_unum = 0;
        if ( ! M_mirror_unum[unum-1]->GetValue().ToLong( & mirror_unum ) )
        {
            std::cerr << __FILE__ << ':' << __LINE__
                      << " *** ERROR *** You have input an invalid player number."
                      << std::endl;
            continue;
        }

#ifdef UNICODE
        std::string role_name = (const char*)(M_role_name[unum-1]->GetValue().mbc_str());
#else
        std::string role_name = (const char*)(M_role_name[unum-1]->GetValue().c_str());
#endif

        if ( role_name.empty() )
        {
            std::cerr << __FILE__ << ':' << __LINE__
                      << " *** ERROR *** You have input an empty role name."
                      << std::endl;
            continue;
        }

        data_ptr->updateRoleData( unum, static_cast< int >( mirror_unum ), role_name );

        double x, y;
        if ( M_pos_x[unum-1]->GetValue().ToDouble( &x )
             && M_pos_y[unum-1]->GetValue().ToDouble( &y ) )
        {
            data_ptr->setPlayerPosition( unum, x, y );
        }
    }

    data_ptr->notifyDraw();
}
