// -*-c++-*-

/*!
  \file debug_painter.cpp
  \brief debug info painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <iostream>

#include <rcsc/param/server_param.h>

#include "main_data.h"

#include "debug_painter.h"


/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::draw( wxDC & dc ) const
{
    int number = M_data.config().getSelectedNumber();
    if ( number == 0 )
    {
        return;
    }
    
    const rcsc::SideID player_side = ( number < 0 ? rcsc::RIGHT : rcsc::LEFT );

    const std::map< long, DebugViewCont > & view_map 
        = ( player_side == rcsc::RIGHT
            ? M_data.getViewHolder().getRightDebugView()
            : M_data.getViewHolder().getLeftDebugView() );

    std::map< long, DebugViewCont >::const_iterator it
        = view_map.find( M_data.getCurrentViewData()->cycle() );

    if ( it == view_map.end() )
    {
        return;
    }

    number = std::abs( number );

    if ( static_cast< int >( it->second.size() ) < number )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error debug view container size is small."
                  << std::endl;
        return;
    }

    const DebugViewPtr view = it->second[number - 1];
    if ( ! view )
    {
        // no data!
        //std::cerr << __FILE__ << ":" << __LINE__
        //          << " Error. No debug view data of " << number
        //          << std::endl;
        return;
    }

    // draw self
    if ( M_data.config().isShownDebugViewSelf() )
    {
        drawSelf( dc, player_side, *view );
    }


    // draw players
    if ( M_data.config().isShownDebugViewPlayers() )
    {
        drawPlayers( dc,
                     player_side,
                     view->getTeammates(),
                     ( M_data.config().isShownDebugViewTarget()
                       ? view->getTargetTeammate()
                       : -1 ),
                     M_data.gdi().getDebugTeammateBrush() );
        drawPlayers( dc,
                     player_side,
                     view->getOpponents(),
                     -1,
                     M_data.gdi().getDebugOpponentBrush() );
        drawPlayers( dc,
                     player_side,
                     view->getUnknownTeammates(),
                     -1,
                     M_data.gdi().getDebugUnknownTeammateBrush() );
        drawPlayers( dc,
                     player_side,
                     view->getUnknownOpponents(),
                     -1,
                     M_data.gdi().getDebugUnknownOpponentBrush() );
        drawPlayers( dc,
                     player_side,
                     view->getUnknownPlayers(),
                     -1,
                     M_data.gdi().getDebugUnknownPlayerBrush() );
    }

    // draw ball
    if ( M_data.config().isShownDebugViewBall()
         && view->getBall() )
    {
        drawBall( dc, player_side, *view );
    }

    // draw lines
    if ( M_data.config().isShownDebugViewFigure() )
    {
        drawLines( dc, player_side, *view );
        drawTriangles( dc, player_side, *view );
        drawRectangles( dc, player_side, *view );
        drawCircles( dc, player_side, *view );
    }

    if ( M_data.config().isShownDebugViewMessage()
         && ! view->getMessage().empty() )
    {
        drawMessage( dc, view->getMessage() );
    }
}

/*-------------------------------------------------------------------*/
/*!
  pos, vel, body, neck, comment
*/
void
DebugPainter::drawSelf( wxDC & dc,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::SelfT > self = view.getSelf();
    if ( ! self )
    {
        std::cerr << "DebugPainter::drawSelf. No self data!" << std::endl;
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const int ix = M_data.config().getScreenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int iy = M_data.config().getScreenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int r = std::max( 2,
                            ( M_data.config().isEnlarged()
                              ? M_data.config().scale( 1.0 )
                              : M_data.config().scale( 0.3 ) ) );
    double body_start_dir = - ( (double)self->body_ + 90.0 );
    if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;
    double body_end_dir = body_start_dir + 180.0;

    // draw targets
    if ( M_data.config().isShownDebugViewTarget()
         &&  view.getTargetPoint() )
    {
        int tx = M_data.config().getScreenX( view.getTargetPoint()->x * reverse );
        int ty = M_data.config().getScreenY( view.getTargetPoint()->y * reverse );

        dc.SetPen( M_data.gdi().getDebugTargetPen() );
        //dc.DrawLine( ix, iy, tx, ty );
        dc.DrawLine( tx - r, ty - r,
                     tx + r, ty + r );
        dc.DrawLine( tx - r, ty + r,
                     tx + r, ty - r );
    }

    // draw body half circle
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( M_data.gdi().getDebugSelfBrush() );
    dc.DrawEllipticArc( ix - r, iy - r,
                        r * 2, r * 2,
                        body_start_dir, body_end_dir );

    // draw edge
    dc.SetPen( M_data.gdi().getDebugPlayerPen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );
    dc.DrawCircle( ix, iy, r );

    // draw comment
    if ( M_data.config().isShownDebugViewComment()
         && ! self->comment_.empty() )
    {
        dc.SetFont( M_data.gdi().getDebugCommentFont() );
        dc.SetTextForeground( M_data.gdi().getDebugCommentFontColor() );
        dc.SetBackgroundMode( wxTRANSPARENT );

        wxString msg( self->comment_.c_str(),
                      *wxConvCurrent );
        dc.DrawText( msg,
                     ix - r, iy + r );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawBall( wxDC & dc,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::BallT > & ball = view.getBall();

    if ( ! ball )
    {
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    rcsc::Vector2D bpos( (double)ball->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    rcsc::Vector2D bvel( (double)ball->vx_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->vy_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const int ix = M_data.config().getScreenX( bpos.x );
    const int iy = M_data.config().getScreenY( bpos.y );
    const int r = std::max( 2,
                            ( M_data.config().isEnlarged()
                              ? M_data.config().scale( 0.25 )
                              : M_data.config().scale( 0.07 ) ) );

    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( M_data.gdi().getDebugBallBrush() );

    dc.DrawCircle( ix, iy, r );

    dc.SetPen( M_data.gdi().getDebugBallTracePen() );
    dc.SetBrush( M_data.gdi().getDebugBallTraceBrush() );

    int bx = ix;
    int by = iy;
    for ( int i = 0; i < 10; ++i )
    {
        bpos += bvel;
        bvel *= ball_decay;
        bx = M_data.config().getScreenX( bpos.x );
        by = M_data.config().getScreenY( bpos.y );
        dc.DrawCircle( bx, by, 2 );
    }
    dc.DrawLine( ix, iy, bx, by );

    // draw comment
    if ( M_data.config().isShownDebugViewComment()
         && ! ball->comment_.empty() )
    {
        dc.SetFont( M_data.gdi().getDebugCommentFont() );
        dc.SetTextForeground( M_data.gdi().getDebugCommentFontColor() );
        dc.SetBackgroundMode( wxTRANSPARENT );

        wxString msg( ball->comment_.c_str(),
                      *wxConvCurrent );
        dc.DrawText( msg,
                     ix - r, iy + r );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawPlayers( wxDC & dc,
                           const rcsc::SideID player_side,
                           const DebugViewData::PlayerCont & players,
                           const int target_number,
                           const wxBrush & body_brush ) const
{
    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const bool comment = M_data.config().isShownDebugViewComment();
    const int r = std::max( 2,
                            ( M_data.config().isEnlarged()
                              ? M_data.config().scale( 1.0 )
                              : M_data.config().scale( 0.3 ) ) );

    const DebugViewData::PlayerCont::const_iterator end = players.end();
    for ( DebugViewData::PlayerCont::const_iterator it = players.begin();
          it != end;
          ++it )
    {
        const int ix = M_data.config().getScreenX( (double)(*it)->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = M_data.config().getScreenY( (double)(*it)->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        if ( (*it)->body_ != -360 )
        {
            double body_start_dir = - ( (double)(*it)->body_ + 90.0 );
            if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;
            double body_end_dir = body_start_dir + 180.0;
            // draw body half circle
            dc.SetPen( *wxTRANSPARENT_PEN );
            dc.SetBrush( body_brush );
            dc.DrawEllipticArc( ix - r, iy - r,
                                r * 2, r * 2,
                                body_start_dir, body_end_dir );
            // draw edge
            dc.SetPen( M_data.gdi().getDebugPlayerPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( ix, iy, r );
        }
        else
        {
            // draw simple circle
            dc.SetPen( *wxTRANSPARENT_PEN );
            dc.SetBrush( body_brush );
            dc.DrawCircle( ix, iy, r );
        }

        if ( (*it)->unum_ > 0 )
        {
            dc.SetFont( M_data.gdi().getDebugCommentFont() );
            dc.SetTextForeground( M_data.gdi().getDebugCommentFontColor() );
            dc.SetBackgroundMode( wxTRANSPARENT );
            wxString num;
            num.Printf( wxT( "%2d" ), (*it)->unum_ );
            dc.DrawText( num, ix + r, iy - r );
        }

        if ( target_number == (*it)->unum_ )
        {
            dc.SetPen( M_data.gdi().getDebugTargetPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( ix, iy, r + 2 );
        }

        if ( comment
             && ! (*it)->comment_.empty() )
        {
            dc.SetFont( M_data.gdi().getDebugCommentFont() );
            dc.SetTextForeground( M_data.gdi().getDebugCommentFontColor() );
            dc.SetBackgroundMode( wxTRANSPARENT );

            wxString msg( (*it)->comment_.c_str(),
                          *wxConvCurrent );
            dc.DrawText( msg, ix - r, iy + r );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawLines( wxDC & dc,
                         const rcsc::SideID player_side,
                         const DebugViewData & view ) const
{
    if ( view.getLines().empty() )
    {
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( M_data.gdi().getDebugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::LineT >::const_iterator end = view.getLines().end();
    for ( std::list< DebugViewData::LineT >::const_iterator it = view.getLines().begin();
          it != end;
          ++it )
    {
        dc.DrawLine( M_data.config().getScreenX( it->x1_ * reverse ),
                     M_data.config().getScreenY( it->y1_ * reverse ),
                     M_data.config().getScreenX( it->x2_ * reverse ),
                     M_data.config().getScreenY( it->y2_ * reverse ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawTriangles( wxDC & dc,
                             const rcsc::SideID player_side,
                             const DebugViewData & view ) const
{
    if ( view.getTriangles().empty() )
    {
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( M_data.gdi().getDebugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::TriangleT >::const_iterator end = view.getTriangles().end();
    for ( std::list< DebugViewData::TriangleT >::const_iterator it = view.getTriangles().begin();
          it != end;
          ++it )
    {
        int x1 = M_data.config().getScreenX( it->x1_ * reverse );
        int y1 = M_data.config().getScreenY( it->y1_ * reverse );
        int x2 = M_data.config().getScreenX( it->x2_ * reverse );
        int y2 = M_data.config().getScreenY( it->y2_ * reverse );
        int x3 = M_data.config().getScreenX( it->x3_ * reverse );
        int y3 = M_data.config().getScreenY( it->y3_ * reverse );

        dc.DrawLine( x1, y1, x2, y2 );
        dc.DrawLine( x2, y2, x3, y3 );
        dc.DrawLine( x3, y3, x1, y1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawRectangles( wxDC & dc,
                              const rcsc::SideID player_side,
                              const DebugViewData & view ) const
{
    if ( view.getRectangles().empty() )
    {
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( M_data.gdi().getDebugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::RectT >::const_iterator end = view.getRectangles().end();
    for ( std::list< DebugViewData::RectT >::const_iterator it = view.getRectangles().begin();
          it != end;
          ++it )
    {
        int left_x = M_data.config().getScreenX( it->left_x_ * reverse );
        int top_y = M_data.config().getScreenY( it->top_y_ * reverse );
        int right_x = M_data.config().getScreenX( it->right_x_ * reverse );
        int bottom_y = M_data.config().getScreenY( it->bottom_y_ * reverse );

        dc.DrawLine( left_x, top_y, right_x, top_y );
        dc.DrawLine( right_x, top_y, right_x, bottom_y );
        dc.DrawLine( right_x, bottom_y, left_x, bottom_y );
        dc.DrawLine( left_x, bottom_y, left_x, top_y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawCircles( wxDC & dc,
                           const rcsc::SideID player_side,
                           const DebugViewData & view ) const
{
    if ( view.getCircles().empty() )
    {
        return;
    }

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( M_data.gdi().getDebugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::CircleT >::const_iterator end = view.getCircles().end();
    for ( std::list< DebugViewData::CircleT >::const_iterator it = view.getCircles().begin();
          it != end;
          ++it )
    {
        dc.DrawCircle( M_data.config().getScreenX( it->center_x_ * reverse ),
                       M_data.config().getScreenY( it->center_y_ * reverse ),
                       M_data.config().scale( it->radius_ ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawMessage( wxDC & dc,
                           const std::string & message ) const
{
    dc.SetFont( M_data.gdi().getDebugMessageFont() );
    dc.SetTextForeground( M_data.gdi().getDebugMessageFontColor() );
    dc.SetBackgroundMode( wxTRANSPARENT );

    wxString msg( message.c_str(),
                  *wxConvCurrent );
    dc.DrawText( msg,
                 10,
                 M_data.config().getScoreBoardHeight() + 2 );
}
