// -*-c++-*-

/*!
  \file view_holder.cpp
  \brief view data repository Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include <cassert>
#include <iostream>
#include <algorithm>
#include <functional>

#include <rcsc/param/player_param.h>
#include <rcsc/param/server_param.h>

#include "id.h"
#include "app_config.h"
#include "main_frame.h"

#include "view_holder.h"

const std::size_t RCG_SIZE_LIMIT = 6000 * 8;

/*-------------------------------------------------------------------*/
/*!
  default constructor.
*/
ViewHolder::ViewHolder( MainFrame * main_frame )
    : M_main_frame( main_frame )
    , M_last_playmode( rcsc::PM_Null )
{
    assert( main_frame );

    rcsc::PlayerType default_type( rcsc::ServerParam::i() );
    M_player_types.assign( 7, default_type );


    M_monitor_view_cont.reserve( 1024 * 8 );

}

/*-------------------------------------------------------------------*/
/*!

*/
ViewHolder::~ViewHolder()
{
    //std::cerr << "ViewHolder Left Debug View Size " << M_left_debug_view.size() << std::endl;
    //std::cerr << "ViewHolder Right Debug View Size " << M_right_debug_view.size() << std::endl;
}

/*-------------------------------------------------------------------*/
/*!
  clear all stored data
*/
void
ViewHolder::clear()
{
    M_last_playmode = rcsc::PM_Null;
    std::memset( M_last_team_left.name, '\0', 16 );
    M_last_team_left.score = 0;
    std::memset( M_last_team_right.name, '\0', 16 );
    M_last_team_right.score = 0;
    M_score_change_indexes.clear();
    M_latest_view_data.reset();
    M_monitor_view_cont.clear();

    M_left_debug_view.clear();
    M_right_debug_view.clear();

    rcsc::PlayerType default_type( rcsc::ServerParam::i() );
    M_player_types.assign( 7, default_type );
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShowInfo( const rcsc::rcg::showinfo_t & show )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show ) );

    addPlayMode( show.pmode );
    addTeamInfo( show.team[0], show.team[1] );

    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShowInfo2( const rcsc::rcg::showinfo_t2 & show2 )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show2 ) );

    addPlayMode( show2.pmode );
    addTeamInfo( show2.team[0], show2.team[1] );

    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addShortShowInfo2( const rcsc::rcg::short_showinfo_t2 & show2 )
{
    if ( M_monitor_view_cont.size() > RCG_SIZE_LIMIT )
    {
        return false;
    }

    MonitorViewPtr ptr( new MonitorViewData( show2,
                                             M_last_playmode,
                                             M_last_team_left,
                                             M_last_team_right ) );
    addMonitorViewData( ptr );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::addMonitorViewData( MonitorViewPtr ptr )
{
    if ( M_latest_view_data )
    {
        // before adding the latest view data,
        // we should remove old BeforeKickOffMode or TimeOver data
        // from data container if exist.

        if ( ! M_monitor_view_cont.empty() )
        {
            if ( ptr->playmode().mode() == rcsc::PM_BeforeKickOff
                 || ptr->playmode().mode() == rcsc::PM_TimeOver )
            {
                rcsc::PlayMode back_pmode
                    = M_monitor_view_cont.back()->playmode().mode();
                if ( back_pmode == rcsc::PM_BeforeKickOff
                     || back_pmode == rcsc::PM_TimeOver )
                {
                    // last data must be swapped.
                    M_monitor_view_cont.pop_back();
                }
            }
            else if ( M_main_frame->isMonitorConnected()
                      && ! AppConfig::instance().timeShiftReplay() )
            {
                M_monitor_view_cont.pop_back();
            }
        }

        M_monitor_view_cont.push_back( M_latest_view_data );
    }

    M_latest_view_data = ptr;
}

/*-------------------------------------------------------------------*/
/*!
  this methos should be called just after finish to load rcg file.
*/
void
ViewHolder::pushBackLatestViewData()
{
    if ( M_latest_view_data )
    {
        M_monitor_view_cont.push_back( M_latest_view_data );
    }
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addMsgInfo( const short board,
                        const std::string & msg )
{
    //std::cerr << "ViewHolder::addMsgInfo" << std::endl;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addDrawInfo( const rcsc::rcg::drawinfo_t & msg )
{
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addPlayMode( const char pmode )
{
    M_last_playmode = pmode;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  push back new data
*/
bool
ViewHolder::addTeamInfo( const rcsc::rcg::team_t & team_l,
                         const rcsc::rcg::team_t & team_r )
{
    if ( M_last_team_left.score != team_l.score
         || M_last_team_right.score != team_r.score )
    {
        M_score_change_indexes.push_back( M_monitor_view_cont.size() );
    }

    M_last_team_left = team_l;
    M_last_team_right = team_r;
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  add new hetero player param
*/
bool
ViewHolder::addPlayerType( const rcsc::rcg::player_type_t & param )
{
    //std::cerr << "ViewHolder::addPlayerType" << std::endl;

    int idx = static_cast< int >( htons( param.id ) );
    if ( idx < 0 || 6 < idx )
    {
        std::cerr << "addPlayerType. over the array size" << std::endl;
        return false;
    }

    M_player_types[idx] = rcsc::PlayerType( rcsc::ServerParam::i(),
                                            param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  set server param
*/
bool
ViewHolder::addServerParam( const rcsc::rcg::server_params_t & param )
{
    //std::cerr << "ViewHolder::addServerParam" << std::endl;
    rcsc::ServerParam::instance().convertFrom( param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  set player param
*/
bool
ViewHolder::addPlayerParam( const rcsc::rcg::player_params_t & param )
{
    //std::cerr << "ViewHolder::addPlayerParam" << std::endl;
    rcsc::PlayerParam::instance().convertFrom( param );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewHolder::addDebugView( const long & cycle,
                          char side,
                          const int unum,
                          DebugViewPtr debug_view )
{
    typedef std::map< long, DebugViewCont >::iterator IterType;

    if ( ! debug_view )
    {
        return;
    }

    if ( unum < 1 || 11 < unum )
    {
        std::cerr << "ViewHolder::addDebugView. Error unum over range"
                  << std::endl;
        return;
    }

    std::map< long, DebugViewCont > * view_map = ( side == 'l'
                                                   ? &M_left_debug_view
                                                   : &M_right_debug_view );

    IterType it = view_map->find( cycle );

    // already exist this cycle data
    if ( it != view_map->end() )
    {
        it->second[unum - 1] = debug_view;
        return;
    }

    // insert new cycle
    std::pair< IterType, bool > p
        = view_map->insert ( std::make_pair( cycle, DebugViewCont( 11 ) ) );

    if ( ! p.second )
    {
        std::cerr << "ViewHolder::addDebugView. Failed to add DebugView"
                  << std::endl;
        return;
    }
    p.first->second[unum - 1] = debug_view;
}

/*-------------------------------------------------------------------*/
/*!
  get view data at specified incex
  \param idx index of data vector
  \return const shared_ptr to monitor view data
*/
const
MonitorViewPtr
ViewHolder::getViewData( const std::size_t idx ) const
{
    if ( M_monitor_view_cont.size() <= idx )
    {
        return MonitorViewPtr
            ( static_cast< MonitorViewData * >( 0 ) );
    }

    return M_monitor_view_cont[idx];
}

/*-------------------------------------------------------------------*/
/*!

*/
std::size_t
ViewHolder::getIndexOf( const long & cycle ) const
{
    MonitorViewCont::size_type len = M_monitor_view_cont.size();
    MonitorViewCont::size_type half;

    MonitorViewCont::const_iterator first = M_monitor_view_cont.begin();
    MonitorViewCont::const_iterator middle;

    // lower bound algorithm
    while ( len > 0 )
    {
        half = len >> 1;
        middle = first;
        middle += half;
        if ( (*middle)->cycle() < cycle )
        {
            first = middle;
            ++first;
            len = len - half - 1;
        }
        else
        {
            len = half;
        }
    }

    if ( first == M_monitor_view_cont.end() )
    {
        return 0;
    }

    return std::distance( M_monitor_view_cont.begin(), first );
}
