// -*-c++-*-

/*!
  \file gdi_config.h
  \brief GDI object manager class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCER_WINDOW_GDI_CONFIG_H
#define SOCCER_WINDOW_GDI_CONFIG_H

#include <boost/shared_ptr.hpp>

#include <rcsc/types.h>

class wxFileConfig;

//! GDI object configuration manager class
class GDIConfig {
public:
    //--------------------------------------------------
    // default color settings
    static const wxColor SCORE_BOARD_COLOR;
    static const wxColor SCORE_BOARD_FONT_COLOR;
    static const wxColor FIELD_COLOR;
    static const wxColor FIELD_DARK_COLOR;
    static const wxColor LINE_COLOR;
    static const wxColor BALL_PEN_COLOR;
    static const wxColor BALL_COLOR;
    static const wxColor BALL_FONT_COLOR;
    static const wxColor PLAYER_PEN_COLOR;
    static const wxColor REAL_BODY_PEN_COLOR;
    static const wxColor LEFT_TEAM_COLOR;
    static const wxColor LEFT_GOALIE_COLOR;
    static const wxColor RIGHT_TEAM_COLOR;
    static const wxColor RIGHT_GOALIE_COLOR;
    static const wxColor PLAYER_SHADOW_COLOR;
    static const wxColor PLAYER_NUMBER_FONT_COLOR;
    static const wxColor PLAYER_STAMINA_FONT_COLOR;
    static const wxColor PLAYER_DETAIL_FONT_COLOR;
    static const wxColor SELECTED_PLAYER_FONT_COLOR;
    static const wxColor EFFORT_DECAYED_PEN_COLOR;
    static const wxColor RECOVERY_DECAYED_PEN_COLOR;
    static const wxColor VIEW_CONE_PEN_COLOR;
    static const wxColor VIEW_AREA_PEN_COLOR;
    static const wxColor KICK_PEN_COLOR;
    static const wxColor KICK_FAULT_COLOR;
    static const wxColor CATCH_COLOR;
    static const wxColor CATCH_FAULT_COLOR;
    static const wxColor COLLIDE_BALL_COLOR;
    static const wxColor COLLIDE_PLAYER_COLOR;
    static const wxColor TACKLE_COLOR;
    static const wxColor TACKLE_FAULT_COLOR;
    static const wxColor MEASURE_COLOR;
    static const wxColor MEASURE_FONT_COLOR;
    static const wxColor DEBUG_BALL_COLOR;
    static const wxColor DEBUG_SELF_COLOR;
    static const wxColor DEBUG_TEAMMATE_COLOR;
    static const wxColor DEBUG_OPPONENT_COLOR;
    static const wxColor DEBUG_UNKNOWN_TEAMMATE_COLOR;
    static const wxColor DEBUG_UNKNOWN_OPPONENT_COLOR;
    static const wxColor DEBUG_UNKNOWN_PLAYER_COLOR;

private:
    //--------------------------------------------------
    // score board

    wxBrush M_score_board_brush;
    wxFont M_score_board_font;
    wxColor M_score_board_font_color;

    //--------------------------------------------------
    // field

    //! field grass brush
    wxBrush M_field_brush;
    //! field dark grass brush
    wxBrush M_field_dark_brush;
    //! line pen
    wxPen M_line_pen;

    //--------------------------------------------------
    // ball

    wxPen M_ball_pen; //!< ball edge
    wxBrush M_ball_brush; //!< ball body paint info
    wxFont M_ball_font; //!< ball info font info
    wxColor M_ball_font_color; //!< ball font foreground color

    //--------------------------------------------------
    // players

    //! player edge pen
    wxPen M_player_pen;
    //! real body edge pen
    wxPen M_real_body_pen;

    //! shadow brush
    wxBrush M_shadow_brush;

    //! left team color pen
    wxPen M_left_team_pen;
    //! left team brush (= left player body brush)
    wxBrush M_left_team_brush;
    //! left goalie color pen
    wxPen M_left_goalie_pen;
    //! left goalie body brush
    wxBrush M_left_goalie_brush;

    //! right team pen
    wxPen M_right_team_pen;
    //! right team brush(= right player body brush)
    wxBrush M_right_team_brush;
    //! right goalie color pen
    wxPen M_right_goalie_pen;
    //! left goalie body brush
    wxBrush M_right_goalie_brush;

    //! player info font
    wxFont M_player_font;
    //! foreground color of the player number font
    wxColor M_player_number_font_color;
    //! foreground color of the player stamina font
    wxColor M_player_stamina_font_color;
    //! foreground color of the player detail
    wxColor M_player_detail_font_color;
    //! foreground color of the selected player font
    wxColor M_selected_player_font_color;

    //! the pen of the effort decayed player indication circle
    wxPen M_effort_decayed_pen;
    //! the pen of the recovery decayed player indication circle
    wxPen M_recovery_decayed_pen;

    //! the pen of players' view cone
    wxPen M_view_cone_pen;
    //! the pen of players' view area
    wxPen M_view_area_pen;


    //--------------------------------------------------
    // player status

    wxPen M_kick_pen; //!< edge color when kicking
    wxPen M_kick_fault_pen; //!< edge color when kick falut
    wxBrush M_kick_fault_brush; //!< body paint brush when kick falut
    wxBrush M_catch_brush; //!< body paint brush when goalie catch
    wxBrush M_catch_fault_brush; //!< body paint brush when goalie catch fault
    wxBrush M_collide_ball_brush; //!< body paint brush when collision with ball
    wxBrush M_collide_player_brush; //!< body paint brush when collision with player
    wxPen M_tackle_pen; //!< edge pen when tackling
    wxBrush M_tackle_brush; //!< body paint brush when tackling
    wxBrush M_tackle_fault_brush; //!< body paint brush when tackle fault

    //! kick accel info pen
    wxPen M_kick_accel_pen;
    //! tackle area rectangle line pen
    wxPen M_tackle_area_pen;

    //--------------------------------------------------
    // misc

    // mouse measure
    wxColor M_measure_color; //!< color of mouse measure pen
    wxPen M_measure_pen;
    wxColor M_measure_font_color; //!< color of mouse measure font
    wxFont M_measure_font;  //!< mouse measure font

    //--------------------------------------------------
    // debug

    // ball
    wxPen M_debug_ball_pen;
    wxBrush M_debug_ball_brush;

    // ball trace
    wxPen M_debug_ball_trace_pen;
    wxBrush M_debug_ball_trace_brush;

    // player
    wxPen M_debug_player_pen; //!< debug info player's body edge pen
    wxPen M_debug_body_dir_pen; //!< debug info player's body direction
    wxBrush M_debug_self_brush;
    wxBrush M_debug_teammate_brush;
    wxBrush M_debug_opponent_brush;
    wxBrush M_debug_unknown_teammate_brush;
    wxBrush M_debug_unknown_opponent_brush;
    wxBrush M_debug_unknown_player_brush;

    // target
    wxPen M_debug_target_pen;

    // line, triangle, rectanble, circle
    wxPen M_debug_figure_pen;

    // message
    wxFont M_debug_comment_font;
    wxColor M_debug_comment_font_color;
    wxFont M_debug_message_font;
    wxColor M_debug_message_font_color; //!< debug message foreground color

public:
    //! constructor
    GDIConfig();
    //! destructor
    ~GDIConfig();

    //--------------------------------------------------
    //! load configuration from file.
    void loadConfig();
    //! save configuration to file.
    void saveConfig();

    void resizeScoreBoardFont( const int size );

private:
    //! convert config string to color object
    wxColor toColor( const wxString & conf_str );
    //! convert clor object to config string
    wxString toString( const wxColor & color );

public:

    //--------------------------------------------------
    // accessor
    //! get const reference
    const
    wxBrush & getScoreBoardBrush() const
      {
          return M_score_board_brush;
      }
    //! get const reference
    const
    wxFont & getScoreBoardFont() const
      {
          return M_score_board_font;
      }
    //! get const reference
    const
    wxColor & getScoreBoardFontColor() const
      {
          return M_score_board_font_color;
      }

    //! get const reference
    const
    wxBrush & getFieldBrush() const
      {
          return M_field_brush;
      }
    //! get const reference
    const
    wxBrush & getFieldDarkBrush() const
      {
          return M_field_dark_brush;
      }
    //! get const reference
    const
    wxPen & getLinePen() const
      {
          return M_line_pen;
      }

    //! get const reference
    const
    wxPen & getBallPen() const
      {
          return M_ball_pen;
      }
    //! get const reference
    const
    wxBrush & getBallBrush() const
      {
          return M_ball_brush;
      }
    //! get const reference
    const
    wxFont & getBallFont() const
      {
          return M_ball_font;
      }
    //! get const reference
    const
    wxColor & getBallFontColor() const
      {
          return M_ball_font_color;
      }

    //! get const reference
    const
    wxPen & getPlayerPen() const
      {
          return M_player_pen;
      }
    //! get const reference
    const
    wxPen & getRealBodyPen() const
      {
          return M_real_body_pen;
      }
    //! get const reference
    const
    wxBrush & getShadowBrush() const
      {
          return M_shadow_brush;
      }
    //! get const reference
    const
    wxPen & getLeftTeamPen() const
      {
          return M_left_team_pen;
      }
    //! get const reference
    const
    wxBrush & getLeftTeamBrush() const
      {
          return M_left_team_brush;
      }
    //! get const reference
    const
    wxPen & getLeftGoaliePen() const
      {
          return M_left_goalie_pen;
      }
    //! get const reference
    const
    wxBrush & getLeftGoalieBrush() const
      {
          return M_left_goalie_brush;
      }

    //! get const reference
    const
    wxPen & getRightTeamPen() const
      {
          return M_right_team_pen;
      }
    //! get const reference
    const
    wxBrush & getRightTeamBrush() const
      {
          return M_right_team_brush;
      }
    //! get const reference
    const
    wxPen & getRightGoaliePen() const
      {
          return M_right_goalie_pen;
      }
    //! get const reference
    const
    wxBrush & getRightGoalieBrush() const
      {
          return M_right_goalie_brush;
      }

    //! get const reference
    const
    wxFont & getPlayerFont() const
      {
          return M_player_font;
      }
    //! get const reference
    const
    wxColor & getPlayerNumberFontColor() const
      {
          return M_player_number_font_color;
      }
    //! get const reference
    const
    wxColor & getPlayerStaminaFontColor() const
      {
          return M_player_stamina_font_color;
      }
    //! get const reference
    const
    wxColor & getPlayerDetailFontColor() const
      {
          return M_player_detail_font_color;
      }
    //! get const reference
    const
    wxColor & getSelectedPlayerFontColor() const
      {
          return M_selected_player_font_color;
      }

    //! get const reference
    const
    wxPen & getEffortDecayedPen() const
      {
          return M_effort_decayed_pen;
      }
    //! get const reference
    const
    wxPen & getRecoveryDecayedPen() const
      {
          return M_recovery_decayed_pen;
      }

    //! get const reference
    const
    wxPen & getViewConePen() const
      {
          return M_view_cone_pen;
      }
    //! get const reference
    const
    wxPen & getViewAreaPen() const
      {
          return M_view_area_pen;
      }

    //! get const reference
    const
    wxPen & getKickPen() const
      {
          return M_kick_pen;
      }
    //! get const reference
    const
    wxPen & getKickFaultPen() const
      {
          return M_kick_fault_pen;
      }
    //! get const reference
    const
    wxBrush & getKickFaultBrush() const
      {
          return M_kick_fault_brush;
      }
    //! get const reference
    const
    wxBrush & getCatchBrush() const
      {
          return M_catch_brush;
      }
    //! get const reference
    const
    wxBrush & getCatchFaultBrush() const
      {
          return M_catch_fault_brush;
      }
    //! get const reference
    const
    wxBrush & getCollideBallBrush() const
      {
          return M_collide_ball_brush;
      }
    //! get const reference
    const
    wxBrush & getCollidePlayerBrush() const
      {
          return M_collide_player_brush;
      }
    //! get const reference
    const
    wxPen & getTacklePen() const
      {
          return M_tackle_pen;
      }
    //! get const reference
    const
    wxBrush & getTackleBrush() const
      {
          return M_tackle_brush;
      }
    //! get const reference
    const
    wxBrush & getTackleFaultBrush() const
      {
          return M_tackle_fault_brush;
      }

    //! get const reference
    const
    wxPen & getKickAccelPen() const
      {
          return M_kick_accel_pen;
      }
    //! get const reference
    const
    wxPen & getTackleAreaPen() const
      {
          return M_tackle_area_pen;
      }

    //! get const reference
    const
    wxColor & getMeasureColor() const
      {
          return M_measure_color;
      }
    //! get const reference
    const
    wxPen & getMeasurePen() const
      {
          return M_measure_pen;
      }

    //! get const reference
    const
    wxColor & getMeasureFontColor() const
      {
          return M_measure_font_color;
      }
    //! get const reference
    const
    wxFont & getMeasureFont() const
      {
          return M_measure_font;
      }

    const
    wxPen & getDebugBallPen() const
      {
          return M_debug_ball_pen;
      }
    const
    wxBrush & getDebugBallBrush() const
      {
          return M_debug_ball_brush;
      }

    const
    wxPen & getDebugBallTracePen() const
      {
          return M_debug_ball_trace_pen;
      }
    const
    wxBrush & getDebugBallTraceBrush() const
      {
          return M_debug_ball_trace_brush;
      }

    const
    wxPen & getDebugPlayerPen() const
      {
          return M_debug_player_pen;
      }
    const
    wxPen & getDebugBodyDirPen() const
      {
          return M_debug_body_dir_pen;
      }
    const
    wxBrush & getDebugSelfBrush() const
      {
          return M_debug_self_brush;
      }
    const
    wxBrush & getDebugTeammateBrush() const
      {
          return M_debug_teammate_brush;
      }
    const
    wxBrush & getDebugOpponentBrush() const
      {
          return M_debug_opponent_brush;
      }
    const
    wxBrush & getDebugUnknownTeammateBrush() const
      {
          return M_debug_unknown_teammate_brush;
      }
    const
    wxBrush & getDebugUnknownOpponentBrush() const
      {
          return M_debug_unknown_opponent_brush;
      }
    const
    wxBrush & getDebugUnknownPlayerBrush() const
      {
          return M_debug_unknown_player_brush;
      }
    const
    wxPen & getDebugTargetPen() const
      {
          return M_debug_target_pen;
      }
    const
    wxPen & getDebugFigurePen() const
      {
          return M_debug_figure_pen;
      }
    const
    wxFont & getDebugCommentFont() const
      {
          return M_debug_comment_font;
      }
    const
    wxColor & getDebugCommentFontColor() const
      {
          return M_debug_comment_font_color;
      }
    const
    wxFont & getDebugMessageFont() const
      {
          return M_debug_message_font;
      }
    const
    wxColor & getDebugMessageFontColor() const
      {
          return M_debug_message_font_color;
      }



    //! get const reference
    const
    wxBrush & getPlayerBodyBrush( const rcsc::SideID side,
                                  const bool goalie ) const;
};

#endif
