// -*-c++-*-

/*!
	\file formation_delaunay.h
	\brief formation data classes using Delaunay Triangulation Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_FORMATION_DELAUNAY_H
#define RCSC_FORMATION_DELAUNAY_H

#include <iostream>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/delaunay_triangulation.h>

#include "formation.h"

/*!
  \class FormationDelaunay
  \brief formation definition using Delaunay Triangulation
*/
class FormationDelaunay
    : public Formation {
public:

    struct Param {
        rcsc::Vector2D M_ball; //!< ball position = focus point
        std::vector< rcsc::Vector2D > M_players; //!< all players' desired position

        /*!
          \brief just allocate memory
         */
        Param()
            : M_ball( 0.0, 0.0 )
            , M_players( 11 )
          { }

        /*!
          \brief initialize with training data snapshot
         */
        Param( const Snapshot & data );

        /*!
          \brief initialize with specified data
          \param ball ball position
          \param players players' position container
         */
        Param( const rcsc::Vector2D & ball,
               const std::vector< rcsc::Vector2D > & players )
            : M_ball( ball )
            , M_players( players )
          { }

        /*!
          \brief initialize with specified data
          \param ball ball position
          \param players players' position container
          \return const reference to itself
         */
        const
        Param & assign( const rcsc::Vector2D & ball,
                        const std::vector< rcsc::Vector2D > & players )
          {
              M_ball = ball;
              M_players = players;
              return *this;
          }

        /*!
          \brief get the specified player's position
          \param unum player number
          \return position value
         */
        rcsc::Vector2D getPosition( const int unum ) const;
    };

private:

    //! player's role names
    std::string M_role_name[11];

    //! set of desired positins used by delaunay triangulation & linear interpolation
    std::vector< Param > M_param;

    //! delaunay triangulation
    rcsc::DelaunayTriangulation M_delaunay;

public:

    /*!
      \brief just call the base class constructor to initialize formation method name
     */
    FormationDelaunay();

    /*!
      \brief static method. get formation method name
      \return method name string
     */
    static
    std::string name()
      {
          return std::string( "Delaunay" );
      }

    /*!
      \brief static factory method. create new object
      \return new object
     */
    static
    Formation * create()
      {
          return ( new FormationDelaunay );
      }

    //--------------------------------------------------------------

    /*!
      \brief get the name of this formation
      \return name string
     */
    virtual
    std::string getMethodName() const
      {
          return FormationDelaunay::name();
      }

    /*!
      \brief create default formation. assign role and initial positions.
      \return snapshow variable for the initial state(ball pos=(0,0)).
     */
    virtual
    Snapshot createDefaultParam();

protected:
    /*!
      \brief create new role parameter.
      \param unum target player's number
      \param role_name new role name
      \param type side type of this parameter
     */
    virtual
    void createNewRole( const int unum,
                        const std::string & role_name,
                        const SideType type );
    /*!
      \brief set the role name of the specified player
      \param unum target player's number
      \param name role name string.
     */
    virtual
    void setRoleName( const int unum,
                      const std::string & name );

    /*!
      \brief set symmetry player info
      \param unum changed player's number
      \param mirror_unum refered player number
    */
    virtual
    bool setMirrorType( const int unum,
                        const int mirror_unum );

public:

    /*!
      \brief get the role name of the specified player
      \param unum target player's number
      \return role name string. if empty string is returned,
      that means no role parameter is assigned for unum.
     */
    virtual
    std::string getRoleName( const int unum ) const;

    /*!
      \brief get position for the current focus point
      \param unum player number
      \param forcus_point current focus point, usually ball position.
     */
    virtual
    rcsc::Vector2D getPosition( const int unum,
                                const rcsc::Vector2D & focus_point ) const;

    /*!
      \brief update formation paramter using training data set
      \param train_data training data container
     */
    virtual
    void train( const std::list< Snapshot > & train_data );

    /*!
      \brief restore data from the input stream.
      \param is reference to the input stream.
      \return parsing result
     */
    virtual
    bool read( std::istream & is );

    /*!
      \brief put data to the output stream.
      \param os reference to the output stream
      \return reference to the output stream
     */
    virtual
    std::ostream & print( std::ostream & os ) const;

};

#endif
