// -*-c++-*-

/*!
  \file feditor_frame.cpp
  \brief formation editor frame class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/spinctrl.h>
#endif

#include <iostream>

#include "xpm/chase.xpm"
#include "xpm/delete.xpm"
#include "xpm/hand.xpm"
#include "xpm/train.xpm"
#include "xpm/record.xpm"
#include "xpm/replace.xpm"
#include "xpm/save.xpm"

#include "id.h"
#include "app_config.h"
#include "feditor_canvas.h"
#include "feditor_data.h"
#include "feditor_dialog.h"

#include "formation_delaunay.h"
#include "formation_bpn.h"
#include "formation_sbsp.h"

#include "feditor_frame.h"

#if defined(__WXGTK__) || defined(__WXX11__) || defined(__WXMOTIF__) || defined(__WXMAC__)
#  include "xpm/soccerwindow2-nostr.xpm"
#endif

/*-------------------------------------------------------------------*/
/*!
  constructing main frame.
*/
FEditorFrame::FEditorFrame()
    : wxFrame( static_cast< wxWindow * >( 0 ), -1,
               wxT( "Formation Editor" ),
               wxDefaultPosition,
               wxSize( 620, 480 ) )
    , M_editor_canvas( static_cast< FEditorCanvas * >( 0 ) )
    , M_editor_dialog( static_cast< FEditorDialog * >( 0 ) )
    , M_train_data_index( static_cast< wxSpinCtrl * >( 0 ) )
{
    // set minimal window size
    this->SetSizeHints( 280, 220 );
    this->SetSize( AppConfig::instance().framePosX(),
                   AppConfig::instance().framePosY(),
                   AppConfig::instance().frameWidth(),
                   AppConfig::instance().frameHeight() );

    createWindows();
    connectEvents();

    SetIcon( wxICON( soccerwindow2_nostr ) );
}

/*-------------------------------------------------------------------*/
/*!
  destructing main frame.
*/
FEditorFrame::~FEditorFrame()
{
    //std::cerr << "delete FEditorFrame" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!
  creates dynamic event signal connect
*/
void
FEditorFrame::connectEvents()
{
    // ---------------------------------------------------
    // register wx event hander
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( FEditorFrame::handleCloseEvent ) );

    // command event : menu
    // file
    Connect( SWID_EDITOR_NEW, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuNew ) );
    Connect( SWID_EDITOR_OPEN, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuOpen ) );
    Connect( SWID_EDITOR_SAVE_ALL, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuSaveAll ) );
    Connect( SWID_EDITOR_SAVE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuSave ) );
    Connect( SWID_EDITOR_SAVE_AS, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuSaveAs ) );
    Connect( SWID_EDITOR_OPEN_TRAIN_DATA, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuOpenTrainData ) );
    Connect( SWID_EDITOR_SAVE_TRAIN_DATA, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuSaveTrainData ) );
    Connect( SWID_EDITOR_SAVE_TRAIN_DATA_AS, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuSaveTrainDataAs ) );
    Connect( SWID_QUIT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuQuit ) );

    // spin ctrl
    //Connect( SWID_EDITOR_SPIN_TRAIN_DATA, wxEVT_COMMAND_TEXT_UPDATED,
    //wxTextEventHandler( FEditorFrame::handleSpinTrainData ) );
    Connect( SWID_EDITOR_SPIN_TRAIN_DATA, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( FEditorFrame::handleSpinTrainData ) );

    // command
    Connect( SWID_EDITOR_RESET_BALL, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleResetBall ) );
    Connect( SWID_EDITOR_PLAYER_AUTO_MOVE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handlePlayerAutoMove ) );

    Connect( SWID_EDITOR_SHOW_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditorFrame::handleMenuShowDialog ) );

    // toolbar tool
    Connect( SWID_EDITOR_BALL_DRAGGABLE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditorFrame::handleBallDraggable ) );

    Connect( SWID_EDITOR_REPLACE_DATA, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditorFrame::handleReplaceTrainData ) );
    Connect( SWID_EDITOR_DELETE_DATA, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditorFrame::handleDeleteTrainData ) );

    Connect( SWID_EDITOR_TRAINING, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditorFrame::handleTrain ) );
    Connect( SWID_EDITOR_RECORD_POSITION, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditorFrame::handleRecordPosition ) );


    ///////////////////////////////////////////////
    // update UI
    Connect( SWID_EDITOR_BALL_DRAGGABLE, wxEVT_UPDATE_UI,
             wxUpdateUIEventHandler( FEditorFrame::handleUpdateBallDraggable ) );
    Connect( SWID_EDITOR_PLAYER_AUTO_MOVE, wxEVT_UPDATE_UI,
             wxUpdateUIEventHandler( FEditorFrame::handleUpdatePlayerAutoMove ) );
}

/*-------------------------------------------------------------------*/
/*!
  \brief create all windows that has this mainframe as a parent
*/
void
FEditorFrame::createWindows()
{
    // create attached tools.
    createMenuBar();
    createToolBar();
    createStatusBar();


    M_editor_canvas = new FEditorCanvas( this );
    M_editor_canvas->SetSize( this->GetClientSize() );

    M_editor_dialog = new FEditorDialog( this );
    M_editor_dialog->Show();
}

/*-------------------------------------------------------------------*/
/*!
  create & realize menu & menu bar.
  called from constructor.
*/
void
FEditorFrame::createMenuBar()
{
    // create menu
    wxMenuBar * menubar = new wxMenuBar();

    {
        wxMenu * menu = new wxMenu;
        menu->Append( SWID_EDITOR_NEW,
                      _( "New formation\tctrl-n" ),
                      _( "Create new formation data" ) );
        menu->AppendSeparator();
        menu->Append( SWID_EDITOR_OPEN,
                      _( "Open formation\tctrl-o" ),
                      _( "Open formation file(" ) );
        menu->Append( SWID_EDITOR_OPEN_TRAIN_DATA,
                      _( "Open training data" ),
                      _( "Open training data file" ) );
        menu->AppendSeparator();
        menu->Append( SWID_EDITOR_SAVE_ALL,
                      _( "Save all\tctrl-s" ),
                      _( "Save formation and training data" ) );
        /*
        menu->Append( SWID_EDITOR_SAVE,
                      _( "Save formation" ),
                      _( "Save formation" ) );
        menu->Append( SWID_EDITOR_SAVE_TRAIN_DATA,
                      _( "Save trainig data" ),
                      _( "Save training data" ) );
        */
        menu->AppendSeparator();
        menu->Append( SWID_EDITOR_SAVE_AS,
                      _( "Save formation as..." ),
                      _( "Save data to the new file" ) );
        menu->Append( SWID_EDITOR_SAVE_TRAIN_DATA_AS,
                      _( "Save trainig data as..." ),
                      _( "Save training data to the new file" ) );
        menu->AppendSeparator();
        menu->Append( SWID_QUIT,
                      _( "&Quit\tctrl-q" ),
                      _( "Quit application" ) );
        menubar->Append( menu, _( "File" ) );
    }
#if 0
    {
        wxMenu * menu = new wxMenu;
        menu->Append( SWID_EDITOR_RESET_BALL,
                      _( "Ball Reset" ),
                      _( "Ball is dropped on the field center point." ) );
        menu->Append( SWID_EDITOR_BALL_DRAGGABLE,
                      _( "Enable Ball Drag" ),
                      _( "Enable to move the ball by mouse" ) );
        menu->Append( SWID_EDITOR_PLAYER_AUTO_MOVE,
                      _( "Player Move" ),
                      _( "Toggle player automatic move mode" ) );
        menubar->Append( menu, _( "Edit" ) );

    }
    {
        wxMenu * menu = new wxMenu;
        menu->Append( SWID_EDITOR_SHOW_DIALOG,
                      _( "Dialog"  ),
                      _( "Toggle control dialog" ) );

        menubar->Append( menu, _( "View" ) );
    }
#endif
    // append menubar to FEditorFrame.
    this->SetMenuBar( menubar );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize toolbar.
  called from constructor.
*/
void
FEditorFrame::createToolBar()
{

    wxToolBar * tbar = this->CreateToolBar( wxTB_HORIZONTAL
                                            | wxTB_FLAT
                                            | wxTB_TEXT
                                            );
    ////////////////////////////////////////////////////////
    tbar->AddTool( SWID_EDITOR_SAVE_ALL,
                   _( "Save" ),
                   wxBitmap( save_xpm ),
                   _( "Save modiried formation and training data" ) );
    tbar->AddTool( SWID_EDITOR_BALL_DRAGGABLE,
                   _( "Ball Drag" ),
                   wxBitmap( hand_xpm ),
                   _( "Toggle ball draggable status" ),
                   wxITEM_CHECK );
    tbar->AddCheckTool( SWID_EDITOR_PLAYER_AUTO_MOVE,
                        _( "Auto Move" ),
                        wxBitmap( chase_xpm ),
                        wxNullBitmap,
                        _( "Toggle player automatic move mode" ) );
    tbar->AddSeparator();

    tbar->AddTool( SWID_EDITOR_REPLACE_DATA,
                   _(  "Replace" ),
                   wxBitmap( replace_xpm ),
                   _( "Replace the current indexed training data by the screen status" ) );
    tbar->AddTool( SWID_EDITOR_DELETE_DATA,
                   _(  "Delete" ),
                   wxBitmap( delete_xpm ),
                   _( "Delete the current indexed training data" ) );

    M_train_data_index = new wxSpinCtrl( tbar, SWID_EDITOR_SPIN_TRAIN_DATA,
                                         wxT( "0" ),
                                         wxDefaultPosition, wxDefaultSize,
                                         wxSP_ARROW_KEYS | wxSP_WRAP,
                                         0, 0, 0 );
    tbar->AddControl( M_train_data_index );

    tbar->AddSeparator();

    tbar->AddTool( SWID_EDITOR_TRAINING,
                   _( "Train" ),
                   wxBitmap( train_xpm ),
                   _( "Perform only train using recorded snapshot" ) );

    tbar->AddTool( SWID_EDITOR_RECORD_POSITION,
                   _( "Record" ),
                   wxBitmap( record_xpm ),
                   _( "Record current players' position as training data and Perform train" ) );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize status bar.
  called from constructor.
*/
void
FEditorFrame::createStatusBar()
{
    // already exist
    if ( this->GetStatusBar() )
    {
        return;
    }

    const int chw = this->GetCharWidth();
    const int widths[] = { -1, 20 * chw };
    this->CreateStatusBar( WXSIZEOF( widths ),
                           wxST_SIZEGRIP,
                           SWID_STATUSBAR_EDITOR );
    this->SetStatusText( wxT( "Ready" ), 0 );
    this->GetStatusBar()->SetStatusWidths( WXSIZEOF( widths ), widths );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::quit()
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    if ( M_editor_data )
    {
        M_editor_data.reset();
    }

    this->Destroy();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    //std::cerr << "FEditorFrame::handleCloseEvent()" << std::endl;
    quit();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuNew( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    // select formation training type
    wxString choices[3] = { wxString( FormationDelaunay::name().c_str(), *wxConvCurrent ),
                            wxString( FormationBPN::name().c_str(), *wxConvCurrent ) //,
                            //wxString( FormationSBSP::name().c_str(), *wxConvCurrent )
    };

    wxSingleChoiceDialog dlg( this,
                              wxT( "Choice a training method:" ),
                              wxT( "Create New Formation" ),
                              WXSIZEOF( choices ),
                              choices );
    if ( dlg.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string type_name =  (const char*)(dlg.GetStringSelection().mbc_str());
#else
    std::string type_name =  (const char*)(dlg.GetStringSelection().c_str());
#endif

    if ( M_editor_data )
    {
        M_editor_data.reset();
    }

    wxString only_filename;
    this->SetTitle( wxT( "Formation Editor - New Formation -" ) );

    // create new data

    M_editor_data = boost::shared_ptr< FEditorData >( new FEditorData( this ) );


    M_editor_data->setCanvas( M_editor_canvas );
    M_editor_data->setDialog( M_editor_dialog );

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );

    M_editor_data->setTrainingTypeName( type_name );
    M_editor_data->createDefaultParam();

    if ( ! M_editor_data->formation() )
    {
        std::cerr << "***ERROR*** Failed to initialize formation data"
                  << std::endl;
        M_editor_data.reset();
        return;
    }

    M_editor_data->moveBallTo( 0.0, 0.0 );
    M_editor_dialog->update();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuOpen( wxCommandEvent & WXUNUSED( event ) )
{
    showOpenDialog();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuOpenTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    showOpenTrainDataDialog();
}


/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuSaveAll( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->save();
        M_editor_data->saveTrainData();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuSave( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->save();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuSaveAs( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveAs();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuSaveTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveTrainData();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuSaveTrainDataAs( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveTrainDataAs();
    }
}


/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuQuit( wxCommandEvent & WXUNUSED( event ) )
{
    //std::cerr << "FEditorFrame::handleCloseEvent()" << std::endl;
    quit();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleResetBall( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->moveBallTo( 0.0, 0.0 );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handlePlayerAutoMove( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->togglePlayerAutoMove();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleUpdatePlayerAutoMove( wxUpdateUIEvent & event )
{
    if ( M_editor_data )
    {
        event.Check( M_editor_data->isPlayerAutoMove() );
    }
    else
    {
        event.Check( false );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditorFrame::handleMenuShowDialog( wxCommandEvent & WXUNUSED( event ) )
{
    M_editor_dialog->Show( ! M_editor_dialog->IsShown() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleBallDraggable( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->toggleBallDraggable();
        M_editor_canvas->draw();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleUpdateBallDraggable( wxUpdateUIEvent & event )
{
    if ( M_editor_data )
    {
        event.Check( M_editor_data->isBallDraggable() );
    }
    else
    {
        event.Check( false );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleSpinTrainData( wxSpinEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->focusTrainData( M_train_data_index->GetValue() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleReplaceTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->replaceTrainData( M_train_data_index->GetValue() );
        M_editor_data->train();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleDeleteTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->deleteTrainData( M_train_data_index->GetValue() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleRecordPosition( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->recordCurrentPosition();
        M_editor_data->train();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::handleTrain( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->train();
    }
}

/*-------------------------------------------------------------------*/
/*!
  \brief show file open dialog, and load view data from .rcg[.gz] file
*/
void
FEditorFrame::showOpenDialog()
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    wxString wild_card( wxT( "Formation file (*.conf)|*.conf"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }
#ifdef UNICODE
    std::string file_path = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string file_path = (const char*)(file_dialog.GetPath().c_str());
#endif

    M_editor_data = boost::shared_ptr< FEditorData >( new FEditorData( this ) );
    if ( ! M_editor_data->open( file_path ) )
    {
        M_editor_data.reset();
        return;
    }

    if ( ! M_editor_data->getFormation() )
    {
        M_editor_data->createDefaultParam();
    }

    M_editor_data->setCanvas( M_editor_canvas );
    M_editor_data->setDialog( M_editor_dialog );

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );
    M_editor_dialog->update();

    M_editor_data->moveBallTo( 0.0, 0.0 );

    int answer = wxMessageBox( wxT( "Do you open a training data?" ),
                               wxT( "Option" ), // title
                               wxYES_NO,
                               this );
    if ( answer != wxYES )
    {
        return;
    }

    showOpenTrainDataDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::showOpenTrainDataDialog()
{
    if ( ! M_editor_data )
    {
        return;
    }

    wxString wild_card( wxT( "Training data file (*.dat)|*.dat"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string file_path = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string file_path = (const char*)(file_dialog.GetPath().c_str());
#endif

    M_editor_data->openTrainData( file_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::setTrainDataIndexMax( const int max_idx )
{
    M_train_data_index->SetRange( 0, max_idx );
    M_train_data_index->SetValue( 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditorFrame::updateStatusText( const double & x, const double & y )
{
    if ( GetStatusBar()
         && GetStatusBar()->IsShown() )
    {
        wxString pos_str;
        pos_str.Printf( wxT( "(%.2f %.2f)" ), x, y );
        GetStatusBar()->SetStatusText( pos_str, 1 );
    }
}
