// -*-c++-*-

/*!
  \file ball_painter.cpp
  \brief ball painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <vector>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/param/server_param.h>

#include "main_data.h"
#include "monitor_view_data.h"

#include "ball_painter.h"

/*-------------------------------------------------------------------*/
/*!
  \param dc reference to the device context
  \param ball painted object
  \param view_conf view configuration (scaling parameter etc.)
  \param sparam soccer server parameter set
*/
void
BallPainter::draw( wxDC & dc ) const
{
    const MonitorViewPtr view = M_data.getCurrentViewData();
    if ( ! view )
    {
        return;
    }

    const ViewConfig & conf = M_data.config();

    const int ix = conf.getScreenX( view->ball().x() );
    const int iy = conf.getScreenY( view->ball().y() );
    // decide radius
    const int ball_radius = std::max( 1,
                                      ( conf.isEnlarged()
                                        ? conf.scale( 0.35 )
                                        : conf.scale( rcsc::ServerParam::i().ballSize() ) )
                                      );

    // set GDI objects
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( M_data.gdi().getBallBrush() );
    // draw ball body
    dc.DrawCircle( ix, iy, ball_radius );

    // draw setplay ball owner team color circle
    if ( view->playmode().isLeftSetPlay() )
    {
        dc.SetBrush( M_data.gdi().getLeftTeamBrush() );
        dc.DrawCircle( ix, iy, std::max( 1, ball_radius - 2 ) );
    }
    else if ( view->playmode().isRightSetPlay() )
    {
        dc.SetBrush( M_data.gdi().getRightTeamBrush() );
        dc.DrawCircle( ix, iy, std::max( 1, ball_radius - 2 ) );
    }

    // draw additional circle
    if ( ! conf.isEnlarged()
         || ball_radius <= 1  )
    {
        dc.SetPen( M_data.gdi().getBallPen() );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        dc.DrawCircle( conf.getScreenX( view->ball().x() ),
                       conf.getScreenY( view->ball().y() ),
                       conf.scale( rcsc::ServerParam::i().defaultKickableMargin() ) );
    }

    // draw future status
    if ( conf.getBallFutureCycle() > 0
         && view->ball().hasDelta() )
    {
        drawFutureState( dc );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainter::drawFutureState( wxDC & dc ) const
{
    // draw future state

    const MonitorViewPtr view = M_data.getCurrentViewData();
    const ViewConfig & conf = M_data.config();
    const rcsc::ServerParam & sparam = rcsc::ServerParam::i();

    rcsc::Vector2D bpos( view->ball().x(),
                         view->ball().y() );
    rcsc::Vector2D bvel( view->ball().deltaX(),
                         view->ball().deltaY() );

    std::vector< wxPoint > points;
    points.push_back( wxPoint( conf.getScreenX( bpos.x ),
                               conf.getScreenY( bpos.y ) ) );
    const int last = conf.getBallFutureCycle();
    for ( int i = 0; i < last; ++i )
    {
        bpos += bvel;
        bvel *= sparam.ballDecay();

        wxPoint pt( conf.getScreenX( bpos.x ),
                    conf.getScreenY( bpos.y ) );
        if ( std::abs( points.back().x - pt.x ) < 1
             && std::abs( points.back().y - pt.y ) < 1 )
        {
            break;
        }
        points.push_back( pt );
    }

    dc.SetPen( M_data.gdi().getBallPen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );
    dc.DrawLine( points.front(), points.back() );

    const std::vector< wxPoint >::iterator end = points.end();
    for ( std::vector< wxPoint >::iterator it = points.begin() + 1;
          it != end;
              ++it )
    {
        dc.DrawCircle( *it, 1 );
    }
}
