// -*-c++-*-

/*!
  \file app_config.cpp
  \brief Application configuration class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/config.h>
#include <wx/fileconf.h>
#include <wx/cmdline.h>
#endif

#include <iostream>
#include <boost/program_options.hpp>

#include "app_config.h"

const double AppConfig::PITCH_LENGTH = 105.0;
const double AppConfig::PITCH_WIDTH = 68.0;
const double AppConfig::PITCH_MARGIN = 5.0;
const double AppConfig::CENTER_CIRCLE_R = 9.15;
const double AppConfig::PENALTY_AREA_LENGTH = 16.5;
const double AppConfig::PENALTY_AREA_WIDTH = 40.32;
const double AppConfig::PENALTY_CIRCLE_R = 9.15; // 7.1
const double AppConfig::PENALTY_SPOT_DIST = 11.0;
const double AppConfig::GOAL_AREA_LENGTH = 5.5;
const double AppConfig::GOAL_AREA_WIDTH = 18.2;
const double AppConfig::GOAL_DEPTH = 2.5;
const double AppConfig::GOAL_WIDTH = 14.02;
const double AppConfig::GOAL_POST_R = 0.06;

const int AppConfig::MIN_SCORE_BOARD_HEIGHT = 20;
const int AppConfig::MAX_SCORE_BOARD_HEIGHT = 40;

/*-------------------------------------------------------------------*/
/*!
  singleton interface
*/
AppConfig &
AppConfig::instance()
{
    static AppConfig s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

*/
AppConfig::AppConfig()
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

*/
AppConfig::~AppConfig()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
AppConfig::init()
{

    M_game_log_file_path = "";
    M_game_log_dir = "";
    M_debug_log_dir = "";

    M_time_shift_replay = true;
    M_auto_loop_mode = false;

    M_connect = false;
    M_host = "localhost";
    M_port = 6000;
    M_client_version = 2;
    M_wait_seconds = 5;
    M_auto_quit_mode = false;
    M_kill_server = false;
    M_server_pid = 0;
    M_server_command = "rcssserver";

    M_maximize = false;
    M_full_screen = false;

    M_pos_x = -1;
    M_pos_y = -1;
    M_width = -1;
    M_height = -1;

    //M_hide_menu_bar;
    M_hide_tool_bar = false;
    M_hide_status_bar = false;

    M_show_player_number = true;
    M_show_player_type = false;
    M_show_view_cone = true;
    M_show_stamina = true;

    M_field_grass_type = "mono";

    M_debug_server_mode = false;
    M_debug_server_port = 6000 + 32;

    M_editor_mode = false;
    M_editor_logging = false;
}

/*-------------------------------------------------------------------*/
/*!
  analyze command line options
*/
bool
AppConfig::parseCmdLine( int argc,
                         char ** argv )
{
    namespace po = boost::program_options;

    po::options_description visibles( "Allowed options:" );

    visibles.add_options()
        ( "help,h",
          "generates this message." )
        ( "version,v",
          "print version." )
        ( "host",
          po::value< std::string >( &M_host )->default_value( "localhost", "localhost" ),
          "set host name to be connected." )
        ( "port",
          po::value< int >( &M_port )->default_value( 6000, "6000" ),
          "set port number for the monitor client." )
        ( "client-version",
          po::value< int >( &M_client_version )->default_value( 2, "2" ),
          "set a monitor client protocol version." )
        ( "wait-seconds",
          po::value< long >( &M_wait_seconds )->default_value( 5, "5" ),
          "set maximal seconds to wait a message from rcssserver." )
        ( "auto-quit-mode",
          po::bool_switch( &M_auto_quit_mode )->default_value( false ),
          "enable automatic quit after game end." )
        ( "kill-server",
          po::bool_switch( &M_kill_server )->default_value( false ),
          "kill rcssserver when soccerwindow2 is quit." )
        ( "server-pid",
          po::value< long >( &M_server_pid )->default_value( 0 ),
          "set rcssserver process ID to kill it." )
        ( "server-path",
          po::value< std::string >( &M_server_command )->default_value( "rcssserver", "rcssserver" ),
          "set a rcssserver command line." )
        ( "connect,c",
          po::bool_switch( &M_connect )->default_value( false ),
          "start as soccer monitor." )
        ( "debug-server-mode,d",
          po::bool_switch( &M_debug_server_mode )->default_value( false ),
          "start as debug server." )
        ( "debug-server-port",
          po::value< long >( &M_debug_server_port )->default_value( 6000 + 32, "6032" ),
          "set port number for the debug server." )
        ( "game-log-dir",
          po::value< std::string >( &M_game_log_dir ), //->default_value( "", "$HOME" ),
          "set a default path where game log files exist." )
        ( "debug-log-dir",
          po::value< std::string >( &M_debug_log_dir ), //->default_value( "", "$HOME" ),
          "set a default path where debug log files exist." )
        ( "time-shift-replay",
          po::value< bool >( &M_time_shift_replay )->default_value( true, "on" ),
          "enable time shift replay mode." )
        ( "auto-loop-mode",
          po::bool_switch( &M_auto_loop_mode )->default_value( false ),
          "enable automatic replay loop." )
        ( "field-grass-type",
          po::value< std::string >( &M_field_grass_type )->default_value( "mono", "mono" ),
          "set a field grass type {mono,lines,checker}." )
        ( "show-player-number",
          po::value< bool >( &M_show_player_number )->default_value( true, "on" ),
          "show player\'s uniform number." )
        ( "show-player-type",
          po::value< bool >( &M_show_player_type )->default_value( false, "off" ),
          "show player\'s heterogeneous type ID." )
        ( "show-view-cone",
          po::value< bool >( &M_show_view_cone )->default_value( true, "on" ),
          "show player\'s view cone." )
        ( "show-stamina",
          po::value< bool >( &M_show_stamina )->default_value( true, "on" ),
          "show player\'s stamina." )
        ( "maximize",
          po::bool_switch( &M_maximize )->default_value( false ),
          "start with a maximized frame." )
        ( "full-screen",
          po::bool_switch( &M_full_screen )->default_value( false ),
          "start with a full screen frame." )
        ( "hide-tool-bar",
          po::bool_switch( &M_hide_tool_bar )->default_value( false ),
          "start without a tool bar." )
        ( "hide-status-bar",
          po::bool_switch( &M_hide_status_bar )->default_value( false ),
          "start without a status bar." )
        ( "pos-x",
          po::value< long >( &M_pos_x )->default_value( -1, "" ),
          "set left x position of the main frame." )
        ( "pos-y",
          po::value< long >( &M_pos_y )->default_value( -1, "" ),
          "set top y position of the main frame." )
        ( "width",
          po::value< long >( &M_width )->default_value( -1, "" ),
          "set width of the main frame." )
        ( "height",
          po::value< long >( &M_height )->default_value( -1, "" ),
          "set height of the main frame." )
        ( "editor-mode",
          po::bool_switch( &M_editor_mode )->default_value( false ),
          "start as formation editor mode." )
        ( "editor-log",
          po::bool_switch( &M_editor_logging )->default_value( false, "off" ),
          "logging formation editor operation" )
        ;

    po::options_description invisibles( "Invisibles" );
    invisibles.add_options()
        ( "rcg-file",
          po::value< std::string >( &M_game_log_file_path )->default_value( "" ),
          "set the path to Game Log file(.rcg) to be loaded."  )
        ;

    po::options_description all_desc( "All options:" );
    all_desc.add( visibles ).add( invisibles );

    po::positional_options_description pdesc;
    pdesc.add( "rcg-file", 1 ); // allowd only one rcg file

    bool help = false;
    try
    {
        po::variables_map vm;
        po::command_line_parser parser( argc, argv );
        parser.options( all_desc ).positional( pdesc );
        po::store( parser.run(), vm );
        po::notify( vm );

        if ( vm.count( "help" ) )
        {
            help = true;
        }
        else if ( vm.count( "version" ) )
        {
            std::cout << PACKAGE_NAME << " Version " << VERSION
                      << std::endl;
            return false;
        }
    }
    catch ( std::exception & e )
    {
        std::cerr << e.what() << std::endl;
        help = true;
    }

    if ( help )
    {
        std::cout << "Usage: " << PACKAGE_NAME
                  << " [options ... ] [<GameLogFile>]\n";
        std::cout << visibles << std::endl;
        return false;
    }

    return true;
}
