// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// For compilers that support precompilation, includes "wx/wx.h".
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/spinctrl.h>
#include <wx/radiobut.h>
#endif

#include <cassert>
#include <iostream>

#include "id.h"
#include "main_data.h"
#include "main_frame.h"
#include "view_config.h"
#include "view_config_control.h"

#include "view_config_dialog.h"

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::ViewConfigDialog( wxWindow * parent,
                                    MainFrame * main_frame,
                                    boost::shared_ptr< ViewConfigControl > control,
                                    ViewConfig & config )
    : wxDialog( parent,
                SWID_DIALOG_FIELD_CANVAS_CONFIG, wxT( "View Preference"),
                wxDefaultPosition,
                wxDefaultSize,
                wxCAPTION | wxSYSTEM_MENU | wxCLOSE_BOX )
    , M_main_frame( main_frame )
    , M_control( control )
    , M_config( config )
    , M_ball_trace_start( new wxTextCtrl( this, SWID_TEXTBOX_BALL_TRACE_START,
                                          wxT( "0" ),
                                          wxDefaultPosition, wxSize( 40, -1 ) ) )
    , M_ball_trace_end( new wxTextCtrl( this, SWID_TEXTBOX_BALL_TRACE_END,
                                        wxT( "0" ),
                                        wxDefaultPosition, wxSize( 40, -1 ) ) )
    , M_player_trace_start( new wxTextCtrl( this, SWID_TEXTBOX_PLAYER_TRACE_START,
                                            wxT( "0" ),
                                            wxDefaultPosition, wxSize( 40, -1 ) ) )
    , M_player_trace_end( new wxTextCtrl( this, SWID_TEXTBOX_PLAYER_TRACE_END,
                                          wxT( "0" ),
                                          wxDefaultPosition, wxSize( 40, -1 ) ) )
    , M_ball_future( new wxSpinCtrl( this, SWID_TEXTBOX_BALL_FUTURE_CYCLE ) )
    , M_player_future( new wxSpinCtrl( this, SWID_TEXTBOX_PLAYER_FUTURE_CYCLE ) )
    , M_canvas_width_text( static_cast< wxTextCtrl * >( 0 ) )
    , M_canvas_height_text( static_cast< wxTextCtrl * >( 0 ) )
{
    assert( main_frame );
    assert( control.get() );

    M_ball_future->SetValue( 0 );
    M_ball_future->SetRange( 0, 50 );
    M_player_future->SetValue( 0 );
    M_player_future->SetRange( 0, 10 );

    createControls();
    connectEvents();

    wxRect cli_rect = ::wxGetClientDisplayRect();
#if 0
    std::cerr << "client display.  "
              << " x = " << cli_rect.GetX()
              << " y = " << cli_rect.GetY()
              << " width = " << cli_rect.GetWidth()
              << " height = " << cli_rect.GetHeight()
              << std::endl;
    std::cerr << "my size = "
              << this->GetSize().x << " "
              << this->GetSize().y  << "  virtual = "
              << this->GetVirtualSize().x  << " "
              << this->GetVirtualSize().y  << std::endl;
#endif
    this->SetSize( 0, cli_rect.GetHeight() - this->GetSize().y - 32,
                   -1, -1 );

    setScale();
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::createControls()
{
    wxBoxSizer * main_sizer = new wxBoxSizer( wxVERTICAL );

    // zoom options
    main_sizer->Add( createZoomControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // canvas size
    main_sizer->Add( createCanvasSizeControls( this ),
                     0, wxLEFT | wxRIGHT, 4 );
    // detail info options
    main_sizer->Add( createPlayersDetailControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // show / hide info
    main_sizer->Add( createShowControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // show / hide computational geometry
    main_sizer->Add( createCompGeomControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // focus selector
    main_sizer->Add( createFocusControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // player selector
    main_sizer->Add( createPlayerSelectionControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // grass type
    main_sizer->Add( createGrassTypeControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // trace cycle span
    main_sizer->Add( createTraceControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // future cycle info
    main_sizer->Add( createInertiaMoveControls(),
                     0, wxLEFT | wxRIGHT, 4 );

    main_sizer->Add( 0, 2 ); // bottom space

    main_sizer->SetSizeHints( this );
    this->SetAutoLayout( true );
    this->SetSizer( main_sizer );
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createZoomControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Zoom" ) );
    wxStaticBoxSizer * top_sizer = new wxStaticBoxSizer( sbox, wxVERTICAL );

    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxButton( this, SWID_CANVAS_ZOOMIN,
                                  wxT( "+" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, 0, 0 );
        sizer->Add( new wxButton( this, SWID_CANVAS_ZOOMOUT,
                                  wxT( "-" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, 0, 0 );
        M_enlarge_cb = new wxCheckBox( this,
                                       SWID_CANVAS_ENLARGE,
                                       wxT( "Enlarge Objects" ) );
        M_enlarge_cb->SetValue( M_config.isEnlarged() );
        sizer->Add( M_enlarge_cb, 0, wxLEFT | wxRIGHT, 4 );
        sizer->Add( new wxButton( this, SWID_CANVAS_UNZOOM,
                                  wxT( "Fit" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, wxLEFT, 10 );

        top_sizer->Add( sizer, 0, 0, 0 );
    }

    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxStaticText( this, -1, wxT( "Scale" ) ),
                    0, wxLEFT | wxRIGHT, 1 );

        M_scale_slider = new wxSlider( this,
                                       SWID_SLIDER_FIELD_SCALE,
                                       80, 10, 4000,  // value, min value, max value
                                       wxDefaultPosition, wxSize( 300, -1 ),
                                       wxSL_HORIZONTAL
                                       | wxSL_AUTOTICKS );
        M_scale_slider->SetLineSize( 1 );
        M_scale_slider->SetPageSize( 1 ); // set the minimal increment/decrement step
        sizer->Add( M_scale_slider, 0, 0, 0 );

        M_scale_text = new wxStaticText( this, -1, wxT( "000.0" ),
                                         wxDefaultPosition,
                                         wxDefaultSize,
                                         wxALIGN_LEFT | wxST_NO_AUTORESIZE );

        sizer->Add( M_scale_text, 0, wxLEFT | wxRIGHT, 1 );

        top_sizer->Add( sizer, 0, 0, 0 );
    }

    return top_sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createCanvasSizeControls( wxWindow * parent )
{
    wxStaticBox * sbox = new wxStaticBox( parent, -1,
                                          wxT( "Canvas Size" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    sizer->Add( new wxStaticText( parent, -1, wxT( " Width:" ) ),
                0, wxALIGN_CENTER_VERTICAL, 0 );

    wxString size_str;

    size_str.sprintf( wxT( "%d" ), M_main_frame->GetClientSize().GetWidth() );
    M_canvas_width_text = new wxTextCtrl( parent, -1, size_str );
    sizer->Add( M_canvas_width_text,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    sizer->Add( new wxStaticText( parent, -1, wxT( " Height:" ) ),
                0, wxALIGN_CENTER_VERTICAL, 0 );

    size_str.sprintf( wxT( "%d" ), M_main_frame->GetClientSize().GetHeight() );
    M_canvas_height_text = new wxTextCtrl( parent, -1, size_str );
    sizer->Add( M_canvas_height_text,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    //M_set_canvas_size_btn = new wxButton( parent, SWID_CANVAS_LOAD_SIZE_BUTTON,
    //wxT( "load" ) );
    //sizer->Add( M_set_canvas_size_btn,
    //0, wxALIGN_CENTER_VERTICAL | wxLEFT, 6 );

    M_set_canvas_size_btn = new wxButton( parent, SWID_CANVAS_SET_SIZE_BUTTON,
                                          wxT( "apply" ) );
    sizer->Add( M_set_canvas_size_btn,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createPlayersDetailControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Players\' Detail" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_player_number_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_PLAYER_NUMBER,
                                             wxT( "Unum") );
        M_player_number_cb->SetValue( M_config.isShownPlayerNumber() );
        sizer->Add( M_player_number_cb );
    }
    {
        M_hetero_number_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_HETERO_NUMBER,
                                             wxT( "Type" ) );
        M_hetero_number_cb->SetValue( M_config.isShownHeteroNumber() );
        sizer->Add( M_hetero_number_cb );
    }
    {
        M_stamina_cb = new wxCheckBox( this,
                                       SWID_CANVAS_DRAW_STAMINA,
                                       wxT( "Stamina" ) );
        M_stamina_cb->SetValue( M_config.isShownStamina() );
        sizer->Add( M_stamina_cb );
    }
    {
        M_view_cone_cb = new wxCheckBox( this,
                                         SWID_CANVAS_DRAW_VIEW_CONE,
                                         wxT( "View Cone" ) );
        M_view_cone_cb->SetValue( M_config.isShownViewCone() );
        sizer->Add( M_view_cone_cb );
    }
    {
        M_control_area_cb = new wxCheckBox( this,
                                            SWID_CANVAS_DRAW_CONTROL_AREA,
                                            wxT( "Control Area" ) );
        M_control_area_cb->SetValue( M_config.isShownControlArea() );
        sizer->Add( M_control_area_cb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createShowControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Show" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_show_score_board_cb = new wxCheckBox( this,
                                                SWID_CANVAS_DRAW_SCORE_BOARD,
                                                wxT( "Score Board" ) );
        M_show_score_board_cb->SetValue( M_config.isShownScoreBoard() );
        sizer->Add( M_show_score_board_cb );
    }
    {
        M_show_ball_cb = new wxCheckBox( this,
                                         SWID_CANVAS_DRAW_BALL,
                                         wxT( "Ball" ) );
        M_show_ball_cb->SetValue( M_config.isShownBall() );
        sizer->Add( M_show_ball_cb );
    }
    {
        M_show_player_cb = new wxCheckBox( this,
                                           SWID_CANVAS_DRAW_PLAYER,
                                           wxT( "Player" ) );
        M_show_player_cb->SetValue( M_config.isShownPlayers() );
        sizer->Add( M_show_player_cb );
    }
    {
        M_show_flags_cb = new wxCheckBox( this,
                                          SWID_CANVAS_DRAW_FLAGS,
                                          wxT( "Flag" ) );
        M_show_flags_cb->SetValue( M_config.isShownFlags() );
        sizer->Add( M_show_flags_cb );
    }
    {
        M_show_offside_line_cb = new wxCheckBox( this,
                                                 SWID_CANVAS_DRAW_OFFSIDE_LINE,
                                                 wxT( "Offside Line" ) );
        M_show_offside_line_cb->SetValue( M_config.isShownOffsideLine() );
        sizer->Add( M_show_offside_line_cb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createCompGeomControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Compututational Geometry" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_show_voronoi_cb = new wxCheckBox( this,
                                            SWID_CANVAS_DRAW_VORONOI_DIAGRAM,
                                            wxT( "VoronoiDiagram" ) );
        M_show_voronoi_cb->SetValue( M_config.isShownVoronoiDiagram() );
        sizer->Add( M_show_voronoi_cb );
    }
    {
        M_show_delaunay_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_DELAUNAY_TRIANGLE,
                                             wxT( "DelaunayTrianglation" ) );
        M_show_delaunay_cb->SetValue( M_config.isShownDelaunayTriangle() );
        sizer->Add( M_show_delaunay_cb );
    }
    {
        wxString choices[] = { wxT( "All" ), wxT( "Left Team" ), wxT( "Right Team" ) };
        M_voronoi_choice = new wxChoice( this,
                                         SWID_CHOICE_VORONOI_TARGET,
                                         wxDefaultPosition,
                                         wxDefaultSize,
                                         WXSIZEOF( choices ),
                                         choices );
        sizer->Add( M_voronoi_choice, 0, wxLEFT, 5 );
        M_voronoi_choice->SetSelection( 0 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createFocusControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Focus" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_focus_ball_rb = new wxRadioButton( this,
                                             SWID_CANVAS_FOCUS_BALL,
                                             wxT( "Ball" ),
                                             wxDefaultPosition,
                                             wxDefaultSize,
                                             wxRB_GROUP );
        M_focus_ball_rb->SetValue( false );
        sizer->Add( M_focus_ball_rb );
    }
    {
        M_focus_player_rb = new wxRadioButton( this,
                                               SWID_CANVAS_FOCUS_PLAYER,
                                               wxT( "Selected Player" ) );
        M_focus_player_rb->SetValue( false );
        sizer->Add( M_focus_player_rb );
    }
    {
        M_focus_fix_rb = new wxRadioButton( this,
                                            SWID_CANVAS_FOCUS_FIX,
                                            wxT( "Fix" ) );
        M_focus_fix_rb->SetValue( true );
        sizer->Add( M_focus_fix_rb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createPlayerSelectionControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Player Selection" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxVERTICAL );
    // selection type
    {
        wxBoxSizer * bsizer = new wxBoxSizer( wxHORIZONTAL );
        {
            M_select_all_rb = new wxRadioButton( this,
                                                 SWID_CANVAS_SELECT_AUTO_ALL,
                                                 wxT( "Auto" ),
                                                 wxDefaultPosition,
                                                 wxDefaultSize,
                                                 wxRB_GROUP );
            M_select_all_rb->SetValue( false );
            bsizer->Add( M_select_all_rb );
        }
        {
            M_select_left_rb = new wxRadioButton( this,
                                                  SWID_CANVAS_SELECT_AUTO_LEFT,
                                                  wxT( "Auto from left" ) );
            M_select_left_rb->SetValue( false );
            bsizer->Add( M_select_left_rb );
        }
        {
            M_select_right_rb = new wxRadioButton( this,
                                                   SWID_CANVAS_SELECT_AUTO_RIGHT,
                                                   wxT( "Auto from right" ) );
            M_select_right_rb->SetValue( false );
            bsizer->Add( M_select_right_rb );
        }
        {
            M_select_fix_rb = new wxRadioButton( this,
                                                 SWID_CANVAS_SELECT_FIX,
                                                 wxT( "Fix" ) );
            M_select_fix_rb->SetValue( false );
            bsizer->Add( M_select_fix_rb );
        }
        {
            M_unselect_rb = new wxRadioButton( this,
                                               SWID_CANVAS_UNSELECT_PLAYER,
                                               wxT( "Unselect" ) );
            M_unselect_rb->SetValue( true );
            bsizer->Add( M_unselect_rb );
        }
        sizer->Add( bsizer, 0, wxLEFT | wxRIGHT, 4 );
    }
    // selection id
    int id_width, id_height;
    this->GetTextExtent( wxT( "L10" ), &id_width, &id_height );
    {
        wxBoxSizer * bsizer = new wxBoxSizer( wxHORIZONTAL );
        for ( int i = 0; i < 11; ++i )
        {
            wxString label;
            label.Printf( wxT( "L%d" ), i + 1 );
            M_left_player_btn[i] = new wxButton( this,
                                                 SWID_CANVAS_SELECT_PLAYER_L1 + i,
                                                 label,
                                                 wxDefaultPosition,
                                                 wxSize( id_width + 12, -1 ) );
            bsizer->Add( M_left_player_btn[i], 1, 0, 0 );
        }

        sizer->Add( bsizer, 0, wxLEFT | wxRIGHT | wxTOP, 2 );
    }
    {
        wxBoxSizer * bsizer = new wxBoxSizer( wxHORIZONTAL );
        for ( int i = 0; i < 11; ++i )
        {
            wxString label;
            label.Printf( wxT( "R%d" ), i + 1 );
            M_right_player_btn[i] = new wxButton( this,
                                                  SWID_CANVAS_SELECT_PLAYER_R1 + i,
                                                  label,
                                                  wxDefaultPosition,
                                                  wxSize( id_width + 12, -1 ) );
            bsizer->Add( M_right_player_btn[i], 1, 0, 0 );
        }

        sizer->Add( bsizer, 0, wxLEFT | wxRIGHT | wxTOP, 2 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createGrassTypeControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Grass Type" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_grass_normal_rb = new wxRadioButton( this,
                                               SWID_CANVAS_GRASS_NORMAL,
                                               wxT( "mono" ),
                                               wxDefaultPosition,
                                               wxDefaultSize,
                                               wxRB_GROUP );
        M_grass_normal_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_NORMAL );
        sizer->Add( M_grass_normal_rb );
    }
    {
        M_grass_line_rb = new wxRadioButton( this,
                                             SWID_CANVAS_GRASS_LINES,
                                             wxT( "lines" ) );
        M_grass_line_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_LINES );
        sizer->Add( M_grass_line_rb );
    }
    {
        M_grass_checker_rb = new wxRadioButton( this,
                                                SWID_CANVAS_GRASS_CHECKER,
                                                wxT( "checker" ) );
        M_grass_checker_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_CHECKER );
        sizer->Add( M_grass_checker_rb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createTraceControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Trace" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );
    {
        wxStaticBox * bsbox = new wxStaticBox( this, -1,
                                               wxT( "Ball" ) );
        wxStaticBoxSizer * bsizer = new wxStaticBoxSizer( bsbox, wxHORIZONTAL );
        bsizer->Add( M_ball_trace_start,
                     0, 0, 0 );
        bsizer->Add( new wxStaticText( this, -1, wxT( "-" ) ),
                     0, wxLEFT | wxRIGHT, 1 );
        bsizer->Add( M_ball_trace_end,
                     0, 0, 0 );
        M_ball_trace_cb = new wxCheckBox( this,
                                          SWID_CANVAS_TOGGLE_BALL_AUTO_TRACE,
                                          wxT( "Auto" ) );
        M_ball_trace_cb->SetValue( false );
        bsizer->Add( M_ball_trace_cb, 1, 0, 0 );

        sizer->Add( bsizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }
    {
        wxStaticBox * psbox = new wxStaticBox( this, -1,
                                               wxT( "Player" ) );
        wxStaticBoxSizer * psizer = new wxStaticBoxSizer( psbox, wxHORIZONTAL );
        psizer->Add( M_player_trace_start,
                     0, 0, 0 );
        psizer->Add( new wxStaticText( this, -1, wxT( "-" ) ),
                     0, wxLEFT | wxRIGHT, 1 );
        psizer->Add( M_player_trace_end,
                     0, 0, 0 );
        M_player_trace_cb =  new wxCheckBox( this,
                                             SWID_CANVAS_TOGGLE_PLAYER_AUTO_TRACE,
                                             wxT( "Auto" ) );
        M_player_trace_cb->SetValue( false );
        psizer->Add( M_player_trace_cb,
                     0, wxALIGN_CENTER_VERTICAL | wxLEFT | wxRIGHT, 2 );

        sizer->Add( psizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }
    sizer->Add( 6, 0 );
    {
        sizer->Add( new wxButton( this, SWID_CANVAS_TOGGLE_TRACE_MODE,
                                  wxT( "Line/Point" ) ),
                    0, wxALIGN_CENTER_VERTICAL | wxRIGHT, 2 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createInertiaMoveControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Inertia Move" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );
    {
        wxStaticBox * bsbox = new wxStaticBox( this, -1,
                                               wxT( "Ball" ) );
        wxStaticBoxSizer * bsizer = new wxStaticBoxSizer( bsbox, wxHORIZONTAL );
        bsizer->Add( M_ball_future,
                     0, 0, 0 );
        sizer->Add( bsizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }
    {
        wxStaticBox * psbox = new wxStaticBox( this, -1,
                                               wxT( "Player" ) );
        wxStaticBoxSizer * psizer = new wxStaticBoxSizer( psbox, wxHORIZONTAL );
        psizer->Add( M_player_future,
                     0, 0, 0 );
        sizer->Add( psizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::connectEvents()
{
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( ViewConfigDialog::handleCloseEvent ) );

    // command event : button
    //Connect( SWID_CANVAS_LOAD_SIZE_BUTTON, wxEVT_COMMAND_BUTTON_CLICKED,
    //wxCommandEventHandler( ViewConfigDialog::handleLoadSizeButton ) );
    // command event : button
    Connect( SWID_CANVAS_SET_SIZE_BUTTON, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::handleSetSizeButton ) );

    // command event : button
    Connect( wxID_ANY, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::handleCommandEvent ) );

    // command event : check box
    Connect( wxID_ANY, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::handleCommandEvent ) );

    // command event : radio button
    Connect( wxID_ANY, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::handleCommandEvent ) );

    // command event : choice
    Connect( wxID_ANY, wxEVT_COMMAND_CHOICE_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::handleCommandEvent ) );

    // slider
    Connect( SWID_SLIDER_FIELD_SCALE, wxEVT_SCROLL_THUMBTRACK,
             wxScrollEventHandler( ViewConfigDialog::handleSliderScroll ) );

    // text ctrl input
    Connect( SWID_TEXTBOX_BALL_TRACE_START, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::handleBallTraceStart ) );
    Connect( SWID_TEXTBOX_BALL_TRACE_END, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::handleBallTraceEnd ) );

    Connect( SWID_TEXTBOX_PLAYER_TRACE_START, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::handlePlayerTraceStart ) );
    Connect( SWID_TEXTBOX_PLAYER_TRACE_END, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::handlePlayerTraceEnd ) );

    Connect( SWID_TEXTBOX_BALL_FUTURE_CYCLE, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( ViewConfigDialog::handleBallFutureCycle ) );
    Connect( SWID_TEXTBOX_PLAYER_FUTURE_CYCLE, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( ViewConfigDialog::handlePlayerFutureCycle ) );
    //Connect( SWID_TEXTBOX_BALL_FUTURE_CYCLE, wxEVT_COMMAND_TEXT_UPDATED,
    //         wxTextEventHandler( ViewConfigDialog::handleBallFutureCycle ) );
    //Connect( SWID_TEXTBOX_PLAYER_FUTURE_CYCLE, wxEVT_COMMAND_TEXT_UPDATED,
    //         wxTextEventHandler( ViewConfigDialog::handlePlayerFutureCycle ) );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    this->Hide();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleCommandEvent( wxCommandEvent & event )
{
    //std::cerr << "ViewConfigDialog::handleCommandEvent" << std::endl;
    const int id = event.GetId();

    if ( SWID_CANVAS_SELECT_PLAYER_L1 <= id
         && id <= SWID_CANVAS_SELECT_PLAYER_L11 )
    {
        int unum = id - SWID_CANVAS_SELECT_PLAYER_L1 + 1;
        M_main_frame->handle( EventMessage( SWID_CANVAS_SELECT_PLAYER,
                                            unum ) );
        return;
    }
    if ( SWID_CANVAS_SELECT_PLAYER_R1 <= id
         && id <= SWID_CANVAS_SELECT_PLAYER_R11 )
    {
        int unum = id - SWID_CANVAS_SELECT_PLAYER_R1 + 1;
        M_main_frame->handle( EventMessage( SWID_CANVAS_SELECT_PLAYER,
                                            -unum ) );
        return;
    }
    if ( id == SWID_CHOICE_VORONOI_TARGET )
    {
        rcsc::SideID side = ( M_voronoi_choice->GetSelection() == 0
                              ? rcsc::NEUTRAL
                              : ( M_voronoi_choice->GetSelection() == 1
                                  ? rcsc::LEFT
                                  : rcsc::RIGHT ) );
        M_main_frame->handle( EventMessage( SWID_CHOICE_VORONOI_TARGET,
                                            static_cast< int >( side ) ) );
        return;
    }

    if ( id == SWID_CANVAS_SET_SIZE_BUTTON )
    {
        handleSetSizeButton( event );
        return;
    }

    M_main_frame->handle( EventMessage( id ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleSliderScroll( wxScrollEvent & event )
{
    double scale = M_scale_slider->GetValue() * 0.1;

    M_control->setScale( scale );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleSetSizeButton( wxCommandEvent & WXUNUSED( event ) )
{
    //std::cerr << "handle set size" << std::endl;

    long width = 0;
    long height = 0;

    if ( ! M_canvas_width_text->GetValue().ToLong( &width ) )
    {
        return;
    }
    if ( ! M_canvas_height_text->GetValue().ToLong( &height ) )
    {
        return;
    }

    if ( width < M_main_frame->GetMinSize().GetWidth() )
    {
        return;
    }
    if ( height < M_main_frame->GetMinSize().GetHeight() )
    {
        return;
    }

    M_main_frame->SetClientSize( static_cast< int >( width ),
                                 static_cast< int >( height ) );
    //setScale();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleBallTraceStart( wxCommandEvent & WXUNUSED(event) )
{
    wxString val = M_ball_trace_start->GetValue();
    unsigned long ucycle = 0;
    if ( ! val.ToULong( &ucycle ) )
    {
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_BALL_TRACE_START,
                                        static_cast< long >( ucycle ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleBallTraceEnd( wxCommandEvent & WXUNUSED(event) )
{
    wxString val = M_ball_trace_end->GetValue();
    unsigned long ucycle = 0;
    if ( ! val.ToULong( &ucycle ) )
    {
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_BALL_TRACE_END,
                                        static_cast< long >( ucycle ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handlePlayerTraceStart( wxCommandEvent & WXUNUSED(event) )
{
    wxString val = M_player_trace_start->GetValue();
    unsigned long ucycle = 0;
    if ( ! val.ToULong( &ucycle ) )
    {
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_PLAYER_TRACE_START,
                                        static_cast< long >( ucycle ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handlePlayerTraceEnd( wxCommandEvent & WXUNUSED(event) )
{
    wxString val = M_player_trace_end->GetValue();
    unsigned long ucycle = 0;
    if ( ! val.ToULong( &ucycle ) )
    {
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_PLAYER_TRACE_END,
                                        static_cast< long >( ucycle ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleBallFutureCycle( wxSpinEvent & WXUNUSED( event ) )
{
    //int cycle = event.GetInt();//M_ball_future->GetValue();
    int cycle = M_ball_future->GetValue();
    if ( M_ball_future->GetMin() <= cycle
         && cycle <= M_ball_future->GetMax() )
    {
        M_main_frame->handle( EventMessage( SWID_SET_BALL_FUTURE_CYCLE,
                                            cycle ) );
    }
    else
    {
        M_ball_future->SetValue( 0 );
    }

    //event.Skip( false );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handlePlayerFutureCycle( wxSpinEvent & WXUNUSED( event ) )
{
    int cycle = M_player_future->GetValue();
    if ( M_player_future->GetMin() <= cycle
         && cycle <= M_player_future->GetMax() )
    {
        M_main_frame->handle( EventMessage( SWID_SET_PLAYER_FUTURE_CYCLE,
                                            cycle ) );
    }
    else
    {
        M_player_future->SetValue( 0 );
    }

    //event.Skip( false );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setGrassType()
{
    M_grass_normal_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_NORMAL );
    M_grass_line_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_LINES );
    M_grass_checker_rb->SetValue( M_config.getGrassType() == ViewConfig::GRASS_CHECKER );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setScale()
{
    M_scale_slider->SetValue( static_cast< int >( M_config.getFieldScale() * 10.0 ) );

    wxString temp;

    temp.Printf( wxT( "%5.1f" ), M_config.getFieldScale() );
    M_scale_text->SetLabel( temp );

    temp.sprintf( wxT( "%d" ), M_main_frame->GetClientSize().GetWidth() );
    M_canvas_width_text->SetValue( temp );
    temp.sprintf( wxT( "%d" ), M_main_frame->GetClientSize().GetHeight() );
    M_canvas_height_text->SetValue( temp );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setEnlarge()
{
    M_enlarge_cb->SetValue( M_config.isEnlarged() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowScoreBoard()
{
    M_show_score_board_cb->SetValue( M_config.isShownScoreBoard() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowBall()
{
    M_show_ball_cb->SetValue( M_config.isShownBall() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowPlayers()
{
    M_show_player_cb->SetValue( M_config.isShownPlayers() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowPlayerNumber()
{
    M_player_number_cb->SetValue( M_config.isShownPlayerNumber() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowStamina()
{
    M_stamina_cb->SetValue( M_config.isShownStamina() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowHeteroNumber()
{
    M_hetero_number_cb->SetValue( M_config.isShownHeteroNumber() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowViewCone()
{
    M_view_cone_cb->SetValue( M_config.isShownViewCone() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowControlArea()
{
    M_control_area_cb->SetValue( M_config.isShownControlArea() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowFlags()
{
    M_show_flags_cb->SetValue( M_config.isShownFlags() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowOffsideLine()
{
    M_show_offside_line_cb->SetValue( M_config.isShownOffsideLine() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowVoronoiDiagram()
{
    M_show_voronoi_cb->SetValue( M_config.isShownVoronoiDiagram() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setShowDelaunayTriangle()
{
    M_show_delaunay_cb->SetValue( M_config.isShownDelaunayTriangle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setVoronoiTarget()
{
    M_voronoi_choice->SetSelection( static_cast< int >( M_config.getVoronoiTarget() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusType()
{
    M_focus_ball_rb->SetValue( M_config.getFocusType() == ViewConfig::FOCUS_BALL );
    M_focus_player_rb->SetValue( M_config.getFocusType() == ViewConfig::FOCUS_PLAYER );
    M_focus_fix_rb->SetValue( M_config.getFocusType() == ViewConfig::FOCUS_POINT );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setPlayerSelectType()
{
    M_select_all_rb->SetValue( M_config.getPlayerSelectType() == ViewConfig::SELECT_AUTO_ALL );
    M_select_left_rb->SetValue( M_config.getPlayerSelectType() == ViewConfig::SELECT_AUTO_LEFT );
    M_select_right_rb->SetValue( M_config.getPlayerSelectType() == ViewConfig::SELECT_AUTO_RIGHT );
    M_select_fix_rb->SetValue( M_config.getPlayerSelectType() == ViewConfig::SELECT_FIX );
    M_unselect_rb->SetValue( M_config.getPlayerSelectType() == ViewConfig::SELECT_UNSELECT );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setBallAutoTrace()
{
    M_ball_trace_cb->SetValue( M_config.isBallAutoTrace() );
}

void
ViewConfigDialog::setPlayerAutoTrace()
{
    M_player_trace_cb->SetValue( M_config.isPlayerAutoTrace() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setLineTrace()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setBallTraceStart()
{
    wxString str;
    str.sprintf( wxT( "%ld" ), M_config.getBallTraceStart() );
    M_ball_trace_start->SetValue( str );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setBallTraceEnd()
{
    wxString str;
    str.sprintf( wxT( "%ld" ), M_config.getBallTraceEnd() );
    M_ball_trace_end->SetValue( str );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setPlayerTraceStart()
{
    wxString str;
    str.sprintf( wxT( "%ld" ), M_config.getPlayerTraceStart() );
    M_player_trace_start->SetValue( str );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setPlayerTraceEnd()
{
    wxString str;
    str.sprintf( wxT( "%ld" ), M_config.getPlayerTraceEnd() );
    M_player_trace_end->SetValue( str );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setBallFutureCycle()
{
    M_ball_future->SetValue( M_config.getBallFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setPlayerFutureCycle()
{
    M_ball_future->SetValue( M_config.getPlayerFutureCycle() );
}
