// -*-c++-*-

/*!
  \file view_config.h
  \brief field canvas configuration class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCER_WINDOW2_VIEW_CONFIG_H
#define SOCCER_WINDOW2_VIEW_CONFIG_H

#include <cmath>

#include <boost/any.hpp>
#include <boost/shared_ptr.hpp>

#include <rcsc/types.h>
#include <rcsc/geom/vector_2d.h>

#include "id.h"

class MainFrame;

//! drawing parameter holder & utility
class ViewConfig {
public:
    enum GrassType {
        GRASS_LINES,
        GRASS_CHECKER,
        GRASS_NORMAL,
    };

    enum PlayerSelectType {
        SELECT_FIX,
        SELECT_AUTO_LEFT,
        SELECT_AUTO_RIGHT,
        SELECT_AUTO_ALL,
        SELECT_UNSELECT,
    };

    enum FocusType {
        FOCUS_BALL,
        FOCUS_PLAYER,
        FOCUS_POINT, // include center
        NO_FOCUS
    };

    //! minimal field scale
    static const double MIN_FIELD_SCALE;
    //! maximal field scale
    static const double MAX_FIELD_SCALE;

private:

    //---------------------------------------------------------
    // canvas draw parameters

    int M_canvas_width; //!< pixel of draw area width
    int M_canvas_height; //!< pixel of draw area height

    GrassType M_grass_type; //!< field grass type

    FocusType M_focus_type; //!< focus type ID
    rcsc::Vector2D M_focus_point; //!< real coordinates

    wxPoint M_field_center; //!< the screen point of field center
    double M_field_scale; //!< field scale rate

    int M_score_board_font_size;
    wxCoord M_score_board_height; //!< screen height of score board

    bool M_zoomed; //!< true if canvas scale is changed.
    bool M_enlarge; //!< true if enlarge mode
    bool M_show_score_board; //!< true if score board is drawon
    bool M_show_ball; //!< true if ball is drawin
    bool M_show_players; //!< true if player is drawin
    bool M_show_player_number; //!< true if uniform number is drawn
    bool M_show_stamina; //!< true if stamina value is drawn
    bool M_show_hetero_number; //!< true if hetero ID is drawon
    bool M_show_view_cone; //!< true if player's view cone is drawn
    bool M_show_control_area; //!< true if control area is drawn
    bool M_show_flags; //!< true if marker flags are drawn
    bool M_show_offside_line; //!< true if offside lines are drawn

    bool M_show_voronoi_diagram; //!< true if voronoi diagram is drawn
    bool M_show_delaunay_triangle; //!< true if delaunay triangle is drawn
    rcsc::SideID M_voronoi_target;

    int M_selected_number; //!< selected player's uniform number.
    PlayerSelectType M_player_select_type; //!< flag for player auto selection

    bool M_ball_auto_trace; //! if true, trace is drawn automatically
    bool M_player_auto_trace; //! if true, trace is drawn automatically
    bool M_line_trace; //! if true, trace is line, else only trace point is drawn
    long M_ball_trace_start; //!< cycle of ball trace start
    long M_ball_trace_end; //!< cycle of ball trace end
    long M_player_trace_start; //!< log index of player trace start
    long M_player_trace_end; //!< log index of player trace end

    int M_ball_future_cycle; //!< specify the cycle to draw ball future point
    int M_player_future_cycle; //!< specify the cycle to draw ball future point

    bool M_show_debug_view;
    bool M_show_debug_view_ball;
    bool M_show_debug_view_self;
    bool M_show_debug_view_players;
    bool M_show_debug_view_comment;
    bool M_show_debug_view_figure;
    bool M_show_debug_view_target;
    bool M_show_debug_view_message;
public:
    //! constructor
    ViewConfig();

    //! reset all settings to default
    void reset();

    /*!
      \brief update parameters with canvas size
      \param canvas_width pixel of canvas width
      \param canvas_height pixel of canvas height
    */
    void updateFieldSize( const int canvas_width,
                          const int canvas_height );

private:

    void updateFieldCenter( const int canvas_width,
                            const int canvas_height );

    /*
      \brief update scoreboard size
      \param canvas_width canvas window width
      \param canvas_height canvas window height
    */
    void updateScoreBoardSize( const int canvas_width,
                               const int canvas_height );

public:

    /*!
      \brief returns current canvas width pixel
      \return current canvas width pixel.
     */
    int getCanvasWidth() const
      {
          return M_canvas_width;
      }

    /*!
      \brief returns current field grass type
      \return grass type Id
     */
    GrassType getGrassType() const
      {
          return M_grass_type;
      }
    /*!
      \brief sets the type of field grass type
     */
    void setGrassType( GrassType type )
      {
          M_grass_type = type;
      }

    /*!
      \brief returns current focust type
      \return focust type ID
     */
    FocusType getFocusType() const
      {
          return M_focus_type;
      }

    /*!
      \brief returns real coordinates of the current focus point
      \returan real coordinates value.
     */
    const
    rcsc::Vector2D & getFocusPoint() const
      {
          return M_focus_point;
      }

    /*!
      \brief sets focus point
     */
    void updateFocusPoint( const double & x,
                           const double & y )
      {
          M_focus_point.x = x;
          M_focus_point.y = y;
      }

    bool isPlayerAutoSelect() const
      {
          return ( M_player_select_type != SELECT_FIX
                   && M_player_select_type != SELECT_UNSELECT );
      }
    PlayerSelectType getPlayerSelectType() const
      {
          return M_player_select_type;
      }

    /*!
      \brief returns current screen point of the field center.
      \return coordinates of the current screen point
     */
    const
    wxPoint & getFieldCenter() const
      {
          return M_field_center;
      }

    /*!
      \brief returns current draw scale rate
      \return current draw scale rate
     */
    const
    double & getFieldScale() const
      {
          return M_field_scale;
      }
    /*!
      \brief sets the value of field scale
     */
    void setFieldScale( const double & d,
                        const bool zoomed = false );

    int getScoreBoardFontSize() const
      {
          return M_score_board_font_size;
      }

    /*!
      \brief returns current score board height
      \return pixel of current score board height
     */
    const
    wxCoord & getScoreBoardHeight() const
      {
          return M_score_board_height;
      }

    /*!
      \brief returns current zoom status.
      \retval true zoomed.
      \retval false not zoomed
     */
    bool isZoomed() const
      {
          return M_zoomed;
      }

    /*!
      \brief returns current enlarge status
      \retval true objects are enlarged.
      \retval false objects are not enlarged.
     */
    bool isEnlarged() const
      {
          return M_enlarge;
      }

    //! check if score board is drawo
    bool isShownScoreBoard() const
      {
          return M_show_score_board;
      }
    //! check if ball is drawo
    bool isShownBall() const
      {
          return M_show_ball;
      }

    //! check if player is drawo
    bool isShownPlayers() const
      {
          return M_show_players;
      }

    /*!
      \brief returns draw mode status for the uniform number.
      \retval true all players' uniform number are drawn.
      \retval false not drawn.
     */
    bool isShownPlayerNumber() const
      {
          return M_show_player_number;
      }

    /*!
      \brief returns draw mode status for hetero ID
      \retval true all players' hetero ID are drawn
      \retval false not drawn
     */
    bool isShownHeteroNumber() const
      {
          return M_show_hetero_number;
      }

    /*!
      \brief returns draw mode status for stamina value
      \retval true all players' stamina values are drawn
      \retval false not drawn
     */
    bool isShownStamina() const
      {
          return M_show_stamina;
      }

    /*!
      \brief returns draw mode status for players' view area
      \retval true all players' view cone are drawn
      \retval false not drawn
     */
    bool isShownViewCone() const
      {
          return M_show_view_cone;
      }

    /*!
      \brief returns draw mode status for players' control area
      \retval true selected player's control area are drawn
      \retval false not drawn
     */
    bool isShownControlArea() const
      {
          return M_show_control_area;
      }

    /*!
      \brief returns draw mode status of marker flags
      \retval true all flags are drawn
      \retval false not drawn
     */
    bool isShownFlags() const
      {
          return M_show_flags;
      }

    /*!
      \brief returns draw mode status of offside line
     */
    bool isShownOffsideLine() const
      {
          return M_show_offside_line;
      }

    /*!
      \brief returns draw mode status of voronoi diagram
     */
    bool isShownVoronoiDiagram() const
      {
          return M_show_voronoi_diagram;
      }

    /*!
      \brief returns draw mode status of delaunay triangle
     */
    bool isShownDelaunayTriangle() const
      {
          return M_show_delaunay_triangle;
      }

    /*!
      \brief get voronoi diagram target player type
     */
    rcsc::SideID getVoronoiTarget() const
      {
          return M_voronoi_target;
      }

    /*!
      \brief returns the selected player ID
      \return uniform number of selected player. negative value means right team.
     */
    int getSelectedNumber() const
      {
          return M_selected_number;
      }

    /*!
      \brief set selected player number
     */
    void setSelectedNumber( rcsc::SideID side,
                            int unum )
      {
          M_selected_number = ( side == rcsc::LEFT ? unum : -unum );
      }

    /*!
      \brief check the player is selected.
    */
    bool isSelectedPlayer( rcsc::SideID side,
                           int unum ) const
      {
          return ( M_selected_number
                   == ( side == rcsc::LEFT ? unum : -unum ) );
      }

    //! get trace mode
    bool isBallAutoTrace() const
      {
          return M_ball_auto_trace;
      }
    //! get trace mode
    bool isPlayerAutoTrace() const
      {
          return M_player_auto_trace;
      }

    //! get trace mode
    bool isLineTrace() const
      {
          return M_line_trace;
      }

    // ! check if ball trace should be drawn
    bool isShownBallTrace() const
      {
          return ( isBallAutoTrace()
                   || ( M_ball_trace_start < M_ball_trace_end ) );
      }
    //! returns start cycle of ball trace
    const
    long & getBallTraceStart() const
      {
          return M_ball_trace_start;
      }
    //! returns end cycle of ball trace
    const
    long & getBallTraceEnd() const
      {
          return M_ball_trace_end;
      }

    // ! check if ball trace should be drawn
    bool isShownPlayerTrace() const
      {
          return ( isPlayerAutoTrace()
                   || ( M_player_trace_start < M_player_trace_end ) );
      }
    //! returns start cycle of player trace
    const
    long & getPlayerTraceStart() const
      {
          return M_player_trace_start;
      }
    //! returns end cycle of player trace
    const
    long & getPlayerTraceEnd() const
      {
          return M_player_trace_end;
      }

    //! return the cycle to draw ball future state
    int getBallFutureCycle() const
      {
          return M_ball_future_cycle;
      }

    //! return the cycle to draw player future state
    int getPlayerFutureCycle() const
      {
          return M_player_future_cycle;
      }

    //////////////////////////////////////////////////////////
    // debug view options

    bool isShownDebugView() const
      {
          return M_show_debug_view;
      }

    bool isShownDebugViewBall() const
      {
          return M_show_debug_view_ball;
      }

    bool isShownDebugViewSelf() const
      {
          return M_show_debug_view_self;
      }

    bool isShownDebugViewPlayers() const
      {
          return M_show_debug_view_players;
      }

    bool isShownDebugViewComment() const
      {
          return M_show_debug_view_comment;
      }

    bool isShownDebugViewFigure() const
      {
          return M_show_debug_view_figure;
      }

    bool isShownDebugViewTarget() const
      {
          return M_show_debug_view_target;
      }

    bool isShownDebugViewMessage() const
      {
          return M_show_debug_view_message;
      }

    //////////////////////////////////////////////////////////
    /*!
      \brief scale input length to screen length.
      \param len real length
      \return screen pixel length
    */
    int scale( const double & len ) const
      {
          return static_cast< int >( ::rint( len * M_field_scale ) );
          //double dlen = len * M_field_scale;
          //if ( std::fabs( dlen ) < 0.5 ) return 0;
          //dlen += ( dlen > 0.0 ) ? 0.5 : -0.5;
          //return static_cast< int >( dlen );
      }
    /*!
      \brief convert 'x' to the screen coordinate X.
      \param x real point
      \return screen point
    */
    wxCoord getScreenX( const double & x ) const
      {
          return M_field_center.x + scale( x );
      }
    /*!
      \brief convert 'y' to the screen coordinate X.
      \param y real point
      \return screen point
    */
    wxCoord getScreenY( const double & y ) const
      {
          return M_field_center.y + scale( y );
      }

    /*!
      \brief convert 'x' of screen to field coordinate x
      \param x screen point value
      \return field real point value
     */
    double getFieldX( const int x ) const
      {
          return ( x - getFieldCenter().x ) / getFieldScale();
      }

    /*!
      \brief convert 'y' of screen to field coordinate y
      \param y screen point value
      \return field real point value
     */
    double getFieldY( const int y ) const
      {
          return ( y - getFieldCenter().y ) / getFieldScale();
      }

    /*!
      \brief convert screen point to the field real coordinate
      \param point screen point value
      \return field real point value
     */
    rcsc::Vector2D getFieldPoint( const wxPoint & point ) const
      {
          return rcsc::Vector2D( getFieldX( point.x ), getFieldY( point.y ) );
      }

    /*!
      \brief reset focus player.
     */
    void unselectPlayer();


    void zoomIn();

    void zoomOut();

    void unzoom();

    void setScale( const double & scale );

    void toggleEnlarge()
      {
          M_enlarge = ! M_enlarge;
      }

    void toggleShowScoreBoard()
      {
          M_show_score_board = ! M_show_score_board;
      }

    void toggleShowBall()
      {
          M_show_ball = ! M_show_ball;
      }

    void toggleShowPlayers()
      {
          M_show_players = ! M_show_players;
      }

    void toggleShowPlayerNumber()
      {
          M_show_player_number = ! M_show_player_number;
      }

    void toggleShowStamina()
      {
          M_show_stamina = ! M_show_stamina;
      }

    void toggleShowHeteroNumber()
      {
          M_show_hetero_number = ! M_show_hetero_number;
      }

    void toggleShowViewCone()
      {
          M_show_view_cone = ! M_show_view_cone;
      }

    void toggleShowControlArea()
      {
          M_show_control_area = ! M_show_control_area;
      }

    void toggleShowFlags()
      {
          M_show_flags = ! M_show_flags;
      }

    void toggleShowOffsideLine()
      {
          M_show_offside_line = ! M_show_offside_line;
      }

    void toggleShowVoronoiDiagram()
      {
          M_show_voronoi_diagram = ! M_show_voronoi_diagram;
      }

    void toggleShowDelaunayTriangle()
      {
          M_show_delaunay_triangle = ! M_show_delaunay_triangle;
      }

    void setVoronoiTarget( const rcsc::SideID side )
      {
          M_voronoi_target = side;
      }

    void setFocusPoint( const int screen_x,
                        const int screen_y );

    void setFocusType( const ViewConfig::FocusType type )
      {
          M_focus_type = type;
      }

    void setPlayerSelectType( const ViewConfig::PlayerSelectType type );

    void toggleBallAutoTrace()
      {
          M_ball_auto_trace = ! M_ball_auto_trace;
      }

    void togglePlayerAutoTrace()
      {
          M_player_auto_trace = ! M_player_auto_trace;
      }

    void toggleLineTrace()
      {
          M_line_trace = ! M_line_trace;
      }

    void setBallTraceStart( const long & cycle );
    void setBallTraceEnd( const long & cycle );


    void setPlayerTraceStart( const long & cycle );
    void setPlayerTraceEnd( const long & cycle );

    void setBallFutureCycle( const int cycle );

    void setPlayerFutureCycle( const int cycle );


    void toggleShowDebugView()
      {
          M_show_debug_view = ! M_show_debug_view;
      }

    void toggleShowDebugViewSelf()
      {
          M_show_debug_view_self = ! M_show_debug_view_self;
      }

    void toggleShowDebugViewBall()
      {
          M_show_debug_view_ball = ! M_show_debug_view_ball;
      }

    void toggleShowDebugViewPlayers()
      {
          M_show_debug_view_players = ! M_show_debug_view_players;
      }

    void toggleShowDebugViewComment()
      {
          M_show_debug_view_comment = ! M_show_debug_view_comment;
      }

    void toggleShowDebugViewFigure()
      {
          M_show_debug_view_figure = ! M_show_debug_view_figure;
      }

    void toggleShowDebugViewTarget()
      {
          M_show_debug_view_target = ! M_show_debug_view_target;
      }

    void toggleShowDebugViewMessage()
      {
          M_show_debug_view_message = ! M_show_debug_view_message;
      }
};

#endif
