// -*-c++-*-

/*!
  \file monitor_view_data.h
  \brief monitor view data type for drawing on soccerwindow Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCER_WINDOW2_MONITOR_VIEW_DATA_H
#define SOCCER_WINDOW2_MONITOR_VIEW_DATA_H

#include <vector>
#include <boost/shared_ptr.hpp>
#include <rcsc/rcg/types.h>

#include "ball.h"
#include "player.h"
#include "team.h"

//! one cycle monitor data snapshot
class MonitorViewData {
public:
    class PlayMode {
    private:
        //! playmode ID
        rcsc::PlayMode M_mode;
    public:
        //! default constructor
        PlayMode()
            : M_mode( rcsc::PM_Null )
          { }

        /*!
          \brief constructor.
          \param pm playmode ID in showinfo_t.
          construct from RCG format data.
        */
        PlayMode( const char pm )
            : M_mode( static_cast< rcsc::PlayMode >( pm ) )
          { }

        //! construct directory
        PlayMode( rcsc::PlayMode pm )
            : M_mode( pm )
          { }

        //! returns mode id
        rcsc::PlayMode mode() const
          {
              return M_mode;
          }

        //! returns mode name string
        const
        std::string & name() const;

        bool isLeftSetPlay() const;

        bool isRightSetPlay() const;

    };

private:
    //! playmode info
    PlayMode M_playmode;
    //! team info (team name & score)
    Team M_teams[2];
    //! ball info
    Ball M_ball;
    //! players info
    std::vector< Player > M_players;
    //! current game cycle
    long M_cycle;

    //! not used
    MonitorViewData();
    //! not used
    MonitorViewData( const MonitorViewData & );
    //! not used
    MonitorViewData & operator=( const MonitorViewData & );
public:
    //! construct from rcg v3 data (this is only for rcg v3)
    MonitorViewData( const rcsc::rcg::short_showinfo_t2 & show,
                     const char pmode,
                     const rcsc::rcg::team_t & team_l,
                     const rcsc::rcg::team_t & team_r );

    //! construct from rcg v3 data (this is only for monitor v2)
    MonitorViewData( const rcsc::rcg::showinfo_t2 & show );

    //! construct from rcg v1,v2 data
    MonitorViewData( const rcsc::rcg::showinfo_t & show );

    /*!
      \brief returns playmode info
      \return const reference to the playmode data object
    */
    const
    PlayMode & playmode() const
      {
          return M_playmode;
      }

    /*!
      \brief returns left team info
      \return const reference to the left team data object
    */
    const
    Team & leftTeam() const
      {
          return M_teams[0];
      }

    /*!
      \brief returns right team info
      \return const reference to the right team data object
    */
    const
    Team & rightTeam() const
      {
          return M_teams[1];
      }

    /*!
      \brief returns ball info
      \return const reference to the ball data object
    */
    const
    Ball & ball() const
      {
          return M_ball;
      }

    /*!
      \brief returns all players info
      \return const reference to the player container
    */
    const
    std::vector< Player > & players() const
      {
          return M_players;
      }

    /*!
      \brief returns game cycle info
      \return const reference to the cycle data
    */
    const
    long & cycle() const
      {
          return M_cycle;
      }

};


// type definitin for convinience
typedef boost::shared_ptr< MonitorViewData > MonitorViewPtr;
typedef std::vector< MonitorViewPtr > MonitorViewCont;

#endif
