// -*-c++-*-

/*!
	\file formation_delaunay.cpp
	\brief Delaunay Triangulation formation data classes Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sstream>

#include <rcsc/math_util.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/segment_2d.h>

#include "formation_delaunay.h"

/*-------------------------------------------------------------------*/
/*!

*/
FormationDelaunay::Param::Param( const Snapshot & data )
    : M_ball( data.ball_ )
    , M_players( data.players_ )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
FormationDelaunay::Param::getPosition( const int unum ) const
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return rcsc::Vector2D( 0.0, 0.0 );
    }

    return M_players[unum - 1];
}

/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/

/*-------------------------------------------------------------------*/
/*!

*/
FormationDelaunay::FormationDelaunay()
    : Formation()
{
    for ( int i = 0; i < 11; ++i )
    {
        M_role_name[i] = "Dummy";
    }
}



/*-------------------------------------------------------------------*/
/*!

*/
Formation::Snapshot
FormationDelaunay::createDefaultParam()
{
#if 1
    // 1: goalie
    // 2: sweeper
    // 3: left side back
    // 4: right side back
    // 5: center back
    // 6: left offensive half
    // 7: right offensive half
    // 8: left side half
    // 9: right side half
    // 10: left forward
    // 11: right forward
    createNewRole( 1, "Goalie", Formation::CENTER );
    createNewRole( 2, "Sweeper", Formation::CENTER );
    createNewRole( 3, "SideBack", Formation::SIDE );
    setRoleName( 4, getRoleName( 3 ) ); setMirrorType( 4, 3 );
    createNewRole( 5, "DefensiveHalf", Formation::CENTER );
    createNewRole( 6, "OffensiveHalf", Formation::SIDE );
    setRoleName( 7, getRoleName( 6 ) ); setMirrorType( 7, 6 );
    createNewRole( 8, "SideHalf", Formation::SIDE );
    setRoleName( 9, getRoleName( 8 ) ); setMirrorType( 9, 8 );
    createNewRole( 10, "Forward", Formation::SIDE );
    setRoleName( 11, getRoleName( 10 ) ); setMirrorType( 11, 10 );

    Snapshot snap;

    snap.ball_.assign( 0.0, 0.0 );
    snap.players_.push_back( rcsc::Vector2D( -50.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, -8.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, 8.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -9.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -5.0, -16.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -5.0, 16.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 0.0, -25.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 0.0, 25.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 10.0, -10.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 10.0, 10.0 ) );
#else
    // 1: goalie
    // 2: sweeper
    // 3: left side back
    // 4: right side back
    // 5: left defensive half
    // 6: right defensive half
    // 7: offensive half
    // 8: left side half
    // 9: right side half
    // 10: left forward
    // 11: right forward
    createNewRole( 1, "Goalie", Formation::CENTER );
    createNewRole( 2, "Sweeper", Formation::CENTER );
    createNewRole( 3, "SideBack", Formation::SIDE );
    setMirrorType( 4, 3 );
    createNewRole( 5, "DefensiveHalf", Formation::SIDE );
    setMirrorType( 6, 5 );
    createNewRole( 7, "OffensiveHalf", Formation::CENTER );
    createNewRole( 8, "SideHalf", Formation::SIDE );
    setMirrorType( 9, 8 );
    createNewRole( 10, "Forward", Formation::SIDE );
    setMirrorType( 11, 10 );

    Snapshot snap;

    snap.ball_.assign( 0.0, 0.0 );
    snap.players_.push_back( rcsc::Vector2D( -50.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, -8.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -15.0, 8.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -5.0, -16.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -5.0, 16.0 ) );
    snap.players_.push_back( rcsc::Vector2D( -9.0, 0.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 0.0, -25.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 0.0, 25.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 10.0, -10.0 ) );
    snap.players_.push_back( rcsc::Vector2D( 10.0, 10.0 ) );
#endif
    return snap;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationDelaunay::setMirrorType( const int unum,
                                  const int mirror_unum )
{
    if ( ! Formation::setMirrorType( unum, mirror_unum ) )
    {
        std::cerr <<  __FILE__ << ":" << __LINE__
                  << " *** ERROR *** Failed to set mirror type "
                  << unum << ',' << mirror_unum
                  << std::endl;
        return false;
    }

    setRoleName( unum, getRoleName( mirror_unum ) );
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FormationDelaunay::setRoleName( const int unum,
                                const std::string & name )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return;
    }

    M_role_name[unum - 1] = name;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FormationDelaunay::getRoleName( const int unum ) const
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return std::string( "" );
    }

    return M_role_name[unum - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationDelaunay::createNewRole( const int unum,
                                  const std::string & role_name,
                                  const Formation::SideType type )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return;
    }

    setRoleName( unum, role_name );

    switch ( type ) {
    case Formation::CENTER:
        setCenterType( unum );
        break;
    case Formation::SIDE:
        setSideType( unum );
        break;
    case Formation::MIRROR:
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** Invalid side type "
                  << std::endl;
        break;
    default:
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
FormationDelaunay::getPosition( const int unum,
                                const rcsc::Vector2D & ball_pos ) const
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " *** ERROR *** invalid unum " << unum
                  << std::endl;
        return rcsc::Vector2D( 0.0, 0.0 );
    }

    rcsc::Vector2D ball_pos_pitch( rcsc::bound( -52.5, ball_pos.x, 52.5 ),
                                   rcsc::bound( -34.0, ball_pos.y, 34.0 ) );

    rcsc::DelaunayTriangulation::TrianglePtr tri
        = M_delaunay.findTriangleContains( ball_pos_pitch );

    if ( ! tri )
    {
        // no triangle!!
        // search the nearest vertex

        const rcsc::DelaunayTriangulation::Vertex * v
            = M_delaunay.findNearestVertex( ball_pos_pitch );
        if ( ! v )
        {
            //std::cerr << __FILE__ << ":" << __LINE__
            //          << " *** ERROR *** No vertex!"
            //          << std::endl;
            return rcsc::Vector2D( 0.0, 0.0 );
        }

        try
        {
            //std::cerr << "found nearest vertex id= " << v->id() << std::endl;
            return M_param.at( v->id() ).getPosition( unum );
        }
        catch ( std::exception & e )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " exception caught! "
                      << e.what() << std::endl;
            return rcsc::Vector2D( 0.0, 0.0 );
        }
    }

    // linear interpolation
    try
    {
        rcsc::Vector2D result_0 = M_param.at( tri->vertex( 0 )->id() ).getPosition( unum );
        rcsc::Vector2D result_1 = M_param.at( tri->vertex( 1 )->id() ).getPosition( unum );
        rcsc::Vector2D result_2 = M_param.at( tri->vertex( 2 )->id() ).getPosition( unum );

        rcsc::Line2D line_0( tri->vertex( 0 )->pos(),
                             ball_pos_pitch );

        rcsc::Segment2D segment_12( tri->vertex( 1 )->pos(),
                                    tri->vertex( 2 )->pos() );

        rcsc::Vector2D intersection_12;

        if ( ! segment_12.intersection( line_0, & intersection_12 ) )
        {
            if ( ball_pos_pitch.dist2( tri->vertex( 0 )->pos() ) < 0.001 )
            {
                return result_0;
            }
            std::cerr << __FILE__ << ":" << __LINE__
                      << "***ERROR*** No intersection!"
                      << " ball=" << ball_pos_pitch
                      << " line_intersection=" << intersection_12
                      << "\n v0=" << tri->vertex( 0 )->pos()
                      << " v1=" << tri->vertex( 1 )->pos()
                      << " v2=" << tri->vertex( 2 )->pos()
                      << std::endl;
            return ( result_0 + result_1 + result_2 ) / 3.0;
        }

        // distance from vertex_? to interxection_12
        double dist_1i = tri->vertex( 1 )->pos().dist( intersection_12 );
        double dist_2i = tri->vertex( 2 )->pos().dist( intersection_12 );

        // interpolation result of vertex_1 & vertex_2
        rcsc::Vector2D result_12
            = result_1
            + ( result_2 - result_1 ) * ( dist_1i / ( dist_1i + dist_2i ) );

        // distance from vertex_0 to ball
        double dist_0b = tri->vertex( 0 )->pos().dist( ball_pos_pitch );
        // distance from interxectin_12 to ball
        double dist_ib = intersection_12.dist( ball_pos_pitch );

        // interpolation result of vertex_0 & intersection_12
        rcsc::Vector2D result_012
            = result_0
            + ( result_12 - result_0 ) * ( dist_0b / ( dist_0b + dist_ib ) );

        return result_012;
    }
    catch ( std::exception & e )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Exception caught!! "
                  << e.what()
                  << std::endl;
        return  rcsc::Vector2D( 0.0, 0.0 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationDelaunay::train( const std::list< Snapshot > & train_data )
{
    rcsc::Rect2D pitch( - 60.0, - 45.0,
                        120.0, 90.0 );
    M_delaunay.init( pitch );
    M_param.clear();

    const std::list< Formation::Snapshot >::const_iterator end
        = train_data.end();
    for ( std::list< Formation::Snapshot >::const_iterator it
              = train_data.begin();
          it != end;
          ++it )
    {
        M_delaunay.addVertex( it->ball_ );
        M_param.push_back( Param( *it ) );
    }

    M_delaunay.compute();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationDelaunay::read( std::istream & is )
{
    M_param.clear();

    int n_line = 0;
    std::string line_buf;


    // read formation type name
    {
        if ( ! std::getline( is, line_buf ) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Failed to read line at "
                      << n_line
                      << std::endl;
            return false;
        }
        ++n_line;

        std::istringstream istr( line_buf );

        std::string tag;
        istr >> tag;

        if ( tag != "Formation" )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Found nvalid tag ["
                      << tag << "] at line"
                      << n_line
                      << std::endl;
            return false;
        }

        std::string name_str;
        istr >> name_str;
        if ( name_str != getMethodName() )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Invalid formation type name. name must be "
                      << getMethodName()
                      << std::endl;
            return false;
        }
    }

    // read role assignment
    {
        if ( ! std::getline( is, line_buf ) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Failed to read line at "
                      << n_line
                      << std::endl;
            return false;
        }

        std::istringstream istr( line_buf );

        std::string role_name;
        int mirror_number;
        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( ! istr.good() )
            {
                std::cerr << __FILE__ << ":" << __LINE__
                          << " *** ERROR *** Failed to read player " << unum
                          << " at line " << n_line
                          << std::endl;
                return false;
            }
            istr >> role_name
                 >> mirror_number;

            //std::cerr << "Read Value " << role_name
            //          << "  " << mirror_number << std::endl;
            const Formation::SideType type = ( mirror_number == 0
                                               ? Formation::CENTER
                                               : mirror_number < 0
                                               ? Formation::SIDE
                                               : Formation::MIRROR );
            if ( type == Formation::CENTER )
            {
                createNewRole( unum, role_name, type );
            }
            else if ( type == Formation::SIDE )
            {
                createNewRole( unum, role_name, type );
            }
            else
            {
                setMirrorType( unum, mirror_number );
            }
        }
    }

#if 0
    std::cerr << "Loaded Role Assignment" << std::endl;
    for ( int unum = 1; unum <= 11; ++unum )
    {
        std::cerr << "Role=" << getRoleName( unum )
                  << " Mirror=" << getMirrorNumber( unum )
                  << std::endl;
    }
#endif
    //---------------------------------------------------
    // read sample data

    while ( std::getline( is, line_buf ) )
    {
        if ( line_buf == "End" )
        {
            break;
        }

        std::istringstream istrm( line_buf );
        ++n_line;

        M_param.push_back( Param() );

        if ( ! istrm.good() )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " *** ERROR *** Invalid formation format{ball} at line "
                      << n_line
                      << std::endl;
            M_param.pop_back();
            return false;
        }

        // read ball pos
        istrm >> M_param.back().M_ball.x
              >> M_param.back().M_ball.y;

        for ( int unum = 1; unum <= 11; ++unum )
        {
            if ( ! istrm.good() )
            {
                std::cerr << __FILE__ << ":" << __LINE__
                          << " *** ERROR *** Invalid formation format{player "
                          << unum << ") at line " << n_line
                          << std::endl;
                M_param.pop_back();
                return false;
            }

            istrm >> M_param.back().M_players[unum - 1].x
                  >> M_param.back().M_players[unum - 1].y;
        }
    }

    if ( is.eof() )
    {
        std::cerr << "Input stream reaches EOF"
                  << std::endl;
    }

    ////////////////////////////////////////////////////////
    {
        rcsc::Rect2D pitch( - 60.0, - 45.0,
                            120.0, 90.0 );
        M_delaunay.init( pitch );

        for ( std::vector< Param >::const_iterator it = M_param.begin();
              it != M_param.end();
              ++it )
        {
            M_delaunay.addVertex( it->M_ball );
        }

        M_delaunay.compute();
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
FormationDelaunay::print( std::ostream & os ) const
{
    // put method type name
    os << "Formation " << getMethodName() << '\n';

    // put role assignment
    for ( int unum = 1; unum <= 11; ++unum )
    {
        os << M_role_name[unum - 1] << ' '
           << M_mirror_number[unum - 1] << ' ';
    }
    os << '\n';

    for ( std::vector< Param >::const_iterator it = M_param.begin();
          it != M_param.end();
          ++it )
    {
        os << it->M_ball.x << ' '
           << it->M_ball.y << ' ';

        for ( std::vector< rcsc::Vector2D >::const_iterator p
                  = it->M_players.begin();
              p != it->M_players.end();
              ++p )
        {
            os << p->x << ' '
               << p->y << ' ';
        }
        os << '\n';
    }

    os << "End" << std::endl;
    return os;
}
