/*************************************************************************************************
 * Raving Bulletin Board System
 *                                                      Copyright (C) 2003-2005 Mikio Hirabayashi
 * This file is part of RBBS, a web-based bulletin board system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "rbbscommon.h"

#define CONFFILE    "rbbs.conf"          /* name of the configuration file */
#define TEXTTYPE    "text/plain"         /* content-type of plain text */
#define HTMLTYPE    "text/html"          /* content-type of HTML */
#define CSSTYPE     "text/css"           /* content-type of CSS */
#define ATOMTYPE    "application/xml"    /* content-type of Atom */
#define RXMLTYPE    "application/xml"    /* content-type of raw XML */
#define ENCODING    "UTF-8"              /* encoding of the page */
#define PETITBNUM   31                   /* bucket number of a petit map */
#define RDATAMAX    67108864             /* max size of data to read */
#define PAGENAME    "rbbs.html"          /* saving filename of the page */
#define FEEDNAME    "rbbs.xml"           /* saving filename of the feed */
#define FEEDCDATE   "cdate"              /* ID for feed ordered by creation date */
#define FEEDMDATE   "mdate"              /* ID for feed ordered by modification date */
#define DUMPNAME    "rbbsdump.xml"       /* saving filename of the dump */
#define DUMPDBCMD   "dumpdb"             /* command name to dump all articles */
#define DELEDBCMD   "deledb"             /* command name to delete the database */
#define CRPREFIX    "crypt:"             /* prefix for password of criptgraph */
#define MEMOATTR    "memo"               /* name of the attribute for memorandum */
#define TOPICCHAR   '!'                  /* beginning character for topic */
#define ITEMCHAR    '*'                  /* beginning character for list item */
#define ASISCHAR    '|'                  /* beginning character for asis text */
#define BREAKSTR    "---"                /* string for breaking line */
#define SINKPREFIX  SINKSTR ":"          /* prefix for sinking responses */

enum {                                   /* enumeration for actions */
  ACTVIEW,                               /* show articles */
  ACTCAL,                                /* show calendars */
  ACTLIST,                               /* show list of articles */
  ACTSEARCH,                             /* show the search form and its result */
  ACTARTFORM,                            /* show the form to post an article */
  ACTARTCNFM,                            /* confirm an article */
  ACTPOSTART,                            /* post an article */
  ACTDELEART,                            /* delete an article */
  ACTPOSTRES,                            /* post a response */
  ACTDUMPDB,                             /* dump all articles */
  ACTDELEDB,                             /* delete the database */
  ACTLISTFILE,                           /* show the list of files */
  ACTSAVEFILE,                           /* upload and save a file */
  ACTDELEFILE,                           /* delete a file */
  ACTSHOWMEMO,                           /* show the memorandum */
  ACTPOSTMEMO,                           /* post the memorandum */
  ACTHELP                                /* show the help page */
};

enum {                                   /* enumeration for focusing article */
  FCSNORMAL,                             /* show whole of the body and a part of responses */
  FCSDETAIL,                             /* show whole of the body and all responses */
  FCSSIMPLE                              /* show the form to post a response */
};


/* global variables for configurations */
const char *g_systemuri = NULL;          /* URI of this system */
const char *g_dbname = NULL;             /* name of the database */
const char *g_helpfile = NULL;           /* path of the help file */
const char *g_stylefile = NULL;          /* path of the style file */
int g_jetlag = DEFJETLAG;                /* jet lag */
const char *g_pagelang = NULL;           /* language of the page */
const char *g_pagetitle = NULL;          /* title of the page */
const char *g_pageauthor = NULL;         /* author of the page */
const char *g_pagesummary = NULL;        /* summary of the page */
int g_norobot = FALSE;                   /* whether to forbid robots */
const char *g_captionfile = NULL;        /* path of the caption file */
const char *g_homepageuri = NULL;        /* URI of the home page */
const char *g_adminpass = NULL;          /* password for administration */
const char *g_writepass = NULL;          /* password for writing */
int g_httpauth = FALSE;                  /* whether to use basic authentication */
int g_postauth = FALSE;                  /* whether to authorize by each article */
const char *g_artflags = NULL;           /* default flags for a new article */
const char *g_resflags = NULL;           /* default flags for a new response */
const char *g_languages = NULL;          /* list of supported languages */
const char *g_categories = NULL;         /* list of supported categories */
const char *g_defsubject = NULL;         /* subject by default */
const char *g_defauthor = NULL;          /* author by default */
int g_sort = 0;                          /* default sorting order of articles */
int g_listnum = 0;                       /* number of shown items in the subject list */
int g_shownum = 0;                       /* number of shown articles in the page */
int g_feednum = 0;                       /* number of shown articles in the feed */
int g_blknum = 0;                        /* number of shown blocks for an article */
int g_resnum = 0;                        /* number of shown responses for an article */
const char *g_homelabel = NULL;          /* label to go to the home page */
const char *g_helplabel = NULL;          /* label to show the help page */
const char *g_postlabel = NULL;          /* label to post a new article */
const char *g_filelabel = NULL;          /* label to manage files */
const char *g_memolabel = NULL;          /* label to show the memo page */
const char *g_feedlabel = NULL;          /* label to get the feed the page or an article */
const char *g_reloadlabel = NULL;        /* label to reload the page */
const char *g_searchlabel = NULL;        /* label to search articles */
const char *g_listlabel = NULL;          /* label to show list of all articles */
const char *g_callabel = NULL;           /* label to show calendars */
const char *g_prevlabel = NULL;          /* label to go to the previous page */
const char *g_nextlabel = NULL;          /* label to go to the next page */
const char *g_editlabel = NULL;          /* label to edit the existing */
const char *g_focuslabel = NULL;         /* label to focus the page or an article */
const char *g_responselabel = NULL;      /* label to write a response */
const char *g_passwordlabel = NULL;      /* label for the password */
const char *g_upstamplabel = NULL;       /* label for the update the time stamp */
const char *g_idlabel = NULL;            /* label for the ID string of an article */
const char *g_sourcelabel = NULL;        /* label for the raw XML of an article */
const char *g_languagelabel = NULL;      /* label for the language of an article */
const char *g_categorylabel = NULL;      /* label for the category of an article */
const char *g_subjectlabel = NULL;       /* label for the subject of an article */
const char *g_authorlabel = NULL;        /* label for the author of an article */
const char *g_cdatelabel = NULL;         /* label for creation date */
const char *g_mdatelabel = NULL;         /* label for modification date */
const char *g_textlabel = NULL;          /* label for the body text of an article */
const char *g_ftslabel = NULL;           /* label for full-text search */
const char *g_orderlabel = NULL;         /* label for showing order */
const char *g_descordlabel = NULL;       /* label for descending order */
const char *g_submitlabel = NULL;        /* label for general submit button */
const char *g_resetlabel = NULL;         /* label for general reset button */
const char *g_backlabel = NULL;          /* label for general back button */
const char *g_anykindlabel = NULL;       /* label for selection of any kind */
const char *g_pathlabel = NULL;          /* label for file path */
const char *g_sizelabel = NULL;          /* label for file size */
const char *g_typelabel = NULL;          /* label for file type */
const char *g_actionlabel = NULL;        /* label for file actions */
const char *g_getlabel = NULL;           /* label for file getting */
const char *g_dellabel = NULL;           /* label for file deleting */


/* global variables for parameters, cookies, and authentication */
const char *p_userpass = NULL;           /* password for administration or writing */
int p_action = ACTVIEW;                  /* number of the action */
int p_page = 0;                          /* number of skipping pages */
int p_focus = FCSNORMAL;                 /* number of the focusing mode */
int p_sort = -1;                         /* sorting order of articles */
int p_upstamp = FALSE;                   /* whether to update the modification time */
const char *p_id = NULL;                 /* ID of an article */
const char *p_language = NULL;           /* language of an article */
const char *p_category = NULL;           /* category of an article */
const char *p_subject = NULL;            /* subject of an article */
const char *p_author = NULL;             /* author of an article */
const char *p_text = NULL;               /* body text of an article */
const char *p_password = NULL;           /* password of an article */
const char *p_hash = NULL;               /* hash code of an article */
int p_year = 1;                          /* creation year of articles to be shown */
const char *p_cdate = NULL;              /* creation date of articles to be shown */
const char *p_feed = NULL;               /* ID of an article for Atom feed */
const char *p_file = NULL;               /* content of the uploaded file */
int p_filelen = 0;                       /* length of the content of the uploaded file */
const char *p_filename = NULL;           /* name of the uploaded file */
const char *c_userpass = NULL;           /* userpass in the cookie */
const char *c_author = NULL;             /* author in the cookie */
const char *a_author = NULL;             /* author via basic authentication */


/* other global variables */
RBBS *g_rbbs = NULL;                     /* database handler */
const char *g_curdate = NULL;            /* current date string */
int g_curyear = 1;                       /* current year */
int g_curmonth = 1;                      /* current month */
int g_curday = 1;                        /* current day */
int g_cannext = FALSE;                   /* whether the next page can be shown */
const char *g_rtryres = "";              /* text for re-try to post a response */
int g_tabidx = 0;                        /* counter for tab indexes */


/* function prototypes */
int main(int argc, char **argv);
const char *skiplabel(const char *str);
CBMAP *getparameters(void);
CBMAP *getcookies(void);
void xprintf(const char *format, ...);
void senderror(const char *msg);
void sendnotfound(void);
void senddump(void);
void sendfeed(void);
void sendasis(const char *id);
const char *getmimetype(const char *name);
void sendhtml(void);
int canadmin(void);
int canwrite(void);
int canartadmin(void);
int canfileadmin(void);
int authorize(void);
void shownote(const char *msg);
void showpagenavi(void);
void showcaption(void);
int matchingart(ARTICLE *art);
void showcalendars(void);
void showmonthcal(int year, int month);
int dayofweek(int year, int month, int day);
int daysofmonth(int year, int month);
void showlist(void);
void showsearchform(void);
void showarticles(void);
void showartwithctl(ARTICLE *art);
void showartform(void);
void showartconfirm(void);
void showartformback(void);
void plaincat(CBDATUM *buf, const char *text, int trim);
void mypostarticle(void);
void mydeletearticle(void);
int mypostresponse(void);
void deletedb(void);
void mylistfiles(void);
void showmemo(void);
void showhelp(void);


/* main routine */
int main(int argc, char **argv){
  CBLIST *lines;
  CBMAP *params, *cooks;
  const char *tmp;
  char *buf;
  int i, ispost;
  /* set configurations */
  signal(SIGINT, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGTERM, SIG_IGN);
  signal(SIGTSTP, SIG_IGN);
  cbstdiobin();
  if(!(lines = cbreadlines(CONFFILE))) senderror("the configuration file is missing.");
  cbglobalgc(lines, (void (*)(void *))cblistclose);
  for(i = 0; i < cblistnum(lines); i++){
    tmp = cblistval(lines, i, NULL);
    if(cbstrfwimatch(tmp, "systemuri:")){
      g_systemuri = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "dbname:")){
      g_dbname = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "helpfile:")){
      g_helpfile = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "stylefile:")){
      g_stylefile = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "jetlag:")){
      g_jetlag = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "pagelang:")){
      g_pagelang = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pagetitle:")){
      g_pagetitle = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pageauthor:")){
      g_pageauthor = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pagesummary:")){
      g_pagesummary = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "norobot:")){
      g_norobot = !strcmp(skiplabel(tmp), "true");
    } else if(cbstrfwimatch(tmp, "captionfile:")){
      g_captionfile = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "homepageuri:")){
      g_homepageuri = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "adminpass:")){
      g_adminpass = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "writepass:")){
      g_writepass = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "httpauth:")){
      g_httpauth = !cbstricmp(skiplabel(tmp), "true");
    } else if(cbstrfwimatch(tmp, "postauth:")){
      g_postauth = !cbstricmp(skiplabel(tmp), "true");
    } else if(cbstrfwimatch(tmp, "artflags:")){
      g_artflags = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "resflags:")){
      g_resflags = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "languages:")){
      g_languages = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "categories:")){
      g_categories = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "defauthor:")){
      g_defauthor = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "defsubject:")){
      g_defsubject = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "sort:")){
      tmp = skiplabel(tmp);
      if(!strcmp(tmp, "cda")){
        g_sort = SORTCDATEA;
      } else if(!strcmp(tmp, "cdd")){
        g_sort = SORTCDATED;
      } else if(!strcmp(tmp, "mda")){
        g_sort = SORTMDATEA;
      } else if(!strcmp(tmp, "mdd")){
        g_sort = SORTMDATED;
      }
    } else if(cbstrfwimatch(tmp, "listnum:")){
      g_listnum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "shownum:")){
      g_shownum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "feednum:")){
      g_feednum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "blknum:")){
      g_blknum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "resnum:")){
      g_resnum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "homelabel:")){
      g_homelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "helplabel:")){
      g_helplabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "postlabel:")){
      g_postlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "filelabel:")){
      g_filelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "memolabel:")){
      g_memolabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "feedlabel:")){
      g_feedlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "reloadlabel:")){
      g_reloadlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "searchlabel:")){
      g_searchlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "listlabel:")){
      g_listlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "callabel:")){
      g_callabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "prevlabel:")){
      g_prevlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "nextlabel:")){
      g_nextlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "editlabel:")){
      g_editlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "focuslabel:")){
      g_focuslabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "responselabel:")){
      g_responselabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "passwordlabel:")){
      g_passwordlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "upstamplabel:")){
      g_upstamplabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "idlabel:")){
      g_idlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "sourcelabel:")){
      g_sourcelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "languagelabel:")){
      g_languagelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "categorylabel:")){
      g_categorylabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "subjectlabel:")){
      g_subjectlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "authorlabel:")){
      g_authorlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "cdatelabel:")){
      g_cdatelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "mdatelabel:")){
      g_mdatelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "textlabel:")){
      g_textlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "ftslabel:")){
      g_ftslabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "orderlabel:")){
      g_orderlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "descordlabel:")){
      g_descordlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "submitlabel:")){
      g_submitlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "resetlabel:")){
      g_resetlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "backlabel:")){
      g_backlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "anykindlabel:")){
      g_anykindlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pathlabel:")){
      g_pathlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "sizelabel:")){
      g_sizelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "typelabel:")){
      g_typelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "actionlabel:")){
      g_actionlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "getlabel:")){
      g_getlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "dellabel:")){
      g_dellabel = skiplabel(tmp);
    }
  }
  if(!g_systemuri) senderror("systemuri is undefined.");
  if(!g_dbname) senderror("dbname is undefined.");
  if(!g_helpfile) senderror("helpfile is undefined.");
  if(!g_stylefile) senderror("stylefile is undefined.");
  jetlag(g_jetlag);
  if(!g_pagelang) senderror("pagelang is undefined.");
  if(!g_pagetitle) senderror("pagetitle is undefined.");
  if(!g_pageauthor) senderror("pageauthor is undefined.");
  if(!g_pagesummary) senderror("pagesummary is undefined.");
  if(!g_captionfile) g_captionfile = "";
  if(!g_homepageuri) senderror("homepageuri is undefined.");
  if(!g_adminpass) senderror("adminpass is undefined.");
  if(!g_writepass) senderror("writepass is undefined.");
  if(!g_artflags) senderror("artflags is undefined.");
  if(!g_resflags) senderror("resflags is undefined.");
  if(!g_languages) senderror("languages is undefined.");
  if(!g_categories) senderror("categories is undefined.");
  if(!g_defsubject) senderror("defsubject is undefined.");
  if(!g_defauthor) senderror("defauthor is undefined.");
  if(g_sort < SORTCDATEA || g_sort >  SORTMDATED) g_sort = SORTCDATEA;
  if(g_listnum < 0) g_listnum = 0;
  if(g_shownum < 0) g_shownum = 0;
  if(g_feednum < 0) g_feednum = 0;
  if(!g_homelabel) senderror("homelabel is undefined.");
  if(!g_helplabel) senderror("helplabel is undefined.");
  if(!g_postlabel) senderror("postlabel is undefined.");
  if(!g_filelabel) senderror("filelabel is undefined.");
  if(!g_memolabel) senderror("memolabel is undefined.");
  if(!g_feedlabel) senderror("feedlabel is undefined.");
  if(!g_reloadlabel) senderror("reloadlabel is undefined.");
  if(!g_searchlabel) senderror("searchlabel is undefined.");
  if(!g_listlabel) senderror("listlabel is undefined.");
  if(!g_callabel) senderror("callabel is undefined.");
  if(!g_prevlabel) senderror("prevlabel is undefined.");
  if(!g_nextlabel) senderror("nextlabel is undefined.");
  if(!g_editlabel) senderror("editlabel is undefined.");
  if(!g_focuslabel) senderror("focuslabel is undefined.");
  if(!g_responselabel) senderror("responselabel is undefined.");
  if(!g_passwordlabel) senderror("passwordlabel is undefined");
  if(!g_upstamplabel) senderror("upstamplabel is undefined");
  if(!g_idlabel) senderror("idlabel is undefined");
  if(!g_sourcelabel) senderror("sourcelabel is undefined");
  if(!g_languagelabel) senderror("languagelabel is undefined");
  if(!g_categorylabel) senderror("categorylabel is undefined");
  if(!g_subjectlabel) senderror("subjectlabel is undefined");
  if(!g_authorlabel) senderror("authorlabel is undefined");
  if(!g_cdatelabel) senderror("cdatelabel is undefined");
  if(!g_mdatelabel) senderror("mdatelabel is undefined");
  if(!g_textlabel) senderror("textlabel is undefined");
  if(!g_ftslabel) senderror("ftslabel is undefined");
  if(!g_orderlabel) senderror("orderlabel is undefined");
  if(!g_descordlabel) senderror("descordlabel is undefined");
  if(!g_submitlabel) senderror("submitlabel is undefined");
  if(!g_resetlabel) senderror("resetlabel is undefined");
  if(!g_backlabel) senderror("backlabel is undefined");
  if(!g_anykindlabel) senderror("anykindlabel is undefined");
  if(!g_pathlabel) senderror("pathlabel is undefined");
  if(!g_sizelabel) senderror("sizelabel is undefined");
  if(!g_typelabel) senderror("typelabel is undefined");
  if(!g_actionlabel) senderror("actionlabel is undefined");
  if(!g_getlabel) senderror("getlabel is undefined");
  if(!g_dellabel) senderror("dellabel is undefined");
  /* read parameters */
  params = getparameters();
  cbglobalgc(params, (void (*)(void *))cbmapclose);
  if((tmp = cbmapget(params, "userpass", -1, NULL)) != NULL) p_userpass = tmp;
  if((tmp = cbmapget(params, "action", -1, NULL)) != NULL){
    if(!strcmp(tmp, DUMPDBCMD)){
      p_action = ACTDUMPDB;
    } else if(!strcmp(tmp, DELEDBCMD)){
      p_action = ACTDELEDB;
    } else {
      p_action = atoi(tmp);
    }
  }
  if((tmp = cbmapget(params, "page", -1, NULL)) != NULL) p_page = atoi(tmp);
  if((tmp = cbmapget(params, "focus", -1, NULL)) != NULL) p_focus = atoi(tmp);
  if((tmp = cbmapget(params, "sort", -1, NULL)) != NULL) p_sort = atoi(tmp);
  if((tmp = cbmapget(params, "upstamp", -1, NULL)) != NULL) p_upstamp = atoi(tmp);
  if((tmp = cbmapget(params, "id", -1, NULL)) != NULL) p_id = tmp;
  if((tmp = cbmapget(params, "language", -1, NULL)) != NULL) p_language = tmp;
  if((tmp = cbmapget(params, "category", -1, NULL)) != NULL) p_category = tmp;
  if((tmp = cbmapget(params, "subject", -1, NULL)) != NULL) p_subject = tmp;
  if((tmp = cbmapget(params, "author", -1, NULL)) != NULL) p_author = tmp;
  if((tmp = cbmapget(params, "text", -1, NULL)) != NULL) p_text = tmp;
  if((tmp = cbmapget(params, "password", -1, NULL)) != NULL) p_password = tmp;
  if((tmp = cbmapget(params, "hash", -1, NULL)) != NULL) p_hash = tmp;
  if((tmp = cbmapget(params, "year", -1, NULL)) != NULL) p_year = atoi(tmp);
  if((tmp = cbmapget(params, "cdate", -1, NULL)) != NULL) p_cdate = tmp;
  if((tmp = cbmapget(params, "feed", -1, NULL)) != NULL) p_feed = tmp;
  if(!p_userpass) p_userpass = "";
  if(p_action < ACTVIEW || p_action > ACTHELP) p_action = ACTVIEW;
  if(p_page < 0) p_page = 0;
  if(p_focus < FCSNORMAL || p_focus > FCSSIMPLE) p_focus = FCSNORMAL;
  if(p_sort < SORTCDATEA || p_sort >  SORTMDATED) p_sort = g_sort;
  if(!p_id) p_id = "";
  if(!p_language) p_language = "";
  if(!p_category) p_category = "";
  if(!p_subject) p_subject = "";
  if(!p_author) p_author = "";
  if(!p_text) p_text = "";
  if(!p_password) p_password = "";
  if(!p_hash) p_hash = "";
  if(p_year < 1 || p_year > 9999) p_year = 1;
  if(!p_cdate) p_cdate = "";
  if(!p_feed) p_feed = "";
  if(!(p_file = cbmapget(params, "file", -1, &p_filelen))){
    p_file = "";
    p_filelen = 0;
  }
  if(!(p_filename = cbmapget(params, "file-filename", -1, NULL))) p_filename = "";
  if((tmp = cbmapget(params, "filename", -1, NULL)) != NULL && tmp[0] != '\0') p_filename = tmp;
  /* read cookies */
  cooks = getcookies();
  cbglobalgc(cooks, (void (*)(void *))cbmapclose);
  if((tmp = cbmapget(cooks, "userpass", -1, NULL)) != NULL) c_userpass = tmp;
  if(!c_userpass) c_userpass = "";
  if((tmp = cbmapget(cooks, "author", -1, NULL)) != NULL) c_author = tmp;
  if(!c_author) c_author = "";
  /* read authentication */
  a_author = "";
  if(g_httpauth && (tmp = getenv("REMOTE_USER")) != NULL) a_author = tmp;
  /* open the database */
  ispost = FALSE;
  if((tmp = getenv("REQUEST_METHOD")) && !strcmp(tmp, "POST")) ispost = TRUE;
  switch(p_action){
  case ACTVIEW:
  case ACTCAL:
  case ACTLIST:
  case ACTSEARCH:
  case ACTARTFORM:
  case ACTARTCNFM:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  case ACTPOSTART:
  case ACTDELEART:
    g_rbbs = rbbsopen(g_dbname, ispost);
    break;
  case ACTPOSTRES:
    g_rbbs = rbbsopen(g_dbname, ispost && p_text != '\0');
    break;
  case ACTDUMPDB:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  case ACTSHOWMEMO:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  case ACTPOSTMEMO:
    g_rbbs = rbbsopen(g_dbname, TRUE);
    break;
  case ACTLISTFILE:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  case ACTSAVEFILE:
    g_rbbs = rbbsopen(g_dbname, p_file && p_filelen > 0);
    break;
  case ACTDELEFILE:
    g_rbbs = rbbsopen(g_dbname, p_id[0] != '\0');
    break;
  default:
    break;
  }
  if(g_rbbs) cbglobalgc(g_rbbs, (void (*)(void *))rbbsclose);
  /* set the current date */
  buf = datecurrent();
  cbglobalgc(buf, free);
  g_curdate = buf;
  buf = dateforwww(g_curdate, TRUE);
  g_curyear = atoi(buf);
  g_curmonth = atoi(buf + 5);
  g_curday = atoi(buf + 8);
  free(buf);
  /* show page frame */
  if(p_action == ACTDUMPDB){
    senddump();
  } else if(p_feed[0] != '\0'){
    sendfeed();
  } else if((tmp = getenv("PATH_INFO")) != NULL){
    if(*tmp == '/') tmp++;
    sendasis(tmp);
  } else {
    sendhtml();
  }
  /* create the database if no */
  if(!g_rbbs && (g_rbbs = rbbsopen(g_dbname, TRUE)) != NULL) rbbsclose(g_rbbs);
  return 0;
}


/* skip the label of a line */
const char *skiplabel(const char *str){
  if(!(str = strchr(str, ':'))) return str;
  str++;
  while(*str != '\0' && (*str == ' ' || *str == '\t')){
    str++;
  }
  return str;
}


/* get a map of the CGI parameters */
CBMAP *getparameters(void){
  CBMAP *map, *attrs;
  CBLIST *pairs, *parts;
  const char *tmp, *body;
  char *buf, *key, *val, *dkey, *dval, *wp, *bound, *fbuf, *aname;
  int i, len, c, blen, flen;
  map = cbmapopenex(PETITBNUM);
  buf = NULL;
  len = 0;
  if((tmp = getenv("REQUEST_METHOD")) != NULL && !strcmp(tmp, "POST") &&
     (tmp = getenv("CONTENT_LENGTH")) != NULL && (len = atoi(tmp)) > 0){
    if(len > RDATAMAX) len = RDATAMAX;
    buf = cbmalloc(len + 1);
    for(i = 0; i < len && (c = getchar()) != EOF; i++){
      buf[i] = c;
    }
    buf[i] = '\0';
    if(i != len){
      free(buf);
      buf = NULL;
    }
  } else if((tmp = getenv("QUERY_STRING")) != NULL){
    buf = cbmemdup(tmp, -1);
    len = strlen(buf);
  }
  if(buf && len > 0){
    if((tmp = getenv("CONTENT_TYPE")) != NULL && cbstrfwmatch(tmp, "multipart/form-data") &&
       (tmp = strstr(tmp, "boundary=")) != NULL){
      tmp += 9;
      bound = cbmemdup(tmp, -1);
      if((wp = strchr(bound, ';')) != NULL) *wp = '\0';
      parts = cbmimeparts(buf, len, bound);
      for(i = 0; i < cblistnum(parts); i++){
        body = cblistval(parts, i, &blen);
        attrs = cbmapopenex(PETITBNUM);
        fbuf = cbmimebreak(body, blen, attrs, &flen);
        if((tmp = cbmapget(attrs, "NAME", -1, NULL)) != NULL){
          cbmapput(map, tmp, -1, fbuf, flen, FALSE);
          aname = cbsprintf("%s-filename", tmp);
          if((tmp = cbmapget(attrs, "FILENAME", -1, NULL)) != NULL)
            cbmapput(map, aname, -1, tmp, -1, FALSE);
          free(aname);
        }
        free(fbuf);
        cbmapclose(attrs);
      }
      cblistclose(parts);
      free(bound);
    } else {
      pairs = cbsplit(buf, -1, "&");
      for(i = 0; i < cblistnum(pairs); i++){
        key = cbmemdup(cblistval(pairs, i, NULL), -1);
        if((val = strchr(key, '=')) != NULL){
          *(val++) = '\0';
          dkey = cburldecode(key, NULL);
          dval = cburldecode(val, NULL);
          cbmapput(map, dkey, -1, dval, -1, FALSE);
          free(dval);
          free(dkey);
        }
        free(key);
      }
      cblistclose(pairs);
    }
  }
  free(buf);
  return map;
}


/* get a map of the CGI cookies */
CBMAP *getcookies(void){
  CBMAP *map;
  CBLIST *pairs;
  char *key, *val, *dkey, *dval;
  const char *buf;
  int i;
  map = cbmapopenex(PETITBNUM);
  if(!(buf = getenv("HTTP_COOKIE"))) return map;
  pairs = cbsplit(buf, -1, "; ");
  for(i = 0; i < cblistnum(pairs); i++){
    key = cbmemdup(cblistval(pairs, i, NULL), -1);
    if((val = strchr(key, '=')) != NULL){
      *(val++) = '\0';
      dkey = cburldecode(key, NULL);
      dval = cburldecode(val, NULL);
      cbmapput(map, dkey, -1, dval, -1, FALSE);
      free(dval);
      free(dkey);
    }
    free(key);
  }
  cblistclose(pairs);
  return map;
}


/* XML-oriented printf */
void xprintf(const char *format, ...){
  va_list ap;
  char *tmp;
  unsigned char c;
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        printf("%s", tmp);
        break;
      case 'd':
        printf("%d", va_arg(ap, int));
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          switch(*tmp){
          case '&': printf("&amp;"); break;
          case '<': printf("&lt;"); break;
          case '>': printf("&gt;"); break;
          case '"': printf("&quot;"); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f))) putchar(*tmp);
            break;
          }
          tmp++;
        }
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            putchar(c);
          } else {
            printf("%%%02X", c);
          }
          tmp++;
        }
        break;
      case '%':
        putchar('%');
        break;
      }
    } else {
      putchar(*format);
    }
    format++;
  }
  va_end(ap);
}


/* show the error page and exit */
void senderror(const char *msg){
  printf("Status: 500 Internal Server Error\r\n");
  printf("Content-Type: %s; charset=%s\r\n", TEXTTYPE, ENCODING);
  printf("\r\n");
  printf("Error: %s\n", msg);
  exit(1);
}


/* show the error page of not found */
void sendnotfound(void){
  printf("Status: 404 Not Found\r\n");
  printf("Content-Type: %s; charset=%s\r\n", TEXTTYPE, ENCODING);
  printf("\r\n");
  printf("The target of the requested URI was not found\n");
}


/* send dump data of all articles */
void senddump(void){
  ARTICLE *art;
  char *tmp, *id, *xml;
  if(!g_rbbs){
    sendnotfound();
    return;
  }
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  printf("Content-Disposition: inline; filename=%s\r\n", DUMPNAME);
  tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
  printf("Last-Modified: %s\r\n", tmp);
  free(tmp);
  printf("Content-Type: %s\r\n", RXMLTYPE);
  printf("\r\n");
  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<!DOCTYPE rbbs SYSTEM \"rbbs.dtd\">\n");
  xprintf("<rbbs version=\"%@\" cdate=\"%@\" mdate=\"%@\">\n",
          RBBS_VERSION, rbbscdate(g_rbbs), rbbsmdate(g_rbbs));
  setsortorder(g_rbbs, SORTCDATEA);
  while((id = getnextid(g_rbbs)) != NULL){
    if((art = getarticle(g_rbbs, id)) != NULL){
      xml = arttoxml(art);
      printf("%s", xml);
      free(xml);
      freearticle(art);
    }
    free(id);
  }
  xprintf("</rbbs>\n");
}


/* send Atom feed */
void sendfeed(void){
  ARTICLE *art;
  char *tmp, *xml;
  int snum;
  if(!g_rbbs){
    sendnotfound();
    return;
  }
  art = NULL;
  if(!strcmp(p_feed, FEEDCDATE)){
    setsortorder(g_rbbs, SORTCDATED);
  } else if(!strcmp(p_feed, FEEDMDATE)){
    setsortorder(g_rbbs, SORTMDATED);
  } else if(!(art = getarticle(g_rbbs, p_feed))){
    sendnotfound();
    return;
  }
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  if(!strcmp(p_feed, FEEDCDATE) || !strcmp(p_feed, FEEDMDATE)){
    printf("Content-Disposition: inline; filename=%s\r\n", FEEDNAME);
  } else {
    printf("Content-Disposition: inline; filename=%s.xml\r\n", p_feed);
  }
  tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
  printf("Last-Modified: %s\r\n", tmp);
  free(tmp);
  printf("Content-Type: %s\r\n", ATOMTYPE);
  printf("\r\n");
  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<feed version=\"0.3\" xmlns=\"http://purl.org/atom/ns#\" xml:lang=\"%@\">\n",
          g_pagelang);
  if(art){
    xprintf("<link rel=\"alternate\" type=\"%@\" href=\"%@?id=%@\"/>\n",
            HTMLTYPE, g_systemuri, artid(art));
    xprintf("<title>%@</title>\n", artsubject(art));
    xprintf("<author>\n");
    xprintf("<name>%@</name>\n", artauthor(art));
    xprintf("</author>\n");
    tmp = dateforwww(artmdate(art), FALSE);
    xprintf("<modified>%@</modified>\n", tmp);
    free(tmp);
    xprintf("<generator>RBBS %@</generator>\n", RBBS_VERSION);
    xml = arttoatom(art, g_systemuri, g_blknum, g_resnum);
    printf("%s", xml);
    free(xml);
    freearticle(art);
  } else {
    xprintf("<link rel=\"alternate\" type=\"%@\" href=\"%@\"/>\n", HTMLTYPE, g_systemuri);
    xprintf("<title>%@</title>\n", g_pagetitle);
    xprintf("<author>\n");
    xprintf("<name>%@</name>\n", g_pageauthor);
    xprintf("</author>\n");
    tmp = dateforwww(rbbsmdate(g_rbbs), FALSE);
    xprintf("<modified>%@</modified>\n", tmp);
    free(tmp);
    xprintf("<tagline>%@</tagline>\n", g_pagesummary);
    xprintf("<generator>RBBS %@</generator>\n", RBBS_VERSION);
    snum = 0;
    while(snum < g_feednum && (tmp = getnextid(g_rbbs)) != NULL){
      if((art = getarticle(g_rbbs, tmp)) != NULL){
        xml = arttoatom(art, g_systemuri, g_blknum, g_resnum);
        printf("%s", xml);
        free(xml);
        freearticle(art);
        snum++;
      }
      free(tmp);
    }
  }
  xprintf("</feed>\n");
}


/* send as-is file */
void sendasis(const char *id){
  CBMAP *attrs;
  const char *filename;
  char *buf, *tmp;
  int i, size;
  if(!g_rbbs) senderror("The database is broken.");
  attrs = cbmapopenex(PETITBNUM);
  if(!(buf = loadfile(g_rbbs, id, &size, attrs))){
    cbmapclose(attrs);
    senderror("The file was not found.");
  }
  if(!(filename = cbmapget(attrs, "subject", -1, NULL))) filename = "";
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  if(filename[0] != '\0'){
    tmp = cbmimeencode(filename, "UTF-8", TRUE);
    printf("Content-Disposition: inline; filename=%s\r\n", tmp);
    free(tmp);
  }
  printf("Content-Length: %d\r\n", size);
  printf("Content-Type: %s\r\n", getmimetype(filename));
  printf("\r\n");
  for(i = 0; i < size; i++){
    putchar(buf[i]);
  }
  free(buf);
  cbmapclose(attrs);
}


/* Get the MIME type of a file. */
const char *getmimetype(const char *name){
  if(!cbstricmp(name, "README")) return "text/plain";
  if(!cbstricmp(name, "COPYING")) return "text/plain";
  if(!cbstricmp(name, "ChangeLog")) return "text/plain";
  if(cbstrbwimatch(name, ".txt")) return "text/plain";
  if(cbstrbwimatch(name, ".asc")) return "text/plain";
  if(cbstrbwimatch(name, ".html")) return "text/html";
  if(cbstrbwimatch(name, ".htm")) return "text/html";
  if(cbstrbwimatch(name, ".css")) return "text/css";
  if(cbstrbwimatch(name, ".js")) return "text/javascript";
  if(cbstrbwimatch(name, ".tsv")) return "text/tab-separated-values";
  if(cbstrbwimatch(name, ".png")) return "image/png";
  if(cbstrbwimatch(name, ".jpg")) return "image/jpeg";
  if(cbstrbwimatch(name, ".jpeg")) return "image/jpeg";
  if(cbstrbwimatch(name, ".gif")) return "image/gif";
  if(cbstrbwimatch(name, ".tif")) return "image/tiff";
  if(cbstrbwimatch(name, ".tiff")) return "image/tiff";
  if(cbstrbwimatch(name, ".bmp")) return "image/bmp";
  if(cbstrbwimatch(name, ".svg")) return "image/svg+xml";
  if(cbstrbwimatch(name, ".xbm")) return "image/x-xbitmap";
  if(cbstrbwimatch(name, ".au")) return "audio/basic";
  if(cbstrbwimatch(name, ".snd")) return "audio/basic";
  if(cbstrbwimatch(name, ".mid")) return "audio/midi";
  if(cbstrbwimatch(name, ".midi")) return "audio/midi";
  if(cbstrbwimatch(name, ".mp3")) return "audio/mpeg";
  if(cbstrbwimatch(name, ".mpg")) return "video/mpeg";
  if(cbstrbwimatch(name, ".mpeg")) return "video/mpeg";
  if(cbstrbwimatch(name, ".xml")) return "application/xml";
  if(cbstrbwimatch(name, ".zip")) return "application/zip";
  if(cbstrbwimatch(name, ".rtf")) return "application/rtf";
  if(cbstrbwimatch(name, ".pdf")) return "application/pdf";
  if(cbstrbwimatch(name, ".ps")) return "application/postscript";
  if(cbstrbwimatch(name, ".eps")) return "application/postscript";
  if(cbstrbwimatch(name, ".doc")) return "application/msword";
  if(cbstrbwimatch(name, ".xls")) return "application/vnd.ms-excel";
  if(cbstrbwimatch(name, ".ppt")) return "application/vnd.ms-powerpoint";
  if(cbstrbwimatch(name, ".xdw")) return "application/vnd.fujixerox.docuworks";
  return "application/octet-stream";
}


/* send HTML page */
void sendhtml(void){
  const char *fstr;
  char *tmp;
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  if(p_id[0] != '\0'){
    printf("Content-Disposition: inline; filename=%s.html\r\n", p_id);
  } else {
    printf("Content-Disposition: inline; filename=%s\r\n", PAGENAME);
  }
  if(g_rbbs && (p_action == ACTVIEW || p_action == ACTCAL || p_action == ACTLIST ||
                p_action == ACTSEARCH)){
    tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
    printf("Last-Modified: %s\r\n", tmp);
    free(tmp);
  }
  if(p_userpass[0] != '\0'){
    tmp = cburlencode(p_userpass, -1);
    printf("Set-Cookie: userpass=%s\r\n", tmp);
    free(tmp);
  }
  if((p_action == ACTPOSTART || p_action == ACTPOSTRES) && p_author[0] != '\0'){
    tmp = cburlencode(p_author, -1);
    printf("Set-Cookie: author=%s; expires=Thu, 1-Jan-2037 00:00:00 GMT\r\n", tmp);
    free(tmp);
  }
  printf("Content-Type: %s; charset=%s\r\n", HTMLTYPE, ENCODING);
  printf("\r\n");
  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\""
          " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n");
  xprintf("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%@\" lang=\"%@\">\n",
          g_pagelang, g_pagelang);
  xprintf("<head>\n");
  xprintf("<title>%@</title>\n", g_pagetitle);
  if(g_rbbs && (p_action == ACTVIEW || p_action == ACTCAL || p_action == ACTLIST ||
                p_action == ACTSEARCH)){
    tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
    xprintf("<meta http-equiv=\"Last-Modified\" content=\"%@\" />\n", tmp);
    free(tmp);
  }
  xprintf("<meta http-equiv=\"Content-Type\" content=\"%@; charset=%@\" />\n",
          HTMLTYPE, ENCODING);
  xprintf("<meta http-equiv=\"Content-Style-Type\" content=\"%@\" />\n", CSSTYPE);
  xprintf("<meta name=\"author\" content=\"%@\" />\n", g_pageauthor);
  xprintf("<meta name=\"generator\" content=\"RBBS %@\" />\n", RBBS_VERSION);
  if(g_norobot) xprintf("<meta name=\"robots\" content=\"NOINDEX,NOFOLLOW\" />\n");
  tmp = dateforhttp(g_curdate, TRUE);
  xprintf("<meta name=\"localtime\" content=\"%@\" />\n", tmp);
  free(tmp);
  xprintf("<link rel=\"contents\" href=\"%@?action=%d\" />\n", g_systemuri, ACTLIST);
  fstr = FEEDCDATE;
  if(g_sort == SORTMDATEA || g_sort == SORTMDATED) fstr = FEEDMDATE;
  if(p_id[0] != '\0') fstr = p_id;
  xprintf("<link rel=\"alternate\" type=\"%@\" href=\"%@?feed=%@\" title=\"Atom\" />\n",
          ATOMTYPE, g_systemuri, fstr);
  if((tmp = cbreadfile(g_stylefile, NULL)) != NULL){
    xprintf("<style type=\"%@\">\n", CSSTYPE);
    xprintf("%s", tmp);
    xprintf("</style>\n");
    free(tmp);
  }
  xprintf("</head>\n");
  xprintf("<body>\n");
  if((p_action == ACTVIEW || p_action == ACTSEARCH) && p_id[0] == '\0' && p_cdate[0] == '\0'){
    showcaption();
    showlist();
  } else if(p_action == ACTHELP){
    showcaption();
  }
  showpagenavi();
  xprintf("<hr />\n");
  if(authorize()){
    switch(p_action){
    case ACTVIEW:
      showarticles();
      break;
    case ACTCAL:
      showcalendars();
      break;
    case ACTLIST:
      showlist();
      break;
    case ACTSEARCH:
      showsearchform();
      showarticles();
      break;
    case ACTARTFORM:
      showartform();
      break;
    case ACTARTCNFM:
      showartconfirm();
      break;
    case ACTPOSTART:
      mypostarticle();
      break;
    case ACTDELEART:
      mydeletearticle();
      break;
    case ACTPOSTRES:
      if(mypostresponse()) showarticles();
      break;
    case ACTDELEDB:
      deletedb();
      break;
    case ACTLISTFILE:
    case ACTSAVEFILE:
    case ACTDELEFILE:
      mylistfiles();
      break;
    case ACTSHOWMEMO:
    case ACTPOSTMEMO:
      showmemo();
      break;
    case ACTHELP:
      showhelp();
      break;
    default:
      break;
    }
  }
  showpagenavi();
  xprintf("<div class=\"sysinfo\">\n");
  tmp = dateforhttp(g_curdate, TRUE);
  xprintf("<div class=\"syscurdate\">%@</div>\n", tmp);
  free(tmp);
  xprintf("<div class=\"sysversion\">Powered by RBBS %@.</div>\n", RBBS_VERSION);
  xprintf("</div>\n");
  xprintf("</body>\n");
  xprintf("</html>\n");
}


/* check whether the user can administrate the system */
int canadmin(void){
  if(g_adminpass[0] == '\0') return TRUE;
  if(cbstrfwmatch(g_adminpass, CRPREFIX)){
    if(matchcrypt(p_userpass, g_adminpass + strlen(CRPREFIX))) return TRUE;
    if(matchcrypt(c_userpass, g_adminpass + strlen(CRPREFIX))) return TRUE;
    return FALSE;
  }
  if(!strcmp(p_userpass, g_adminpass)) return TRUE;
  if(!strcmp(c_userpass, g_adminpass)) return TRUE;
  return FALSE;
}


/* check whether the user can write articles */
int canwrite(void){
  if(canadmin()) return TRUE;
  if(g_writepass[0] == '\0') return TRUE;
  if(cbstrfwmatch(g_writepass, CRPREFIX)){
    if(matchcrypt(p_userpass, g_writepass + strlen(CRPREFIX))) return TRUE;
    if(matchcrypt(c_userpass, g_writepass + strlen(CRPREFIX))) return TRUE;
    return FALSE;
  }
  if(!strcmp(p_userpass, g_writepass)) return TRUE;
  if(!strcmp(c_userpass, g_writepass)) return TRUE;
  return FALSE;
}


/* check whether the user can administrate the article */
int canartadmin(void){
  ARTICLE *art;
  int match;
  if(!g_rbbs || !g_postauth || !(art = getarticle(g_rbbs, p_id))) return FALSE;
  match = matchcrypt(p_userpass, artpassword(art));
  freearticle(art);
  return match;
}


/* check whether the user can administrate the file */
int canfileadmin(void){
  CBMAP *attrs;
  const char *tmp;
  int match;
  attrs = cbmapopenex(PETITBNUM);
  if(!g_rbbs || !g_postauth) return FALSE;
  loadfile(g_rbbs, p_id, NULL, attrs);
  match = (tmp = cbmapget(attrs, "password", -1, NULL)) && matchcrypt(p_userpass, tmp);
  cbmapclose(attrs);
  return match;
}


/* check the password and show the form to input a password */
int authorize(void){
  int admin, artadm;
  admin = FALSE;
  artadm = FALSE;
  switch(p_action){
  case ACTVIEW:
  case ACTCAL:
  case ACTLIST:
  case ACTSEARCH:
  case ACTHELP:
  case ACTPOSTRES:
  case ACTSHOWMEMO:
  case ACTPOSTMEMO:
    return TRUE;
  case ACTARTFORM:
  case ACTARTCNFM:
  case ACTPOSTART:
  case ACTDELEART:
    if(p_id[0] != '\0'){
      admin = TRUE;
      artadm = TRUE;
    }
    break;
  case ACTDELEDB:
    admin = TRUE;
    break;
  case ACTLISTFILE:
  case ACTSAVEFILE:
    break;
  case ACTDELEFILE:
    admin = TRUE;
    break;
  default:
    break;
  }
  if(admin){
    if(canadmin()) return TRUE;
    if(p_action == ACTDELEFILE && canfileadmin()) return TRUE;
    if(artadm && canartadmin()) return TRUE;
  } else {
    if(canwrite()) return TRUE;
  }
  if(p_userpass[0] != '\0'){
    shownote("The password did not match.  Try again.");
  } else if(p_action == ACTDELEFILE && g_postauth){
    shownote("The password of the file is needed.");
  } else if(artadm && g_postauth){
    shownote("The password of the article is needed.");
  } else if(admin){
    shownote("The password of the administrator is needed.");
  } else {
    shownote("The password for writers is needed.");
  }
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_passwordlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"password\" name=\"userpass\" value=\"\" size=\"16\""
          " class=\"password\" tabindex=\"%d\" />\n", ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
          " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", p_action);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("</div>\n");
  xprintf("</form>\n");
  return FALSE;
}


/* show a note line */
void shownote(const char *msg){
  xprintf("<p class=\"note\">%@</p>\n", msg);
}


/* show the page navigation */
void showpagenavi(void){
  static int fst = TRUE;
  CBDATUM *buf;
  const char *fstr, *cond;
  char idbuf[NUMBUFSIZ];
  buf = cbdatumopen("", 0);
  if(p_action == ACTSEARCH){
    xsprintf(buf, "&amp;action=%d", p_action);
    xsprintf(buf, "&amp;language=%@", p_language);
    xsprintf(buf, "&amp;category=%@", p_category);
    xsprintf(buf, "&amp;subject=%@", p_subject);
    xsprintf(buf, "&amp;author=%@", p_author);
    xsprintf(buf, "&amp;text=%@", p_text);
    xsprintf(buf, "&amp;sort=%d", p_sort);
    cond = cbdatumptr(buf);
  } else {
    cond = "";
  }
  xprintf("<div class=\"pagenavi\">\n");
  if(g_homepageuri[0] != '\0')
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_homepageuri, ++g_tabidx, g_homelabel);
  xprintf("<a href=\"%@?action=%d\" rel=\"help\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTHELP, ++g_tabidx, g_helplabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\" accesskey=\"9\">%@</a>\n",
          g_systemuri, ACTARTFORM, ++g_tabidx, g_postlabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTLISTFILE, ++g_tabidx, g_filelabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTSHOWMEMO, ++g_tabidx, g_memolabel);
  fstr = FEEDCDATE;
  if(g_sort == SORTMDATEA || g_sort == SORTMDATED) fstr = FEEDMDATE;
  xprintf("<a href=\"%@?feed=%@\" type=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, fstr, ATOMTYPE, ++g_tabidx, g_feedlabel);
  xprintf("<a href=\"%@?action=%d\" rel=\"top\" class=\"navianc\""
          " tabindex=\"%d\" accesskey=\"0\">%@</a>\n",
          g_systemuri, ACTVIEW, ++g_tabidx, g_reloadlabel);
  xprintf("<a href=\"%@?action=%d\" rel=\"search\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTSEARCH, ++g_tabidx, g_searchlabel);
  xprintf("<a href=\"%@?action=%d\" rel=\"contents\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTLIST, ++g_tabidx, g_listlabel);
  sprintf(idbuf, "M%04d%02d", g_curyear, g_curmonth);
  xprintf("<a href=\"%@?action=%d&amp;year=%d#%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTCAL, g_curyear, idbuf, ++g_tabidx, g_callabel);
  if(p_action == ACTCAL){
    xprintf("<a href=\"%@?action=%d&amp;year=%d\" rel=\"next\""
            " class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ACTCAL, p_year - 1, ++g_tabidx, g_prevlabel);
  } else if(p_page > 0){
    xprintf("<a href=\"%@?page=%d%s\" rel=\"prev\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, p_page - 1, cond, ++g_tabidx, g_prevlabel);
  } else {
    xprintf("<span class=\"navivoid\">%@</span>\n", g_prevlabel);
  }
  if(p_action == ACTCAL){
    xprintf("<a href=\"%@?action=%d&amp;year=%d\" rel=\"next\""
            " class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ACTCAL, p_year + 1, ++g_tabidx, g_nextlabel);
  } else if(g_cannext && p_action != ACTLIST){
    xprintf("<a href=\"%@?page=%d%s\" rel=\"next\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, p_page + 1, cond, ++g_tabidx, g_nextlabel);
  } else {
    xprintf("<span class=\"navivoid\">%@</span>\n", g_nextlabel);
  }
  xprintf("</div>\n");
  cbdatumclose(buf);
  fst = FALSE;
}


/* show the caption of the page */
void showcaption(void){
  char *buf;
  xprintf("<div class=\"pagecaption\">\n");
  if(g_captionfile[0] != '\0' && (buf = cbreadfile(g_captionfile, NULL)) != NULL){
    xprintf("%s", buf);
    free(buf);
  } else {
    xprintf("<h1>%@</h1>\n", g_pagetitle);
    xprintf("<p>%@</p>\n", g_pagesummary);
  }
  xprintf("</div>\n");
  xprintf("<hr />\n");
}


/* check whether an article matches the narrowing condition. */
int matchingart(ARTICLE *art){
  CBLIST *words;
  const char *word;
  char *text;
  int i, hit;
  if(p_language[0] != '\0' && strcmp(artlanguage(art), p_language)) return FALSE;
  if(p_category[0] != '\0' && strcmp(artcategory(art), p_category)) return FALSE;
  if(p_subject[0] != '\0' && strcmp(artsubject(art), p_subject)) return FALSE;
  if(p_author[0] != '\0' && strcmp(artauthor(art), p_author)) return FALSE;
  if(p_text[0] != '\0'){
    text = arttotext(art);
    words = cbsplit(p_text, -1, " \t\n");
    hit = TRUE;
    for(i = 0; i < cblistnum(words); i++){
      word = cblistval(words, i, NULL);
      if(word[0] == '\0') continue;
      if(!strstr(text, word)){
        hit = FALSE;
        break;
      }
    }
    cblistclose(words);
    free(text);
    if(!hit) return FALSE;
  }
  return TRUE;
}


/* show calendars */
void showcalendars(void){
  ARTICLE *art;
  CBLIST *list;
  char *date, *pv;
  int i, j, year, month, day;
  if(!g_rbbs){
    shownote("There is no corresponding article.");
    return;
  }
  year = p_year;
  month = 1;
  xprintf("<div class=\"calenders\">\n");
  xprintf("<table summary=\"calenders\" class=\"calenders\">\n");
  for(i = 0; i < 12; i++){
    xprintf("<tr>\n");
    xprintf("<td class=\"cal\">\n");
    showmonthcal(year, month);
    xprintf("</td>\n");
    xprintf("<td class=\"list\">\n");
    for(day = 1; day <= 31; day++){
      list = getcdatelist(g_rbbs, year, month, day);
      for(j = 0; j < cblistnum(list); j++){
        if(!(art = getarticle(g_rbbs, cblistval(list, j, NULL)))) continue;
        date = cbmemdup(artcdate(art), -1);
        if((pv = strchr(date, ' ')) != NULL) *pv = '\0';
        xprintf("<div class=\"dlist\">\n");
        xprintf("<span class=\"date\">%@</span>\n", date);
        xprintf("<a href=\"%@?id=%@\" class=\"subject\">%@</a>\n",
                g_systemuri, artid(art), artsubject(art));
        xprintf("</div>\n");
        free(date);
        freearticle(art);
      }
      cblistclose(list);
    }
    xprintf("</td>\n");
    xprintf("</tr>\n");
    month++;
    if(month > 12){
      year++;
      month = 1;
    }
  }
  xprintf("</table>\n");
  xprintf("</div>\n");
}


/* show the calendar of a month */
void showmonthcal(int year, int month){
  CBLIST *list;
  char numbuf[NUMBUFSIZ], idbuf[NUMBUFSIZ];
  int i, dm, dw, day;
  dw = dayofweek(year, month, 1);
  dm = daysofmonth(year, month);
  day = 1;
  xprintf("<div class=\"mcal\">\n");
  sprintf(numbuf, "%04d-%02d", year, month);
  sprintf(idbuf, "M%04d%02d", year, month);
  xprintf("<div id=\"%@\" class=\"mcap\">%@</div>\n", idbuf, numbuf);
  xprintf("<table summary=\"%@\">\n", numbuf);
  xprintf("<tr>\n");
  for(i = 0; i < 42; i++){
    if(i > 0 && i % 7 == 0){
      xprintf("</tr>\n");
      xprintf("<tr>\n");
    }
    if(i >= dw && day <= dm){
      xprintf("<td class=\"%@%@\">", i % 7 == 0 ? "sunday" : i % 7 == 6 ? "saturday" : "weekday",
              year == g_curyear && month == g_curmonth && day == g_curday ? " today" : "");
      list = getcdatelist(g_rbbs, year, month, day);
      if(cblistnum(list) > 0){
        sprintf(numbuf, "%04d-%02d-%02d", year, month, day);
        xprintf("<a href=\"%@?sort=%d&amp;cdate=%@\">%d</a>",
                g_systemuri, SORTCDATEA, numbuf, day);
      } else {
        xprintf("%d", day);
      }
      cblistclose(list);
      xprintf("</td>\n");
      day++;
    } else {
      xprintf("<td class=\"padding\">~</td>\n");
    }
  }
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
}


/* get the day of week of a day */
int dayofweek(int year, int month, int day){
  return year -= month < 3,
    (year + year / 4 - year / 100 + year / 400 + "032503514624"[month-1] - '0' + day) % 7;
}


/* get the number of days of a month */
int daysofmonth(int year, int month){
  switch(month){
  case 2:
    if(year % 400 == 0 || (year % 4 == 0 && year % 100 != 0)) return 29;
    return 28;
  case 4: case 6: case 9: case 11:
    return 30;
  }
  return 31;
}


/* show titles of articles */
void showlist(void){
  ARTICLE *art;
  char *id;
  int i, skip, snum;
  if(g_listnum < 1 || !g_rbbs || rbbsanum(g_rbbs) < 1) return;
  setsortorder(g_rbbs, p_sort);
  snum = 0;
  skip = 0;
  xprintf("<div class=\"artlist\">\n");
  for(i = 0; (p_action == ACTLIST || snum < g_listnum) && (id = getnextid(g_rbbs)) != NULL; i++){
    if(p_action != ACTSEARCH){
      if(skip++ < p_page * g_shownum){
        free(id);
        continue;
      }
    }
    if((art = getarticle(g_rbbs, id)) != NULL){
      if(matchingart(art) && skip++ >= p_page * g_shownum){
        xprintf("<span class=\"artitem\">\n");
        xprintf("<span class=\"artseq\">%d</span>\n", i + 1);
        xprintf("<a href=\"%@?id=%@\">%@</a>\n", g_systemuri, artid(art), artsubject(art));
        xprintf("<span class=\"resnum\">%d</span>\n", artresnum(art));
        xprintf("</span>\n");
        snum++;
      }
      freearticle(art);
    }
    free(id);
  }
  if(snum < 1) shownote("There is no corresponding article.");
  xprintf("</div>\n");
  xprintf("<hr />\n");
  if(snum > g_shownum) g_cannext = TRUE;
}


/* show the search form */
void showsearchform(void){
  CBLIST *list;
  const char *tmp;
  int i;
  xprintf("<form method=\"get\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_languagelabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"language\" value=\"\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          ++g_tabidx, p_language[0] == '\0' ? " checked=\"checked\"" : "", g_anykindlabel);
  list = cbsplit(g_languages, -1, ",");
  for(i = 0; i < cblistnum(list); i++){
    tmp = cblistval(list, i, NULL);
    if(tmp[0] == '\0') continue;
    xprintf("<label><input type=\"radio\" name=\"language\" value=\"%@\""
            " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
            tmp, ++g_tabidx, !strcmp(tmp, p_language) ? " checked=\"checked\"" : "", tmp);

  }
  cblistclose(list);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_categorylabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"category\" value=\"\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          ++g_tabidx, p_language[0] == '\0' ? " checked=\"checked\"" : "", g_anykindlabel);
  list = cbsplit(g_categories, -1, ",");
  for(i = 0; i < cblistnum(list); i++){
    tmp = cblistval(list, i, NULL);
    if(tmp[0] == '\0') continue;
    xprintf("<label><input type=\"radio\" name=\"category\" value=\"%@\""
            " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
            tmp, ++g_tabidx, !strcmp(tmp, p_category) ? " checked=\"checked\"" : "", tmp);
  }
  cblistclose(list);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_subjectlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"subject\" value=\"%@\" size=\"48\""
          " class=\"text\" tabindex=\"%d\" />\n", p_subject, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_authorlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"16\""
          " class=\"text\" tabindex=\"%d\" />\n", p_author, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_ftslabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"text\" value=\"%@\" size=\"48\""
          " class=\"text\" tabindex=\"%d\" />\n", p_text, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_orderlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          SORTCDATEA, ++g_tabidx, p_sort == SORTCDATEA ? " checked=\"checked\"" : "",
          g_cdatelabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@ %@</label>\n",
          SORTCDATED, ++g_tabidx, p_sort == SORTCDATED ? " checked=\"checked\"" : "",
          g_cdatelabel, g_descordlabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          SORTMDATEA, ++g_tabidx, p_sort == SORTMDATEA ? " checked=\"checked\"" : "",
          g_mdatelabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@ %@</label>\n",
          SORTMDATED, ++g_tabidx, p_sort == SORTMDATED ? " checked=\"checked\"" : "",
          g_mdatelabel, g_descordlabel);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\">");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\" tabindex=\"%d\" />",
          g_submitlabel, ++g_tabidx);
  xprintf("<input type=\"reset\" value=\"%@\" class=\"reset\" tabindex=\"%d\" />",
          g_resetlabel, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTSEARCH);
  xprintf("</div>\n");
  xprintf("</form>\n");
  xprintf("<hr />\n");
}


/* show articles */
void showarticles(void){
  CBLIST *list;
  ARTICLE *art;
  char *id;
  int i, skip, snum, year, month, day;
  snum = 0;
  if(g_rbbs && p_id[0] != '\0'){
    if((art = getarticle(g_rbbs, p_id)) != NULL){
      showartwithctl(art);
      snum++;
      freearticle(art);
    }
  } else if(g_rbbs && p_cdate[0] != '\0'){
    year = atoi(p_cdate);
    month = strlen(p_cdate) > 5 ? atoi(p_cdate + 5) : 1;
    day = strlen(p_cdate) > 8 ? atoi(p_cdate + 8) : 1;
    list = getcdatelist(g_rbbs, year, month, day);
    for(i = 0; i < cblistnum(list); i++){
      if((art = getarticle(g_rbbs, cblistval(list, i, NULL))) != NULL){
        showartwithctl(art);
        snum++;
        freearticle(art);
      }
    }
    cblistclose(list);
  } else if(g_rbbs){
    setsortorder(g_rbbs, p_sort);
    skip = 0;
    while(snum < g_shownum && (id = getnextid(g_rbbs)) != NULL){
      if(p_action != ACTSEARCH){
        if(skip++ < p_page * g_shownum){
          free(id);
          continue;
        }
      }
      if((art = getarticle(g_rbbs, id)) != NULL){
        if(matchingart(art) && skip++ >= p_page * g_shownum){
          showartwithctl(art);
          snum++;
        }
        freearticle(art);
      }
      free(id);
    }
  }
  if(snum < 1) shownote("There is no corresponding article.");
}


/* show one article with the navigation control */
void showartwithctl(ARTICLE *art){
  static int anum = 0;
  char buf[NUMBUFSIZ], *html;
  if(++anum < 9){
    sprintf(buf, "%d", anum);
  } else {
    sprintf(buf, "#");
  }
  xprintf("<div class=\"artctl\">\n");
  xprintf("<div class=\"artnavi\">\n");
  xprintf("<a href=\"%@?action=%d&amp;id=%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTARTFORM, artid(art), ++g_tabidx, g_editlabel);
  xprintf("<a href=\"%@?feed=%@\" type=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, artid(art), ATOMTYPE, ++g_tabidx, g_feedlabel);
  xprintf("<a href=\"%@?id=%@&amp;focus=%d\" class=\"navianc\" tabindex=\"%d\">"
          "%@</a>\n", g_systemuri, artid(art), FCSDETAIL, ++g_tabidx, g_focuslabel);
  if(!strchr(artflags(art), LONECHAR)){
    xprintf("<a href=\"%@?id=%@&amp;focus=%d#%@R\" class=\"navianc\""
            " tabindex=\"%d\" accesskey=\"%s\">%@</a>\n",
            g_systemuri, artid(art), FCSSIMPLE, artid(art), ++g_tabidx, buf, g_responselabel);
  } else {
    xprintf("<span class=\"navivoid\">%@</span>\n", g_responselabel);
  }
  xprintf("</div>\n");
  html = arttohtml(art, g_systemuri, p_focus != FCSSIMPLE ? 0 : g_blknum,
                   p_focus == FCSDETAIL ? 0 : g_resnum);
  xprintf("%s", html);
  free(html);
  if(p_focus == FCSSIMPLE && !strchr(artflags(art), LONECHAR)){
    if(g_rtryres[0] != '\0') shownote("Some responses are crossing.  Try again.");
    xprintf("<form method=\"post\" action=\"%@#%@R\" id=\"%@R\">\n",
            g_systemuri, artid(art), artid(art));
    xprintf("<div class=\"resform\">\n");
    xprintf("<span class=\"formitem\">\n");
    if(a_author[0] != '\0'){
      xprintf("<span class=\"resauthor\">%@</span>\n", a_author);
    } else {
      xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"8\""
              " class=\"text\" tabindex=\"%d\" />\n",
              p_author[0] != '\0' ? p_author : c_author, ++g_tabidx);
    }
    xprintf("</span>\n");
    xprintf("<span class=\"formitem\">\n");
    xprintf("<input type=\"text\" name=\"text\" value=\"%@\" size=\"80\""
            " class=\"text\" tabindex=\"%d\" />\n", g_rtryres, ++g_tabidx);
    xprintf("</span>\n");
    xprintf("<span class=\"formitem\">\n");
    xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" />\n", g_submitlabel, ++g_tabidx);
    xprintf("</span>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTPOSTRES);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", artid(art));
    xprintf("<input type=\"hidden\" name=\"hash\" value=\"%@\" />\n", arthash(art));
    xprintf("<input type=\"hidden\" name=\"focus\" value=\"%d\" />\n", FCSSIMPLE);
    xprintf("</div>\n");
    xprintf("</form>\n");
  }
  xprintf("</div>\n");
  xprintf("<hr />\n");
}


/* show the form to post an article */
void showartform(void){
  ARTICLE *art;
  CBLIST *list;
  const char *tmp;
  char *xml;
  int i, fst;
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  if(p_id[0] != '\0'){
    xml = NULL;
    if(p_text[0] != '\0'){
      xml = cbbasedecode(p_text, NULL);
    } else {
      if(g_rbbs && (art = getarticle(g_rbbs, p_id)) != NULL){
        xml = arttoxml(art);
        freearticle(art);
      }
    }
    if(!xml) xml = cbmemdup("", 0);
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
    xprintf("<td class=\"tbody\">%@</td>\n", p_id);
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_sourcelabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<textarea name=\"text\" cols=\"80\" rows=\"20\" tabindex=\"%d\">%@</textarea>\n",
            ++g_tabidx, xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    free(xml);
  } else {
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_languagelabel);
    xprintf("<td class=\"tbody\">\n");
    list = cbsplit(g_languages, -1, ",");
    fst = p_language[0] == '\0';
    for(i = 0; i < cblistnum(list); i++){
      tmp = cblistval(list, i, NULL);
      if(tmp[0] == '\0') continue;
      xprintf("<label><input type=\"radio\" name=\"language\" value=\"%@\""
              " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
              tmp, ++g_tabidx, fst || !strcmp(tmp, p_language) ? " checked=\"checked\"" : "", tmp);
      fst = FALSE;
    }
    cblistclose(list);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_categorylabel);
    xprintf("<td class=\"tbody\">\n");
    list = cbsplit(g_categories, -1, ",");
    fst = p_category[0] == '\0';
    for(i = 0; i < cblistnum(list); i++){
      tmp = cblistval(list, i, NULL);
      if(tmp[0] == '\0') continue;
      xprintf("<label><input type=\"radio\" name=\"category\" value=\"%@\""
              " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
              tmp, ++g_tabidx, fst || !strcmp(tmp, p_category) ? " checked=\"checked\"" : "", tmp);
      fst = FALSE;
    }
    cblistclose(list);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_subjectlabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<input type=\"text\" name=\"subject\" value=\"%@\" size=\"48\""
            " class=\"text\" tabindex=\"%d\" />\n",
            p_subject[0] != '\0' ? p_subject : g_defsubject, ++g_tabidx);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_authorlabel);
    xprintf("<td class=\"tbody\">\n");
    if(a_author[0] != '\0'){
      xprintf("%@\n", a_author);
    } else {
      xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"16\""
              " class=\"text\" tabindex=\"%d\" />\n",
              p_author[0] != '\0' ? p_author : g_defauthor, ++g_tabidx);
    }
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_textlabel);
    xprintf("<td class=\"tbody\">\n");
    xml = cbbasedecode(p_text, NULL);
    xprintf("<textarea name=\"text\" cols=\"80\" rows=\"20\" tabindex=\"%d\">%@</textarea>\n",
            ++g_tabidx, xml);
    free(xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    if(g_postauth){
      xprintf("<tr>\n");
      xprintf("<td class=\"tlabel\">%@:</td>\n", g_passwordlabel);
      xprintf("<td class=\"tbody\">\n");
      xprintf("<input type=\"password\" name=\"password\" value=\"%@\" size=\"16\""
              " class=\"text\" tabindex=\"%d\" />\n", p_password, ++g_tabidx);
      xprintf("</td>\n");
      xprintf("</tr>\n");
    }
  }
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\">");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\" tabindex=\"%d\" />",
          g_submitlabel, ++g_tabidx);
  xprintf("<input type=\"reset\" value=\"%@\" class=\"reset\" tabindex=\"%d\" />",
          g_resetlabel, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"userpass\" value=\"%@\" />\n", p_userpass);
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTARTCNFM);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("</div>\n");
  xprintf("</form>\n");
  xprintf("<hr />\n");
}


/* show the confirmation to post an article */
void showartconfirm(void){
  ARTICLE *art;
  CBDATUM *buf;
  CBLIST *lines;
  const char *pline, *cline, *nline;
  char *xml, *tmp;
  int i, len;
  buf = cbdatumopen("", 0);
  if(p_id[0] != '\0'){
    xml = strnormalize(p_text, FALSE, TRUE);
    xsprintf(buf, "%s", xml);
    free(xml);
  } else {
    xsprintf(buf, "<article password=\"%@\">\n", makecrypt(p_password));
    xsprintf(buf, "<head>\n");
    xsprintf(buf, "<flags>%@</flags>\n", g_artflags);
    xsprintf(buf, "<language>%@</language>\n", p_language);
    xsprintf(buf, "<category>%@</category>\n", p_category);
    xsprintf(buf, "<subject>%@</subject>\n", p_subject);
    xsprintf(buf, "<author>%@</author>\n", a_author[0] != '\0' ? a_author : p_author);
    xsprintf(buf, "<cdate>%@</cdate>\n", g_curdate);
    xsprintf(buf, "<mdate>%@</mdate>\n", g_curdate);
    xsprintf(buf, "</head>\n");
    xsprintf(buf, "<body>\n");
    xml = strnormalize(p_text, FALSE, FALSE);
    if(xml[0] != '<'){
      lines = cbsplit(xml, -1, "\n");
      for(i = 0; i < cblistnum(lines); i++){
        pline = i > 0 ? cblistval(lines, i - 1, NULL) : "";
        cline = cblistval(lines, i, NULL);
        nline = i < cblistnum(lines) - 1 ? cblistval(lines, i + 1, NULL) : "";
        if(cline[0] != '\0'){
          if(cline[0] == TOPICCHAR){
            xsprintf(buf, "<topic>");
            plaincat(buf, cline + 1, TRUE);
            xsprintf(buf, "</topic>\n");
          } else if(cline[0] == ITEMCHAR){
            if(pline[0] != ITEMCHAR) xsprintf(buf, "<list>\n");
            xsprintf(buf, "<item>");
            plaincat(buf, cline + 1, TRUE);
            xsprintf(buf, "</item>\n");
            if(nline[0] != ITEMCHAR) xsprintf(buf, "</list>\n");
          } else if(cline[0] == ASISCHAR){
            if(pline[0] != ASISCHAR) xsprintf(buf, "<asis>");
            plaincat(buf, cline + 1, FALSE);
            xsprintf(buf, "\n");
            if(nline[0] != ASISCHAR) xsprintf(buf, "</asis>\n");
          } else if(!strcmp(cline, BREAKSTR)){
            xsprintf(buf, "<break/>\n");
          } else {
            xsprintf(buf, "<para>");
            plaincat(buf, cline, TRUE);
            xsprintf(buf, "</para>\n");
          }
        }
      }
      cblistclose(lines);
    } else {
      xsprintf(buf, "%s", xml);
    }
    free(xml);
    xsprintf(buf, "</body>\n");
    xsprintf(buf, "<tail>\n");
    xsprintf(buf, "</tail>\n");
    xsprintf(buf, "</article>\n");
  }
  len = cbdatumsize(buf);
  art = makearticle(cbdatumptr(buf));
  cbdatumclose(buf);
  if(p_id[0] != '\0' && len < 1){
    shownote("The article is being deleted.");
    xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
    xprintf("<div class=\"comform\">\n");
    xprintf("<table summary=\"form table\">\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
    xprintf("<td class=\"tbody\">%@</td>\n", p_id);
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\"></td>\n");
    xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
    xprintf("</tr>\n");
    xprintf("</table>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"userpass\" value=\"%@\" />\n", p_userpass);
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTDELEART);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
    xprintf("</div>\n");
    xprintf("</form>\n");
  } else if(!articleisok(art)){
    shownote(artemsg(art));
  } else {
    shownote("Confirm the content then submit it.");
    if(g_postauth && p_id[0] == '\0' && p_password[0] == '\0')
      shownote("Warning: The password is empty and everyone can edit the article.");
    xml = arttoxml(art);
    xprintf("<div class=\"artctl\">\n");
    tmp = arttohtml(art, g_systemuri, 0, 0);
    xprintf("%s", tmp);
    free(tmp);
    xprintf("</div>\n");
    xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
    xprintf("<div class=\"comform\">\n");
    xprintf("<table summary=\"form table\">\n");
    if(p_id[0] != '\0'){
      xprintf("<tr>\n");
      xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
      xprintf("<td class=\"tbody\">%@</td>\n", p_id);
      xprintf("</tr>\n");
    }
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_sourcelabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<pre class=\"source\">%@</pre>", xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    if(p_id[0] != '\0'){
      xprintf("<tr>\n");
      xprintf("<td class=\"tlabel\"></td>\n");
      xprintf("<td class=\"tbody\">\n");
      xprintf("<label><input type=\"checkbox\" name=\"upstamp\" value=\"%d\""
              " class=\"checkbox\" tabindex=\"%d\" checked=\"checked\" />%@</label>\n",
              TRUE, ++g_tabidx, g_upstamplabel);
      xprintf("</td>\n");
      xprintf("</tr>\n");
    }
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\"></td>\n");
    xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
    xprintf("</tr>\n");
    xprintf("</table>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"userpass\" value=\"%@\" />\n", p_userpass);
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTPOSTART);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
    xprintf("<input type=\"hidden\" name=\"author\" value=\"%@\" />\n", p_author);
    tmp = cbbaseencode(xml, -1);
    xprintf("<input type=\"hidden\" name=\"text\" value=\"%@\" />\n", tmp);
    free(tmp);
    xprintf("</div>\n");
    xprintf("</form>\n");
    free(xml);
  }
  freearticle(art);
  showartformback();
  xprintf("<hr />\n");
}


/* show a button to go back to the article form */
void showartformback(void){
  char *tmp;
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"todo\">\n");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\""
          " tabindex=\"%d\" />\n", g_backlabel, ++g_tabidx);
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"userpass\" value=\"%@\" />\n", p_userpass);
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTARTFORM);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("<input type=\"hidden\" name=\"password\" value=\"%@\" />\n", p_password);
  xprintf("<input type=\"hidden\" name=\"language\" value=\"%@\" />\n", p_language);
  xprintf("<input type=\"hidden\" name=\"category\" value=\"%@\" />\n", p_category);
  xprintf("<input type=\"hidden\" name=\"subject\" value=\"%@\" />\n", p_subject);
  xprintf("<input type=\"hidden\" name=\"author\" value=\"%@\" />\n", p_author);
  tmp = cbbaseencode(p_text, -1);
  xprintf("<input type=\"hidden\" name=\"text\" value=\"%@\" />\n", tmp);
  free(tmp);
  xprintf("</div>\n");
  xprintf("</form>\n");
}


/* concatenate a plain text to an XML buffer */
void plaincat(CBDATUM *buf, const char *text, int trim){
  const char *prefs[] = {
    "http://", "https://", "ftp://", "gopher://", "ldap://", "file://", NULL
  };
  const char *pref, *pv, *rp;
  char *rtext, *tmp, *mp;
  int i, j;
  rtext = cbmemdup(text, -1);
  if(trim) cbstrtrim(rtext);
  for(i = 0; rtext[i] != '\0'; i++){
    if(cbstrfwmatch(rtext + i, "[[") && (pv = strstr(rtext + i, "]]")) != NULL){
      tmp = cbmemdup(rtext + i + 2, pv - rtext - i - 2);
      switch(tmp[0]){
      case '^':
        xsprintf(buf, "<emp>%@</emp>", tmp + 1);
        break;
      case '~':
        xsprintf(buf, "<cite>%@</cite>", tmp + 1);
        break;
      case '+':
        xsprintf(buf, "<ins>%@</ins>", tmp + 1);
        break;
      case '-':
        xsprintf(buf, "<del>%@</del>", tmp + 1);
        break;
      default:
        if((mp = strchr(tmp, '|')) != NULL){
          *mp = '\0';
          xsprintf(buf, "<link to=\"%@\">%@</link>", mp + 1, tmp);
        } else {
          xsprintf(buf, "%@", tmp);
        }
        break;
      }
      free(tmp);
      i = pv - rtext + 1;
    } else {
      pref = NULL;
      for(j = 0; prefs[j]; j++){
        if(cbstrfwimatch(rtext + i, prefs[j])){
          pref = prefs[j];
        }
      }
      if(pref){
        rp = rtext + i;
        while(*rp != '\0' && ((*rp >= '0' && *rp <= '9') || (*rp >= 'A' && *rp <= 'Z') ||
                              (*rp >= 'a' && *rp <= 'z') || strchr("-_.!~*'();/?:@&=+$,%#", *rp))){
          rp++;
        }
        tmp = cbmemdup(rtext + i, rp - rtext - i);
        xsprintf(buf, "<link to=\"%@\">%@</link>", tmp, tmp);
        free(tmp);
        i = rp - rtext - 1;
      } else {
        switch(rtext[i]){
        case '&': xsprintf(buf, "&amp;"); break;
        case '<': xsprintf(buf, "&lt;"); break;
        case '>': xsprintf(buf, "&gt;"); break;
        default: xsprintf(buf, "%c", rtext[i]); break;
        }
      }
    }
  }
  free(rtext);
}


/* post an article */
void mypostarticle(void){
  ARTICLE *art;
  char *xml;
  if(!g_rbbs){
    shownote("The database is broken or not writable.");
    return;
  }
  xml = cbbasedecode(p_text, NULL);
  art = makearticle(xml);
  if(p_id[0] != '\0'){
    if(p_upstamp) artsetmdate(art, g_curdate);
  } else {
    artsetcdate(art, g_curdate);
    artsetmdate(art, g_curdate);
  }
  if(!articleisok(art)){
    shownote(artemsg(art));
  } else if(p_id[0] != '\0' && strcmp(p_id, artid(art))){
    shownote("The ID can not be changed.");
  } else if(g_postauth && p_id[0] != '\0' && !canadmin() &&
            !matchcrypt(p_userpass, artpassword(art))){
    shownote("The password can not be changed.");
  } else if(a_author[0] != '\0' && strcmp(artauthor(art), a_author)){
    shownote("The author is invalid.");
  } else if(!postarticle(g_rbbs, art, p_id[0] != '\0' ? p_id : NULL)){
    shownote(rbbsemsg(g_rbbs));
  } else {
    shownote("Posting the article has been completed.");
    xprintf("<div class=\"todo\">\n");
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ++g_tabidx, g_reloadlabel);
    xprintf("</div>\n");
  }
  freearticle(art);
  free(xml);
  xprintf("<hr />\n");
}


/* delete an article */
void mydeletearticle(void){
  if(!g_rbbs){
    shownote("The database is broken or not writable.");
    return;
  }
  if(!deletearticle(g_rbbs, p_id)){
    shownote(rbbsemsg(g_rbbs));
  } else {
    shownote("Deleting the article has been completed.");
    xprintf("<div class=\"todo\">\n");
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ++g_tabidx, g_reloadlabel);
    xprintf("</div>\n");
  }
}


/* post a response */
int mypostresponse(void){
  ARTICLE *art;
  char flags[PATHBUFSIZ], *wp, *author, *ap, *text;
  int err;
  art = NULL;
  err = TRUE;
  if(!g_rbbs || !(art = getarticle(g_rbbs, p_id))){
    shownote("There is no corresponding article.");
    return FALSE;
  }
  if(strcmp(arthash(art), p_hash)){
    freearticle(art);
    g_rtryres = p_text;
    return TRUE;
  }
  author = strnormalize(a_author[0] != '\0' ? a_author : p_author, TRUE, TRUE);
  text = strnormalize(p_text, TRUE, TRUE);
  if(author[0] == '\0' || text[0] == '\0'){
    free(text);
    free(author);
    freearticle(art);
    return TRUE;
  }
  wp = flags;
  wp += sprintf(wp, "%s", g_resflags);
  ap = author;
  if(cbstrfwimatch(author, SINKPREFIX)){
    if(!strchr(flags, SINKCHAR)) *(wp++) = SINKCHAR;
    ap += strlen(SINKPREFIX);
  }
  *wp = '\0';
  if(!strchr(flags, SINKCHAR)) artsetmdate(art, g_curdate);
  artaddres(art, flags, ap, g_curdate, text);
  err = FALSE;
  if(!postarticle(g_rbbs, art, p_id)){
    shownote(rbbsemsg(g_rbbs));
    err = TRUE;
  }
  free(text);
  free(author);
  freearticle(art);
  return err ? FALSE : TRUE;
}


/* delete the database directory */
void deletedb(void){
  if(rbbsremove(g_dbname)){
    shownote("The database was deleted.");
  } else {
    shownote("The database could not be deleted.");
  }
}


/* list the files */
void mylistfiles(void){
  CBLIST *list;
  CBMAP *attrs;
  const char *name, *tmp, *id;
  char *date;
  int i;
  if(!g_rbbs) return;
  if(p_action == ACTSAVEFILE && p_filelen > 0){
    name = p_filename;
    if((tmp = strrchr(name, '\\')) != NULL) name = tmp + 1;
    if((tmp = strrchr(name, '/')) != NULL) name = tmp + 1;
    if(name[0] == '\0'){
      shownote("The subject of the file is needed.");
    } else if(p_author[0] == '\0'){
      shownote("The author of the file is needed.");
    } else if(savefile(g_rbbs, p_password, name, p_author, g_curdate, p_file, p_filelen)){
      shownote("A new file was saved.");
      if(g_postauth && p_password[0] == '\0')
        shownote("Warning: The password is empty and everyone can delete the file.");
    } else {
      shownote("Saving was failed.");
    }
  } else if(p_action == ACTDELEFILE && p_id[0] != '\0'){
    if(deletefile(g_rbbs, p_id)){
      shownote("A file was deleted.");
    } else {
      shownote("Deleting was failed.");
    }
  }
  xprintf("<form method=\"post\" enctype=\"multipart/form-data\" action=\"%s\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_pathlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"file\" name=\"file\" size=\"64\" class=\"file\" tabindex=\"%d\" />\n",
          ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_subjectlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"filename\" value=\"\" size=\"32\""
          " class=\"text\" tabindex=\"%d\" />\n", ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_authorlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"16\""
          " class=\"text\" tabindex=\"%d\" />\n", p_author, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  if(g_postauth){
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_passwordlabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<input type=\"password\" name=\"password\" value=\"%@\" size=\"16\""
            " class=\"password\" tabindex=\"%d\" />\n", p_password, ++g_tabidx);
    xprintf("</td>\n");
    xprintf("</tr>\n");
  }
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
          " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTSAVEFILE);
  xprintf("<input type=\"hidden\" name=\"userpass\" value=\"%@\" />\n", p_userpass);
  xprintf("</div>\n");
  xprintf("</form>\n");
  xprintf("<div class=\"flist\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"flabel\">%@</td>\n", g_subjectlabel);
  xprintf("<td class=\"flabel\">%@</td>\n", g_authorlabel);
  xprintf("<td class=\"flabel\">%@</td>\n", g_cdatelabel);
  xprintf("<td class=\"flabel\">%@</td>\n", g_sizelabel);
  xprintf("<td class=\"flabel\">%@</td>\n", g_typelabel);
  xprintf("<td class=\"flabel\">%@</td>\n", g_actionlabel);
  xprintf("</tr>\n");
  list = listfiles(g_rbbs);
  attrs = cbmapopenex(PETITBNUM);
  for(i = 0; i < cblistnum(list); i++){
    id = cblistval(list, cblistnum(list) - i - 1, NULL);
    loadfile(g_rbbs, id, NULL, attrs);
    date = dateformen(cbmapget(attrs, "cdate", -1, NULL));
    xprintf("<tr>\n");
    xprintf("<td class=\"fbody\">%@</td>\n", cbmapget(attrs, "subject", -1, NULL));
    xprintf("<td class=\"fbody\">%@</td>\n", cbmapget(attrs, "author", -1, NULL));
    xprintf("<td class=\"fbody\">%@</td>\n", date);
    xprintf("<td class=\"fbody\">%@</td>\n", cbmapget(attrs, "size", -1, NULL));
    xprintf("<td class=\"fbody\">%@</td>\n", getmimetype(cbmapget(attrs, "subject", -1, NULL)));
    xprintf("<td class=\"fbody\">");
    xprintf("<a href=\"%@/%?\" class=\"actanc\" tabindex=\"%d\">%@</a> ",
            g_systemuri, id, ++g_tabidx, g_getlabel);
    xprintf("<a href=\"%@?action=%d&amp;id=%?\" class=\"actanc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ACTDELEFILE, id, ++g_tabidx, g_dellabel);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    free(date);
  }
  cbmapclose(attrs);
  cblistclose(list);
  xprintf("</table>\n");
  xprintf("</div>\n");
}


/* show the memorandum */
void showmemo(void){
  char *value;
  if(!g_rbbs) return;
  if(p_action == ACTPOSTMEMO){
    if(g_writepass[0] != '\0' && !canwrite()){
      shownote("The password did not match.  Try again.");
    } else if(setattribute(g_rbbs, MEMOATTR, p_text)){
      shownote("The memorandum was updated.");
    } else {
      shownote("The memorandum could not be updated.");
    }
  }
  if(!(value = getattribute(g_rbbs, MEMOATTR))) value = cbmemdup("", 0);
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_textlabel);
  xprintf("<td class=\"tbody\">");
  xprintf("<textarea name=\"text\" cols=\"80\" rows=\"20\" tabindex=\"%d\">%@</textarea>\n",
          ++g_tabidx, p_action == ACTPOSTMEMO ? p_text : value);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  if(g_writepass[0] != '\0'){
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_passwordlabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<input type=\"password\" name=\"userpass\" value=\"%@\" size=\"16\""
            " class=\"password\" tabindex=\"%d\" />\n", p_userpass, ++g_tabidx);
    xprintf("</td>\n");
  }
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\">");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\" tabindex=\"%d\" />",
          g_submitlabel, ++g_tabidx);
  xprintf("<input type=\"reset\" value=\"%@\" class=\"reset\" tabindex=\"%d\" />",
          g_resetlabel, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTPOSTMEMO);
  xprintf("</div>\n");
  xprintf("</form>\n");
  free(value);
}


/* show the help file */
void showhelp(void){
  ARTICLE *art;
  char *xml, *html;
  if(!(xml = cbreadfile(g_helpfile, NULL))){
    shownote("The help file was not found.");
    return;
  }
  art = makearticle(xml);
  if(!articleisok(art)){
    shownote(artemsg(art));
    freearticle(art);
    free(xml);
    return;
  }
  xprintf("<div class=\"artctl\">\n");
  html = arttohtml(art, g_systemuri, 0, 0);
  xprintf("%s", html);
  free(html);
  xprintf("</div>\n");
  freearticle(art);
  free(xml);
  xprintf("<hr />\n");
}



/* END OF FILE */
