#include "config.h"
#include "pipes/pipes.h"
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <dirent.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <oniguruma.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <limits.h>

static sObject* gReadlineBlock;
static sObject* gCompletionArray;

static BOOL name_sort(void* left, void* right)
{
    char* lfname = left;
    char* rfname = right;

    if(strcmp(lfname, ".") == 0) return TRUE;
    if(strcmp(lfname, "..") == 0) {
        if(strcmp(rfname, ".") == 0) return FALSE;

        return TRUE;          
    }
    if(strcmp(rfname, ".") == 0) return FALSE;
    if(strcmp(rfname, "..") == 0) return FALSE;

    return strcasecmp(lfname, rfname) < 0;
}

static sObject* gUserCompletionNextout;
static sObject* gReadlineCurrentObject;

static char* message_completion(const char* text, int stat)
{
    static int index, wordlen;

    if(stat == 0) {
        sStatment* statment = SBLOCK(gReadlineBlock).mStatments + SBLOCK(gReadlineBlock).mStatmentsNum - 1;
        
        sCommand* command = statment->mCommands + statment->mCommandsNum-1;

        gCompletionArray = VECTOR_NEW_STACK(16);

        sObject* current = gReadlineCurrentObject;
        sObject* object;

        sObject* messages = STRING_NEW_STACK("");

        while(1) {
            object = hash_item(current, command->mMessages[0]);
            if(object || current == gRootObject) break;
            current = SUOBJECT((current)).mParent;
        }

        string_put(messages, command->mMessages[0]);
        string_push_back(messages, "->");

        if(object && TYPE(object) == T_UOBJECT) {
            int i;
            for(i=1; i<command->mMessagesNum; i++) {
                object = hash_item(object, command->mMessages[i]);
                if(object == NULL || TYPE(object) != T_UOBJECT) break;
                string_push_back(messages, command->mMessages[i]);
                string_push_back(messages, "->");
            }

            if(object && TYPE(object) == T_UOBJECT) {
                hash_it* it = hash_loop_begin(object);
                while(it) {
                    char* key = hash_loop_key(it);
                    sObject* item = hash_loop_item(it);

                    sObject* candidate = STRING_NEW_STACK(string_c_str(messages));
                    string_push_back(candidate, key);

                    if(TYPE(item) == T_UOBJECT) {
                        vector_add(gCompletionArray, string_c_str(candidate));

                        sObject* candidate2 = STRING_NEW_STACK(string_c_str(messages));
                        string_push_back(candidate2, key);
                        string_push_back(candidate2, "->");

                        vector_add(gCompletionArray, string_c_str(candidate2));
                    }
                    else {
                        vector_add(gCompletionArray, string_c_str(candidate));
                    }

                    it = hash_loop_next(it);
                }

                vector_sort(gCompletionArray, name_sort);
            }
        }

        wordlen = strlen(text);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = vector_item(gCompletionArray, index);
        index++;

        if(!strncmp(text, candidate, wordlen)) {
            int len = strlen(candidate);
            if(len > 2 && candidate[len-2] == '-' && candidate[len-1] == '>') {
                rl_completion_append_character = 0;
            }
            else {
                rl_completion_append_character = ' ';
            }
            return strdup(candidate);
        }
    }

    return NULL;
}

static char* all_program_completion(const char* text, int stat)
{
    static int index, wordlen;

    if(stat == 0) {
        gCompletionArray = VECTOR_NEW_STACK(16);
        sObject* hash = HASH_NEW_STACK(16);

        sObject* current = gReadlineCurrentObject;
        while(1) {
            hash_it* it = hash_loop_begin(current);
            while(it) {
                char* key = hash_loop_key(it);
                sObject* object = hash_loop_item(it);

                if(hash_item(hash, key) == NULL) {
                    hash_put(hash, key, object);

                    sObject* candidate = STRING_NEW_STACK(hash_loop_key(it));
                    if(TYPE(object) == T_UOBJECT) {
                        if(hash_item(object, "main")) {
                            vector_add(gCompletionArray, string_c_str(candidate));
                        }
                        else {
                            sObject* candidate2 = STRING_NEW_STACK(hash_loop_key(it));
                            string_push_back(candidate2, "->");
                            vector_add(gCompletionArray, string_c_str(candidate2));
                        }
                    }
                    else {
                        vector_add(gCompletionArray, string_c_str(candidate));
                    }
                }

                it = hash_loop_next(it);
            }
            
            if(current == gRootObject) break;

            current = SUOBJECT(current).mParent;
        }
        vector_sort(gCompletionArray, name_sort);

        wordlen = strlen(text);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = vector_item(gCompletionArray, index);
        index++;

        return strdup(candidate);
    }

    return NULL;
}

static char* program_completion(const char* text, int stat)
{
    static int index, wordlen;

    if(stat == 0) {
        gCompletionArray = VECTOR_NEW_STACK(16);
        sObject* hash = HASH_NEW_STACK(16);

        sObject* current = gReadlineCurrentObject;
        while(1) {
            hash_it* it = hash_loop_begin(current);
            while(it) {
                char* key = hash_loop_key(it);
                sObject* object = hash_loop_item(it);

                if(hash_item(hash, key) == NULL) {
                    hash_put(hash, key, object);

                    sObject* candidate = STRING_NEW_STACK(hash_loop_key(it));
                    if(TYPE(object) == T_UOBJECT) {
                        if(hash_item(object, "main")) {
                            vector_add(gCompletionArray, string_c_str(candidate));
                        }
                        else {
                            sObject* candidate2 = STRING_NEW_STACK(hash_loop_key(it));
                            string_push_back(candidate2, "->");
                            vector_add(gCompletionArray, string_c_str(candidate2));
                        }
                    }
                    else {
                        vector_add(gCompletionArray, string_c_str(candidate));
                    }
                }

                it = hash_loop_next(it);
            }
            
            if(current == gRootObject) break;

            current = SUOBJECT(current).mParent;
        }
        vector_sort(gCompletionArray, name_sort);

        wordlen = strlen(text);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = vector_item(gCompletionArray, index);
        index++;

        if(!strncmp(text, candidate, wordlen)) {
            int len = strlen(candidate);
            if(len > 2 && candidate[len-2] == '-' && candidate[len-1] == '>') {
                rl_completion_append_character = 0;
            }
            else {
                rl_completion_append_character = ' ';
            }
            return strdup(candidate);
        }
    }

    return NULL;
}

static void make_text_myself(char* text2, char* text, int size)
{
    char* p = text;
    char* p2 = text2;
    BOOL squote = FALSE;
    BOOL dquote = FALSE;
    while(*p) {
        if(!squote && *p == '"') {
            p++;
            dquote = !dquote;
        }
        else if(!dquote && *p =='\'') {
            p++;
            squote = !squote;
        }
        else if(squote || dquote) {
            if(p2 - text2 + 1 < size) {
                *p2++ = *p++;
            }
        }
        else if(*p == '\\') {
            if(p2 - text2 + 1< size) {
                *p2++ = *p++;
                *p2++ = *p++;
            }
        }
        else if(*p == ' ' || *p == '\t') {
            p++;
            p2 = text2;
        }
        else {
            if(p2 - text2 + 1< size) {
                *p2++ = *p++;
            }
        }
    }

    *p2 = 0;
}

static char* user_completion(const char* text, int stat)
{
    static int index, wordlen;
    static char text2[1024];

    if(stat == 0) {
        gCompletionArray = VECTOR_NEW_STACK(16);

        int i;
        for(i=0; i<vector_size(SFD(gUserCompletionNextout).mLines); i++) {
            sObject* candidate = STRING_NEW_STACK(vector_item(SFD(gUserCompletionNextout).mLines, i));
            string_chomp(candidate);
            vector_add(gCompletionArray, string_c_str(candidate));
        }

        vector_sort(gCompletionArray, name_sort);

        make_text_myself(text2, rl_line_buffer, 1024);
//printf("text2 (%s) rl_line_buffer (%s)\n", text2, rl_line_buffer);

        wordlen = strlen(text2);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = vector_item(gCompletionArray, index);
        index++;

        if(!strncmp(text2, candidate, wordlen)) {
            if(candidate[strlen(candidate)-1] == '/') {
                rl_completion_append_character = 0;
            }
            else {
                rl_completion_append_character = ' ';
            }
            return strdup(candidate);
        }
    }

    return NULL;
}

static char* tilda_completion(const char* text, int stat)
{
    static int index, wordlen;
    static sObject* pw_dirs = NULL;

    if(stat == 0) {
        gCompletionArray = VECTOR_NEW_STACK(16);
        pw_dirs = HASH_NEW_STACK(16);

        struct passwd* pw;

        vector_add(gCompletionArray, STRING_NEW_STACK("~/"));
        pw = getpwuid(getuid());
        hash_put(pw_dirs, "~/", STRING_NEW_STACK(pw->pw_dir));

        while((pw = getpwent()) != NULL) {
            sObject* str = STRING_NEW_STACK("~");
            string_push_back(str, pw->pw_name);
            string_push_back(str, "/");
            vector_add(gCompletionArray, str);

            hash_put(pw_dirs, string_c_str(str), STRING_NEW_STACK(pw->pw_dir));
        }

        endpwent();

        vector_sort(gCompletionArray, name_sort);

        wordlen = strlen(text);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = string_c_str((sObject*)vector_item(gCompletionArray, index));
        index++;

        if(!strncmp(text, candidate, wordlen)) {
            rl_completion_append_character = 0;
            sObject* dir = hash_item(pw_dirs, candidate);
            //rl_insert_text(string_c_str(dir));

            return strdup(candidate);
        }
    }

    return NULL;
}

/*
#if !HAVE_DECL_ENVIRON
            extern char **environ;
#endif
*/

static char* env_completion(const char* text, int stat_)
{
    static int index, wordlen;

    if(stat_ == 0) {
        gCompletionArray = VECTOR_NEW_STACK(16);
        sObject* hash = HASH_NEW_STACK(16);

        sObject* current = gReadlineCurrentObject;
        while(1) {
            hash_it* it = hash_loop_begin(current);
            while(it) {
                char* key = hash_loop_key(it);
                sObject* object = hash_loop_item(it);

                if(hash_item(hash, key) == NULL) {
                    if(TYPE(object) == T_STRING || TYPE(object) == T_VECTOR || TYPE(object) == T_HASH) {
                        hash_put(hash, key, object);

                        sObject* candidate = STRING_NEW_STACK("");
                        string_push_back(candidate, hash_loop_key(it));

                        vector_add(gCompletionArray, string_c_str(candidate));
                    }
                }

                it = hash_loop_next(it);
            }
            
            if(current == gRootObject) break;

            current = SUOBJECT(current).mParent;

/*
            char** p;
            for(p = environ; *p; p++) {
                char env_name[PATH_MAX];

                char* p2 = env_name;
                char* p3 = *p;

                while(*p3 != '=') {
                    *p2++ = *p3++;
                }

                char* env = getenv(*p);
                struct stat estat;
                if(stat(env, &estat) >= 0) {
                    if(S_ISDIR(estat.st_mode)) {
                        *p2++ = '/';
                    }
                }
                *p2 = 0;

                vector_add(gCompletionArray, STRING_NEW_STACK(env_name));
            }
*/
        }

        vector_sort(gCompletionArray, name_sort);

        wordlen = strlen(text);
        index = 0;
    }

    while(index < vector_size(gCompletionArray)) {
        char* candidate = vector_item(gCompletionArray, index);
        index++;

        if(!strncmp(text, candidate, wordlen)) {
            int len = strlen(candidate);
            if(len > 2 && candidate[len-2] == '-' && candidate[len-1] == '>') {
                rl_completion_append_character = 0;
            }
            else {
                rl_completion_append_character = ' ';
            }
            return strdup(candidate);
        }
    }

    return NULL;
}

static void get_current_completion_object(sObject** completion_object, sObject** current_object)
{
    if(*current_object != gRootObject) {
        sObject* parent_object = SUOBJECT(*current_object).mParent;
        get_current_completion_object(completion_object, &parent_object);
        if(*completion_object) *completion_object = hash_item(*completion_object, SUOBJECT(*current_object).mName);
    }
}

static sObject* access_object_compl(char* name, sObject** current)
{
    sObject* object;

    while(1) {
        object = hash_item(*current, name);

        if(object || *current == gCompletionObject) { return object; }

        *current = SUOBJECT((*current)).mParent;
    }
}


char** readline_on_complete(const char* text, int start, int end)
{
    stack_start_stack();

    gReadlineBlock = BLOCK_NEW_STACK();

    sObject* cmdline = STRING_NEW_STACK("");
    string_push_back3(cmdline, rl_line_buffer, end);

    int sline = 1;
    gReadlineCurrentObject = gCurrentObject;
    BOOL result = parse(string_c_str(cmdline), "readline", &sline, gReadlineBlock, &gReadlineCurrentObject, FALSE);


    /// in the block? get the block
    if(!result && (SBLOCK(gReadlineBlock).mCompletionFlags & (COMPLETION_FLAGS_BLOCK|COMPLETION_FLAGS_ENV_BLOCK))) {
        while(1) {
            if(SBLOCK(gReadlineBlock).mStatmentsNum > 0) {
                sStatment* statment = SBLOCK(gReadlineBlock).mStatments + SBLOCK(gReadlineBlock).mStatmentsNum - 1;

                if(statment->mCommandsNum > 0) {
                    sCommand* command = statment->mCommands + statment->mCommandsNum-1;

                    int num = SBLOCK(gReadlineBlock).mCompletionFlags & 0xFF;

                    if(num > 0) {
                        if(SBLOCK(gReadlineBlock).mCompletionFlags & COMPLETION_FLAGS_ENV_BLOCK) {
                            sEnv* env = command->mEnvs + num -1;
                            gReadlineBlock = env->mBlock;
                        }
                        else {
                            gReadlineBlock = *(command->mBlocks + num -1);
                        }
                    }
                    else {
                        break;
                    }
                }
                else {
                    break;
                }
            }
            else {
                break;
            }
        }
    }

    /// expand tilda ///
    if(SBLOCK(gReadlineBlock).mCompletionFlags & COMPLETION_FLAGS_TILDA) {
        if(SBLOCK(gReadlineBlock).mStatmentsNum > 0) {
            sStatment* statment = SBLOCK(gReadlineBlock).mStatments + SBLOCK(gReadlineBlock).mStatmentsNum - 1;

            if(statment->mCommandsNum > 0) {
                sCommand* command = statment->mCommands + statment->mCommandsNum-1;

                char* arg = command->mArgs[command->mArgsNum-1];

                if(command->mArgsNum > 0 && strstr(arg, "/")) {
                    char* new_arg;
                    tilda_expasion(arg, ALLOC &new_arg);

                    rl_point -= rl_delete_text(strlen(rl_line_buffer) - strlen(arg), strlen(rl_line_buffer));
                    if(rl_point < 0) { rl_point = 0; }
                    rl_insert_text(new_arg);

                    FREE(new_arg);
                }
                else {
                    char** result = rl_completion_matches(text, tilda_completion);
                    stack_end_stack();
                    return result;
                }
            }
            else {
                char** result = rl_completion_matches(text, tilda_completion);
                stack_end_stack();
                return result;
            }
        }
        else {
            char** result = rl_completion_matches(text, tilda_completion);
            stack_end_stack();
            return result;
        }
    }

    if(SBLOCK(gReadlineBlock).mStatmentsNum == 0) {
        char** result = rl_completion_matches(text, all_program_completion);
        stack_end_stack();
        return result;
    }
    else {
        sStatment* statment = SBLOCK(gReadlineBlock).mStatments + SBLOCK(gReadlineBlock).mStatmentsNum - 1;

        if(statment->mCommandsNum == 0 || SBLOCK(gReadlineBlock).mCompletionFlags & COMPLETION_FLAGS_LAST_CHAR_IS_PIPE) {
            char** result = rl_completion_matches(text, all_program_completion);
            stack_end_stack();
            return result;
        }
        else {
            sCommand* command = statment->mCommands + statment->mCommandsNum-1;

            /// get user completion ///
            sObject* ucompletion;
            if(command->mArgsNum > 0) {
                sObject* completion_object = gCompletionObject;
                sObject* current_object = gCurrentObject;

                get_current_completion_object(&completion_object, &current_object);

                if(completion_object == NULL) completion_object = gCompletionObject;

                sObject* object;
                if(command->mMessagesNum > 0) {
                    sObject* reciever = completion_object;
                    object = access_object_compl(command->mMessages[0], &reciever);

                    int i;
                    for(i=1; i<command->mMessagesNum; i++) {
                        if(object && TYPE(object) == T_UOBJECT) {
                            object = hash_item(object, command->mMessages[i]);
                        }
                        else {
                            break;
                        }
                    }

                    if(object && TYPE(object) == T_UOBJECT) {
                        ucompletion = hash_item(object, command->mArgs[0]);

                        if(ucompletion == NULL) {
                            ucompletion = hash_item(object, "__all__");;
                        }
                    }
                    else {
                        ucompletion = NULL;
                    }
                }
                else {
                    sObject* reciever = completion_object;
                    ucompletion = access_object_compl(command->mArgs[0], &reciever);

                    if(ucompletion == NULL) {
                        reciever = completion_object;
                        ucompletion = access_object_compl("__all__", &reciever);
                    }
                }
            }
            else {
                ucompletion = NULL;
            }

            char last_char = rl_line_buffer[strlen(rl_line_buffer)-1];

            if(SBLOCK(gReadlineBlock).mCompletionFlags & COMPLETION_FLAGS_ENV) {
                char** result = rl_completion_matches(text, env_completion);
                stack_end_stack();
                return result;
            }
            else if(command->mArgsNum == 0 
                || command->mArgsNum == 1 && last_char != ' ' && last_char != '"' && last_char != '\'')
            {
                if(command->mMessagesNum > 0) {
                    char** result = rl_completion_matches(text, message_completion);
                    stack_end_stack();
                    return result;
                }
                else {
                    char** result = rl_completion_matches(text, program_completion);
                    stack_end_stack();
                    return result;
                }
            }
            else if(ucompletion && TYPE(ucompletion) == T_COMPLETION) {
                sObject* nextin = FD_NEW_STACK();
                if(!fd_write(nextin, string_c_str(cmdline))) {
                    stack_end_stack();
                    return NULL;
                }
                sObject* nextout = FD_NEW_STACK();

                sObject* fun = FUN_NEW_STACK(NULL);
                sObject* stackframe = UOBJECT_NEW_GC(8, gPipesObject, "_stackframe", FALSE);
                vector_add(gStackFrames, stackframe);
                //uobject_init(stackframe);
                SFUN(fun).mLocalObjects = stackframe;

                sObject* argv = VECTOR_NEW_GC(16, FALSE);
                if(command->mArgsNum == 1 || (SBLOCK(gReadlineBlock).mCompletionFlags & COMPLETION_FLAGS_LAST_CHAR_IS_SPACE)) 
                {
                    vector_add(argv, STRING_NEW_GC(command->mArgs[0], FALSE));
                    vector_add(argv, STRING_NEW_GC("", FALSE));
                }
                else if(command->mArgsNum > 1) {
                    vector_add(argv, STRING_NEW_GC(command->mArgs[0], FALSE));

                    /// if parser uses PARSER_MAGIC_NUMBER_OPTION, convert it
                    char* str = command->mArgs[command->mArgsNum-1];
                    char* new_str = MALLOC(strlen(str) + 1);
                    xstrncpy(new_str, str, strlen(str) + 1);
                    if(new_str[0] == PARSER_MAGIC_NUMBER_OPTION) {
                        new_str[0] = '-';
                    }
                    vector_add(argv, STRING_NEW_GC(new_str, FALSE));
                    FREE(new_str);
                }
                else {
                    vector_add(argv, STRING_NEW_GC("", FALSE));
                    vector_add(argv, STRING_NEW_GC("", FALSE));
                }
                hash_put(SFUN(fun).mLocalObjects, "ARGV", argv);

                int rcode = 0;
                pipes_set_signal();
                if(!run(ucompletion, nextin, nextout, &rcode, gReadlineCurrentObject, fun)) {
                    readline_signal();
                    fprintf(stderr, "\nrun time error\n");
                    fprintf(stderr, "%s", string_c_str(gErrMsg));
                    vector_pop_back(gStackFrames);
                    stack_end_stack();
                    return NULL;
                }
                readline_signal();
                vector_pop_back(gStackFrames);

                fd_split(nextout, kLF);

                gUserCompletionNextout = nextout;
                char** result = rl_completion_matches(text, user_completion);
                stack_end_stack();
                return result;
            }
        }
    }

    stack_end_stack();
    return NULL;
}

BOOL cmd_completion(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    sCommand* command = runinfo->mCommand;

    if(command->mArgsNumRuntime >= 2 && command->mBlocksNum == 1) {
        sObject* block = command->mBlocks[0];

        int i;
        for(i=1; i<command->mArgsNumRuntime; i++) {
            sObject* object = gCompletionObject;

            sObject* str = STRING_NEW_GC("", FALSE);
            char* p = command->mArgsRuntime[i];
            while(*p) {
                if(*p == '-' && *(p+1) == '>') {
                    if(string_c_str(str)[0] == 0) {
                        err_msg("invalid object name", runinfo->mSName, runinfo->mSLine, command->mArgs[0]);
                        return FALSE;
                    }
                    p+=2;

                    sObject* object2 = hash_item(object, string_c_str(str));
                    if(object2 == NULL || TYPE(object2) !=T_UOBJECT) {
                        sObject* new_object = UOBJECT_NEW_GC(8, object, string_c_str(str), TRUE);
                        hash_put(object, string_c_str(str), new_object);
                        uobject_init(new_object);

                        object = new_object;
                    }
                    else {
                        object = object2;
                    }
                    string_put(str, "");
                }
                else {
                    string_push_back2(str, *p);
                    p++;
                }
            }
            if(object && TYPE(object) == T_UOBJECT && string_c_str(str)[0] != 0) {
                hash_put(object, string_c_str(str), block_clone_gc(block, T_COMPLETION, FALSE));
            }
            else {
                err_msg("There is no object", runinfo->mSName, runinfo->mSLine, command->mArgsRuntime[i]);

                return FALSE;
            }
        }

        runinfo->mRCode = 0;
    }

    return TRUE;
}

BOOL cmd_readline_clear_screen(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    mclear_immediately();
    rl_forced_update_display();

    runinfo->mRCode = 0;

    return TRUE;
}

BOOL cmd_readline_point_move(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    sCommand* command = runinfo->mCommand;

    if(command->mArgsNumRuntime == 2) {
        int n = atoi(command->mArgsRuntime[1]);

        if(n < 0) n += strlen(rl_line_buffer) + 1;
        if(n < 0) n = 0;
        if(n > strlen(rl_line_buffer)) n = strlen(rl_line_buffer);

        rl_point = n;

        runinfo->mRCode = 0;
    }

    return TRUE;
}

BOOL cmd_readline_insert_text(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    sCommand* command = runinfo->mCommand;

    if(runinfo->mFilter) {
        (void)rl_insert_text(SFD(nextin).mBuf);
#if defined(__FREEBSD__)
//extern int rl_reset_terminal PARAMS((const char *));
    mclear_immediately();
    rl_forced_update_display();
#endif

        runinfo->mRCode = 0;
    }

    return TRUE;
}

BOOL cmd_readline_delete_text(sObject* nextin, sObject* nextout, sRunInfo* runinfo)
{
    sCommand* command = runinfo->mCommand;

    if(command->mArgsNumRuntime == 3) {
        int n = atoi(command->mArgsRuntime[1]);
        int m = atoi(command->mArgsRuntime[2]);
        if(n < 0) n += strlen(rl_line_buffer) + 1;
        if(n < 0) n= 0;
        if(m < 0) m += strlen(rl_line_buffer) + 1;
        if(m < 0) m = 0;
        rl_point -= rl_delete_text(n, m);
        if(rl_point < 0) {
            rl_point = 0;
        }
#if defined(__FREEBSD__)
    mclear_immediately();
    rl_forced_update_display();
#endif

        runinfo->mRCode = 0;
    }

    return TRUE;
}

static int readline_macro(int count, int key)
{
    stack_start_stack();

    sObject* nextout2 = FD_NEW_STACK();

    int rcode = 0;
    sObject* block = BLOCK_NEW_STACK();
    int sline = 1;
    if(parse("root->macro", "macro", &sline, block, NULL, TRUE)) {
        pipes_set_signal();

        sObject* fun = FUN_NEW_STACK(NULL);
        sObject* stackframe = UOBJECT_NEW_GC(8, gPipesObject, "_stackframe", FALSE);
        vector_add(gStackFrames, stackframe);
        //uobject_init(stackframe);
        SFUN(fun).mLocalObjects = stackframe;

        sObject* nextin2 = FD_NEW_STACK();

        (void)fd_write3(nextin2, rl_line_buffer, rl_point);

        if(!run(block, nextin2, nextout2, &rcode, gRootObject, fun)) {
            if(rcode == RCODE_BREAK) {
                fprintf(stderr, "invalid break. Not in a loop\n");
            }
            else if(rcode == RCODE_RETURN) {
                fprintf(stderr, "invalid return. Not in a function\n");
            }
            else if(rcode == RCODE_EXIT) {
            }
            else {
                fprintf(stderr, "run time error\n");
                fprintf(stderr, "%s", string_c_str(gErrMsg));
            }
        }
        readline_signal();
        //pipes_restore_signal_default();
        (void)vector_pop_back(gStackFrames);
    }
    else {
        fprintf(stderr, "parser error\n");
        fprintf(stderr, "%s", string_c_str(gErrMsg));
    }

    rl_insert_text(SFD(nextout2).mBuf);
#if defined(__FREEBSD__)
    mclear_immediately();
    rl_forced_update_display();
#endif

    stack_end_stack();

    return 0;
}
static int skip_quoted(const char *s, int i, char q)
{
   while(s[i] && s[i]!=q) 
   {
      if(s[i]=='\\' && s[i+1]) i++;
      i++;
   }
   if(s[i]) i++;
   return i;
}

static int lftp_char_is_quoted(const char *string, int eindex)
{
  int i, pass_next;

  for (i = pass_next = 0; i <= eindex; i++)
    {
      if (pass_next)
        {
          pass_next = 0;
          if (i >= eindex)
            return 1;
          continue;
        }
      else if (string[i] == '"' || string[i] == '\'')
        {
          char quote = string[i];
          i = skip_quoted (string, ++i, quote);
          if (i > eindex)
            return 1;
          i--;
        }
      else if (string[i] == '\\')
        {
          pass_next = 1;
          continue;
        }
    }
  return (0);
}

char* readline_filename_completion_null_generator(const char* a, int b)
{
    return NULL;
}

void pipes_readline_init(BOOL runtime_script)
{
    rl_attempted_completion_function = readline_on_complete;
    rl_completion_entry_function = readline_filename_completion_null_generator;
    rl_completer_quote_characters = "\\\"'";
    rl_completer_word_break_characters = " \t\n\"'|!&;()=$";
    rl_completion_append_character= ' ';
    rl_char_is_quoted_p = (rl_linebuf_func_t*)lftp_char_is_quoted;

//rl_basic_word_break_characters = " ";
//      "\"'",		      // rl_completer_quote_characters
//      " \t\n\"'",	      // rl_completer_word_break_characters
    //rl_filename_quote_characters = " ";
    //rl_completer_quote_characters = " \\\t\n\"'|$*?[]{}&;#()";
    //rl_completer_quote_characters = "\"'";
    //rl_completer_word_break_characters = " \t\n\"'|!&;()=";
    //rl_filename_quote_characters = " \\\t\n\"'|$*?[]{}&;#()";

    rl_bind_key('x'-'a'+1, readline_macro);
}

void pipes_readline_final()
{
}
