<?php
/**
 * phpoot - template engine for php
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @author Haruki Setoyama <haruki@planewave.org>
 * @copyright Copyright &copy; 2003-2004, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @version 0.6- $Id: phpoot.parser.php,v 1.22 2004/05/12 10:52:01 haruki Exp $
 * @link http://phpoot.sourceforge.jp/
 * @package phpoot
 * @subpackage _parser
 */
/**
* attribute name for Variable Tag
*/
@define('PHPOOT_VAR', 'var');
/**
* base class of template parser classes
* This class and its sub classes are accessed only by phpoot_handler class.
* This class is also used as null object.
* @see phpoot_handler
* @package phpoot
* @subpackage _parser
*/
class phpoot_parser {
    /**
     * indent string at the beginning
     * @var string
     */
    var $indent = '';

    /**
     * line feed at the end
     * @var string
     */
    var $linefeed = '';

    /**
     * content data
     * @access private
     * @var string
     */
    var $_content = '';

    /**
     * parse data and compile it into PHP script
     * @return string
     */
    function parse()
    {
        return '';
    }

    /**
     * returns if this content needs indent and linefeed
     * @return string
     */
    function need_indent()
    {
        return false;
    }

    /**
     * returns indent for next contents
     * @return string|null
     */
    function indent_for_next()
    {
        return '';
    }

    /**
     * returns linefeed "\n" for previous content
     * @return string|null
     */
    function linefeed_for_prev()
    {
        return '';
    }

    /**
     * returns indent for close tag
     * @return string|null
     */
    function indent_for_tag()
    {
        return '';
    }

    /**
     * returns linefeed "\n" for open tag
     * @return string|null
     */
    function linefeed_for_tag()
    {
        return '';
    }

    /**
     * returns if it is empty
     * @return string
     */
    function is_empty()
    {
        return false;
    }

    /**
     * returns if this class can have alternative content(s)
     * @return string
     */
    function is_alternative()
    {
        return false;
    }

    /**
     * returns if this content represent format to display
     * @return array|false
     */
    function formatable()
    {
        return false;
    }
}

////////////////////////////////////////

/**
* parser for data
* @package phpoot
*/
class phpoot_parser_data extends phpoot_parser {

    function phpoot_parser_data($data)
    {
        $this->_content = $data;
    }

    function parse()
    {
        return $this->_content;
    }

    function indent_for_next()
    {
        if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->_content, $match)) {
            $this->_content = $match[1];
            return $match[2];
        } else {
            return '';
        }
    }

    function linefeed_for_prev()
    {
        if (preg_match('/^([\040\011]*\n)/sD', $this->_content, $match)) {
            $this->_content = substr($this->_content, strlen($match[1]));
            return $match[1];
        } else {
            return '';
        }
    }
/*
    function indent_for_tag()
    {
        if (preg_match('/^(.*)(\n[\040\011]*)$/sD', $this->_content, $match)) {
            $this->_content = $match[1];
            return $match[2];
        } else {
            return '';
        }
    }

    function linefeed_for_tag()
    {
        if (preg_match('/^([\040\011]*\n[\040\011]*)/sD', $this->_content, $match)) {
            $this->_content = substr($this->_content, strlen($match[1]));
            return $match[1];
        } else {
            return '';
        }
    }
    */

    function is_empty()
    {
        return ($this->_content == '');
    }

    function formatable()
    {
        $data = trim($this->_content);
        if (preg_match('/^\{([a-z_]+)\s+"([^"]*)"\}$/D', $data, $match)
        && file_exists(PHPOOT_FORMAT . $match[1]. '.php')) {
            return array($match[1], $match[2]);
        }
        return false;
    }
}

/**
* parser for escape
* @package phpoot
*/
class phpoot_parser_escape extends phpoot_parser {

    function phpoot_parser_escape($data)
    {
        $this->_content = $data;
    }

    function parse()
    {
        return '<!' . $this->_content . '>';
    }
}

/**
* parser for pi
* @package phpoot
*/
class phpoot_parser_pi extends phpoot_parser {

    var $_target;

    function phpoot_parser_pi($target,$data)
    {
        $this->_content = $data;
        $this->_target = $target;
    }

    function parse()
    {
        return '<?' . $this->_target . ' ' . $this->_content . '?>';
    }
}

/**
* parser for jasp
* @package phpoot
*/
class phpoot_parser_jasp extends phpoot_parser {

    function phpoot_parser_jasp($data)
    {
        $this->_content = $data;
    }


    function parse()
    {
        return '<%' . $this->_content . '%>';
    }
}

/**
* parser for removed data
* @package phpoot
*/
class phpoot_parser_remove extends phpoot_parser {

    function need_indent()
    {
        return true;
    }
}

/**
* parser for empty data
* @package phpoot
*/
class phpoot_parser_empty extends phpoot_parser_remove {

    function is_empty()
    {
        return true;
    }
}

////////////////////////////////////////

/**
 * Base of parser for tag
 * @package phpoot
 * @abstract
 */
class phpoot_parser_tag extends phpoot_parser {
    /**
     * data in this tag
     * @var array of phpoot_parser
     */
    var $data = '';

    /**
     * information about variable tag
     * @var array
     */
    var $info;

    /**
     * options
     * @var array
     */
    var $option;

    /**
     * common information in phpoot_parser_tag of all level
     */
    var $common;

    /**
     * name of open tag
     * @access private
     */
    var $_open_tag;
    /**
     * attribute in the open tag
     * @var array
     * @access private
     */
    var $_attrs;
    /**
     * if this tag can have data or not
     * @var bool
     * @access private
     */
    var $_empty_tag = false;
    /**
     * position when tag is opend
     * @access private
     */
    var $_position;
    /**
     * tags that will be deleted when no attribute
     * @var array
     */
    var $_delete = array('span' => 1);
    /**
     * if content is to be formated or not
     * @access private
     */
    var $_format = null;
    /**
     * @access private
     */
    var $_linefeed_open_tag = '', $_indent_close_tag = '';
    /**
     * @access private
     */
    var $_closed = false;

    /**
     * judges if this class accept the tag
     * @abstract
     * @access public
     * @return bool
     */
    function accept($open_tag, $attrs)
    {
        return false;
    }

    /**
     * initialize when the tag is opened
     * @abstract
     */
    function open($position, $name, $attrs, &$info) {}

    /**
     * common initializations
     * @access private
     */
    function _open($position, $name, $attrs)
    {
        $this->_position = $position;
        $this->_open_tag = $name;
        $this->_attrs = $attrs;

        if (isset($this->option['empty'][$name])) {
            $this->_empty_tag = true;
        }

        $this->data = array();
        $this->data[] = new phpoot_parser; // null obj
    }

    /**
     * parse all the data in $this->data stack
     * and set its compiled PHP script into $this->_content.
     * $this->_format
     * @access public
     */
    function close($position)
    {
        if ($this->_closed) {
            exit('close again!'); // for debug.
        }

        if ($this->_position == $position) {
            $this->_empty_tag = true;
        }

        switch(count($this->data)){
        case 1:
            $this->_content = '';
            break;
        case 2:
            if (($format = $this->data[1]->formatable()) != false) {
                $this->_format = $format;
            }
            $this->_indent_close_tag = $this->data[1]->indent_for_next();
            $this->_content = $this->data[1]->parse();
            break;
        default:
            $this->data[] = new phpoot_parser;
            $num = count($this->data) -1;

            // get indent and linefeed for pritty print
            $this->_indent_close_tag = $this->data[$num-1]->indent_for_next();
            for ($i = $num-1; $i > 0; $i--) {
                if ($this->data[$i]->need_indent()) {
                    $this->data[$i]->indent = $this->data[$i -1]->indent_for_next();
                    $this->data[$i]->linefeed = $this->data[$i +1]->linefeed_for_prev();
                }
            }
            $this->_linefeed_open_tag = $this->data[1]->linefeed_for_prev();

            // altarnative & parse
            $this->_content = '';
            for ($i=1; $i < $num; $i++) {
                if ($this->data[$i]->is_alternative() == true
                && $this->data[$i+1]->is_empty()) {
                    $alts = $this->_get_alternative($i, $num, $this->data[$i]);
                    if ($alts > 0) {
                        $this->_content .= $this->_parse_alternative($i, $alts);
                        $i += $alts*2;
                        continue;
                    }
                }
                $this->_content .= $this->data[$i]->parse();
            }

            $this->_closed = true; // for debug.
        }
    }

    /**
     * get alternative tags
     * @access private
     * @return int  the number of alternatives
     */
    function _get_alternative($i, $num, &$phpoot_parser) {
        $open_tag = $phpoot_parser->open_tag();
        $variable = $phpoot_parser->info['name'];
        $alts = 0;
        $i += 2;
        while($i < $num){
            if ($this->data[$i]->is_alternative() == true
            && $this->data[$i]->open_tag() == $open_tag
            && $this->data[$i]->info['name'] == $variable) {
                $alts += 1;
                if (! $this->data[$i+1]->is_empty()) {
                    return $alts;
                }
                $i += 2;
            } else {
                return $alts;
            }
        } // while
        return $alts;
    }

    /**
     * parse the tag with alternative
     * @access private
     * @return string
     */
    function _parse_alternative($i, $alts)
    {
        $alts += 1;

        $level      = $this->data[$i]->info['level'];
        $_var       = $this->data[$i]->info['name'];
        $_each      = '$_each' . $level;
        $_valprev   = '$_val'  . ($level -1);
        $_valcurr   = '$_val'  . $level;
        $_num       = '$_num'  . $level;

        $parsed
            = '<?php '
            . "$_each = phpoot_each($_valprev, '$_var'); "
            . "$_num = 0; "
            . "foreach ($_each as $_valcurr) { ";

        for ($j=0; $j < $alts; $j++) {
            $parsed
                .= "if ($_num%$alts == $j) { "
                . $this->data[$i+$j*2]->parse_in_loop()
                . "continue; } ";
        }

        $parsed
            .= "} ?>\n";

        return $parsed;
    }

    /**
     * returns name of open tag
     * @return string
     */
    function open_tag()
    {
        return $this->_open_tag;
    }

    /**
     * returns position of open tag
     * @return int
     */
    function open_position()
    {
        return $this->_position;
    }

    /**
     * parse tag and set $this->_parsed_close_tag and $this->_parsed_open_tag
     * @access private
     */
    function _parse_tag()
    {
        // open tag
        $open_tag = '<' . $this->_open_tag;
        $attr_variables = array();
        $attr_statics = 0;
        $_val = '$_val' . ($this->info['level'] -1);

        foreach ($this->_attrs as $key => $value) {
            if ($key == PHPOOT_VAR && !is_string($value)) {
                $_val = '$_val' . $this->info['level'];
                continue;
            }

            if (preg_match('/^\{([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)\}$/D', $value, $match)) {
            // variable attribute, bar="{foo}" style
                $_varibale = $match[1];
                $open_tag .= "<?php echo phpoot_attr('$key', $_val, '$_varibale'); ?>\n";
                $attr_variables[] = "phpoot_attr('$key', $_val, '$_varibale')";
            }
            else {
            // static attribute
                $open_tag .= ' ' . $key . '=' . $this->_double_quote_html($value);
                $attr_statics++;
            }
        }

        if ($this->_empty_tag == true) {
            $open_tag  .= ' />';
        } else {
            $open_tag  .= '>';
        }

        // close tag
        $close_tag = $this->_indent_close_tag();
        if ($this->_empty_tag == true) {
            $close_tag  = '';
        } else {
            $close_tag .= '</' . $this->_open_tag . '>';
        }

        // deletable tag?
        if ($attr_statics == 0 && isset($this->_delete[$this->_open_tag])) {
            if (empty($attr_variables)) {
                $open_tag = (strlen($this->_linefeed_open_tag) > 0)
                    ? ''
                    : $this->indent;
            } else {
                $_condition = implode(" != '' || ", $attr_variables) . " != ''";
                if (strlen($this->_linefeed_open_tag) > 0) {
                    $open_tag = "<?php if ($_condition) { ?>\n" . $this->indent
                        . $open_tag . $this->_linefeed_open_tag . "<?php } ?>\n";
                } else {
                    $open_tag = $this->indent . "<?php if ($_condition) { ?>\n"
                        . $open_tag . "<?php } ?>\n";
                }
            }
            if (empty($attr_variables) || $close_tag == '') {
                $close_tag = (strlen($this->_indent_close_tag) > 0)
                    ? ''
                    : $this->linefeed;
            } else {
                if (strlen($$this->_indent_close_tag) > 0) {
                    $close_tag = "<?php if ($_condition) { ?>\n"
                        . $this->_indent_close_tag . $close_tag . $this->linefeed . "<?php } ?>\n";
                } else {
                    $close_tag = "<?php if ($_condition) { ?>\n"
                        . $close_tag . "<?php } ?>\n" . $this->linefeed;
                }
            }
        }
        else {
            $open_tag = $this->indent . $open_tag . $this->_linefeed_open_tag;
            $close_tag = $this->_indent_close_tag . $close_tag . $this->linefeed;
        }

        return array($open_tag, $close_tag);
    }

    /**
     * get indent for close tag
     * @access private
     * @return string
     * @see _parse_tag()
     */
    function _indent_close_tag() {
        // for static tag
        return '';
    }

    /**
     * double quote for html attribute
     * @access private
     * @return string
     */
    function _double_quote_html($str)
    {
        return '"' . str_replace(array('"', '<', '>'), array('&quot', '&lt;', '&gt;'), $str) . '"';
    }

    /**
     * single quote for php
     * @access private
     * @return string
     */
    function _single_quote($str)
    {
        return '\'' . str_replace('\'', '\\\'', $str) . '\'';
    }

}

/**
 * Parser for root level of the stack
 *
 * @package phpoot
 */
class phpoot_parser_tag_root extends phpoot_parser_tag {

    /**
     * information of variable tag in each level
     */
    var $info = array('level' => 0, 'nests' => false);

    function phpoot_parser_tag_root()
    {
        $this->data = array();
        $this->data[] = new phpoot_parser; // null obj
    }

    function close($position)
    {
        // do nothing. Infact, this is not called.
        return;
    }

    function parse()
    {
        parent::close(0);
        foreach($this->common['format'] as $type) {
            $this->_content =
                "<?php require_once PHPOOT_FORMAT.'$type'.'.php'; ?>\n"
                . $this->_content;
        }
        return $this->_content;
    }
}


/**
 * Parser for tags without variable attribute
 *
 * @package phpoot
 */
class phpoot_parser_tag_static extends phpoot_parser_tag {

    function accept($open_tag, $attrs)
    {
        return true;
    }

    function open($position, $name, $attrs, &$info)
    {
        $this->_open($position, $name, $attrs);
        $this->info =& $info;
    }

    function parse()
    {
        list($opentag, $closetag) = $this->_parse_tag();
        return $opentag . $this->_content . $closetag;
    }
}

/**
 * Parser for variable tag
 * variable tag means the one with var attribute.
 *
 * @package phpoot
 */
class phpoot_parser_tag_variable extends phpoot_parser_tag {
   /**
     * flag for alternation
     * @access private
     */
    var $_alt = 0;

    function accept($open_tag, $attrs)
    {
        if (isset($attrs[PHPOOT_VAR])
        && preg_match('/^\{[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*\}\+?$/D', $attrs[PHPOOT_VAR])) {
            return true;
        }
        return false;
    }

    function open($position, $name, $attrs, &$info)
    {
        $this->info['level'] = (int)$info['level'] +1;
        $this->info['nests'] = false;
        $info['nests'] = true;
        if (preg_match('/^\{([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)\}(\+?)$/D', $attrs[PHPOOT_VAR], $match)) {
            $this->info['name'] = $match[1];
            $this->_alt = strlen($match[2]);
        }

        $this->_open($position, $name, $attrs);

        $this->_attrs[PHPOOT_VAR] = true;
    }

    function parse()
    {
        $_var       = $this->info['name'];
        $_each      = '$_each' . $this->info['level'];
        $_valprev   = '$_val'  . ($this->info['level'] -1);
        $_valcurr   = '$_val'  . $this->info['level'];
        $_num       = '$_num'  . $this->info['level'];

        $parsed
            = '<?php '
            . "$_each = phpoot_each($_valprev, '$_var'); "
            . "$_num = 0; "
            . "foreach ($_each as $_valcurr) { "
            . $this->parse_in_loop()
            . "} ?>\n";

        return $parsed;
    }

    function parse_in_loop() {
        list($open_tag, $close_tag) = $this->_parse_tag();

        $_var       = $this->info['name'];
        $_each      = '$_each' . $this->info['level'];
        $_valcurr   = '$_val'  . $this->info['level'];
        $_num       = '$_num'  . $this->info['level'];

        $parsed
            = "if ($_valcurr === null || $_valcurr === false) { "
            . "if (!\$_quiet) { ?>\n"
            . $this->indent
            . "<!-- NO DATA for '$_var' -->"
            . $this->linefeed
            . "<?php } "
            . "} else { $_num++; ";

        if (! $this->info['nests']) {
            $parsed
                .= "?>\n"
                .  $open_tag;

            if (! isset($this->_format)) {
            // normal data
                $_func = 'phpoot_string';
                foreach ($this->option['raw'] as $ext) {
                    if (substr($_var, -strlen($ext)) == $ext) {
                        $_func = 'phpoot_raw';
                    }
                }

                $parsed
                    .= "<?php if ($_valcurr === true) { ?>\n"
                    .  $this->_content
                    .  "<?php } else { $_func($_valcurr, '$_var', \$_quiet); } ?>\n"
                    .  $close_tag;
            } else {
            // data to be formated
                $_func = 'phpoot_format_' . $this->_format[0];
                $_format = $this->_single_quote($this->_format[1]);
                $parsed
                    .= "<?php $_func($_valcurr, '$_var', $_format, \$_quiet); ?>\n"
                    .  $close_tag;
                $this->common['format'][$this->_format[0]] = $this->_format[0];
            }
        } else {
            $parsed
                .= "phpoot_to_array($_valcurr, '$_var'); ?>\n"
                 . $open_tag
                 . $this->_content
                 . $close_tag;
        }

        $parsed
            .= '<?php } ';

        return $parsed;
    }

    function _indent_close_tag()
    {
        // indent before closetag
        if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->_content, $match)
        && substr($match[1], -3) != ' ?>') {
            $this->_content = $match[1];
            return $match[2];
        } else {
            return '';
        }
    }

    function need_indent()
    {
        return true;
    }

    function is_alternative()
    {
        return (bool)$this->_alt;
    }
}

/**
 * Parser for tags with '#num' of attribute for variable
 *
 * @package phpoot
 */
class phpoot_parser_tag_num extends phpoot_parser_tag_static {
    function accept($open_tag, $attrs)
    {
        if (@$attrs[PHPOOT_VAR] == '{#num}') {
                return true;
        }
        return false;
    }

    function open($position, $name, $attrs, &$info)
    {
        parent::open($position, $name, $attrs, $info);
        unset($this->_attrs[PHPOOT_VAR]);
    }

    function parse()
    {
        list($open_tag, $close_tag) = $this->_parse_tag();

        $_num = '$_num' . $this->info['level'];
        return $open_tag . "<?php echo $_num; ?>\n" . $close_tag;
    }
}

/**
 * Parser for tags with '#num' of attribute for variable
 *
 * @package phpoot
 */
class phpoot_parser_tag_notop extends phpoot_parser_tag_static {
    function accept($open_tag, $attrs)
    {
        if (@$attrs[PHPOOT_VAR] == '{#notop}') {
                return true;
        }
        return false;
    }

    function open($position, $name, $attrs, &$info)
    {
        parent::open($position, $name, $attrs, $info);
        unset($this->_attrs[PHPOOT_VAR]);
    }

    function parse()
    {
        list($open_tag, $close_tag) = $this->_parse_tag();

        $_num = '$_num' . $this->info['level'];
        $indent = (strlen($this->linefeed) > 0)
            ? ''
            : $this->indent;
        return "<?php if($_num>1) { ?>\n"
            . $open_tag . $this->_content . $close_tag
            . "<?php } else { ?>\n"
            . $indent
            . "<?php } ?>\n";
    }

    function need_indent()
    {
        return true;
    }
}
?>