<?php
/**
 * phpoot - template engine for php
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * @author Haruki Setoyama <haruki@planewave.org> 
 * @copyright Copyright &copy; 2003-2004, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @version 0.3 $Id: phpoot.parser.php,v 1.10 2004/02/26 14:38:54 haruki Exp $
 * @link http://phpoot.sourceforge.jp/
 * @package phpoot
 * @subpackage _parser
 */
/**
* Requires PEAR
*/
require_once 'PEAR.php';
/**
* XML_HTMLSax Package is Required
*/
require_once 'XML/XML_HTMLSax.php';

/**
* attribute name for Variable Tag
*/
@define('PHPOOT_VAR', 'var');
    
/**
* handler for HTMLSax
* This class is accessed only by phpoot class
* @package phpoot
* @subpackage _parser
*/
class phpoot_handler {
    /**
     * template text 
     * @var string
     */
    var $template = '';
	
    /**
     * option 
	 * 'keep_pi'		when false, remove all the <? ... ?>
	 * 'keep_jasp'		when false, remove all the <% ... %>
	 * 'empty_tags'
     * @var array
     */
    var $option = array(
			'empty_tags' => array(
		        'br' => 1,
		        'param' => 1,
		        'hr' => 1,
		        'input' => 1,
		        'col' => 1,
		        'img' => 1,
		        'area' => 1,
		        'frame' => 1,
		        'meta' => 1,
		        'link' => 1,
		        'base' => 1,
		        'basefont' => 1,
		        'isindex' => 1
		    )
		);
	
	/**
     * private parameters
     * @access private 
     */
    var $raw_parsers = array();
    var $level = 0;
    var $parser_stack = array();
    var $error = array();
	var $open_position = 0;
	var $data_prefetch = 0;

    /**
     * constructor
     * @access public 
     */
    function phpoot_handler($parsers = array('num', 'alt', 'var'))
    {
        $this->raw_parsers = array();
        $this->num_raw_parsers = 0;
        foreach ($parsers as $parser) {
            $class = 'phpoot_parser_' . $parser;
            $this->raw_parsers[$this->num_raw_parsers] = new $class;
            $this->raw_parsers[$this->num_raw_parsers]->error =& $this->error;
            $this->num_raw_parsers++;
        } 
        $this->raw_parsers[$this->num_raw_parsers] = new phpoot_parser_general;
        $this->raw_parsers[$this->num_raw_parsers]->error =& $this->error;
        $this->num_raw_parsers++;
    }
	
	/**
     * set option by array
     * @access public 
     */
	function setOption($option)
	{
		$this->option = array_merge($this->option, $option);
	}

    /**
     * initialize before HTMLSax parse
     * @access public 
     */ 
    function init(&$template)
    {
        $this->level = 0;
        $this->error = array();
		$this->template =& $template;
		
		$this->parser_stack = array();
        $this->parser_stack[0] = new phpoot_parser_root;
        $this->parser_stack[0]->error =& $this->error;
        $this->parser_stack[0]->level = 0;
    }

    /**
     * set handler on HTMLSax parser
     * @access public 
     */     
    function configParser(&$parser)
    {
        $parser->set_element_handler('openHandler', 'closeHandler');
        $parser->set_data_handler('dataHandler');
        $parser->set_escape_handler('escapeHandler');
        $parser->set_pi_handler('piHandler');
        $parser->set_jasp_handler('jaspHandler');
        $parser->set_option('XML_OPTION_FULL_ESCAPES'); 
    } 

    /**
     * get parsed template, that is PHP script 
     * @access public 
     */ 
    function getParsed()
    {
        $this->parser_stack[$this->level]->close(true, false);
		
		if (empty($this->error)) {
			return $this->parser_stack[0]->data;
		} else {
		    $errmsg = "<?php \$_no_error = false; if (\$_error) { echo '<!-- ERROR IN TEMPLATE: \n";
	        foreach ($this->error as $error) {
	            $line = substr_count(substr($this->template, 0, $error[0]), "\n")+1;
	            $errmsg .= str_replace('\'', '\\\'', "Line $line: $error[1] \n");
	        }
			$errmsg .= " -->\n'; } ?>\n";
			
			return $errmsg . $this->parser_stack[0]->data;
		}
    } 
	
    /**
     * handler method for tag open 
     * @access public 
     */
    function openHandler(&$parser, $name, $attrs)
    { 
        $name = strtolower($name);
        $attrs = array_change_key_case($attrs, CASE_LOWER);
		$this->open_position = $parser->get_current_position();
        
        for ($i =0; $i < $this->num_raw_parsers; $i++) {
            if ($this->raw_parsers[$i]->accept($name, $attrs)) {
                $this->level++;
                unset($this->parser_stack[$this->level]);
                $this->parser_stack[$this->level] 				= $this->raw_parsers[$i];
				$this->parser_stack[$this->level]->prev 		=& $this->parser_stack[$this->level -1];
				
                $this->parser_stack[$this->level]->open_tag 	= $name;
				$this->parser_stack[$this->level]->empty_tag 	= isset($this->option['empty_tags'][$name]);
                $this->parser_stack[$this->level]->attrs 		= $attrs;
                $this->parser_stack[$this->level]->level 		= $this->level;
                $this->parser_stack[$this->level]->position 	= $this->open_position;
                $this->parser_stack[$this->level]->open();
                return;
            } 
        } 
    } 

    /**
     * handler method for tag close 
     * @access public 
     */
    function closeHandler(&$parser, $close_tag)
    {
        $close_tag = strtolower($close_tag);
		
		if ($parser->get_current_position() == $this->open_position) {
		// for <br /> style tag
		    $this->parser_stack[$this->level]->empty_tag = true;
		}
		
		if ($this->parser_stack[$this->level]->open_tag != $close_tag) {
			$err_msg = 'tag <'.$this->parser_stack[$this->level]->open_tag.'> unclosed. ';
			$this->error[] = array($this->open_position, $err_msg);
		    
			$indent_real_close_tag = '';
			if (preg_match('/^(.*)(\n[\040\011]*)$/sD', $this->parser_stack[$this->level]->data, $match) 
	        && substr($match[1], -3) != ' ?>') {
			// "\n" + indent
	            $this->parser_stack[$this->level]->data = $match[1];
	            $indent_real_close_tag = $match[2];
	        }
			if ($this->parser_stack[$this->level]->close($close_tag, false)) {
				$this->level--;
				$this->parser_stack[$this->level]->data .= $indent_real_close_tag;
				$this->closeHandler($parser, $close_tag);
			} else {
				$this->parser_stack[$this->level]->data .= $indent_real_close_tag;
			}
		} else {
			// prefetch for "\n" after the close tag
			$line_feed = $this->_prefetch_linefeed($parser->get_current_position());
			if ($this->parser_stack[$this->level]->close($close_tag, $line_feed)) {
				$this->level--;
			}
		}
    } 

    /**
     * handler method for data 
     * @access public 
     */
    function dataHandler(&$parser, $data)
    {
		if ($this->data_prefetch > 0) {
		    $data = substr($data, $this->data_prefetch);
			$this->data_prefetch = 0;
		}
        $this->parser_stack[$this->level]->data($data);
    } 

    /**
     * handler method for escape 
     * @access public 
     */
    function escapeHandler(&$parser, $data)
    {
        $this->parser_stack[$this->level]->escape($data);
    } 

    /**
     * handler method for pi 
     * @access public 
     */
    function piHandler(&$parser, $target, $data)
    {
		if ($this->option['keep_pi'] == true) {
			$this->parser_stack[$this->level]->pi($target, $data);
		} else {
            $this->error[] = array($parser->get_current_position(), 'Pi removed. ');
            $this->parser_stack[$this->level]->data .= "<?php if (\$_error) { echo '<!-- Pi removed -->'; } ?>\n";
        }
    } 

    /**
     * handler method for jasp 
     * @access public 
     */
    function jaspHandler(&$parser, $data)
    {
		if ($this->option['keep_jasp'] == true) {
			$this->parser_stack[$this->level]->jasp($data);
		} else {
            $this->error[] = array($parser->get_current_position(), 'Jasp removed. ');
            $this->parser_stack[$this->level]->data .= "<?php if (\$_error) { echo '<!-- Jasp removed -->'; } ?>\n";
        }
    } 
	
	/**
     * prefetch linefeed 
     * @access private 
     */
	function _prefetch_linefeed($position)
	{
		if (substr($this->template, $position, 1) == "\n") {
			$this->data_prefetch = 1; 
			return true; 
		} else {
			return false;
		}
	}
} 

/**
 * Base of Tag Parser
 * This class is accessed only by phpoot_handler class
 * @see phpoot_handler
 * @package phpoot
 * @subpackage _parser
 * @abstract 
 */
class phpoot_parser {

    /**
     * PHP script compiled 
     * @var string
     */
    var $data = '';
    
    /**
     * nest level of tag 
     * @var int
     */
    var $level;
    
    /**
     * attribute in the open tag 
     * @var array
     */
    var $attrs;
    
    /**
     * name of open tag 
     * @var string
     */
    var $open_tag;
	
    /**
     * name of open tag 
     * @var string
     */
    var $close_tag;
    
    /**
     * position when open() is called 
     * @var int
     */
    var $position;
	
    /**
     * if the tag is 'empty tag' like <br /> 
     * @var bool
     */
    var $empty_tag = false;
	
    /**
     * "\n" after close tag 
     * @var bool
     */
    var $lf = false;
    
    /**
     * informations of the variable 
     * @var array
     */
    var $var_info;
    
    /**
     * phpoot_parser object at previous nest level of tag 
     * @var phpoot_parser
     */
    var $prev;
    
    /**
     * error massages. reference to phpoot_handler->error 
     * @var string
     */
    var $error;
	
    /**
     * @access private 
     */
    var $_rws;

    /**
     * judge if this class accept the tag
     * @abstract 
     * @access public
     * @return bool
     */
    function accept($open_tag, $attrs) 
    {
        return false;
    }

    /**
     * first part of the parser called when tha tag open
     * @abstract
     * @access public 
     */
    function open() {} 

    /**
     * last part of the parser called when tha tag close
     * @abstract 
     * @access public
     * @return int nest level of tags
     */    
    function close($close_tag, $line_feed) {} 

    /**
     * parses data
     * @access public 
     */
    function data($data)
    {
        $this->data .= $data;
    } 

    /**
     * parses escape
     * @access public 
     */
    function escape($data)
    {
        $this->data .= '<!'.$data.'>';
    } 

    /**
     * parses pi , that is <? ... ?>
     * @access public 
     */
    function pi($target, $data)
    {
        $this->data .= '<?'.$target.' '.$data.'?>';
    } 

    /**
     * parses jasp , that is <% ... %>
     * @access public 
     */
    function jasp($data, $position)
    {
        $this->data .= '<%'.$data.'%>';
    } 
        
} 

/**
 * Parser for root level
 * 
 * @package phpoot
 */
class phpoot_parser_root extends phpoot_parser {

    function close($close_tag, $line_feed)
    {
        if ($close_tag !== true) {
            $this->error[] = array($position, "tag </$close_tag> unopend. ");
        }
        return false;
    }
}

/**
 * Parser for tags without id attribute
 * 
 * @package phpoot 
 */
class phpoot_parser_general extends phpoot_parser {

    /**
     * @access private 
     */
    var $_parsed_open_tag;
    
    /**
     * @access private 
     */
    var $_parsed_close_tag;
	
	/**
     * @access private 
     */
    var $_indent_open_tag;
	
	/**
     * @access private 
     */
    var $_linefeed_close_tag;
	
	/**
	 * tags if no attribute is
     * @var array 
     */
	var $delete = array('span' => 1);

    function accept($open_tag, $attrs)
    {
        return true;
    } 

    function open()
    {
        $this->var_info =& $this->prev->var_info;
    } 

    function close($close_tag, $line_feed)
    {
        $this->_parse_tag($line_feed, false);
        $this->prev->data 
			.= $this->_parsed_open_tag 
			.  $this->data 
			.  $this->_parsed_close_tag;
        return true;
    } 

	function _indent_open_tag() {
		// indent before opentag
        if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->prev->data, $match) 
        && substr($match[1], -3) != ' ?>') {
            $this->prev->data = $match[1];
            return $match[2];
        } else {
            return '';
        }
	}
	
	function _indent_close_tag() {
		// indent before closetag
		if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->data, $match) 
        && substr($match[1], -3) != ' ?>') {
            $this->data = $match[1];
            return $match[2];
        } else {
            return '';
        }
	}
	
    /**
     * parse tag and set $this->_parsed_close_tag and $this->_parsed_open_tag
     * @access private 
     */ 
    function _parse_tag($line_feed =false, $loop =false)
    {
		// indent befor open tag
		$this->_indent_open_tag = ($line_feed && $loop) ? $this->_indent_open_tag() : '';
		
		// linefeed after close tag
		$this->_linefeed_close_tag = ($line_feed) ? "\n" : '';
		
		// open tag
        $this->_parsed_open_tag = '<' . $this->open_tag;
        $attr_variables = array();
        $attr_statics = 0;
        $_val = '$_val' . (int)$this->prev->var_info['level'];
		
        foreach ($this->attrs as $key => $val) {
            if ($key == PHPOOT_VAR 
			&& !is_string($val)) {
                $_val = '$_val' . (int)$this->var_info['level'];
                continue;
            }
            
            if (preg_match('/^\{([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)\}$/D', $val, $match)) {
			// variable attribute, var="{foo}" style
                $_valcurr = '$_val' . (int)$this->var_info['level'];
                $_var = $match[1];
                $this->_parsed_open_tag .= "<?php echo phpoot_attr('$key', $_val, '$_var'); ?>\n";
                $attr_variables[] = "phpoot_attr('$key', $_val, '$_var')";
            } 
            else {
			// static attribute
                $this->_parsed_open_tag .= ' ' . $key . '=' . $this->_double_quote_html($val);
                $attr_statics++;
            } 
        }
		
        if ($this->empty_tag == true) {
            $this->_parsed_open_tag  .= ' />';
        } else {
            $this->_parsed_open_tag  .= '>';
        }
		
		// close tag
		$this->_parsed_close_tag = $this->_indent_close_tag();
        if ($this->empty_tag == true) {
            $this->_parsed_close_tag  = '';
        } else {
            $this->_parsed_close_tag .= '</' . $this->open_tag . '>';
        } 
		
		// deletable tag?
        if ($attr_statics == 0 && isset($this->delete[$this->open_tag])) {
            if (empty($attr_variables)) {
                $this->_parsed_open_tag  = '';
                $this->_parsed_close_tag = '';
            } else {
                $_condition = implode(" != '' || ", $attr_variables) . " != ''";
                $this->_parsed_open_tag = "<?php if ($_condition) { ?>\n" . $this->_parsed_open_tag . "<?php } ?>\n";
                if ($this->_parsed_close_tag != '') {
					$this->_parsed_close_tag = "<?php if ($_condition) { ?>\n" . $this->_parsed_close_tag . "<?php } ?>\n";
            	}
			} 
        }
		
		//
		$this->_parsed_open_tag   = $this->_indent_open_tag . $this->_parsed_open_tag;
		$this->_parsed_close_tag .= $this->_linefeed_close_tag;
    } 

    /**
     * double quote for html attribute
     * @access private 
     * @return string
     */
    function _double_quote_html($str)
    {
        return '"' . str_replace(array('"', '<', '>'), array('&quot', '&lt;', '&gt;'), $str) . '"';
    } 
} 

/**
 * Parser for variable tag
 * variable tag means the one with var attribute.
 * 
 * @package phpoot
 */
class phpoot_parser_var extends phpoot_parser_general {

    function accept($open_tag, $attrs)
    {
        if (! isset($attrs[PHPOOT_VAR])) {
            return false;
        } 
        if (! preg_match('/^[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*$/D', $attrs[PHPOOT_VAR])) {
            return false;
        } 
        return true;
    } 

    function open()
    {
        $this->var_info['level'] = (int)$this->prev->var_info['level'] +1;
        $this->var_info['nests'] = false;
        $this->prev->var_info['nests'] = true;
        $this->var_info['callback'] = array();
        $this->var_info['name'] = $this->attrs[PHPOOT_VAR];
        $this->attrs[PHPOOT_VAR] = true;
    } 
    
    function close($close_tag, $line_feed)
    {
        $this->_parse_tag($line_feed, true);

        $_var 		= $this->var_info['name'];
        $_each 		= '$_each' . (int)$this->var_info['level'];
        $_valcurr 	= '$_val'  . (int)$this->var_info['level'];
        $_valprev 	= '$_val'  . (int)$this->prev->var_info['level'];
        $_attr 		= '$_attr' . (int)$this->var_info['level'];

        $this->prev->data 
			.= "<?php $_each = phpoot_each($_valprev, '$_var'); "
			.  $this->_before()
			.  "foreach ($_each as $_valcurr) { "
			.  "if ($_valcurr === null || $_valcurr === false) { "
			.  "if (\$_error) { ?>\n"
			.  $this->_indent_open_tag
			.  "<!-- NO DATA for '$_var' -->"
			.  $this->_linefeed_close_tag
			.  "<?php } " 
			.  "} else { ";
        
        if (! $this->var_info['nests']) {
			$this->prev->data 
                .= "?>\n" 
				.  $this->_parsed_open_tag;
				
            if (substr($_var, 4) == '_raw') {
                $_func = 'phpoot_raw';
            } else {
                $_func = 'phpoot_string';
            }
        
            $this->prev->data
                .= "<?php if ($_valcurr === true) { ?>\n"
                .  $this->data
                .  "<?php } else { $_func($_valcurr, '$_var', \$_error); } ?>\n"
            	.  $this->_parsed_close_tag;
        } else {
            $this->prev->data 
                .= "phpoot_to_array($_valcurr, '$_var'); ?>\n"
                 . $this->_parsed_open_tag
                 . $this->data
                 . $this->_parsed_close_tag;
        } 
        
        $this->prev->data 
            .= '<?php ' . $this->_in() . '} } '
            .  $this->_after()
            .  "?>\n";
			
		return true;
    } 
    
    /**
     * calls _before() method of callback classes
     * @access private 
     * @return string
     */
    function _before()
    {
        $ret = '';
        foreach ($this->var_info['callback'] as $cb) {
            $ret .= $cb->_before();
        }
        return $ret;
    }

    /**
     * calls _in() method of callback classes
     * @access private 
     * @return string
     */
    function _in()
    {
        $ret = '';
        foreach ($this->var_info['callback'] as $cb) {
            $ret .= $cb->_in();
        }
        return $ret;
    }
    
    /**
     * calls _after() method of callback classes
     * @access private 
     * @return string
     */ 
    function _after()
    {
        $ret = '';
        foreach ($this->var_info['callback'] as $cb) {
            $ret .= $cb->_after();
        }
        return $ret;
    }
} 

/**
 * Parser for tags with '#num' id attribute
 * 
 * @package phpoot
 */
class phpoot_parser_num extends phpoot_parser_var {
    function accept($open_tag, $attrs)
    {
        if ($attrs[PHPOOT_VAR] == '#num') {
                return true;
        } 
        return false;
    }
    
    function open()
    {
        parent::open();
        if (! isset($this->prev->var_info['num'])) {
            $this->prev->var_info['num'] = true;
            $cnt = (int)count($this->prev->var_info['callback']);
            $this->prev->var_info['callback'][$cnt] = new phpoot_parser_num_cb;
            $this->prev->var_info['callback'][$cnt]->level =& $this->prev->var_info['level'];
        }
    }
    
    function close($close_tag, $line_feed)
    {
        $this->_parse_tag($line_feed, false);
		
        if ($this->var_info['nests']) {
            $this->error[] = array($this->position, "'#num' nest variable tag(s). ");
        }   
    
        $_num = '$_num' . (int)$this->prev->var_info['level'];
        $this->prev->data 
			.= $this->_parsed_open_tag 
            .  "<?php echo $_num; ?>\n"
            .  $this->_parsed_close_tag;
		
		return true;
    }
}

/**
 * base class of Call back class for phpoot_parser_var
 * 
 * @package phpoot
 * @subpackage _parser
 * @abstract
 */
class phpoot_parser_cb {
    function _before() {}
    
    function _in() {}
    
    function _after()  {}
}

/**
 * Call back class for phpoot_parser_num
 * 
 * @package phpoot
 * @see phpoot_parser_num
 */
class phpoot_parser_num_cb extends phpoot_parser_cb {
    
    var $level;
    
    function _before()
    {
        $_num = '$_num' . (int)$this->level;
        return "$_num = 1; ";
    }
    
    function _in()
    {
        $_num = '$_num' . (int)$this->level;
        return "$_num++; ";
    }
    
    function _after()
    {
        $_num = '$_num' . (int)$this->level;
        return "unset($_num); ";
    }
}

/**
 * Parser for '#alt' var attribute
 * 
 * @package phpoot
 */
class phpoot_parser_alt extends phpoot_parser_general {
    var $invalid = false;

    function accept($open_tag, $attrs)
    {
        if ($attrs[PHPOOT_VAR] == '#alt') {
            return true;
        } 
        return false;
    } 

    function open()
    {
        unset($this->attrs[PHPOOT_VAR]);
        parent::open();
        if (! isset($this->prev->var_info['alt'])) {
            $this->prev->var_info['alt'] = 1;
            $this->prev->var_info['alt_open'] = true;
            $cnt = (int)count($this->prev->var_info['callback']);
            $this->prev->var_info['callback'][$cnt] = new phpoot_parser_alt_cb;
            $this->prev->var_info['callback'][$cnt]->var_info =& $this->prev->var_info;
        } else {
            if ($this->prev->var_info['alt_open'] != true) {
                $this->prev->var_info['alt']++;
            } else {
                $this->error[] = array($this->position, "'#alt' nested. ");
                $this->invalid = true;
            }
        }
    } 
    
    function close($close_tag, $line_feed)
    {
		$this->delete = array('span'=>1, 'div'=>1);
        $this->_parse_tag($line_feed, true);
		
        $_alt  = '$_alt' . (int)$this->prev->var_info['level'];
        $_case = (int)$this->prev->var_info['alt']-1;
        $this->prev->data 
			.= "<?php if ($_case == (int)$_alt) { ?>\n"
			.  $this->_parsed_open_tag 
			.  $this->data 
			.  $this->_parsed_close_tag
			.  "<?php } ?>\n";
        
        $this->prev->var_info['alt_open'] = false;
		
		return true;
    } 
}

/**
 * Call back class for phpoot_parser_alt
 * 
 * @package phpoot
 * @see phpoot_parser_alt
 */
class phpoot_parser_alt_cb  extends phpoot_parser_cb {
    
    var $var_info;
    
    function _before()
    {
        $_alt = $this->_alt();
        return "$_alt = 0; ";
    }
    
    function _in()
    {
        $_alt = $this->_alt();
        $_num = $this->var_info['alt'];
        if ($_num > 1) {
            return "$_alt = ($_alt +1)%$_num; ";
        } else {
            return '';
        }
    }
    
    function _after()
    {
        $_alt = $this->_alt();
        return "unset($_alt); ";
    }
	
	function _alt()
	{
		return '$_alt' . (int)$this->var_info['level'];
	}
} 
?>