<?php
// phpoot, PHP Object Orient style Template engine
//
// Copyright (C) Haruki Setoyama <haruki@planewave.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
include_once 'phpoot.parsers.php';
/**
* core parts of the parser for phpoot compiler
*
* @author Haruki Setoyama  <haruki@planewave.org>
* @version $Id: phpoot.parser.core.php,v 1.3 2002/12/23 01:09:00 haruki Exp $
* @package phpoot
**/
/**
* phpoot_parser
*
* @access private
**/
class phpoot_parser extends phpoot_tokenize {

    var $error;
    var $buffer;
    var $manager;

    /**
    * constructor
    * @access public
    **/
    function phpoot_parser() {

    }

    /**
    * main function of parsers.
    * OVERRIDE THIS function in child classes!
    * @access private (abstruct)
    * @param array      one part of array done by _scan_template()
    * @return bool      if this class do not handle the type of the part, this returns false.
    **/
    function parse($scaned) {
        return false;
    }

    /**
    * error handling function
    * @access private
    * @param string $message
    * @return void
    */
    function _put_error($message='') {
        if($message == '')
            $message = $this->error;

        if(PHPOOT_DEBUG) {
            $this->buffer->setText('<b>phpoot error: </b>'.$message.'<br>');
        }
        return;
    }

    /**
    * interface to tokenize a parameter. OVERRIDE THIS IF NEED!
    * @param array $tokenized
    * @return array
    * @access private
    **/
    function _tokenize_parameter($str) {
        return $this->_tokenize_parameter_sub($str);
    }

    /**
    * main part of tokenization of a parameter.
    * @param array $tokenized
    * @return array
    * @access private
    **/
    function _tokenize_parameter_sub($str, $other_tokens='') {
        $tokens[] = array(REGEX_DOUBLE_QUOTED,'double-quoted');
        $tokens[] = array(REGEX_SINGLE_QUOTED,'single-quoted');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_FRONT,'variable');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_BODY.'+','variable-nohead');
        $tokens[] = array(REGEX_NUMBER,'number');
        $tokens[] = array('\s*\(\s*','open-paren');
        $tokens[] = array('\s*\)\s*','close-paren');
        $tokens[] = array('\s*,\s*','comma');
        $tokens[] = array('\s*(?:\+|\-|\*|\/|\^|\.)\s*','operator');

        if(is_array($other_tokens)) {
            foreach($other_tokens as $val) {
                array_push($tokens, $val);
            }
        }
        $tokenized = $this->_tokenize($str, $tokens);
        $ret = array();
        $token_num = count($tokenized);
        $j=0;
        for($i=0; $i < $token_num; $i++) {
            $ret[$j] = $tokenized[$i];
            if(preg_match('/^variable/',$tokenized[$i]['type'])) {
                $vartype = $this->_get_variable_type($tokenized[$i]['value']);
                $ret[$j]['vartype'] = $vartype;

                if($vartype == 'method') {
                    if($i+1 >= $token_num || $tokenized[$i+1]['type'] != 'open-paren') {
                        $j++;
                        $ret[$j] = array('type'=>'open-paren','value'=>'(');
                        $j++;
                        $ret[$j] = array('type'=>'close-paren','value'=>')');
                    }
                }
            }
            $j++;
        }
        return $ret;
    }

    /**
    * interface of scaning a parameter. OVERRIDE THIS IF NEED!
    * @param array $tokenized
    * @return array
    * @access private
    **/
    function _scan_parameter($str) {
        return $this->_scan_parameter_sub($this->_tokenize_parameter($str));
    }

    /**
    * main part of scaning a parameter
    * @param array $tokenized
    * @return array
    * @access private
    **/
    function _scan_parameter_sub($tokenized) {
        $ret = array();
        $token_num = count($tokenized);
        $j=0;
        for($i=0; $i < $token_num; $i++) {
            $ret[$j] = $tokenized[$i];
            if(preg_match('/^variable/',$tokenized[$i]['type'])
                && $this->_get_variable_type($tokenized[$i]['value'])=='method') {

                unset($ret[$j]['value']);
                $ret[$j]['value'][] = $tokenized[$i];
                $ret[$j]['value'][] = array('type'=>'open-paren','value'=>'(');
    
                $i++;
                $parens = 1;
                while($i < $token_num-1 && $parens > 0) {
                    $i++;
                    if($tokenized[$i]['type'] == 'open-paren')
                        $parens++;
                    if($tokenized[$i]['type'] == 'close-paren')
                        $parens--;
                    if($tokenized[$i]['type'] == 'etc' || $tokenized[$i]['type'] == 'variable-nohead') {
                        $this->_put_error('Invalid parameter of method call.');
                        return false;
                    }
                    $ret[$j]['value'][] = $tokenized[$i];
                }
                if($parens != 0) {
                        $this->_put_error('Invalid number of paren.');
                        return false;
                }
            }
            $j++;
        }
        return $ret;
    }

    /**
    * get the type of a variable
    * @param string $name   variable name
    * @return string    'method','property','array' or false
    * @access private
    **/
    function _get_variable_type($name) {
        if(preg_match('/('.REGEX_PHPOOT_VARIABLE_METHOD.')$/', trim($name))) {
            return 'method';
        }
        if(preg_match('/('.REGEX_PHPOOT_VARIABLE_PROPERTY.')$/', trim($name))) {
            return 'property';
        }
        if(preg_match('/('.REGEX_PHPOOT_VARIABLE_ARRAY.')$/', trim($name))) {
            return 'array';
        }
        if(preg_match('/('.REGEX_PHPOOT_VARIABLE_LOOP.')$/', trim($name))) {
            return 'loop';
        }
        return false;
    }
}

/**
* phpoot_parser_variable_manager
*
* This class parse a variable, and manage variables for loop.
* @access public
**/
class phpoot_parser_variable_manager  extends phpoot_parser {

    var $_loop_var = array();
    var $_stack_loop = array();

    /**
    * constructor
    * @param object     phpoot_buffer class
    * @access public
    **/
    function phpoot_parser_variable_manager(&$buffer) {
        $this->buffer =& $buffer;
    }

    /**
    * push a loop tag name
    * @param string $name   variable name
    * @access public
    **/
    function pushLoop($tag) {
        array_push($this->_stack_loop, $tag);
    }

    /**
    * pop a loop tag name
    * @return string    variable name
    * @access public
    **/
    function popLoop() {
        if(count($this->_stack_loop) >0)
            return array_pop($this->_stack_loop);
        else
            return false;
    }

    /**
    * get current loop name
    * @return string    variable name
    * @access public
    **/
    function getCurrentLoop() {
        if(count($this->_stack_loop) >0)
            return $this->_stack_loop[count($this->_stack_loop)-1];
        else
            return false;
    }

    /**
    * push a loop variable to stack
    * @param string $name   variable name
    * @access public
    **/
    function pushLoopVariable($scaned, $varname='') {
        if($scaned['type']!='variable') return false;
        $loop_var = is_array($scaned['value']) ? $scaned['value'][0]['value'] : $scaned['value'];

        if($varname == '') $varname = '$loop[\''.str_replace('$','',$loop_var).'\']';
        array_push($this->_loop_var, array($loop_var, $varname));
    }

    /**
    * pop a loop variable from stack
    * @return string    variable name
    * @access public
    **/
    function popLoopVariable() {
        if(count($this->_loop_var) >0)
            return array_pop($this->_loop_var);
        else
            return false;
    }

    /**
    * get a array of loop variables
    * @return string    variable name
    * @access public
    **/
    function getLoopVariable() {
        if(count($this->_loop_var) >0)
            return $this->_loop_var[count($this->_loop_var)-1];
        else
            return false;
    }

    /**
    * parse a variable
    * @param string $name   variable name
    * @return string
    * @access public
    **/
    function parseVariable($scaned) {
        if($scaned['type']!='variable' && $scaned['type']!='variable-nohead')
            return false;

        if(is_array($scaned['value'])) {
        // object method
            $parsed = '';
            $i=0;
            $flag = false;
            $parens = 0;
            foreach($scaned['value'] as $part) {
            //while($i<count($scaned['value'])) {
                if($flag != false) {
                    if($part['type']=='open-paren')
                        $parens++;
                    elseif($part['type']=='close-paren')
                        $parens--;
                    elseif($parens == 0)
                        $flag = false;
                }
                if($flag != true) {
                    $var = $this->_parse_variable_front($part);
                    if(! $var) return false;
                    if($part['type'] == 'variable'
                            && $this->_get_variable_type($part['value']) == 'method'
                            && !preg_match('/\->'.REGEX_WORD.'$/',$var)) {
                        $flag=true;
                    }
                    $parsed .= $var;
                }

                $i++;
            }
        }
        else{
        // other
            $var = $this->_parse_variable_front($scaned);
            if(! $var) return false;
            $parsed .= $var;
        }
        return $parsed;
    }

    /**
    * parse a loop variable
    * @param string $name   variable name
    * @return string
    * @access public
    **/
    function parseLoopVariable($scaned, $basename='loop') {
        if($scaned['type']!='variable') return false;
        $loop_var = is_array($scaned['value']) ? $scaned['value'][0]['value'] : $scaned['value'];
        if(false != $this->_is_loop_variable($loop_var)) {
            $this->_put_error('This is already used as a loop variable.');
            return false;
        }
        return '$'.$basename.'[\''.str_replace('$','',$loop_var).'\']';
    }

/////////////////////////////
///// private functions /////
/////////////////////////////

    function _is_loop_variable($name) {
        foreach($this->_loop_var as $var) {
            if($name == $var[0]) {
                return $var[1];
            }
        }
        return false;
    }

    function _parse_variable_front($scaned) {
        if($scaned['type']!='variable' && $scaned['type']!='variable-nohead')
            return $scaned['value'];

        $scaned = $this->_scan_variable_front($scaned['value']);
        if(! $scaned) return false;

        $base = '';
        $i = 0;

        if($scaned['type']!='variable-nohead') {

            // handling loop variables
            for($i = count($scaned)-1; $i >= 0; $i--) {
                if(false != ($loop_var = $this->_is_loop_variable($scaned[$i]['joined']))) {
                    $base = $loop_var;
                    break;
                }
            }
            $i++;
            // check first part of variable is correct.
            if($i == 0 && $scaned[$i]['type'] == 'loop') {
                $this->_put_error('A loop variable "'.$scaned[$i]['value'].'" is not declared.');
                return false;
            }
        }

        // parser main part
        if($i < count($scaned)) {
            $lest = $this->_parse_variable_front_raw($scaned, $i);
            if(! $lest) return false;
            $base .= $lest;
        }

        return $base;
    }

    function _parse_variable_front_raw($scaned, $i=0) {
        $ret = '';
        while($i < count($scaned)) {
            switch($scaned[$i]['type']) {
            case 'method':
                $ret .= '->'.$scaned[$i]['value'];
                $i++;
                break 2;

            case 'array':
                $ret .= '[\''.$scaned[$i]['value'].'\']';
                break;

            case 'property':
                $ret .= '->'.$scaned[$i]['value'];
                break;

            case 'head':
                $ret = '$val[\''.substr($scaned[$i]['value'],1).'\']';
                break;

            default:
                $this->_put_error('Invalid variable.(1)');
                return false;
            }
            $i++;
        }

        if($i < count($scaned)) {
            $this->_put_error('Invalid variable.');
            return false;
        }

        return $ret;
    }

    function _scan_variable_front($str) {
        $tokenized = $this->_tokenize_variable_front($str);

        $joined = '';
        for($i=0; $i<count($tokenized); $i++) {
            if($tokenized[$i]['type']=='etc') {
                return false;
            }
            if($i>0 && ($tokenized[$i]['type']=='head' || $tokenized[$i]['type']=='loop')) {
                return false;
            }
            $joined .= $tokenized[$i]['value'];
            $tokenized[$i]['joined'] = $joined;
            $tokenized[$i]['value'] = str_replace(array('.','/','%'), array('','',''), $tokenized[$i]['value']);
        }

        return $tokenized;
    }

    function _tokenize_variable_front($str) {
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_HEAD_ARRAY,'head');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_HEAD_LOOP,'loop');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_METHOD,'method');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_ARRAY,'array');
        $tokens[] = array(REGEX_PHPOOT_VARIABLE_PROPERTY,'property');

        return $this->_tokenize($str, $tokens);
    }

}
?>