/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.facesdevfilter.http;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Map;

import javax.portlet.ActionResponse;
import javax.portlet.PortletMode;
import javax.portlet.PortletModeException;
import javax.portlet.PortletResponse;
import javax.portlet.PortletURL;
import javax.portlet.RenderResponse;
import javax.portlet.WindowState;
import javax.portlet.WindowStateException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

import jp.sf.pal.facesdevfilter.FacesDevConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class HttpServletPortletResponseWrapper extends
        HttpServletResponseWrapper implements PortletResponse, ActionResponse,
        RenderResponse {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(HttpServletPortletResponseWrapper.class);

    private HttpServletRequest request;

    private HttpServletResponse response;

    private String currentContentType;

    private ByteArrayOutputStream outputStream = null;

    private PrintWriter printWriter = null;

    public HttpServletPortletResponseWrapper(HttpServletRequest request,
            HttpServletResponse response) {
        super(response);
        this.request = request;
        this.response = response;
        outputStream = new ByteArrayOutputStream();
    }

    public byte[] getBytes() {
        return outputStream.toByteArray();
    }

    public String getString() {
        try {
            if (printWriter != null) {
                printWriter.flush();
            }
            outputStream.flush();
        } catch (IOException e) {
            log.warn("Could not flush the output stream.", e);
        }

        try {
            return outputStream.toString(getCharacterEncoding());
        } catch (UnsupportedEncodingException e) {
            log.warn("Unsupported encoding: " + getCharacterEncoding(), e);
            return outputStream.toString();
        }
    }

    public PrintWriter getWriter() {
        if (printWriter == null) {
            OutputStreamWriter streamWriter = new OutputStreamWriter(
                    outputStream, Charset.forName(getCharacterEncoding()));
            printWriter = new PrintWriter(streamWriter, true);
        }
        return printWriter;
    }

    public ServletOutputStream getOutputStream() {
        return new ServletOutputStreamWrapper(outputStream);
    }

    public void setContentLength(int contentLength) {
        // nothing
    }

    public void flushBuffer() throws IOException {
        outputStream.flush();
    }

    public void finishResponse() {
        try {
            if (printWriter != null) {
                printWriter.close();
            } else {
                if (outputStream != null) {
                    outputStream.close();
                }
            }
        } catch (IOException e) {
            log.error("I/O exception.", e);
        }
    }

    private class ServletOutputStreamWrapper extends ServletOutputStream {

        private ByteArrayOutputStream outputStream_;

        public ServletOutputStreamWrapper(ByteArrayOutputStream outputStream) {
            this.outputStream_ = outputStream;
        }

        public void write(int b) {
            outputStream_.write(b);
        }

        public void write(byte[] bytes) throws IOException {
            outputStream_.write(bytes);
        }

        public void write(byte[] bytes, int off, int len) {
            outputStream_.write(bytes, off, len);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.ServletResponseWrapper#setContentType(java.lang.String)
     */
    public void setContentType(String contentType) {
        super.setContentType(contentType);
        currentContentType = contentType;
    }

    // PortletResponse

    public void addProperty(String arg0, String arg1) {
        // TODO nothing right now.. in the future, need to put it in session
    }

    public void setProperty(String arg0, String arg1) {
        // TODO nothing right now.. in the future, need to put it in session
    }

    // ActionResponse

    public void setPortletMode(PortletMode mode) throws PortletModeException {
        request.getSession().setAttribute(FacesDevConstants.PORTLET_MODE, mode);
    }

    public void setRenderParameter(String arg0, String arg1) {
        // TODO nothing right now.. in the future, need to put it in session
    }

    public void setRenderParameter(String arg0, String[] arg1) {
        // TODO nothing right now.. in the future, need to put it in session
    }

    public void setRenderParameters(Map arg0) {
        // TODO nothing right now.. in the future, need to put it in session
    }

    public void setWindowState(WindowState state) throws WindowStateException {
        request.getSession()
                .setAttribute(FacesDevConstants.WINDOW_STATE, state);
    }

    // RenderResponse

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#createActionURL()
     */
    public PortletURL createActionURL() {
        // TODO nothing right now.. but it may need to be returned
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#createRenderURL()
     */
    public PortletURL createRenderURL() {
        // TODO nothing right now.. but it may need to be returned
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#getContentType()
     */
    public String getContentType() {
        return currentContentType;
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#getNamespace()
     */
    public String getNamespace() {
        return (String) request.getSession().getAttribute(
                FacesDevConstants.NAMESPACE);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#getPortletOutputStream()
     */
    public OutputStream getPortletOutputStream() throws IOException {
        return response.getOutputStream();
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.RenderResponse#setTitle(java.lang.String)
     */
    public void setTitle(String title) {
        request.setAttribute(FacesDevConstants.PORTLET_TITLE, title);
    }

}
