/*
 * Copyright 2005-2006 Portal Application Laboratory project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.facesdevfilter.services;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.portlet.PortletMode;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import jp.sf.pal.facesdevfilter.portlet.descriptors.PortletNode;
import jp.sf.pal.facesdevfilter.portlet.descriptors.PortletInfoNode;
import jp.sf.pal.facesdevfilter.portlet.descriptors.RoleRefNode;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * @author shinsuke
 * 
 */
public class PortletConfigService {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(PortletConfigService.class);

    private Document _doc;

    private List portletList;

    private PortletConfigService() {
    }

    public PortletConfigService(InputStream ios) {
        _doc = getDocument(ios);
        createPortletDDList();
    }

    private Document getDocument(InputStream ios) {
        boolean validation = false;

        boolean ignoreWhitespace = false;
        boolean ignoreComments = false;
        boolean putCDATAIntoText = false;
        boolean createEntityRefs = false;

        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

        dbf.setValidating(validation);
        dbf.setIgnoringComments(ignoreComments);
        dbf.setIgnoringElementContentWhitespace(ignoreWhitespace);
        dbf.setCoalescing(putCDATAIntoText);
        dbf.setExpandEntityReferences(!createEntityRefs);

        DocumentBuilder db = null;
        Document doc = null;
        try {
            db = dbf.newDocumentBuilder();
            doc = db.parse(ios);
        } catch (ParserConfigurationException e) {
            log.error("Parser Configuration Exception.", e);
        } catch (SAXException e) {
            log.error("SAX Exception.", e);
        } catch (IOException e) {
            log.error("I/O Exception.", e);
        }
        return doc;
    }

    private void createPortletDDList() {
        portletList = new ArrayList();
        NodeList portlets = _doc.getElementsByTagName("portlet");
        Element portlet = null;
        NodeList roleRefs = null;
        ArrayList refList = null;
        Element ref = null;
        if (portlets != null) {
            for (int i = 0; i < portlets.getLength(); i++) {
                PortletNode portletEntry = new PortletNode();
                portlet = (Element) portlets.item(i);

                // portlet-name is required
                if (portlet.getElementsByTagName("portlet-name").item(0) != null) {
                    String name = portlet.getElementsByTagName("portlet-name")
                            .item(0).getChildNodes().item(0).getNodeValue();
                    if (name == null) {
                        IllegalStateException e = new IllegalStateException(
                                "The portlet-name element is required in portlet.xml");
                        log.error(e);
                        throw e;
                    }
                    portletEntry.setPortletName(name);
                }

                // portlet-class is required
                if (portlet.getElementsByTagName("portlet-class").item(0) != null) {
                    String className = portlet.getElementsByTagName(
                            "portlet-class").item(0).getChildNodes().item(0)
                            .getNodeValue();
                    if (className == null) {
                        IllegalStateException e = new IllegalStateException(
                                "The portlet-class element is required in portlet.xml");
                        log.error(e);
                        throw e;
                    }
                    portletEntry.setPortletClass(className);
                }

                // init-param
                NodeList initParams = portlet
                        .getElementsByTagName("init-param");
                Map initParamMap = new HashMap();
                if (initParams != null) {
                    for (int j = 0; j < initParams.getLength(); j++) {
                        Element initParam = (Element) initParams.item(j);
                        if (initParam.getElementsByTagName("name") != null
                                && initParam.getElementsByTagName("value") != null) {
                            String n = initParam.getElementsByTagName("name")
                                    .item(0).getChildNodes().item(0)
                                    .getNodeValue();
                            String v = initParam.getElementsByTagName("value")
                                    .item(0).getChildNodes().item(0)
                                    .getNodeValue();
                            if (n != null && v != null) {
                                initParamMap.put(n, v);
                            }
                        }
                    }
                }
                portletEntry.setInitParams(initParamMap);

                // supports
                if (portlet.getElementsByTagName("supports").item(0) != null) {
                    Element supports = (Element) portlet.getElementsByTagName(
                            "supports").item(0);
                    // mime-type
                    ArrayList mimeTypeList = new ArrayList();
                    NodeList mimeTypes = supports
                            .getElementsByTagName("mime-type");
                    if (mimeTypes != null) {
                        for (int j = 0; j < mimeTypes.getLength(); j++) {
                            mimeTypeList.add(mimeTypes.item(j).getChildNodes()
                                    .item(0).getNodeValue());
                        }
                    } else {
                        mimeTypeList.add("text/html");
                    }
                    portletEntry.setMimeTypes(mimeTypeList);

                    // portlet-mode
                    ArrayList portletModeList = new ArrayList();
                    NodeList portletModes = supports
                            .getElementsByTagName("portlet-mode");
                    if (portletModes != null) {
                        for (int j = 0; j < portletModes.getLength(); j++) {
                            portletModeList.add(new PortletMode(portletModes
                                    .item(j).getChildNodes().item(0)
                                    .getNodeValue().toLowerCase()));
                        }
                    } else {
                        portletModeList.add(new PortletMode("view"));
                    }
                    portletEntry.setPortletModes(portletModeList);
                }

                // resource-bundle
                if (portlet.getElementsByTagName("resource-bundle").item(0) != null) {
                    String resourceBundle = portlet.getElementsByTagName(
                            "resource-bundle").item(0).getChildNodes().item(0)
                            .getNodeValue();
                    portletEntry.setResourceBundle(resourceBundle);
                }

                // portlet-info
                if (portlet.getElementsByTagName("portlet-info").item(0) != null) {
                    Element portletInfo = (Element) portlet
                            .getElementsByTagName("portlet-info").item(0);
                    String title = null;
                    String shortTitle = null;
                    String keywords = null;
                    // title
                    if (portletInfo.getElementsByTagName("title").item(0) != null) {
                        title = portletInfo.getElementsByTagName("title").item(
                                0).getChildNodes().item(0).getNodeValue();
                    }
                    // short-title
                    if (portletInfo.getElementsByTagName("short-title").item(0) != null) {
                        shortTitle = portletInfo.getElementsByTagName(
                                "short-title").item(0).getChildNodes().item(0)
                                .getNodeValue();
                    }
                    // keywords
                    if (portletInfo.getElementsByTagName("keywords").item(0) != null) {
                        keywords = portletInfo.getElementsByTagName("keywords")
                                .item(0).getChildNodes().item(0).getNodeValue();
                    }
                    if (title != null || shortTitle != null || keywords != null) {
                        PortletInfoNode portletInfoEntry = new PortletInfoNode();
                        portletInfoEntry.setTitle(title);
                        portletInfoEntry.setShortTitle(shortTitle);
                        portletInfoEntry.setKeywords(keywords);
                        portletEntry.setPortletInfo(portletInfoEntry);
                    }
                }

                // TODO display-name
                // TODO description
                // TODO expiration-cache
                // TODO supported-locale
                // TODO portlet-info
                // TODO portlet-preferences

                // security-role-ref
                roleRefs = portlet.getElementsByTagName("security-role-ref");
                refList = new ArrayList();
                if (roleRefs != null && roleRefs.getLength() != 0) {
                    for (int j = 0; j < roleRefs.getLength(); j++) {
                        ref = (Element) roleRefs.item(j);
                        String refName = ref.getElementsByTagName("role-name")
                                .item(0).getChildNodes().item(0).getNodeValue();
                        // role-link is optional
                        String refLink = null;
                        if (ref.getElementsByTagName("role-link").item(0) != null) {
                            refLink = ref.getElementsByTagName("role-link")
                                    .item(0).getChildNodes().item(0)
                                    .getNodeValue();
                        }
                        refList.add(new RoleRefNode(refName, refLink));
                    }
                }

                portletEntry.setSecurityRoleRefs(refList);
                portletList.add(portletEntry);

            }
        } else {
            IllegalStateException e = new IllegalStateException(
                    "The portlet.xml file does not contain portlet elements.");
            log.error(e);
            throw e;
        }

    }

    public List getPortletList() {
        return portletList;
    }

    public PortletNode getPortletByName(String name) {
        if (name == null) {
            return null;
        }
        Iterator ite = getPortletList().iterator();
        while (ite.hasNext()) {
            PortletNode portlet = (PortletNode) ite.next();
            if (name.equals(portlet.getPortletName())) {
                return portlet;
            }
        }
        return null;
    }

    public PortletNode getDefaultPortlet() {
        if (getPortletList() == null || getPortletList().size() == 0) {
            throw new IllegalStateException(
                    "Could not find a portlet in the portlet descriptor.");
        }
        return (PortletNode) getPortletList().get(0);
    }
}
