/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.jstock.reader.impl;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.cyberneko.html.parsers.DOMParser;
import org.w3c.dom.Node;

import jp.sf.pal.jstock.dto.StockDataDto;
import jp.sf.pal.jstock.exception.JStockRuntimeException;
import jp.sf.pal.jstock.reader.StockDataReader;
import jp.sf.pal.jstock.util.MessageUtil;
import jp.sf.pal.jstock.util.PriceUtil;

/**
 * Get the Information of Stock Prices from Livedoor Finance<br>
 * 
 * ライブドアファイナンス (http://finance.livedoor.com/) から株価情報を取得します．
 * ライブドアファイナンスの HTML 構造に依存しているので，デザインが変わったりすると正常に動きません．
 * 
 * @author KATOH Yasufumi <karma@prog.club.ne.jp>
 * @version $Id$
 */
public class LivedoorStockDataReader implements StockDataReader {
    
    private static final Log log = LogFactory.getLog(LivedoorStockDataReader.class);
    
    /** 銘柄 */
    public static final int STOCK_NAME_INDEX = 5;
    /** 市場 */
    public static final int STOCK_MARKET_INDEX = 6;
    /** 業種 */
    public static final int STOCK_CATEGORY_INDEX = 7;
    /** ストップ高／安 */
    public static final int STOP_INDEX = 8;
    /** 時刻 */
    public static final int TIME_INDEX = 9;
    /** 取引値*/
    public static final int PRICE_INDEX = 10;
    /** 前日比 */
    public static final int CHANGE_INDEX = 11;
    /** 前日比 (パーセント) */
    public static final int COMPARE_PERCENT_INDEX = 12;
    /** 前日終値 */
    public static final int YESTERDAY_PRICE_INDEX = 13;
    /** 気配値 */
    public static final int INDICATIVE_PRICE_INDEX = 14;
    /** 出来高 */
    public static final int VOLUME_INDEX = 15;
    
    /** 情報を取得する際の URL の固定部分 */
    private static final String URL_BASE = "http://finance.livedoor.com/quote/format/c/";
    
    /** @see org.cyberneko.html.parsers.DOMParser */
    private DOMParser parser;
    
    /** 取得する株の証券コードのリスト */
    private List codes;
    
    /**
     * @param code 証券コード
     */
    public LivedoorStockDataReader(List codes) {
        this.parser = new DOMParser();
        this.codes = codes;
    }
    
    public void setCodes(List codes) {
        this.codes = codes;
    }

    /**
     * 株式情報の取得
     * @return 株式情報
     */
    public StockDataDto getStockData(String code) 
        throws JStockRuntimeException { 
        
        String url = URL_BASE + code;
        List stockInfoList = new ArrayList();
        String[] args = {code};
        
        if (log.isDebugEnabled()) {
            log.debug("Get Stock Data:" + code);
        }
        
        try {
            this.parser.parse(url);
        } catch (Exception e) {
            throw new JStockRuntimeException("jp.sf.pal.jstock.ReadStockInfoError", args);
        }
        this.scanChildNodes(parser.getDocument(), code, stockInfoList);
        
        if (stockInfoList.size() < VOLUME_INDEX) {
            throw new JStockRuntimeException("jp.sf.pal.jstock.ReadStockInfoError", args);
        }
        
        StockDataDto stockDataDto = new StockDataDto();
        stockDataDto.setName((String)stockInfoList.get(STOCK_NAME_INDEX));
        stockDataDto.setTime((String)stockInfoList.get(TIME_INDEX));
        
        // 株価，前日比が数値でない場合 (商いがなかったとか) null をセットする
        try {
            stockDataDto.setPrice(PriceUtil.convertToBigDecimal((String)stockInfoList.get(PRICE_INDEX)));
            stockDataDto.setChange(PriceUtil.convertToBigDecimal((String)stockInfoList.get(CHANGE_INDEX)));
        } catch(ParseException e) {
            stockDataDto.setPrice(null);
            stockDataDto.setChange(null);
        }
            
        return stockDataDto;
    }
    
    public List getStockDataList() {
        
        List dataList = new ArrayList();
        
        for (int i = 0; i < this.codes.size(); i++) {
            String[] args = {(String)this.codes.get(i)};
            try {
                dataList.add(this.getStockData((String)this.codes.get(i)));
            } catch (JStockRuntimeException e) {
                log.error(MessageUtil.getErrorMessage(e.getMessageId(), e.getArgs()));
            }
        }
        
        return dataList;
    }
    
    
    /**
     * DOM のツリー構造をスキャンし，株式の情報が含まれる &lt;tr&gt; タグノードを抜き出す．<br>
     * ここの実装は livedoor finance の実装 (画面) 依存
     * @param document org.w3c.dom.Node
     * @return
     */
    private void scanChildNodes(Node node, String code, List stockInfoList) {
        Node child = node.getFirstChild();
        
        // 現在のノードの値がコードと同じ場合，その親の親のノードを返す
        // (親の親が株式情報が含まれる <tr> タグ)
        // (今はページ内に証券コードは二度現れるけど，後に現れる方が目的のモノだからとりあえず何もせずそのまま使う ^^;)
        if (this.isStockNameNode(child, code)) {
            Node stockInfoNode = node.getParentNode().getParentNode();
            setStockInfoArray(stockInfoNode, stockInfoList);
        }
        
        // さらに下の階層を探し，なければ次のノードへ
        while (child != null) {
            scanChildNodes(child, code, stockInfoList);
            child = child.getNextSibling();
        }

    }
    
    /**
     * ArrayList に株式情報を納める
     * @param node
     */
    private void setStockInfoArray(Node node, List stockInfoList) {
        Node child = node.getFirstChild();
        
        if (child != null && child.getNodeValue() != null) {
            stockInfoList.add(child.getNodeValue());
        }
        
        while (child != null) {
            setStockInfoArray(child, stockInfoList);
            child = child.getNextSibling();
        }

    }
    
    /**
     * 与えられたノードの値が，code で指定された証券コードと同じかどうか判定する．
     * @param node Node
     * @return 同じ場合は true を返す．
     */
    private boolean isStockNameNode(Node node, String code) {
        if (node == null || node.getNodeValue() == null) {
            return false;
        }
        if (node.getNodeValue().equals(code)) {
            return true;
        }
        return false;
    }

}
