/*-
 * Copyright (C) 2008 Speecys Corporation
 *		Toshiaki Nozawa <nozawa@speecys.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the Speecys Corporation.
 * 4. Neither the name of The Speecys Corporation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE SPEECYS CORPORATION AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE CORPORATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <sys/time.h>
#include <assert.h>
#include <limits.h>
#include <string.h>
#include <strings.h>

#include "libutil.h"
#include "servo.h"
#include "rs485.h"

#include "errcheck.h"
//#define FUNC_CALL_TRACE_ON
#include "trace.h"

#include "nerveMain.h"
//#include "libled.h"

/**
 * variables, structur declaration
 */
static bool ledChg[NUM_e_LedBoardID] = {false, false, false};

static const u_char blank[LED_DATA_LEN];

static const u_char slantStripeMix1[LED_DATA_LEN] = {
    /*@ignore@*/
    0x58, 0xa9, 0x15,
    0x56, 0x6a, 0xa5,
    0x95, 0x5a, 0xa9,
    0xa5, 0x56, 0x6a,
    0xa8, 0x95, 0x1a
    /*@end@*/
};

static u_char slantStripeMix2[sizeof slantStripeMix1];

static const u_char dualDiaMix[LED_DATA_LEN] = {
    /*@ignore@*/
    0x20, 0x00, 0x08,
    0x98, 0x00, 0x26,
    0x56, 0x82, 0x95,
    0x98, 0x00, 0x26,
    0x20, 0x00, 0x08
    /*@end@*/
};

static const u_char centCirclMix[LED_DATA_LEN] = {
    /*@ignore@*/
    0x00, 0x14, 0x00,
    0x06, 0x69, 0x90,
    0x1a, 0x69, 0xa4,
    0x06, 0x69, 0x90,
    0x00, 0x14, 0x00
    /*@end@*/
};

static const u_char checkerMix1[LED_DATA_LEN] = {
    /*@ignore@*/
    0x64, 0x66, 0x26,
    0x99, 0x99, 0x99,
    0x66, 0x66, 0x66,
    0x99, 0x99, 0x99,
    0x64, 0x66, 0x26
    /*@end@*/
};

static u_char checkerMix2[sizeof checkerMix1];

static const u_char ripple[][LED_DATA_LEN] = {
    /*@ignore@*/
    {	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00	},

    {	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x14, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00	},

    {	0x00, 0x00, 0x00,
	0x00, 0xaa, 0x00,
	0x80, 0x96, 0x02,
	0x00, 0xaa, 0x00,
	0x00, 0x00, 0x00	},

    {	0x40, 0x55, 0x01,
	0x50, 0xaa, 0x05,
	0x94, 0x96, 0x16,
	0x50, 0xaa, 0x05,
	0x40, 0x55, 0x01	},

    {	0x68, 0x55, 0x29,
	0x5a, 0xaa, 0xa5,
	0x96, 0x96, 0x96,
	0x5a, 0xaa, 0xa5,
	0x68, 0x55, 0x29	}
    /*@end@*/
};

static const u_char invasion[][sizeof checkerMix2] = {
    /*@ignore@*/
    {	0x00, 0x00, 0x00,
	0x01, 0x00, 0x80,
	0x01, 0x00, 0x80,
	0x01, 0x00, 0x80,
	0x00, 0x00, 0x00	},

    {	0x14, 0x00, 0x28,
	0x15, 0x00, 0xa8,
	0x15, 0x00, 0xa8,
	0x15, 0x00, 0xa8,
	0x14, 0x00, 0x28	},

    {	0x54, 0xa5, 0x2a,
	0x55, 0xa5, 0xaa,
	0x55, 0xa5, 0xaa,
	0x55, 0xa5, 0xaa,
	0x54, 0xa5, 0x2a	},

    {	0x40, 0xa5, 0x02,
	0x40, 0xa5, 0x02,
	0x40, 0xa5, 0x02,
	0x40, 0xa5, 0x02,
	0x40, 0xa5, 0x02	},

    {	0x00, 0x24, 0x00,
	0x00, 0x24, 0x00,
	0x00, 0x24, 0x00,
	0x00, 0x24, 0x00,
	0x00, 0x24, 0x00	},

    {	0x00, 0x18, 0x00, //6
	0x00, 0x18, 0x00,
	0x00, 0x18, 0x00,
	0x00, 0x18, 0x00,
	0x00, 0x18, 0x00	},

    {	0x80, 0x5a, 0x01,
	0x80, 0x5a, 0x01,
	0x80, 0x5a, 0x01,
	0x80, 0x5a, 0x01,
	0x80, 0x5a, 0x01	},

    {	0xa8, 0x5a, 0x15,
	0xaa, 0x5a, 0x55,
	0xaa, 0x5a, 0x55,
	0xaa, 0x5a, 0x55,
	0xa8, 0x5a, 0x15	},

    {	0x28, 0x00, 0x14,
	0x2a, 0x00, 0x54,
	0x2a, 0x00, 0x54,
	0x2a, 0x00, 0x54,
	0x28, 0x00, 0x14	},

    {	0x00, 0x00, 0x00,
	0x02, 0x00, 0x40,
	0x02, 0x00, 0x40,
	0x02, 0x00, 0x40,
	0x00, 0x00, 0x00	}
    /*@end@*/
};

static const u_char mesuringWorm[][sizeof checkerMix2] = {
    /*@ignore@*/
    {	0xa8, 0x0a, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x50, 0x15	},

    {	0xa8, 0x0a, 0x00,
	0xaa, 0x0a, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x50, 0x55,
	0x00, 0x50, 0x15	},

    {	0xa8, 0x0a, 0x00,
	0xaa, 0x0a, 0x00,
	0xaa, 0x5a, 0x55,
	0x00, 0x50, 0x55,
	0x00, 0x50, 0x15	},

    {	0x00, 0x00, 0x00,
	0xaa, 0x0a, 0x00,
	0xaa, 0x5a, 0x55,
	0x00, 0x50, 0x55,
	0x00, 0x00, 0x00	},

    {	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0xaa, 0x5a, 0x55,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00	},

    {	0x00, 0x00, 0x00, //6
	0x00, 0x00, 0x00,
	0x55, 0xa5, 0xaa,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00	},

    {	0x00, 0x00, 0x00,
	0x55, 0x05, 0x00,
	0x55, 0xa5, 0xaa,
	0x00, 0xa0, 0xaa,
	0x00, 0x00, 0x00	},

    {	0x54, 0x05, 0x00,
	0x55, 0x05, 0x00,
	0x55, 0xa5, 0xaa,
	0x00, 0xa0, 0xaa,
	0x00, 0xa0, 0x2a	},

    {	0x54, 0x05, 0x00,
	0x55, 0x05, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0xa0, 0xaa,
	0x00, 0xa0, 0x2a	},

    {	0x54, 0x05, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0xa0, 0x2a	}
    /*@end@*/
};

static const u_char ok[LED_DATA_LEN] = {
    /*@ignore@*/
    0x00, 0xa8, 0x00,
    0x00, 0x02, 0x02,
    0x00, 0x02, 0x02,
    0x00, 0x02, 0x02,
    0x00, 0xa8, 0x00
    /*@end@*/
};

static const u_char ng[LED_DATA_LEN] = {
    /*@ignore@*/
    0x00, 0x01, 0x01,
    0x00, 0x44, 0x00,
    0x00, 0x10, 0x00,
    0x00, 0x44, 0x00,
    0x00, 0x01, 0x01
    /*@end@*/
};

static const u_char ok_ng[LED_DATA_LEN] = {
    /*@ignore@*/
    0xa8, 0x10, 0x10,
    0x02, 0x42, 0x04,
    0x02, 0x02, 0x01,
    0x02, 0x42, 0x04,
    0xa8, 0x10, 0x10
    /*@end@*/
};

static const u_char *patTbl[NUM_e_ledPattern] = {
    [BLANK] = blank,
    [SL_STRIPE_MIX1] = slantStripeMix1,
    [SL_STRIPE_MIX2] = slantStripeMix2,
    [DUAL_DIA_MIX] = dualDiaMix,
    [CENT_CIRCLE_MIX] = centCirclMix,
    [CHECKER_MIX1] = checkerMix1,
    [CHECKER_MIX2] = checkerMix2,
    [RIPPLE_MIX1st] = ripple[0],
    [RIPPLE_MIX2nd] = ripple[1],
    [RIPPLE_MIX3rd] = ripple[2],
    [RIPPLE_MIX4th] = ripple[3],
    [INVASION_MIX1st] = invasion[0],
    [INVASION_MIX2nd] = invasion[1],
    [INVASION_MIX3rd] = invasion[2],
    [INVASION_MIX4th] = invasion[3],
    [INVASION_MIX5th] = invasion[4],
    [INVASION_MIX6th] = invasion[5],
    [INVASION_MIX7th] = invasion[6],
    [INVASION_MIX8th] = invasion[7],
    [INVASION_MIX9th] = invasion[8],
    [INVASION_MIX10th] = invasion[9],
    [MESURINGWORM1st] = mesuringWorm[0],
    [MESURINGWORM2nd] = mesuringWorm[1],
    [MESURINGWORM3rd] = mesuringWorm[2],
    [MESURINGWORM4th] = mesuringWorm[3],
    [MESURINGWORM5th] = mesuringWorm[4],
    [MESURINGWORM6th] = mesuringWorm[5],
    [MESURINGWORM7th] = mesuringWorm[6],
    [MESURINGWORM8th] = mesuringWorm[7],
    [MESURINGWORM9th] = mesuringWorm[8],
    [MESURINGWORM10th] = mesuringWorm[9],
    [OK_MARK] = ok,
    [NG_MARK] = ng,
    [OK__NG_MARK] = ok_ng
};

/**
 * prototype declaration
 */
static void 
reversePattern(u_char *dst, const u_char *src, size_t size);

/**
 * functions
 */
void
initLedPtn(void)
{
    TRACE_LOG("func start -----");

    reversePattern(slantStripeMix2, slantStripeMix1, sizeof slantStripeMix2);
    reversePattern(checkerMix2, checkerMix1, sizeof checkerMix2);

    TRACE_LOG("----- func end");
}

static void 
reversePattern(u_char *dst, const u_char *src, size_t size)
{
    const int	    mask = 0x3;
    const int	    width = 2;
    size_t bcnt;

    for (bcnt = 0; bcnt < size; bcnt++) {
	u_int shift;
	u_char tmp;
	for (shift = 0, tmp = (u_char)0; shift < (u_int)(sizeof(char) * 8);
	    shift += width)
	{
	    const u_char bit = (src[bcnt] >> shift) & mask;
	    if (bit != (u_char)0)
		tmp |= (u_char)(~bit & mask) << shift;
	}

	dst[bcnt] = tmp;
    }
}

bool 
ledPixelCheck(int row, int column)
{
    return !((column == 0 && (row == 0 || row == LED_ROW - 1))
	   || (column == LED_COLUMN - 1 && (row == 0 || row == LED_ROW - 1))
    );
}

/**
 * edit LED display pattern to reqB.
 *
 * return: edit packet size.
 */
size_t
editLedDispReq(
    enum e_LedBoardID ledId,/** LED board device ID	*/
    const u_char *pat,	/** pattern table(must 15 byte)	*/
    u_char *buf,	/** packet data buffer	*/
    size_t bLen		/** buf max length(checking buffer-overflow*/
)
{
    static u_char dispPat[NUM_e_LedBoardID][LED_DATA_LEN];
    int	    len = 0;

    assert(ledId >= BREST_LED && ledId <= HEAD_LED);

    if (ledChg[GET_e_LedBoardID_OFFSET(ledId)] ||
	memcmp(dispPat[GET_e_LedBoardID_OFFSET(ledId)], pat,
	       sizeof dispPat[GET_e_LedBoardID_OFFSET(ledId)]) != 0)
    {
	ledChg[GET_e_LedBoardID_OFFSET(ledId)] = false;
	/*@+enumindex@*/
	memcpy(/*@i@*/dispPat[GET_e_LedBoardID_OFFSET(ledId)], pat,
	       sizeof dispPat[GET_e_LedBoardID_OFFSET(ledId)]);

	len = rs485_make_packet(buf, (u_char)ledId, (u_char)0x0
		, (u_char)LMM_LedMatrixData
		, (u_char)ARR_LEN(dispPat[0]), (u_char)1,
		dispPat[GET_e_LedBoardID_OFFSET(ledId)]);
	assert(len == 23);
	assert(len <= (int)bLen);
    }
    return len;
}

/**
 * edit LED display pattern from fix pattern table to reqB.
 */
size_t
editLedDispFixpat(
		  enum e_LedBoardID ledId,	/** LED board device ID	*/
		  enum e_ledPattern pat,	/** pattern ID	*/
		  u_char *buf,	/** packet data buffer	*/
		  size_t bLen	/** reqB max length(checking buffer-overflow*/
)
{
    return editLedDispReq(ledId, patTbl[pat], buf, bLen);
}

void
ledPset(int row, int clm, enum e_LedColorCode col, u_char *pat)
{
    const int rowSize = LED_DATA_LEN / LED_ROW;
    const int byte_p = (clm / LED_PIX_IN_BYTE) + (row * rowSize);
    const u_int	shift_w = (u_int)clm % LED_PIX_IN_BYTE * LED_PIXEL_DEPTH_BIT;

    assert(byte_p < LED_DATA_LEN);
    if (ledPixelCheck(row, clm) == true) {
	pat[byte_p] = (u_char)col << shift_w;
    }
}

/**
 * vartical line request
 */
void 
ledVertLineDisp(
    int clm,	/** column from left	*/
    enum e_LedColorCode col,	/** line color	*/
    u_char *pat	/** pattern output buffer	*/
)
{
    int row;

    memset(pat, 0, (size_t)LED_DATA_LEN);
    for (row = 0; row < LED_ROW; row++) {
	ledPset(row, clm, col, pat);
    }
}

/**
 * perform a notice of update of the LED
 * when you DO NOT depend on "editLedDispReq()"
 */
void
notifyLedDisp(
    enum e_LedBoardID id	/** RS485 device ID	*/
)
{
    ERR_CHK(id, < BREST_LED);
    ERR_CHK(id, > RIGHT_LED);
    ledChg[GET_e_LedBoardID_OFFSET(id)] = true;
}
