/*-
 * Copyright (C) 2008 Speecys Corporation
 *		Toshiaki Nozawa <nozawa@speecys.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the Speecys Corporation.
 * 4. Neither the name of The Speecys Corporation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE SPEECYS CORPORATION AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE CORPORATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <netinet/in.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <netdb.h>
#include <stdbool.h>
#include <assert.h>
#include <sys/stat.h>
#include <string.h>
#include <err.h>
#include <sys/param.h>

#include "nerveMain.h"
#include "rs485devs.h"
#include "libutil.h"

#include "errcheck.h"

//#define FUNC_CALL_TRACE_ON
#include "trace.h"

/**
 * convert usec to 'struct timeval'
 *
 * @ret	return {@link tv}
 */
struct timeval
*usec2tv(
    time_t usec, 	/** value in usec.	*/
    struct timeval *tv /** pointer to set return value	*/
) {
    tv->tv_sec = (long)(usec / (time_t)1e6);
    tv->tv_usec = (long)(usec - tv->tv_sec * (time_t)1e6);

    return /*@i2@*/tv;
}

/**
 * open client socket.
 *
 * return socket descriptor.
 */
int
open_cli_sock(
    const char *srvName,/** IP address or server name. IPv6 is effective. */
    const char *srvPort	/**
    			 * server port number or service name(if effective).
			 *
			 * default server port is '6809'.
			 */
) {
    // bring from 'man getaddrinfo' EXAMPLE
    struct addrinfo hints, *res, *res0;
    int error;
    int s;
    const char *cause = NULL;

    TRACE_LOG("cli_sock() start !! -----");

    memset(&hints, 0, sizeof(hints));
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    error = getaddrinfo(srvName, srvPort, &hints, /*@i1@*/&res0);
    if (error != 0) {
	   TRACE_LOG("getaddrinfo() error");
	   errx(1, "%s", gai_strerror(error));
	   /*@notreached@*/
    }
    s = -1;
    for (res = /*@i2@*/res0; res; res = res->ai_next) {
	s = socket(res->ai_family, res->ai_socktype,
	   res->ai_protocol);
	if (s < 0) {
	    perror("socket error!!");
	    cause = "socket";
	    continue;
	}

	if (connect(s, res->ai_addr, res->ai_addrlen) < 0) {
	    perror("connect error!!");
	    cause = "connect";
	    (void)close(s);
	    s = -1;
	    continue;
	}

	break;  /* okay we got one */
    }
    if (s < 0) {
	assert(cause != NULL);
	err(1, "%s", cause);
	/*@notreached@*/
    }
    freeaddrinfo(res0);

    PRT_INT(s);

    TRACE_LOG("----- cli_sock() end !!");

    return s /*@i3@*/;
    /*
    @i3@ is ignoring follows ---
Only storage hints.ai_canonname (type char *) derived from
    variable declared in this scope is not released (memory leak)
  A storage leak due to incomplete deallocation of a structure or deep pointer
  is suspected. Unshared storage that is reachable from a reference that is
  being deallocated has not yet been deallocated. Splint assumes when an object
  is passed as an out only void pointer that the outer object will be
  deallocated, but the inner objects will not. (Use -compdestroy to inhibit
  warning)
Only storage hints.ai_addr (type struct sockaddr *) derived
    from variable declared in this scope is not released (memory leak)
Only storage hints.ai_next (type struct addrinfo *) derived
    from variable declared in this scope is not released (memory leak)
    */
}

/**
 * read packet from 'd' to buffer 'iovec' likely readv(2).
 * The different is that data size is decided after read header
 * , and set iovec[PcData].iov_len to 'header.datLen'.
 *
 * return to read(2) status.
 */
ssize_t
readPacket(
    int d,			/** descriptor	*/
    struct iovec iovec[NUM_e_packetCat],/** read buffer vector.
				 *
				 * 'iovec[PcData].iov_len' will be set
				 * after read(2) header.
				 */
    bool ntohF,			/** ntoh(byteorder) flag to recv(2) data.
    				 *  effective only in header part.
				 */
    bool readBlkF		/** blocking read flag
    				 *
    				 * true: block to complete read,
				 * false: non-block read
				 */
)
{
    struct stat sb;
    ssize_t nb = 0;
    enum e_packetCat cat;
    int err;

    ERR_CHK(err = fstat(d, &sb), == -1);

    for (cat = PcHdr;
	err != -1 && cat < NUM_e_packetCat && iovec[cat].iov_len > 0;
	cat++)
    {
	ssize_t rb = 0;
	if (S_ISSOCK(sb.st_mode)) {
	    TRACE_LOG("socket recv(2)");
	    if (readBlkF) {
		rb = recv(d, iovec[cat].iov_base, iovec[cat].iov_len
		    , MSG_WAITALL);
	    } else {
		rb = recv(d, iovec[cat].iov_base, iovec[cat].iov_len
		    , MSG_WAITALL | ((cat == PcHdr) ? MSG_DONTWAIT : 0));
	    }
	} else {
	    TRACE_LOG("pipe read(2)");
	    if (readBlkF) {
		u_int zero = 0;
		while(rb < (ssize_t)iovec[cat].iov_len && zero < 5) {
		    ssize_t r = read(d, &((u_char *)iovec[cat].iov_base)[rb]
			, iovec[cat].iov_len - rb);
		    if (r < 0) return r;
		    else if (r == 0)	zero++;
		    else {
			rb += r;
			zero = 0;
		    }

		    (void)usleep(0);
		}
	    } else {
		rb = read(d, iovec[cat].iov_base, iovec[cat].iov_len);
	    }
	}
	if (rb <= 0)	return rb;

	if (cat == PcHdr) {
	    /**
	     * iovec[PcData].iov_len is max size of iovec[PcData].iov_base,
	     * that will over write to request data size in readPacket().
	     */
	    // pickup data length
	    size_t len;
	    struct cmnHeader *hd = iovec[PcHdr].iov_base;
	    hd->seq = (ntohF) ? ntohl(hd->seq) : hd->seq;
	    hd->pri = (ntohF) ? ntohs(hd->pri) : hd->pri;
	    hd->cat = (ntohF) ? ntohs(hd->cat) : hd->cat;
	    hd->flg = (ntohF) ? ntohs(hd->flg) : hd->flg;
	    hd->datLen = (ntohF) ? ntohs(hd->datLen) : hd->datLen;

	    len = hd->datLen;
	    // limit check
	    assert(len <= iovec[PcData].iov_len);
	    if (len <= iovec[PcData].iov_len) {
		// set request size
		iovec[PcData].iov_len = len;
	    }
	}

	nb += rb;
    }

    return nb;
}

/**
 * edit and send packet to 'd'
 *
 * return write size(byte).
 */
ssize_t
writePacket(
    int d,		/** write descriptor	*/
    uint32_t seq,	/** sequence no	*/
    uint16_t pri,	/** priority	*/
    uint16_t cat,	/** category	*/
    uint16_t flg,	/** flags	*/
    uint16_t datLen,	/**
    			 * data length.
			 *
			 * ATTENTION!!: OpenRoads-nerve server can receive
			 * MAX_DATA_PART  bytes MAX.
    			 */
    void *dat,		/** data	*/
    bool htonF		/** hton(byteorder) flag to edit packet.
			 *  effective only in header part.
			 */
) {
    struct cmnHeader hd;
    struct iovec buf[] = {
	[PcHdr] = {&hd, sizeof hd},
	[PcData] = {dat, datLen}
    };

    hd.seq = (htonF) ? htonl(seq) : seq;
    hd.pri = (htonF) ? htons(pri) : pri;
    hd.cat = (htonF) ? htons(cat) : cat;
    hd.flg = (htonF) ? htons(flg) : flg;
    hd.datLen = (htonF) ? htons(datLen) : datLen;

    return writev(d, buf, (int)ARR_LEN(buf));
}

/**
 *
 */
size_t
rs485_make_packet(
    u_char *pcmdbuf,	/** data buf of send data	*/
    u_char id,		/** dev ID	*/
    u_char flag,	/** flag in header	*/
    u_char address,	/** device map address	*/
    u_char length,	/** pdata length	*/
    u_char count,	/** count in header	*/
    const u_char *pdata	/** body of send data	*/
)
{
    size_t pdataLen;
    u_char sum = (u_char)0;
    u_int i;

    pcmdbuf[PK_HEAD0] = (u_char)0xFA;
    pcmdbuf[PK_HEAD1] = (u_char)0xAF;
    pcmdbuf[PK_IDNUM] = id;
    pcmdbuf[PK_FLG] = flag;
    pcmdbuf[PK_ADR] = address;
    pcmdbuf[PK_LEN] = length;
    pcmdbuf[PK_CNT] = count;

    if (pdata == NULL) {
	pdataLen = 0; //read packet request
    } else {
	pdataLen = (size_t)length;
	memcpy(&pcmdbuf[(PK_DATA0)], pdata, pdataLen);
    }

    for (i = PK_IDNUM; i < (u_int)(PK_DATA0 + pdataLen); i++) {
	sum = sum ^ pcmdbuf[i];
    }
    pcmdbuf[(PK_DATA0 + pdataLen)] = sum;

    return PK_DATA0 + pdataLen + 1;
}

/**
 *
 */
int
rs485_make_long_packet(
    u_char *pcmdbuf,
    u_char flag,
    u_char address,
    u_char length,
    u_char count,
    const u_char *pdata
)
{
    u_char sum = (u_char)0;
    u_int i;

    //TODO integrate with rs485_make_packet()
    pcmdbuf[PK_HEAD0] = (u_char)0xFA;
    pcmdbuf[PK_HEAD1] = (u_char)0xAF;
    pcmdbuf[PK_IDNUM] = (u_char)0x00;
    pcmdbuf[PK_FLG] = flag;
    pcmdbuf[PK_ADR] = address;
    pcmdbuf[PK_LEN] = length;
    pcmdbuf[PK_CNT] = count;

    memcpy(&pcmdbuf[(PK_DATA0)], pdata, (size_t)(length * count));

    for (i = PK_IDNUM; i < (u_int)PK_DATA0 + (u_int)(length * count); i++) {
	sum = sum ^ pcmdbuf[i];
    }
    pcmdbuf[(u_int)PK_DATA0 + (u_int)(length * count)] = sum;

    return (int)((u_int)PK_DATA0 + (u_int)(length * count) + 1);
}

/**
 * set short data to send RS485 device
 *
 * return src value
 */
u_short
set485devShort(
    u_char dst[sizeof(short)],
    u_short src
)
{
    *dst = (u_char)(src & 0xff);
    *(dst + 1) = (u_char)((src >> NBBY) & 0xff);
    return src;
}

/**
 * get short data from RS485 device
 *
 * return decode value
 */
u_short
get485devShort(
	u_char src[sizeof(short)]
)
{
	return (u_short)(src[0] | src[1] << NBBY);
}
