//
// xbuftest.cpp --- test for xbuf.h
//

#include <cstdio>
#include <string>
#include <omt/xbuf.h>
#include <omt/utest.h>

using namespace std;
using namespace omt;

// -----------------------------------------------------------------------------
void print( const xstr& x )
{
	size_t	n = x.length();

	if ( n > 32 ) 
		printf( "size = %ld, length = %ld, str = '%32.32s...\n", x.size(), n, x.c_str());
	else
		printf( "size = %ld, length = %ld, str = '%s'\n", x.size(), n, x.c_str());
}

// -----------------------------------------------------------------------------
void perf_test( const char* cmd, size_t n )
{
	char	buf[ 256 ];

	printf( "*** put/push_back performance test -- %lu times ***\n", n );
	printf( "\nomt::xbuf::put()\n" );
	snprintf( buf, 256, "time %s -x %lu", cmd, n );
	system( buf );
	printf( "\nstd::string::push_back()\n" );
	snprintf( buf, 256, "time %s -s %lu", cmd, n );
	system( buf );
}

void perf_x( size_t n )
{
	const char* 	dt = "0123456789abcdef";
	xstr		s;

	for ( int i = 0; i < n; ++i ) {
		for ( const char* p = dt; *p; ++p ) s.put( *p );
	}
	print( s );
}

void perf_s( size_t n )
{
	const char* 	dt = "0123456789abcdef";
	string		s;

	for ( int i = 0; i < n; ++i ) {
		for ( const char* p = dt; *p; ++p ) s.push_back( *p );
	}
	printf( "size = %ld, length = %ld, str = '%32.32s...\n", s.size(), s.length(), s.c_str());
}

// -----------------------------------------------------------------------------
class xbuftest : public unittest
{
	void test01();
	void test02();
	void test03();
	void test04();
	void test05();

    public:
	xbuftest() : unittest( "test for xbuf.h" ) { }
	void run();
};

void xbuftest::run()
{
	test01();
	test02();
	test03();
	test04();
	test05();
}

// -----------------------------------------------------------------------------
void xbuftest::test01()
{
	this->set_subtitle( "constructor" );

	const char* nul = 0;

	xstr	s0;
	xstr	s1( 32 );
	xstr	s2( "Hello, world!" );
	xstr	s3( s2.c_str());
	xstr	s4( nul );

	AssertTrue( s0.size() == 256 && s0.length() == 0 );
	AssertTrue( s1.size() == 32 && s1.length() == 0 );
	AssertTrue( s2.size() == 14 && s2.length() == 13 &&
			strncmp( s2.buf(), "Hello, world!", s2.length()) == 0 );
	AssertTrue( s3.size() == 14 && s3.length() == 13 &&
			strncmp( s3.buf(), "Hello, world!", s3.length()) == 0 );
	AssertTrue( s4.size() == 256 && s4.length() == 0 );

	s0 = s2.c_str();

	AssertTrue( s0.size() == 256 && s0.length() == 13 &&
			strncmp( s0.buf(), "Hello, world!", s0.length()) == 0 );
}

// -----------------------------------------------------------------------------
void xbuftest::test02()
{
	this->set_subtitle( "put" );

	xstr		s;
	const char* 	nul = 0;

	s.append( "Hello, world!", 5 );
	s.put( ',' );
	s.put( ' ' );
	s.append( "world!" );
	AssertTrue( s.size() == 256 && s.length() == 13 &&
			strncmp( s.buf(), "Hello, world!", s.length()) == 0 );

	s.set( "oh!" );
	AssertTrue( s.size() == 256 && s.length() == 3 &&
			strncmp( s.buf(), "oh!", s.length()) == 0 );

	s.set( "Hello, world!!!!", 13 );
	AssertTrue( s.size() == 256 && s.length() == 13 &&
			strncmp( s.buf(), "Hello, world!", s.length()) == 0 );

	s.append( nul );
	AssertTrue( s.size() == 256 && s.length() == 13 &&
			strncmp( s.buf(), "Hello, world!", s.length()) == 0 );

	s.set( nul );
	AssertTrue( s.size() == 256 && s.length() == 0 );
}

// -----------------------------------------------------------------------------
void xbuftest::test03()
{
	this->set_subtitle( "get" );

	xstr	s( "Hello, world!" );
	size_t	n;
	int	i;

	AssertTrue( s.get() == 'H' && s.get() == 'e' && s.peek() == 'l' );
	AssertTrue( !s.is_eos() && ( n = s.tell()) == 2 );
	s.unget();
	AssertTrue( s.peek() == 'e' );
	s.seek( n );
	AssertTrue( s.peek() == 'l' );
	for ( i = 0; s.get() >= 0; ++i ) ;
	AssertTrue( i == 11 && s.is_eos() );
	s.reset();
	AssertTrue( s.get() == 'H' );
}

// -----------------------------------------------------------------------------
void xbuftest::test04()
{
	this->set_subtitle( "utility operator" );

	xstr		s;
	const xstr	c( "Hello, world!" );

	s = "Hello, world!";
	AssertTrue( s[ 0 ] == 'H' && s[ 9 ] == 'r' && s[ 13 ] == '\0' && s[ 20 ] == '\0' );
	AssertTrue( c[ 0 ] == 'H' && c[ 9 ] == 'r' && c[ 13 ] == '\0' && c[ 20 ] == '\0' );
	AssertTrue( s );
	while ( s.get() >= 0 );
	AssertTrue( !s );

	s.clear();
	s << "Hello";
	s << ',' << ' ';
	s << "world" << '!';
	AssertTrue( s.size() == 256 && s.length() == 13 &&
			strncmp( s.buf(), "Hello, world!", s.length()) == 0 );
}

// -----------------------------------------------------------------------------
void xbuftest::test05()
{
	this->set_subtitle( "ext" );

	xstr	s( "Hello, world!" );

	void*	p = s;
	s.put( '!' );
	void*	q = s;
	
	AssertTrue( s.size() == 28 && s.length() == 14 &&
			strncmp( s.buf(), "Hello, world!!", s.length()) == 0 );
	AssertTrue( p != q );
}

// -----------------------------------------------------------------------------
int main( int argc, char** argv )
{
	const char* usage = "xbuftest [ -p [ n ]| -x n | -s n ]";

	if ( argc > 1 ) {
		size_t	n = ( argc > 2 ) ? strtoul( argv[ 2 ], 0, 0 ) : 65535;
		if ( *argv[ 1 ] == '-' ) {
			switch ( *( argv[ 1 ] + 1 )) {
				case 'p': perf_test( argv[ 0 ], n ); break;
				case 'x': perf_x( n ); break;
				case 's': perf_s( n ); break;
				default:
					fprintf( stderr, "unknown option %s\n", argv[ 1 ] );
				case 'h':
				case '?':
					fprintf( stderr, "usage: %s\n", usage );
					exit( 1 );
			}
		} else {
			fprintf( stderr, "illegal option %s\n", argv[ 1 ] );
			exit( 1 );
		}
	} else {
		xbuftest	t;

		t.run();
	}
	return 0;
}
