//
// srztest.cpp --- test for srz.h
//
// -----------------------------------------------------------------------------
#include <cstdlib>
#include <cstdio>

#ifndef NO_CPPSTD
#include <iosfwd>
#include <fstream>
#include <string>
#endif

#include <omt/srz.h>
#include <omt/utest.h>

using namespace std;
using namespace omt;

// User defined class
// -----------------------------------------------------------------------------
class test {
        int	m_int;
        char*	m_str;

        static int cntr;

    public:
        test() : m_int( 0 ), m_str( 0 )
        {
                ++cntr;
#ifdef _DEBUG
                printf( "%03d: ==> alloc test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
        }
        test( int n, const char* b ) : m_int( n ), m_str( dup( b ))
        {
                ++cntr;
#ifdef _DEBUG
                printf( "%03d: ==> alloc test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
        }
        ~test()
        {
#ifdef _DEBUG
                printf( "%03d: <== free test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
                --cntr;
		delete [] m_str;
        }

        void print() const
        {
                printf( "(%d, \"%s\")", m_int, m_str );
        }

        const char*     cstr() const { return m_str; }
        int             num()  const { return m_int; }

	void deserialize( deserializer& in ) 
	{
		char*	p;
		in >> begin >> m_int >> p >> end;

		if ( m_str ) delete [] m_str;
		m_str = dup( p );
	}
	void serialize( serializer& out ) const
	{
		out << begin << m_int << m_str << end;
	}

        static int getCntr() { return cntr; }

    private:
        test( const test& t );
        test& operator=( const test& t );
};

int test::cntr = 0;

// -----------------------------------------------------------------------------
bool operator== ( const test& a, const test& b )
{
        return a.num() == b.num() && eql_fn<const char*>()( a.cstr(), b.cstr());
}

// -----------------------------------------------------------------------------
test* dup( const test* p )
{
        return p ? new test( p->num(), p->cstr()) : 0;
}

// -----------------------------------------------------------------------------
namespace omt {
        template<> struct hash_fn<const test*>
        {
                unsigned long operator()( const test* p ) const
                {
                        return hash( p->cstr()) + hash( p->num()) * 7;
                }
        };
        template<> struct eql_fn<const test*>
        {
                bool operator()( const test* p, const test* q ) const
                {
                        return !p && !q || p && q && *p == *q;
                }
        };
	template<> struct cmp_fn<const test*>
        {
                int operator()( const test* p, const test* q ) const
                {
                        return ( !p && !q ) ? 0 :
                                !p ? -1 :
                                !q ? 1 :
                                ( p->num() < q->num()) ? -1 :
                                ( p->num() > q->num()) ? 1 :
                                cmp_fn<const char*>()( p->cstr(), q->cstr());
                }
        };
}

// -----------------------------------------------------------------------------
bool readFile( const char* fn, xstr& buf )
{
	bool	r = false;

	if ( FILE* f = fopen( fn, "r" )) {
		int	a;

		while (( a = getc( f )) > 0 ) buf.put( a );
		r = true;
	}
	return r;
}

// Unit Test Class
// -----------------------------------------------------------------------------
class srztest : public unittest
{
	// test functions
	void test11();
	void test12();
	void test13();
	void test14();
#ifndef NO_CPPSTD
	void test15();
	void test16();
#endif

	void test21();
	void test22();
	void test23();

	void test31();

	void test41();
	void test42();
	void test43();

	void test51();
	void test52();
	void test53();

	void test61();
	void test62();
	void test63();

	void test71();
	void test72();
	void test73();

	// test data
	typedef plist< long > 			    		testnlist;
	typedef plist< const char*, store_policy<const char*> > testcstrl_s;
	typedef plist< const char*,  copy_policy<const char*> > testcstrl_c;
	typedef plist< const char*, refer_policy<const char*> > testcstrl_r;
	typedef plist< test*, store_policy<test*> >		testplist_s;
	typedef plist< test*,  copy_policy<test*> >		testplist_c;
	typedef plist< test*, refer_policy<test*> >		testplist_r;
	
	typedef phash< long >			    		testnhash;
	typedef phash< const char*, store_policy<const char*> > testcstrh_s;
	typedef phash< const char*,  copy_policy<const char*> > testcstrh_c;
	typedef phash< const char*, refer_policy<const char*> > testcstrh_r;
	typedef phash< test*, store_policy<test*> >		testphash_s;
	typedef phash< test*,  copy_policy<test*> >		testphash_c;
	typedef phash< test*, refer_policy<test*> >		testphash_r;
	typedef ptree< long > 	  		    		testntree;
	typedef ptree< const char*, store_policy<const char*> > testcstrt_s;
	typedef ptree< const char*,  copy_policy<const char*> > testcstrt_c;
	typedef ptree< const char*, refer_policy<const char*> > testcstrt_r;
	typedef ptree< test*, store_policy<test*> >		testptree_s;
	typedef ptree< test*,  copy_policy<test*> >		testptree_c;
	typedef ptree< test*, refer_policy<test*> >		testptree_r;

	size_t	m_cnt;

    public:
	srztest() : unittest( "test for srz.h" ) 
        {
                m_cnt = test::getCntr();
        }
        ~srztest()
        {
                this->set_subtitle( "test class allocation" );
                AssertNonZero( test::getCntr() == m_cnt );
        }

	void run();
};

// external stream test
// -----------------------------------------------------------------------------
void srztest::test11()
{
	this->set_subtitle( "serialize to/deseralize from xstr" );

	xstr		s;

	xstr		ds, ss( "\"hello, world!\"\\n" );
	binary		dx, sx( "01234567890123456789012345678901234567890123456789"
		  	     	"01234567890123456789012345678901234567890123456789"
				"01234567890123456789012345678901234567890123456789"
				"01234567890123456789012345678901234567890123456789"
				"01234567890123456789012345678901234567890123456789"
				"01234567890123456789012345678901234567890123456789" );

	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	AssertTrue( "version:", sz.writeVersion( "test_version" ));

	sz << begin << ss << sx << si << sf << sl
	   << begin << n << n << n << end
	   << sb
	   << end << newline;

	// --- stream ---
	AssertZero( "serz stream:", strcmp( s.c_str(), "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"#300'MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5'-468 0.123(1 2 3)(0 0 0)1)\n" ));

	// --- deserialize ---
	deserializer	dz( s );

	long	seq = dz.checkVersion( "test_version" );

	dz >> begin >> ds >> dx >> di >> df >> dl
	   >> begin >> n >> n >> n >> end
	   >> db
	   >> end;

	// --- check ---
	AssertEqual( seq, 1L );
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

// -----------------------------------------------------------------------------
void srztest::test12()
{
	this->set_subtitle( "serialize to/deserialize from FILE*" );

	const char*	fn = "test.srz";

	xstr		ds, ss( "\"hello, world!\"\\n" );
	binary		dx, sx( "zxcvb" );
	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;
	long		seq;

	// --- serialize ---
	if ( FILE* out = fopen( fn, "w" )) {
	
		serializer sz( out );
		AssertTrue( "version:", sz.writeVersion( "test_version" ));
	
		sz << begin << ss << sx << si << sf << sl
		   << begin << n << n << n << end
		   << sb
		   << end << newline;

		fclose( out );
	} else {
		fprintf( stderr, "file %s cannot open", fn );
		exit( -1 );
	}

	// --- serialized stream ---
	xstr	buf, srz( "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0 0)1)\n" );
	readFile( fn, buf );
	AssertEqual( "serz stream:", buf, srz );

	// --- deserialize ---
	if ( FILE* in = fopen( fn, "r" )) {

		deserializer	dz( in );

		seq = dz.checkVersion( "test_version" );

		dz >> begin >> ds >> dx >> di >> df >> dl
		   >> begin >> n >> n >> n >> end
		   >> db
		   >> end;

		fclose( in );
	} else {
		fprintf( stderr, "file %s cannot open", fn );
		exit( -1 );
	}

	// --- check ---
	AssertEqual( seq, 1L );
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

// -----------------------------------------------------------------------------
void srztest::test13()
{
	this->set_subtitle( "deseralize from const char*" );

	xstr		s;

	xstr		ds, ss( "\"hello, world!\"\\n" );
	binary		dx, sx( "zxcvb" );
	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	AssertTrue( "version:", sz.writeVersion( "test_version" ));

	sz << begin << ss << sx << si << sf << sl
	   << begin << n << n << n << end
	   << sb
	   << end << newline;

	// --- serialized stream ---
	xstr	srz( "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0 0)1)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---

	deserializer	dz( s.buf() );		// s has enough size for deserz.

	long	seq = dz.checkVersion( "test_version" );

	dz >> begin >> ds >> dx >> di >> df >> dl
	   >> begin >> n >> n >> n >> end
	   >> db
	   >> end;

	// --- check ---
	AssertEqual( seq, 1L );
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

// -----------------------------------------------------------------------------
void srztest::test14()
{
	this->set_subtitle( "serialize to/deseralize from xstr (xbuf<char>)" );

	xstr		s( 8 );		// force expanding size on deserz.

	xstr		ds, ss( "\"hello, world!\"\\n" );
	binary		dx, sx( "zxcvb" );
	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	AssertTrue( "version:", sz.writeVersion( "test_version" ));

	sz << begin << ss << sx << si << sf << sl
	   << begin << n << n << n << end
	   << sb
	   << end << newline;

	// --- serialized stream ---
	xstr	srz( "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0 0)1)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---
	deserializer	dz( s );

	long	seq = dz.checkVersion( "test_version" );

	dz >> begin >> ds >> dx >> di >> df >> dl
	   >> begin >> n >> n >> n >> end
	   >> db
	   >> end;

	// --- check ---
	AssertEqual( seq, 1L );
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
void srztest::test15()
{
	this->set_subtitle( "serialize to/deseralize from std::string" );

	std::string	s;

	std::string	ds, ss( "\"hello, world!\"\\n" );
	binary		dx, sx( "zxcvb" );
	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	AssertTrue( "version:", sz.writeVersion( "test_version" ));

	sz << begin << ss << sx << si << sf << sl
	   << begin << n << n << n << end
	   << sb
	   << end << newline;

	// --- serialized stream ---
	std::string srz( "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0 0)1)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---
	deserializer	dz( s );

	long	seq = dz.checkVersion( "test_version" );

	dz >> begin >> ds >> dx >> di >> df >> dl
	   >> begin >> n >> n >> n >> end
	   >> db
	   >> end;

	// --- check ---
	AssertEqual( seq, 1L );
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

// -----------------------------------------------------------------------------
void srztest::test16()
{
        this->set_subtitle( "serialize/deserialize std::fstream" );

        const char*     fn = "test.srz";

        std::string     ds, ss( "\"hello, world!\"\\n" );
        binary          dx, sx( "zxcvb" );
        long            di, si( -468 );
        double          df, sf( 0.123 );
        bool            db, sb = true;
        plist<int>      dl, sl;
        sl.enq( 1 );
        sl.enq( 2 );
        sl.enq( 3 );
        null            n;
        long            seq;

        // --- serialize ---
        {
                ofstream        out( fn );
                AssertTrue( out );

                serializer sz( out );

                AssertTrue( "version:", sz.writeVersion( "test_version" ));

                sz << begin << ss << sx << si << sf << sl
                   << begin << n << n << n << end
                   << sb
                   << end << newline;

		// destruct ofstream
        }
        // --- serialized stream ---
        xstr    buf, srz( "test_version 1\n(\"\\\"hello, world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0 0)1)\n" );
        readFile( fn, buf );
        AssertEqual( "serz stream:", buf, srz );

        // --- deserialize ---
        {
                ifstream        in( fn );
                AssertTrue( in );

                deserializer    dz( in );

                seq = dz.checkVersion( "test_version" );

                dz >> begin >> ds >> dx >> di >> df >> dl
                   >> begin >> n >> n >> n >> end
                   >> db
                   >> end;
        }
        // --- check ---
        AssertEqual( seq, 1L );
        AssertEqual( ds, ss );
        AssertEqual( dx, sx );
        AssertEqual( di, si );
        AssertEqual( df, sf );
        AssertEqual( db, sb );
        AssertEqual( dl, sl );
}
#endif // NO_CPPSTD

// serialize constants/null, scan
// -----------------------------------------------------------------------------
void srztest::test21()
{
	this->set_subtitle( "serialize constants" );

	xstr		s;

	const char*	ds;
	const char*	ss = "\"hello, world!\"\\n";
	long		di, si( -468 );
	double		df, sf( 0.123 );
	bool		db, sb = true;
	plist<int>	dl, sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	sz << begin << "\"hello, world!\"\\n" << -468 << 0.123 
	     << begin << 1 << 2 << 3 << end
	     << begin << 0 << 0 << 0 << end
	     << true 
	   << end << newline;

	// --- serialized stream ---
	xstr	srz( 
		"(_1@\"\\\"hello, world!\\\"\\\\n\"-468 0.123(1 2 3)(0 0 0)1)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---
	deserializer	dz( s );		// s has enough size for deserz.

	dz >> begin >> ds >> di >> df
	     >> dl
	     >> begin >> n >> n >> n >> end
	     >> db
	   >> end;

	// --- check ---
	AssertZero( strcmp( ds, ss ));
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
}

// -----------------------------------------------------------------------------
void srztest::test22()
{
	this->set_subtitle( "Simple Datatype with Null Value" );

	xstr		s;

	xstr		ds, ss;
	binary		dx, sx;
	long		di, si = 0;
	double		df, sf = 0;
	bool		db, sb = 0;
	plist<int>	dl, sl;
	phash<int>	dh, sh;
	null		n;

	// --- serialize ---
	serializer 	sz( s );

	sz << begin << ss << sx << si << sf << sl << sh << n << sb << end << newline;

	// --- serialized stream ---
	xstr	srz( "(\"\"''0 0()()0 0)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---
	deserializer	dz( s );

	dz >> begin >> ds >> dx >> di >> df >> dl >> dh >> n >> db >> end;

	// --- check ---
	AssertEqual( ds, ss );
	AssertEqual( dx, sx );
	AssertEqual( di, si );
	AssertEqual( df, sf );
	AssertEqual( db, sb );
	AssertEqual( dl, sl );
	AssertEqual( dh, sh );
}

// -----------------------------------------------------------------------------
void srztest::test23()
{
	this->set_subtitle( "deseralizer::scan()" );

	xstr		s;

	xstr		ss( "\"hello, (secret) world!\"\\n" );
	binary		sx( "zxcvb" );
	long		si( -468 );
	double		sf( 0.123 );
	bool		sb = true;
	plist<int>	sl;
	sl.enq( 1 );
	sl.enq( 2 );
	sl.enq( 3 );
	null		n;
	xstr		ds, dx, di, df, dl, dn, db;
	token		t;

	// --- serialize ---
	serializer 	sz( s );

	sz << begin << ss << sx << si << sf << sl
	   << begin << n << n << begin << sl << ss << sx << end << n << end
	   << sb
	   << end << newline;

	// --- serialized stream ---
	xstr	srz( "(\"\\\"hello, (secret) world!\\\"\\\\n\"'enhjdmI='-468 0.123(1 2 3)(0 0((1 2 3)\"\"'')0)1)\n" );
	AssertEqual( "serz stream:", s, srz );

	// --- deserialize ---
	deserializer	dz( s );

	// --- check ---
	dz >> begin;
	t = dz.scan( ds );
	AssertZero( deserializer::token2id( t ), strcmp( ds.c_str(), "\"\\\"hello, (secret) world!\\\"\\\\n\"" ));
	t = dz.scan( dx );
	AssertZero( deserializer::token2id( t ), strcmp( dx.c_str(), "'enhjdmI='" ));
	t = dz.scan( di );
	AssertZero( deserializer::token2id( t ), strcmp( di.c_str(), "-468" ));
	t = dz.scan( df );
	AssertZero( deserializer::token2id( t ), strcmp( df.c_str(), " 0.123" ));
	t = dz.scan( dl );
	AssertZero( deserializer::token2id( t ), strcmp( dl.c_str(), "(1 2 3)" ));
	t = dz.scan( dn );
	AssertZero( deserializer::token2id( t ), strcmp( dn.c_str(), "(0 0((1 2 3)\"\"'')0)" ));
	t = dz.scan( db );
	AssertZero( deserializer::token2id( t ), strcmp( db.c_str(), "1" ));
	dz >> end;
}

// serialize/deserialize pointer 
// -----------------------------------------------------------------------------
void srztest::test31()
{
	this->set_subtitle( "pointer test for long* / char* / test*" );

	xstr		s;
	short		m = 2043;
	long		n = -468;
	test		t1( 1, "abc" );
	test		t2( 2, "cde" );

	long*		si = &n;
	short*		sj = &m;
	unsigned long*	su = 0;
	char*		sn = 0;
	char*		ss = "hello";
	const char*	sc = "world";
	test*		sv = 0;
	test*		st = &t1;
	test*		sd = &t1;
	const test*	sb = &t2;
	
	long*		di;
	short*		dj;
	unsigned long*	du;
	char*		dn;
	char*		ds;
	const char*	dc;
	test*		dv;
	test*		dt;
	test*		dd;
	test*		db;
	
	// --- serialize ---
	serializer 	sz( s );

	sz << si << sj << su << sn << ss << sc << sv << st << sd << sb;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> di >> dj >> du >> dn >> ds >> dc >> dv >> dt >> dd >> db;

	// --- check ---
	AssertEqual( *di, *si );
	AssertEqual( *dj, *sj );
	AssertZero( du );
	AssertZero( dn );
	AssertEqual( *ds, *ss );
	AssertEqual( *dc, *sc );
	AssertZero( dv );
	AssertEqual( *dt, *st );
	AssertEqual( *dd, *sd );
	AssertEqual( *const_cast<const test*>( db ), *sb );
	AssertEqual( "pointer equality:", dd, dt );
}

// pcontainer: plist
// -----------------------------------------------------------------------------
void srztest::test41()
{
	this->set_subtitle( "plist for long" );

	xstr		s;

	testnlist	la;
	la.enq( 0 );
	la.enq( 1 );
	la.enq( 2 );
	la.enq( 3 );
	la.enq( 4 );
	testnlist	ln;

	testnlist	da;
	testnlist	dn;

	// --- serialize ---
	serializer 	sz( s );

	sz << la << ln;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> da >> dn;

	// --- check ---
	AssertEqual( la, da );
	AssertEqual( ln, dn );
}

// -----------------------------------------------------------------------------
void srztest::test42()
{
	this->set_subtitle( "plist for char*" );

	xstr		s;

	testcstrl_s	ls, ds;
	testcstrl_c	lc, dc;
	testcstrl_r	lr, dr;

	ls.enq( dup( "a" ));
	ls.enq( dup( "bcd" ));
	ls.enq( dup( "ef" ));
	ls.enq( 0 );
	ls.enq( dup( "" ));
	ls.enq( dup( "uvwxyz" ));

	lc.enq( "a" );
	lc.enq( "bcd" );
	lc.enq( "ef" );
	lc.enq( 0 );
	lc.enq( "" );
	lc.enq( "uvwxyz" );

	lr.enq( "a" );
	lr.enq( "bcd" );
	lr.enq( "ef" );
	lr.enq( 0 );
	lr.enq( "" );
	lr.enq( "uvwxyz" );
	
	// --- serialize ---
	serializer 	sz( s );

	sz << ls << lc << lr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( ls, ds );
	AssertEqual( lc, dc );
	AssertEqual( lr, dr );
}

// -----------------------------------------------------------------------------
void srztest::test43()
{
	this->set_subtitle( "plist for test*" );

	xstr		s;
  
	test		t1( 1, "abc" );
	test		t2( 2, "cde" );
	test		t3( 3, "efg" );
	test		t4( 4, "ghi" );
	test		t5( 5, "ijk" );

	testplist_s	ls, ds;
	testplist_c	lc, dc;
	testplist_r	lr, dr;

	ls.enq( dup( &t1 ));
	ls.enq( dup( &t2 ));
	ls.enq( dup( &t3 ));
	ls.enq( 0 );

	lc.enq( &t2 );
	lc.enq( &t3 );
	lc.enq( 0 );
	lc.enq( &t4 );

	lr.enq( 0 );
	lr.enq( &t3 );
	lr.enq( &t4 );
	lr.enq( &t5 );

	// --- serialize ---
	serializer 	sz( s );

	sz << ls << lc << lr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( ls, ds );
	AssertEqual( lc, dc );
	AssertEqual( lr, dr );
}

// pcontainer: phash 
// -----------------------------------------------------------------------------
void srztest::test51()
{
	this->set_subtitle( "phash for long" );

	xstr		s;

	testnhash	ha;
	ha.set( "abc", 0 );
	ha.set( "cde", 1 );
	ha.set( "efg", 2 );
	ha.set( "ghi", 3 );
	ha.set( "ijk", 4 );
	testnhash	hn;

	testnhash	da;
	testnhash	dn;

	// --- serialize ---
	serializer 	sz( s );

	sz << ha << hn;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> da >> dn;

	// --- check ---
	AssertEqual( ha, da );
	AssertEqual( hn, dn );
}

// -----------------------------------------------------------------------------
void srztest::test52()
{
	this->set_subtitle( "phash for char*" );

	xstr		s;

	testcstrh_s	hs, ds;
	testcstrh_c	hc, dc;
	testcstrh_r	hr, dr;

	hs.set( "abc", dup( "a" ));
	hs.set( "cde", dup( "bcd" ));
	hs.set( "efg", dup( "ef" ));
	hs.set( "ghi", 0 );
	hs.set( "ijk", dup( "" ));
	hs.set( "klm", dup( "uvwxyz" ));

	hc.set( "abc", "a" );
	hc.set( "cde", "bcd" );
	hc.set( "efg", "ef" );
	hc.set( "ghi", 0 );
	hc.set( "ijk", "" );
	hc.set( "klm", "uvwxyz" );

	hr.set( "abc", "a" );
	hr.set( "cde", "bcd" );
	hr.set( "efg", "ef" );
	hr.set( "ghi", 0 );
	hr.set( "ijk", "" );
	hr.set( "klm", "uvwxyz" );

	// --- serialize ---
	serializer 	sz( s );

	sz << hs << hc << hr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( hs, ds );
	AssertEqual( hc, dc );
	AssertEqual( hr, dr );
}

// -----------------------------------------------------------------------------
void srztest::test53()
{
	this->set_subtitle( "phash for test*" );

	xstr		s;

	testphash_s	hs, ds;
	testphash_c	hc, dc;
	testphash_r	hr, dr;

	test		t1( 1, "abc" );
	test		t2( 2, "cde" );
	test		t3( 3, "efg" );
	test		t4( 4, "ghi" );
	test		t5( 5, "ijk" );

	hs.set( t1.cstr(), dup( &t1 ));
	hs.set( t2.cstr(), dup( &t2 ));
	hs.set( t3.cstr(), dup( &t3 ));
	hs.set( t4.cstr(), 0 );
	hs.set( t5.cstr(), dup( &t5 ));

	hc.set( t1.cstr(), &t1 );
	hc.set( t2.cstr(), &t2 );
	hc.set( t3.cstr(), &t3 );
	hc.set( t4.cstr(), &t4 );
	hc.set( t5.cstr(), 0 );

	hr.set( t1.cstr(), &t1 );
	hr.set( t2.cstr(), 0 );
	hr.set( t3.cstr(), &t3 );
	hr.set( t4.cstr(), &t4 );
	hr.set( t5.cstr(), &t5 );

	// --- serialize ---
	serializer 	sz( s );

	sz << hs << hc << hr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( hs, ds );
	AssertEqual( hc, dc );
	AssertEqual( hr, dr );
}

// pcontainer: ptree
// -----------------------------------------------------------------------------
void srztest::test61()
{
	this->set_subtitle( "ptree for long" );

	xstr		s;

	testntree	ta;
	ta.set( "abc", 0 );
	ta.set( "cde", 1 );
	ta.set( "efg", 2 );
	ta.set( "ghi", 3 );
	ta.set( "ijk", 4 );
	testntree	tn;

	testntree	da;
	testntree	dn;

	// --- serialize ---
	serializer 	sz( s );

	sz << ta << tn;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> da >> dn;

	// --- check ---
	AssertEqual( ta, da );
	AssertEqual( tn, dn );
}

// -----------------------------------------------------------------------------
void srztest::test62()
{
	this->set_subtitle( "ptree for char*" );

	xstr		s;

	testcstrt_s	ts, ds;
	testcstrt_c	tc, dc;
	testcstrt_r	tr, dr;

	ts.set( "abc", dup( "a" ));
	ts.set( "cde", dup( "bcd" ));
	ts.set( "efg", dup( "ef" ));
	ts.set( "ghi", 0 );
	ts.set( "ijk", dup( "" ));
	ts.set( "klm", dup( "uvwxyz" ));

	tc.set( "abc", "a" );
	tc.set( "cde", "bcd" );
	tc.set( "efg", "ef" );
	tc.set( "ghi", 0 );
	tc.set( "ijk", "" );
	tc.set( "klm", "uvwxyz" );

	tr.set( "abc", "a" );
	tr.set( "cde", "bcd" );
	tr.set( "efg", "ef" );
	tr.set( "ghi", 0 );
	tr.set( "ijk", "" );
	tr.set( "klm", "uvwxyz" );

	// --- serialize ---
	serializer 	sz( s );

	sz << ts << tc << tr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( ts, ds );
	AssertEqual( tc, dc );
	AssertEqual( tr, dr );
}

// -----------------------------------------------------------------------------
void srztest::test63()
{
	this->set_subtitle( "ptree for test*" );

	xstr		s;

	testptree_s	ts, ds;
	testptree_c	tc, dc;
	testptree_r	tr, dr;

	test		t1( 1, "abc" );
	test		t2( 2, "cde" );
	test		t3( 3, "efg" );
	test		t4( 4, "ghi" );
	test		t5( 5, "ijk" );

	ts.set( t1.cstr(), dup( &t1 ));
	ts.set( t2.cstr(), dup( &t2 ));
	ts.set( t3.cstr(), dup( &t3 ));
	ts.set( t4.cstr(), 0 );
	ts.set( t5.cstr(), dup( &t5 ));

	tc.set( t1.cstr(), &t1 );
	tc.set( t2.cstr(), &t2 );
	tc.set( t3.cstr(), &t3 );
	tc.set( t4.cstr(), &t4 );
	tc.set( t5.cstr(), 0 );

	tr.set( t1.cstr(), &t1 );
	tr.set( t2.cstr(), 0 );
	tr.set( t3.cstr(), &t3 );
	tr.set( t4.cstr(), &t4 );
	tr.set( t5.cstr(), &t5 );

	// --- serialize ---
	serializer 	sz( s );

	sz << ts << tc << tr;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> ds >> dc >> dr;

	// --- check ---
	AssertEqual( ts, ds );
	AssertEqual( tc, dc );
	AssertEqual( tr, dr );
}

// compound data
// -----------------------------------------------------------------------------
void srztest::test71()
{
	this->set_subtitle( "compound data - store policy" );

	xstr		s;

	phash<testplist_s*, store_policy<testplist_s*> > g;
	phash<testplist_s*, store_policy<testplist_s*> > h;
	testptree_s	t;
	testptree_s	u;
	const char*	d = "abcdefghijklmnopqrstuvwxyz";
	
	for ( int i = 0; i < 3; i++ ) {
		testplist_s*	l = new testplist_s;
		const char*	k;

		for ( int j = 0; j < 5; j++ ) {
			xstr	s( &d[ i * 5 + j ], 3 );	
			test*		p = new test( i * 10 + j, s.c_str());

			if ( j == 0 ) k = p->cstr();
			l->enq( p );
			t.set( p->cstr(), dup( p ));
		}
		h.set( k, l );
	}

	// --- serialize ---
	serializer 	sz( s );

	sz << h << t;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> g >> u; 

	// --- check ---
	AssertEqual( g, h );
	AssertEqual( t, u );
}

// -----------------------------------------------------------------------------
void srztest::test72()
{
	// only for test, copy policy is not efficient in this case.

	this->set_subtitle( "compound data - copy policy" );

	xstr		s;

	phash<testplist_c*, copy_policy<testplist_c*> >	g;
	phash<testplist_c*, copy_policy<testplist_c*> >	h;
	testptree_c	v;
	testptree_c	u;
	const char*	d = "abcdefghijklmnopqrstuvwxyz";
	
	for ( int i = 0; i < 3; i++ ) {
		testplist_c	l;
		const char*	k;

		for ( int j = 0; j < 5; j++ ) {
			xstr	s( &d[ i * 5 + j ], 3 );	
			test		t( i * 10 + j, s.c_str());

			if ( j == 0 ) k = dup( t.cstr());
			l.enq( &t );
			v.set( t.cstr(), &t );
		}
		h.set( k, &l );
		delete [] k;
	}

	// --- serialize ---
	serializer 	sz( s );

	sz << h << v;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> g >> u; 

	// --- check ---
	AssertEqual( g, h );
	AssertEqual( v, u );
}

// -----------------------------------------------------------------------------
void srztest::test73()
{
	this->set_subtitle( "compound data - refer policy" );

	xstr		s;

	phash<testplist_r*, refer_policy<testplist_r*> >	g;
	phash<testplist_r*, refer_policy<testplist_r*> >	h;
	testptree_r	t;
	testptree_r	u;

	testplist_r	l[ 3 ];
	testplist_s	b;
	const char*	d = "abcdefghijklmnopqrstuvwxyz";
	
	for ( int i = 0; i < 3; i++ ) {
		const char*	k;

		for ( int j = 0; j < 5; j++ ) {
			xstr		s( &d[ i * 5 + j ], 3 );	
			test*		p = new test( i * 10 + j, s.c_str());

			if ( j == 0 ) k = p->cstr();
			b.enq( p );
			l[ i ].enq( p );
			t.set( p->cstr(), p );
		}
		g.set( k, &l[ i ] );
	}

	// --- serialize ---
	serializer 	sz( s );

	sz << g << t;

	// --- deserialize ---
	deserializer	dz( s );

	dz >> h >> u; 

	// --- check ---
	AssertEqual( g, h );
	AssertEqual( t, u );
}

// -----------------------------------------------------------------------------
void srztest::run()
{
	test11();
	test12();
	test13();
	test14();
#ifndef NO_CPPSTD
	test15();
	test16();
#endif

	test21();
	test22();
	test23();

	test31();

	test41();
	test42();
	test43();
	
	test51();
	test52();
	test53();
	
	test61();
	test62();
	test63();
	
	test71();
	test72();
	test73();
}


// -----------------------------------------------------------------------------
int main( int argc, char** argv )
{
	srztest	t;

	t.run();

	return 0;
}

