// 
// xbuf.h --- extensible string buffer
//
//      Version 1.1
//
//      Copyright (C) 2005, 2006 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_XSTR_H
#define __OMT_XSTR_H

#include <omt/common.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

template <typename C> struct char_traits
{
	typedef C		char_type;
	typedef unsigned long	int_type;

	static const char_type* copy( char_type* s1, const char_type* s2, size_t n )
	{
                if ( s1 && s2 ) 
			for ( size_t i = 0; i < n; ++i ) { *s1++ = *s2; if ( !*s2++ ) break; }
                	return s1;
	} 
	static const char_type* move( char_type* s1, const char_type* s2, size_t n ) 
	{
                	if ( s1 && s2 ) {
			if ( s1 < s2 && !( s1 + n < s2 ))
				for ( size_t i = n; 0 < i; --i ) *( s1 + i ) = *( s2 + i );
			else if ( s1 != s2 )
				for ( size_t i = 0; i < n; ++i ) { *s1++ = *s2; if ( !*s2++ ) break; }
		}
                	return s1;
	}
	static size_t length( const char_type* p )
	{
		size_t	r = 0;
		if ( p ) for ( ; *p; ++r, ++p ) ;
		return r;
	}
	static int_type eof() { return static_cast<int_type>( -1 ); }
};

template <> struct char_traits<char>
{
	typedef char	char_type;
	typedef int	int_type;

	static const char_type* copy( char_type* d, const char_type* s, size_t n )
	{
		memcpy( d, s, n );
	} 
	static const char_type* move( char_type* d, const char_type* s, size_t n ) 
	{
		memmove( d, s, n );
	}
	static size_t length( const char_type* p )
	{
		size_t	r = 0;
		if ( p ) r = strlen( p );
		return r;
	}
	static int_type eof() { return static_cast<int_type>( -1 ); }
};
template <> struct char_traits<wchar_t>
{
	typedef wchar_t	char_type;
	typedef wint_t	int_type;

	static const char_type* copy( char_type* d, const char_type* s, size_t n )
	{
		wmemcpy( d, s, n );
	} 
	static const char_type* move( char_type* d, const char_type* s, size_t n ) 
	{
		wmemmove( d, s, n );
	}
	static size_t length( const char_type* p )
	{
		size_t	r = 0;
		if ( p ) r = wcslen( p );
		return r;
	}
	static int_type eof() { return static_cast<int_type>( -1 ); }
};

//-----------------------------------------------------------------------------
template <typename charT> class xbuf 
{
#ifndef NO_CPPSTD
	typedef std::char_traits<charT> trait; 
#else
	typedef omt::char_traits<charT> trait;
#endif

	size_t		m_size;		// allocated size of m_buf
	mutable	charT*	m_buf;
	size_t		m_pos;		// for put
	mutable size_t	m_cur;		// for get/peek

	static const size_t default_size = 256;

    public:
	explicit xbuf( size_t sz = 0 )
		: m_size( sz ? sz : default_size ), 
		  m_buf( new charT[ m_size ] ),
		  m_pos( 0 ), m_cur( 0 )
	{ }
	explicit xbuf( const charT* s )
		: m_size( s ? trait::length( s ) + 1 : default_size ),
		  m_buf( new charT[ m_size ] ),
		  m_pos( s ? m_size - 1 : 0 ), m_cur( 0 )
	{
		if ( s ) trait::copy( m_buf, s, m_pos );
	}
	xbuf( const charT* s, size_t sz )	// s != 0, and allocated >= sz, 
		: m_size(( sz < default_size ) ? default_size : sz + 1 ),
		  m_buf( new charT[ m_size ] ),
		  m_pos( sz ), m_cur( 0 )
	{
		trait::copy( m_buf, s, m_pos );
	}
	~xbuf() { delete [] m_buf; }

	// stream operations
	void put( const charT a )
	{
		if ( m_pos >= m_size - 1 ) reserve( m_size + m_size );
		m_buf[ m_pos++ ] = a;
	}
	void append( const charT* s )
	{
		if ( s ) append( s, trait::length( s ));
	}
	void append( const charT* s, size_t n )
	{
		// s must not null
		if ( n >= m_size - m_pos - 1 ) { 
			reserve( m_size + (( n >= m_size ) ? n : m_size ));
			trait::copy( &m_buf[ m_pos ], s, n );
		} else {
			trait::move( &m_buf[ m_pos ], s, n );
		}
		m_pos += n;
	}
	void append( const xbuf& x ) { append( x.buf(), x.length()); }

	void set( const charT* const& s )
	{
		clear();
		append( s );
	}
	void set( const charT* const& s, size_t n )
	{
		// s must not null
		clear();
		append( s, n );
	}
	void set( const xbuf& x )
	{
		clear();
		append( x );
	}

	typename trait::int_type pop()
	{
		return ( m_cur < m_pos ) ? m_buf[ --m_pos ] : trait::eof();
	}

	typename trait::int_type peek() const
	{
		return ( m_cur < m_pos ) ? m_buf[ m_cur ] : trait::eof();
	}
	typename trait::int_type get() const
	{
		return ( m_cur < m_pos ) ? m_buf[ m_cur++ ] : trait::eof();
	}
	void unget() const
	{
		if ( m_cur > 0 ) m_cur--;
	}

	bool is_eos() const { return m_cur >= m_pos; }

	size_t tell() { return m_cur; }
	void seek( size_t c ) { m_cur = ( m_pos < c ) ? m_pos : c; }

	void clear() { m_pos = m_cur = 0; }
	void reset() { m_cur = 0; }

	// miscellaneous operation
	// Note: return of buf()/c_str() will be unavailable after reserve().
	const charT* buf() const { return m_buf; }
	size_t length() const { return m_pos; }
	size_t size() const { return m_size; }
	size_t max_length() const { return m_size - 1; }
	const charT* c_str() const
	{
		m_buf[ m_pos ] = charT();
		return m_buf;
	}

	// utility operators
	operator void* () const { return ( m_cur < m_pos ) ? m_buf + m_cur : 0; }
	operator bool() const   { return !is_eos(); } 
	bool operator !() const { return  is_eos(); } 

	charT& operator[] ( size_t n )
	{
		if ( n >= m_pos ) { n = m_pos; m_buf[ m_pos ] = charT(); }
		return m_buf[ n ];
	}
	const charT& operator[] ( size_t n ) const
	{
		if ( n >= m_pos ) { n = m_pos; m_buf[ m_pos ] = charT(); }
		return m_buf[ n ];
	}

	xbuf& operator= ( const charT* const& s ) { set( s ); return *this; }

	// buffer size management
	void reserve( size_t sz )
	{
		if ( sz > m_size ) {
			charT* p = new charT[ sz ];
			trait::copy( p, m_buf, m_pos );
			delete [] m_buf;
			m_buf = p;
			m_size = sz;
		}
	}

    private:
	xbuf( const xbuf& );
	xbuf& operator=( const xbuf& );
};

//-----------------------------------------------------------------------------
template<typename charT>
inline bool operator== ( const xbuf<charT>& a, const xbuf<charT>& b )
{
	return a.length() == b.length() && eql_fn<charT*>()( a.c_str(), b.c_str());
}

template<typename charT>
xbuf<charT>& operator<< ( xbuf<charT>& b, const charT a )
{
	b.put( a );
	return b;
}

template<typename charT>
xbuf<charT>& operator<< ( xbuf<charT>& b, const charT* s )
{
	b.append( s );
	return b;
}

template<typename charT>
xbuf<charT>& operator<< ( xbuf<charT>& b, const xbuf<charT>& x )
{
	b.append( x );
	return b;
}

// typedefs
//-----------------------------------------------------------------------------
typedef xbuf<char>		xstr;
typedef xbuf<unsigned char>	xustr;
typedef xbuf<wchar_t>		xwstr;

#ifndef NO_NAMESPACE
}
#endif

#endif // __OMT_XSTR_H

