//
// srz.h -- serializer/deserializer for Open Web Middleware 
//
//	Version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
////////////////////////////////////////////////////////////////////////////////
// External Stream Definition
//
// Serialized Stream Symtax
//	serialized_stream ::= [ version sequence ][ term ]*
//	version		::= SYM
//	sequence	::= INT		// 0 or negative: error
//	term		::= [ control ]* data
//	control		::= PGM [ INT | SYM ]
//	data		::= list | string | binary | integer | floating | null | pointer
//	list            ::= LPR [ term ]* RPR
//	string          ::= STR 
//	binary          ::= BIN
//	integer         ::= INT
//	floating        ::= FLT
//	bool		::= BOL
//	null		::= NUL
//	pointer		::= SYM [ LBL term ]
//
// Token and Character Set ( tokenizer setting ):
//	INT ::= /[-][0-9][0-9]*/
//	      | /0[0-7][0-7]*/
//	      | /0x[0-9A-F][0-9A-F]*/
//	FLT ::= /[-][0-9][0-9]*[.[0-9]*][e[-][0-9][0-9]*]/
//	      | /[-][0-9]*.[0-9][0-9]*[e[-][0-9][0-9]*]/ 
//	STR ::=	'"' escaped_cstr '"'
//	BIN ::= '\'' decode64 '\''
//	LPR ::= '('
//	RPR ::= ')'
//	LBL ::= '@'
//	PGM ::= '#'
//	BOL ::= '0' | '1'
//	NUL ::= '0'
//	SYM ::= /[a-zA-Z:_][0-9a-zA-Z.-:_]*/
//
// Controls
//	PGM INT              resize token buffer 
//	PGM SYM	             not implemented ( for future extension )
//
////////////////////////////////////////////////////////////////////////////////

#ifndef __OMT_SRZ_H
#define __OMT_SRZ_H

#include <cstdlib>
#include <cstdarg>

#ifndef NO_CPPSTD
#include <iosfwd>
#include <string>
#endif

#include <omt/common.h>
#include <omt/filter.h>
#include <omt/tkstream.h>
#include <omt/plist.h>
#include <omt/phash.h>
#include <omt/ptree.h>
#include <omt/gstream.h>
#include <omt/xbuf.h>
		
#ifndef NO_NAMESPACE
namespace omt {
#endif

//
// ---- specification ----
//

// token
// -----------------------------------------------------------------------------
typedef int token;

const int tkERR = -1;
const int tkEOS = 0;
const int tkLCM = ';';
const int tkLPR = '(';
const int tkRPR = ')';
const int tkSTR = '"';
const int tkBIN = '\'';
const int tkLBL = '@';
const int tkPGM = '#';
const int tkSYM = 256;
const int tkINT = 257;
const int tkFLT = 258;

// data pool (hashtable) size
// -----------------------------------------------------------------------------
const unsigned DPoolSize = 1024;

// utility classes
// -----------------------------------------------------------------------------
class null { };

class rawstream : public xstr
{
    public:
	rawstream() { } 
	rawstream( const char* cbuf, size_t n ) : xstr( cbuf, n ) { }
	rawstream( const char* cstr ) : xstr( cstr ) { }
	rawstream( const xstr& s ) : xstr( s.c_str(), s.length()) { }
#ifndef NO_CPPSTD
	rawstream( const std::string& s ) : xstr( s.c_str(), s.length()) { }
#endif
};

class binary : public xstr
{
    public:
	binary() { } 
	binary( const char* cbuf, size_t n ) : xstr( cbuf, n ) { }
	binary( const char* cstr ) : xstr( cstr ) { }
	binary( const xstr& s ) : xstr( s.c_str(), s.length()) { }
#ifndef NO_CPPSTD
	binary( const std::string& s ) : xstr( s.c_str(), s.length()) { }
#endif
};

// Deserializer
// =============================================================================
class deserializer
{
	// --- data pool -------------------------------------------------------
	class pbase
	{
	    public:
		virtual ~pbase() { }
		virtual void read( deserializer& in ) = 0;
	};

	template <typename T> class pdata : public pbase
	{
		T*		m_ptr;

	    public:
		pdata( T* p ) : m_ptr( p ) { }	// store p as m_ptr
		~pdata() { delete m_ptr; }	// m_ptr may be 0

		T* get() { return m_ptr; }
		void read( deserializer& in ) { if ( m_ptr ) in >> *m_ptr; }
	};

	typedef phash<pbase*, store_policy<pbase*>, DPoolSize > pool;	

	// --- tokenizer ----------------------------------------------------------
	static const unsigned char ct_delm = 0x01;	// delimiter-char
	static const unsigned char ct_tknc = 0x04; 	// token-char
	static const unsigned char ct_sym1 = 0x10;	// symbol-char-first
	static const unsigned char ct_sym2 = 0x20;	// symbol-char-cont
	static const unsigned char ct_prnt = 0x80;	// valid character for tokenizing

	static const unsigned char ctable[ 256 ];	// character table

	typedef enum { ng, ok, un } state;		// input status
	static const size_t ibufsize = 1024;		// initial read buffer size

	// --- members ------------------------------------------------------------
	pool  		m_pool;		// data pool for read pointers
	itkstream	m_in;		// input stream
	xstr	  	m_buf;		// input buffer
	token		m_token;	// current token
	state		m_state;	// input status

    public:
	explicit deserializer( FILE* in = stdin ) 
		: m_in( in ), m_buf( ibufsize ), m_token( tkERR ), m_state( ok ) { }
	explicit deserializer( const char* in ) 
		: m_in( in ), m_buf( ibufsize ), m_token( tkERR ), m_state( ok ) { }
	explicit deserializer( const xstr& in ) 
		: m_in( in ), m_buf( ibufsize ), m_token( tkERR ), m_state( ok ) { }
#ifndef NO_CPPSTD
	explicit deserializer( std::istream& in ) 
		: m_in( in ), m_buf( ibufsize ), m_token( tkERR ), m_state( ok ) { }
	explicit deserializer( const std::string& in ) 
		: m_in( in ), m_buf( ibufsize ), m_token( tkERR ), m_state( ok ) { }
#endif

	~deserializer() { } 		// do not derive this class

	// specific deserializing 
	bool beginList() { return get_if( tkLPR ); }
	bool endList()   { return get_if( tkRPR ); }
	bool getIfStr( xstr& s );
	bool getIfBin( binary& s );
	bool getIfInt( long& n );
	bool getIfFlt( double& n );
	bool getIfBool();
	bool getIfNull(); 		// check INT(0), for read pointer
	template <typename T> bool getIfPtr( T*& p );

	// for deserialize operator
	void readStr( xstr& s ) { if ( !getIfStr( s )) setfail( "token STR required" ); }
	void readBin( binary& x ) { if ( !getIfBin( x )) setfail( "token BIN required" ); }
	long readInt()   { long   r; if ( !getIfInt( r )) setfail( "token INT required" ); return r; }
	double readFlt() { double r; if ( !getIfFlt( r )) setfail( "token FLT required" ); return r; }
	bool readBool()  { bool r = getIfBool(); if ( !r ) setfail( "token INT(0 or 1) required" ); return r; }
	void readNull();		// skip one deserialize unit
	template <typename T>
	T* readPtr() { T* r; if ( !getIfPtr( r )) setfail( "token SYM required" ); return r; }

#ifndef NO_CPPSTD
	bool getIfStr( std::string& s );
	void readStr( std::string& s ) { if ( !getIfStr( s )) setfail( "token STR required" ); }
#endif

	// raw stream support -- read one deserialize-unit
	token scan( xstr& s );

	// input status
	operator bool() { return is_ok() && !is_eof(); }
	const char* getError() { return m_buf.c_str(); }
	void resetfail() { m_state = ok; }

	// test version and sequence of the stream
	long checkVersion( const char* ver );

	// for access by serializer
	static bool isdelm( int c ) { return ( ctable[ c & 0xFF ] & ct_delm ) ? true : false; }
	static bool istknc( int c ) { return ( ctable[ c & 0xFF ] & ct_tknc ) ? true : false; }
	static bool issym1( int c ) { return ( ctable[ c & 0xFF ] & ct_sym1 ) ? true : false; }
	static bool issym2( int c ) { return ( ctable[ c & 0xFF ] & ct_sym2 ) ? true : false; }
	static bool isprnt( int c ) { return ( ctable[ c & 0xFF ] & ct_prnt ) ? true : false; }

	// --- for debug ---
	static const char* token2id( token tk );

    private:
	deserializer( const deserializer& );
	deserializer& operator=( const deserializer& );

	// --- tokenizer -------------------------------------------------------
	token get();
	bool get_if( token t );
	void unget() { unget_token(); }

	token get_token()
	{
		if      ( m_state == ok ) m_token = tokenize();
		else if ( m_state == un ) m_state = ok;

		return ( m_state != ng ) ? m_token : tkERR;
	} 
	token peek_token()
	{
		if ( m_state == ok ) { m_token = tokenize(); m_state = un; }

		return ( m_state != ng ) ? m_token : tkERR;
	}
	void unget_token()
	{
		if ( m_state == ok ) m_state = un;
		else		     m_state = ng;
	}

	token tokenize();
	token tokenizeSymbol( int a );

	// tokenizer status
	void setfail( const char* msg, ... );
	bool is_ok() { return m_state != ng; }
	bool is_eof() { return m_state == ng || m_state == ok && m_in.is_eof(); }

	// retrieve
	long getint() const { return strtol( m_buf.c_str(), 0, 0 ); }
	double getflt() const { return strtod( m_buf.c_str(), 0 ); }

	// miscellaneous
	unsigned getline() const { return m_in.line(); }

	int skip() { return m_in.skip(); }
	int skip( char end ) { return m_in.skip( end ); }
	int skip( const char* pat ) { return m_in.skip( pat ); }

	void reserve( long sz ) { m_buf.reserve( sz ); }
};

// Serializer
// =============================================================================
class serializer
{
	// --- data pool -------------------------------------------------------
	typedef phash<unsigned, refer_policy<unsigned>, DPoolSize, const void*, refer_policy<const void*> > pool;

	// --- members ------------------------------------------------------------
	gostream*	m_out;
	bool		m_space;
	pool		m_pool;
	unsigned	m_cnt;

    public:
	explicit serializer( FILE* out = stdout )
		: m_out( new gout<FILE*>( out )), m_space( false ), m_cnt( 0 ) { }
	explicit serializer( gostream& out )
		: m_out( new gout<gostream>( out )), m_space( false ), m_cnt( 0 ) { }
	explicit serializer( xstr& out )
		: m_out( new gout<xstr>( out )), m_space( false ), m_cnt( 0 ) { }
#ifndef NO_CPPSTD
	explicit serializer( std::ostream& out )
		: m_out( new gout<std::ostream>( out )), m_space( false ), m_cnt( 0 ) { }
	explicit serializer( std::string& out )
		: m_out( new gout<std::string>( out )), m_space( false ), m_cnt( 0 ) { }
#endif
	virtual ~serializer() { delete m_out; }

	void beginList() { nospace(); m_out->put( '(' ); }
	void endList()   { nospace(); m_out->put( ')' ); }

	void writeStr( const char* s ) { xstr x( s ); writeStr( x ); }
	void writeStr( const xstr& s );
	void writeBin( const char* b, size_t n ) { xstr x( b, n ); writeBin( x ); }
	void writeBin( const xstr& x );
#ifndef NO_CPPSTD
	void writeStr( const std::string& s ) { xstr x( s.c_str()); writeStr( x ); }
	void writeBin( const std::string& s ) { xstr x( s.c_str(), s.length()); writeBin( x ); }
#endif
	void writeInt( const long n ) { space(); m_out->write_int( n ); }
	void writeFlt( const double n ) { space(); m_out->write_flt( n ); }
	void writeBool( const bool b ) { space(); m_out->put( b ? '1' : '0' ); }
	void writeNull() { space(); m_out->put( '0' ); }

	// raw stream support
	void copy( const xstr& x ) { m_out->write_str( x.c_str()); }

	// PTR/LBL support
	template <typename T> void writePtr( const T* p );

	// output version and sequence into the stream
	bool writeVersion( const char* ver, long seq = 1 );

	// for stream formatting
	void writeNewLine() { nospace(); m_out->put( '\n' ); }

    private:
	serializer( const serializer& out );
	serializer& operator=( const serializer& out );

	// ---------------------------------------------------------------------
	void space()
	{
		if ( m_space ) m_out->put( ' ' );
		m_space = true;
	}
	void nospace() { m_space = false; }
	bool is_symbol( const char* sym )
	{
		const char*	p = sym;
		bool		r = true;

		if      ( !p ) r = false;
		else if ( !deserializer::issym1( *p++ )) r = false;
		else while ( *p ) {
			if ( !deserializer::issym2( *p++ )) {
				r = false;
				break;
			}
		}
		return r;
	}
};

// 
// implementation 
//

// -----------------------------------------------------------------------------
// deserializer
// -----------------------------------------------------------------------------
inline token deserializer::get()
{
	token	r;

	while (( r = get_token()) == tkLCM ) m_in.skip();

	if ( r == tkPGM ) {
		if (( r = get_token()) == tkINT ) {
			reserve( getint());
		} else {
			setfail( "undefined pragma" );
		}
		r = get();
	}
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::get_if( token t )
{
	bool	r = false;

	if ( get() == t ) {
		 if ( is_ok()) r = true;
	} else {
		 unget_token();
	}
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::getIfStr( xstr& s )
{
	token	t = get();
	bool	r = true;

	if      ( t == tkSTR ) s.set( m_buf );
	else if ( t == tkINT && getint() == 0 ) s.clear();
	else {
		r = false;
		unget_token();
	}
	return r;
}

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
inline bool deserializer::getIfStr( std::string& s )
{
	token	t = get();
	bool	r = true;

	if      ( t == tkSTR ) s = m_buf.c_str();
	else if ( t == tkINT && getint() == 0 ) s.clear();
	else {
		r = false;
		unget_token();
	}
	return r;
}
#endif

// -----------------------------------------------------------------------------
inline bool deserializer::getIfBin( binary& x )
{
	token	t = get();
	bool	r = true;

	if      ( t == tkBIN ) x.set( m_buf );
	else if ( t == tkINT && getint() == 0 ) x.clear();
	else {
		r = false;
		unget_token();
	}
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::getIfInt( long& l )
{
	bool	r = ( get() == tkINT );

	if ( r ) l = getint(); else unget_token();
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::getIfFlt( double& d )
{
	token	t = get();
	bool	r = true;

	if      ( t == tkFLT ) d = getflt();
	else if ( t == tkINT ) d = getint();
	else {
		r = false;
		unget_token();
	}
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::getIfBool()
{
	bool	r = false;

	if ( get() == tkINT ) r = ( getint() != 0 ); else unget_token();
	return r;
}

// -----------------------------------------------------------------------------
inline bool deserializer::getIfNull()
{
	bool	r = ( get() == tkINT && getint() == 0 );

	if ( !r ) unget_token();
	return r;
}

// -----------------------------------------------------------------------------
template <typename T>
inline bool deserializer::getIfPtr( T*& pr )
{
	bool	r = true;
	token	t;

	if (( t = get()) == tkSYM ) {
		if ( pool::dref dr = m_pool.get( m_buf.c_str())) {
			if ( pdata<T>* p = dynamic_cast<pdata<T>*>( dr.get() )) {
				pr = p->get();
			} else {
				//omt_error( ERR_ILLEGAL_OP, "bad cast on read pointer" );
				fprintf( stderr, "bad cast on read pointer" );
				exit( -1 );
			}
		} else {
			pr = new T();
		 	dr.set( new pdata<T>( pr ));
			if ( get_if( tkLBL )) *this >> *pr;
		}
	} else if ( t == tkINT && getint() == 0 ) {
		pr = 0;
	} else {
		r = false;
	}
	return r;
}

// -----------------------------------------------------------------------------
template <>
inline bool deserializer::getIfPtr<char>( char*& pr )
{
	bool	r = true;
	token	t;

	if (( t = get()) == tkSYM ) {
		if ( pool::dref dr = m_pool.get( m_buf.c_str())) {
			if ( pdata<char>* p = dynamic_cast<pdata<char>*>( dr.get() )) {
				pr = p->get();
			} else {
				//omt_error( ERR_ILLEGAL_OP, "bad cast on read pointer" );
				fprintf( stderr, "bad cast on read pointer" );
				exit( -1 );
			}
		} else if ( get_if( tkLBL )) {
			if (( t = get()) == tkSTR ) {
				pr = dup( m_buf.c_str());
				dr.set( new pdata<char>( pr ));
			} else if ( t == tkINT && getint() == 0 ) {
				pr = 0;
			} else {
				//omt_error( ERR_PARSER, "token STR required for C-string" );
				fprintf( stderr, "token STR required for C-string" );
				exit( -1 );
			}
		} else {
			//omt_error( ERR_ILLEGAL_OP, "unregistered pointer" );
			fprintf( stderr, "unregistered pointer" );
			exit( -1 );
		}
	} else if ( t == tkINT && getint() == 0 ) {
		pr = 0;
	} else {
		r = false;
	}
	return r;
}

// -----------------------------------------------------------------------------
inline void deserializer::readNull()
{
	unsigned	c = 0;
	token		t;

	do {
		t = get();
		if      ( t == tkLPR ) ++c;
		else if ( t == tkRPR ) --c;
	}
	while ( c > 0 && t != tkEOS && t != tkERR );
}

// -----------------------------------------------------------------------------
inline token deserializer::scan( xstr& s ) 
{
	unsigned	c = 0;
	token		r;

	m_in.keep();

	do {
		r = get();
		if      ( r == tkLPR ) ++c;
		else if ( r == tkRPR ) --c;
	}
	while ( c > 0 && r != tkEOS && r != tkERR );

	s.set( m_in.kept());

	return r;
}

// -----------------------------------------------------------------------------
inline long deserializer::checkVersion( const char* ver )
{
	long	r = 0;

	if ( get_token() == tkSYM && ver && strcmp( m_buf.c_str(), ver ) == 0 ) {
		if ( get_token() == tkINT ) 
			r = getint();
		else
			setfail( "INT required as version sequence" );
	} else {
		//omt_error( ERR_PARSER, "version error: %s/%s", m_buf.c_str(), ver );
		fprintf( stderr, "version error: %s/%s", m_buf.c_str(), ver );
		exit( -1 );
	}
	return r;
}

// Operator Overlode for Deserializer
// -----------------------------------------------------------------------------
template <typename T>
inline deserializer& operator>>( deserializer& in, T& objref )
{
	// deserialize user defined class 
	if ( in ) objref.deserialize( in );
	return in;
}

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <std::string> ( deserializer& in, std::string& s )
{
	if ( in ) in.readStr( s ); 
	return in;
}
#endif

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <xstr> ( deserializer& in, xstr& s )
{
	if ( in ) in.readStr( s );
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <binary> ( deserializer& in, binary& x )
{
	if ( in ) in.readBin( x );
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <long> ( deserializer& in, long& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <short> ( deserializer& in, short& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <int> ( deserializer& in, int& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <char> ( deserializer& in, char& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <unsigned long> ( deserializer& in, unsigned long& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <unsigned int> ( deserializer& in, unsigned int& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <unsigned short> ( deserializer& in, unsigned short& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <unsigned char> ( deserializer& in, unsigned char& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <signed char> ( deserializer& in, signed char& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <wchar_t> ( deserializer& in, wchar_t& l )
{
	if ( in ) l = in.readInt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <float> ( deserializer& in, float& f ) 
{
	if ( in ) f = in.readFlt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <double> ( deserializer& in, double& f ) 
{
	if ( in ) f = in.readFlt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <long double> ( deserializer& in, long double& f ) 
{
	if ( in ) f = in.readFlt();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <bool> ( deserializer& in, bool& b ) 
{
	if ( in ) b = in.readBool();
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <null> ( deserializer& in, null& n ) 
{
	if ( in ) in.readNull();
	return in;
}

// rawstream
// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <rawstream> ( deserializer& in, rawstream& r ) 
{
	if ( in ) in.scan( r );
	return in;
}

// -----------------------------------------------------------------------------
template <>
inline deserializer& operator>> <serializer> ( deserializer& in, serializer& out ) 
{
	xstr	s;

	if ( in ) in.scan( s );
	out.copy( s );
	return in;
}

// pointer
// -----------------------------------------------------------------------------
template <typename T>
inline deserializer& operator>> ( deserializer& in, T*& p ) 
{
	if ( in ) p = in.readPtr<T>();
	return in;
}

// -----------------------------------------------------------------------------
template <typename T>
inline deserializer& operator>> ( deserializer& in, const T*& p ) 
{
	if ( in ) p = in.readPtr<T>();
	return in;
}

// plist
// -----------------------------------------------------------------------------
template<typename T, typename P>
inline deserializer& operator>> ( deserializer& in, plist<T,P>& l ) 
{
	if ( in.beginList()) {
		l.clear();
		while ( !in.endList()) { T p; in >> p; l.enq( P::copy( p )); }
	}
	return in;
}

// phash
// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename C>
inline deserializer& operator>> ( deserializer& in, phash<T,P,Z,K,Q,H,C>& h ) 
{
	T	p;
	K	k;

	if ( in.beginList()) {
		h.clear();
		while ( !in.endList()) { in >> k >> p; h.set( k, P::copy( p )); }
	}
	return in;
}

// ptree
// -----------------------------------------------------------------------------
template<typename T, typename P, typename K, typename Q, typename C>
inline deserializer& operator>> ( deserializer& in, ptree<T,P,K,Q,C>& t ) 
{
	T	p;
	K	k;

	if ( in.beginList()) {
		t.clear();
		while ( !in.endList()) { in >> k >> p; t.set( k, P::copy( p )); }
		t.balance();
	}
	return in;
}

// Manipurator
// -----------------------------------------------------------------------------
typedef deserializer& ( *desrzmanip )( deserializer& );

// -----------------------------------------------------------------------------
inline deserializer& begin( deserializer& in )
{
	in.beginList();
	return in;
}

// -----------------------------------------------------------------------------
inline deserializer& end( deserializer& in )
{
	in.endList();
	return in;
}

// -----------------------------------------------------------------------------
inline deserializer& operator>> ( deserializer& in, const desrzmanip& f )
{
	return ( *f )( in );
}

// plist
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// serializer
// -----------------------------------------------------------------------------
inline void serializer::writeStr( const xstr& x )
{
	unsigned long n = x.length();

	nospace();

	if ( n > 256 ) {
		m_out->put( '#' );
		m_out->write_int( n );
	}
	m_out->put( '\"' );
	gin<xstr> in( x );
	escape_cstr( in, *m_out, '"' );
	m_out->put( '\"' );
}

// -----------------------------------------------------------------------------
inline void serializer::writeBin( const xstr& x )
{
	unsigned long n = x.length();

	nospace();

	if ( n > 256 ) {
		m_out->put( '#' );
		m_out->write_int( n );
	}
	m_out->put( '\'' );
	gin<xstr> in( x );
	encode64( in, *m_out, '\'' );
	m_out->put( '\'' );
}

// -----------------------------------------------------------------------------
template <typename T>
inline void serializer::writePtr( const T* p )
{
	// p must be not null
	space();

	if ( pool::dref dr = m_pool.get( p )) {
		m_out->put( '_' );
		m_out->write_int( dr.get());
	} else {
		dr.set( ++m_cnt );
		m_out->put( '_' );
		m_out->write_int( m_cnt );
		nospace();
		m_out->put( '@' );
		*this << *p;
	}
}

// -----------------------------------------------------------------------------
template <>
inline void serializer::writePtr<char>( const char* p )
{
	// p must be not null
	space();

	if ( pool::dref dr = m_pool.get( p )) {
		m_out->put( '_' );
		m_out->write_int( dr.get());
	} else {
		dr.set( ++m_cnt );
		m_out->put( '_' );
		m_out->write_int( m_cnt );
		nospace();
		m_out->put( '@' );
		writeStr( p );
	}
}

// -----------------------------------------------------------------------------
inline bool serializer::writeVersion( const char* ver, long seq )
{
	bool	r;

	if ( r = is_symbol( ver )) {
		nospace();
		m_out->write_str( ver );
		m_out->put( ' ' );
		m_out->write_int( seq );
		m_out->put( '\n' );
	}
	return r;
}

// Operator Overlode for Serlializer
// -----------------------------------------------------------------------------
template<typename T>
inline serializer& operator<< ( serializer& out, const T& objref )
{
	// serialize user defined class 
	objref.serialize( out );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <xstr> ( serializer& out, const xstr& s )
{
	out.writeStr( s );
	return out;
}

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <std::string> ( serializer& out, const std::string& s )
{
	out.writeStr( s );
	return out;
}
#endif

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <binary> ( serializer& out, const binary& x )
{
	out.writeBin( x );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <long> ( serializer& out, const long& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <int> ( serializer& out, const int& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <short> ( serializer& out, const short& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <char> ( serializer& out, const char& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <unsigned long> ( serializer& out, const unsigned long& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <unsigned int> ( serializer& out, const unsigned int& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <unsigned short> ( serializer& out, const unsigned short& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <unsigned char> ( serializer& out, const unsigned char& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <signed char> ( serializer& out, const signed char& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <wchar_t> ( serializer& out, const wchar_t& l )
{
	out.writeInt( l );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <double> ( serializer& out, const double& f )
{
	out.writeFlt( f );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <long double> ( serializer& out, const long double& f )
{
	out.writeFlt( f );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <float> ( serializer& out, const float& f )
{
	out.writeFlt( f );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <bool> ( serializer& out, const bool& b )
{
	out.writeBool( b );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <null> ( serializer& out, const null& n )
{
	out.writeNull();
	return out;
}

// rawstream
// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <rawstream> ( serializer& out, const rawstream& r )
{
	out.copy( r );
	return out;
}

// -----------------------------------------------------------------------------
template<>
inline serializer& operator<< <deserializer> ( serializer& out, const deserializer& in )
	// Note: the argument 'in' is capped 'const' for templeate matching, but not const
{
	xstr	s;

	const_cast<deserializer&>( in ).scan( s );
	out.copy( s );
	return out;
}

// pointer
// -----------------------------------------------------------------------------
template<typename T>
inline serializer& operator<< ( serializer& out, T* p )
{
	if ( p ) out.writePtr( p ); else out.writeNull();
	return out;
}

// -----------------------------------------------------------------------------
template<typename T>
inline serializer& operator<< ( serializer& out, const T* p )
{
	if ( p ) out.writePtr( p ); else out.writeNull();
	return out;
}

// plist
// -----------------------------------------------------------------------------
template<typename T, typename P>
inline serializer& operator<<( serializer& out, const plist<T,P>& l ) 
{
	typename plist<T,P>::const_itor i( l );	

	out.beginList();
	for ( i.init(); i.cont(); i.next()) {
		if ( T v = i.get()) out << v; else out.writeNull();
	}
	out.endList();

	return out;
}

// phash
// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename C>
inline serializer& operator<<( serializer& out, const phash<T,P,Z,K,Q,H,C>& h ) 
{
	typename phash<T,P,Z,K,Q,H,C>::const_itor i( h );	

	out.beginList();
	for ( i.init(); i.cont(); i.next()) {
		if ( K k = i.key()) out << k; else {
			out.writeNull();
#ifdef _DEBUG
			fprintf( stderr, "Null key on phash\n" );
#endif
		}
		if ( T v = i.get()) out << v; else out.writeNull(); 
	}
	out.endList();

	return out;
}

// ptree
// -----------------------------------------------------------------------------
template<typename T, typename P, typename K, typename Q, typename C>
inline serializer& operator<<( serializer& out, const ptree<T,P,K,Q,C>& t ) 
{
	typename ptree<T,P,K,Q,C>::const_itor i( t );	

	out.beginList();
	for ( i.init(); i.cont(); i.next()) {
		if ( K k = i.key()) out << k; else {
			out.writeNull();
#ifdef _DEBUG
			fprintf( stderr, "Null key on ptree\n" );
#endif
		}
		if ( T v = i.get()) out << v; else out.writeNull(); 
	}
	out.endList();

	return out;
}

// manipurator
// -----------------------------------------------------------------------------
typedef serializer& ( *srzmanip )( serializer& );

// -----------------------------------------------------------------------------
inline serializer& begin( serializer& out )
{
	out.beginList();
	return out;
}

// -----------------------------------------------------------------------------
inline serializer& end( serializer& out )
{
	out.endList();
	return out;
}

// -----------------------------------------------------------------------------
inline serializer& newline( serializer& out )
{
	out.writeNewLine();
	return out;
}

// -----------------------------------------------------------------------------
inline serializer& operator<< ( serializer& out, const srzmanip& f )
{
	return ( *f )( out );
}

#ifndef NO_NAMESPACE
}
#endif

#endif

