//
// phash.h -- pointer based hash table template for Open Middleware
//
//	version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_PHASH_H
#define __OMT_PHASH_H

#include <omt/common.h>
#include <omt/plist.h>
#include <omt/pcontainer.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

//
// Specification
//
// -----------------------------------------------------------------------------
template<typename T,
         typename P = refer_policy< T >, 
	 size_t Z = 256,
	 typename K = const char*,
         typename Q = copy_policy< K >,
	 typename H = hash_fn< K >,
	 typename E = eql_fn< K > >
class phash  
{
	class itor_base;
	struct item;

	item**					m_table;
	plist<item*, store_policy<item*> >	m_removed;

    public:
	class dref;

	typedef	T	value_type;
	typedef K	key_type;
	typedef P	policy_type;
	typedef Q	key_policy_type;
	typedef H	hash_function_type;
	typedef E	eql_function_type;

	// --- constructor/destructor ------------------------------------------
	phash();
	~phash();

	// --- phash operations ------------------------------------------------
        const T find( const K& k ) const;
	dref get( const K& k ) const;

        T set( const K& k, const T& p );
        bool replace( const K& k, const T& p );
        bool insert( const K& k, const T& p );

	T remove( const K& k );
	void clear();
	size_t size() const { return Z; }
	size_t length() const;
	void swap( phash<T,P,Z,K,Q,H,E>& ph );

	// --- omt style iterator ----------------------------------------------
	class const_itor : private itor_base
	{
	    public:
		explicit const_itor( const phash<T,P,Z,K,Q,H,E>& rh )
			: itor_base( const_cast<phash<T,P,Z,K,Q,H,E>&>( rh )) { }
		~const_itor() { }

		void init() { itor_base::init(); }
		void next() { itor_base::next(); }
		bool cont() const { return itor_base::cont(); }

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }

	    private:
		const_itor( const const_itor& );
		const_itor& operator=( const const_itor& );
	};
	class itor : private itor_base
	{
		using itor_base::m_hitem;
		using itor_base::m_index;
		using itor_base::m_rhash;
		bool  		 m_dflag;

	    public:
		explicit itor( phash<T,P,Z,K,Q,H,E>& rh )
			: itor_base( rh ), m_dflag( false ) { }
		~itor() { }

		void init() { itor_base::init(); }
		void next();
		bool cont() const { return itor_base::cont(); }

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }
		T set( const T& p );	// return old data pointer if P is refer_policy
		T del() { m_dflag = true; return get(); }

	    private:
		itor( const itor& );
		itor& operator=( const itor& );
	};

	// --- data reference class for return value of get() ------------------
	class dref
	{
		friend class phash<T,P,Z,K,Q,H,E>;

		typedef typename phash<T,P,Z,K,Q,H,E>::item*	pitem;

		mutable pitem*		m_hitem;
		phash<T,P,Z,K,Q,H,E>*	m_phash;
		K	       		m_key;

	    public:
		dref() : m_hitem( 0 ), m_phash( 0 ), m_key( K() ) { }
		dref( const dref& r )
			: m_hitem( r.m_hitem ), m_phash( r.m_phash ), m_key( Q::copy( r.m_key ))
		{ }
		dref& operator=( const dref& r );
		~dref() { Q::release( m_key ); }

		// check also the container had modified
		bool check() const;

		const K key() const { return check() ? ( *m_hitem )->m_key : K(); }
		const T get() const { return check() ? ( *m_hitem )->m_val : T(); }
		T set( const T& p );
		T del();

		// utility
		operator const T() const { return get(); }
		operator bool() const { return check(); }
		const T operator=( const T& p ) { set( p ); return p; }

	     private:
		// called from phash::get()
		dref( pitem* hitem, const phash<T,P,Z,K,Q,H,E>* ph, const K key )
			: m_hitem( hitem ), m_phash( const_cast<phash<T,P,Z,K,Q,H,E>*>( ph )), m_key( key )
		{ }
	};

#ifdef _DEBUG
	void print()
	{
		for ( size_t i = 0; i < Z; ++i ) {
			printf( "%3d: ", i );
			for ( item* p = m_table[ i ]; p; p = p->m_next ) printf( "%p, ", p->m_key );
			printf( "\n", i );
		}
	}
#endif

    private:
	phash( const phash& );
	phash& operator=( const phash& );

	item** find_item( const K& k ) const { return find_item( k, &m_table[ H()( k ) % Z ] ); }
	item** find_item( const K& k, item** p ) const;

	// ---------------------------------------------------------------------
	struct item
	{
		K	 m_key;
		T	 m_val;
		item*	 m_next;

		item( const K k, const T& p )
			: m_key( Q::set( k )), m_val( P::set( p )), m_next( 0 )
		{ }
		~item() { P::del( m_val ); Q::del( m_key ); }
	
		item* dup() const { return new item( Q::copy( m_key ), P::copy( m_val )); }
		T setdata( const T& p )
		{
			T r = P::del( m_val );
			m_val = P::set( p );
			return	r;
		}
		T remove()
		{
			m_val = P::del( m_val );
			m_key = Q::del( m_key );
			return m_val;
		}	

            private:
                item( const item& );
                item& operator= ( const item& );
	};

	// ---------------------------------------------------------------------
	class itor_base
	{
	    protected:
		typename phash<T,P,Z,K,Q,H,E>::item*const* m_hitem;
		size_t					   m_index;
		phash<T,P,Z,K,Q,H,E>&		 	   m_rhash;

		explicit itor_base( phash<T,P,Z,K,Q,H,E>& rh )
			: m_hitem( 0 ), m_index( 0 ), m_rhash( rh )
		{ }
		~itor_base() { }

		void init();
		void next();
		bool cont() const { return ( m_hitem && *m_hitem ) ? true : false; }

		const K key() const { return cont() ? ( *m_hitem )->m_key : K(); }
		const T get() const { return cont() ? ( *m_hitem )->m_val : T(); }

	    private:
		itor_base( const itor_base& );
		itor_base& operator=( itor_base& );
	};
};

// utility templates
// -----------------------------------------------------------------------------
template<>
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
struct dup_fn<phash<T,P,Z,K,Q,H,E>*>
{
        // default: new with copy constructor
        phash<T,P,Z,K,Q,H,E>* operator()( const phash<T,P,Z,K,Q,H,E>* p )
        {
                phash<T,P,Z,K,Q,H,E>*	r = 0;

                if ( p ) {
                        r = new phash<T,P,Z,K,Q,H,E>;

                        typename phash<T,P,Z,K,Q,H,E>::const_itor    i( *p );
                        for ( i.init(); i.cont(); i.next())
				r->set( Q::copy( i.key()), P::copy( i.get()) );
                }
                return r;
        }
};

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
bool operator==( const phash<T,P,Z,K,Q,H,E>& a, const phash<T,P,Z,K,Q,H,E>& b )
{
	typename phash<T,P,Z,K,Q,H,E>::const_itor	i( a );
	typename phash<T,P,Z,K,Q,H,E>::const_itor	j( b );
	eql_fn<T>					f_eql;

	for ( i.init(), j.init(); i.cont() && j.cont(); i.next(), j.next())
        	if ( !f_eql( i.get(), j.get()) ) break;

	return !( i.cont() || j.cont());
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void swap( phash<T,P,Z,K,Q,H,E>& a, phash<T,P,Z,K,Q,H,E>& b )
{
        a.swap( b );
}

//
// Implementation
//
// -----------------------------------------------------------------------------

// phash member functions
// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline phash<T,P,Z,K,Q,H,E>::phash()
{
	m_table = new item*[ Z ];
	memset( m_table, 0, sizeof( item* ) * Z );
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline phash<T,P,Z,K,Q,H,E>::~phash()
{
	clear();
	delete [] m_table;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline const T phash<T,P,Z,K,Q,H,E>::find( const K& k ) const
{
        item**  h = find_item( k );

        return ( *h ) ? ( *h )->m_val : T();
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline typename phash<T,P,Z,K,Q,H,E>::dref phash<T,P,Z,K,Q,H,E>::get( const K& k ) const
{
	item**	t = &m_table[ H()( k ) % Z ];
	return dref( find_item( k, t ), this, k );
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline T phash<T,P,Z,K,Q,H,E>::set( const K& k, const T& p )
{
        item**  h = find_item( k );
        T       r = T();

        if ( *h ) {
                r = ( *h )->setdata( p );
		Q::release( k );
        } else {
                *h = new item( k, p );
	}
        return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline bool phash<T,P,Z,K,Q,H,E>::replace( const K& k, const T& p )
{
        item**  h = find_item( k );
        bool    r = *h ? true : false;

        if ( *h ) 
		( *h )->setdata( p );
	else
		P::release( p );

        return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline bool phash<T,P,Z,K,Q,H,E>::insert( const K& k, const T& p )
{
        item**  h = find_item( k );
        bool    r = *h ? false : true;

        if ( *h ) {
		P::release( p );
		Q::release( k );
	} else {
		*h = new item( k, p );
	}
        return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline T phash<T,P,Z,K,Q,H,E>::remove( const K& k )
{
	item**	b = &m_table[ H()( k ) % Z ];
	item**	h = find_item( k, b );
        T       r = T();

        if ( h && *h ) {
                item* p = *h;
                *h = p->m_next;
		p->m_next = *b;

                r = p->remove();
		m_removed.enq( p );
        }
        return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline size_t phash<T,P,Z,K,Q,H,E>::length() const
{
	const_itor	i( *this );
	size_t		r = 0;

	for ( i.init(); i.cont(); i.next()) r++;
	return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void phash<T,P,Z,K,Q,H,E>::swap( phash<T,P,Z,K,Q,H,E>& ph )
{
	item**	ht = m_table;
	m_table = ph.m_table;
	ph.m_table = ht;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void phash<T,P,Z,K,Q,H,E>::clear()
{
	for ( size_t i = 0; i < Z; i++ ) {
		if ( item* p = m_table[ i ] ) {
			while ( p ) {
				item* b = p;
				p = p->m_next;
				delete b;
			}
			m_table[ i ] = 0;
		}
	}
	m_removed.clear();
}	

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline typename phash<T,P,Z,K,Q,H,E>::item** 
	phash<T,P,Z,K,Q,H,E>::find_item( const K& k, phash<T,P,Z,K,Q,H,E>::item** r ) const
{
	while ( *r ) {
		if ( E()(( *r )->m_key, k )) break;
 		r = &( *r )->m_next; 
	}
	return r;	// never null
}

// iterator member functions
// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void phash<T,P,Z,K,Q,H,E>::itor_base::init() 
{
	for ( m_index = 0; m_index < Z; m_index++ ) 
		if ( m_rhash.m_table[ m_index ] ) break;

	m_hitem = ( m_index < Z ) ? &m_rhash.m_table[ m_index ] : 0;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void phash<T,P,Z,K,Q,H,E>::itor_base::next()
{
	if ( m_hitem ) {
		if ( *m_hitem && ( *m_hitem )->m_next ) {
			m_hitem = &( *m_hitem )->m_next;
		} else {
			for ( ++m_index; m_index < Z; ++m_index )
				if ( m_rhash.m_table[ m_index ] ) break;
			m_hitem = ( m_index < Z ) ? &m_rhash.m_table[ m_index ] : 0;
		}
	}
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline void phash<T,P,Z,K,Q,H,E>::itor::next() 
{
	typedef typename phash<T,P,Z,K,Q,H,E>::item* pitem;

	if ( m_dflag ) {
		if ( m_hitem && *m_hitem ) { 
			pitem&	h = const_cast<pitem&>( *m_hitem );
			pitem	p = h; 

			if (( h = h->m_next ) == 0 ) {
				for ( ++m_index; m_index < Z; ++m_index )
					if ( m_rhash.m_table[ m_index ] ) break;
				m_hitem = ( m_index < Z ) ? &m_rhash.m_table[ m_index ] : 0;
			}
			p->m_next = m_rhash.m_table[ m_index ];
                	p->remove();
			m_rhash.m_removed.enq( p );
		}
		m_dflag = false;
	} else {
		itor_base::next();
	}
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline T phash<T,P,Z,K,Q,H,E>::itor::set( const T& p )
{
       	return ( m_hitem && *m_hitem ) ? ( *m_hitem )->setdata( p ) : T();
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline typename phash<T,P,Z,K,Q,H,E>::dref&
	phash<T,P,Z,K,Q,H,E>::dref::operator=( const phash<T,P,Z,K,Q,H,E>::dref& r )
{
	if ( &r != this ) {
		m_hitem = r.m_hitem;
		m_phash = r.m_phash;
		Q::release( m_key );
		m_key = Q::copy( r.m_key );
	}
	return *this;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline bool phash<T,P,Z,K,Q,H,E>::dref::check() const
{
        if ( m_hitem && m_key ) {
		while ( *m_hitem && E()(( *m_hitem )->m_key, m_key ) == false )
			m_hitem = &( *m_hitem )->m_next;
	}
	return ( m_hitem && *m_hitem ) ? true : false;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline T phash<T,P,Z,K,Q,H,E>::dref::set( const T& p )
{
	T	r = T();

        if ( m_hitem ) {
                if ( check() ) 
                        r = ( *m_hitem )->setdata( p );
		else 
                        *m_hitem = new item( Q::copy( m_key ), p );
	} else {
		P::release( p );
	}
        return r;
}

// -----------------------------------------------------------------------------
template<typename T, typename P, size_t Z, typename K, typename Q, typename H, typename E> 
inline T phash<T,P,Z,K,Q,H,E>::dref::del()
{
        T       r = T();

        if ( check() ) {
                typename phash<T,P,Z,K,Q,H,E>::item* p = *m_hitem;
		*m_hitem = p->m_next;
		p->m_next = m_phash->m_table[ H()( m_key ) % Z ];

                r = p->remove();
		m_phash->m_removed.enq( p );
        }
        return r;
}

#ifndef NO_NAMESPACE
}
#endif
#endif // __OMT_PHASH_H

