//
// pcontainer.h --- common declaration of pointer container for Open Web Middleware
//
//	version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_PCONTAINER_H
#define __OMT_PCONTAINER_H

#ifndef NO_NAMESPACE
namespace omt {
#endif

// Common Memory Management Policy
// ============================================================================

// refer policy
// ----------------------------------------------------------------------------
template<typename T> struct refer_policy 
{
	static T set( const T& d ) { return d; }
	static T copy( const T& d ) { return d; }
	static T del( const T& d ) { return d; }

	static T release( const T& d ) { return 0; }	// release unused argument
};

// store policy
// ----------------------------------------------------------------------------
template<typename T> struct store_policy_base
{
	static T set( T p ) { return p; }
	static T copy( T p ) { return p ? dup( p ) : 0; }
};
template<typename T> struct store_policy { /* only for pointer */ };
template<typename T> struct store_policy<T*> : public store_policy_base<T*>
{
	static T* del( T* p ) { if ( p ) delete p; return 0; }
	static T* release( T* p ) { if ( p ) delete p; return 0; }
};
template<typename T> struct store_policy<T[]> : public store_policy_base<T[]>
{
	static T* del( T p[] ) { if ( p ) delete [] p; return 0; }
	static T* release( T p[] ) { if ( p ) delete [] p; return 0; }
};
template<> struct store_policy<char*> : public store_policy_base<char*>
{
	static char* del( char* p ) { if ( p ) delete [] p; return 0; }
	static char* release( char* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct store_policy<const char*> : public store_policy_base<const char*>
{
	static const char* del( const char* p ) { if ( p ) delete [] p; return 0; }
	static const char* release( const char* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct store_policy<wchar_t*> : public store_policy_base<wchar_t*>
{
	static wchar_t* del( wchar_t* p ) { if ( p ) delete [] p; return 0; }
	static wchar_t* release( wchar_t* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct store_policy<const wchar_t*> : public store_policy_base<const wchar_t*>
{
	static const wchar_t* del( const wchar_t* p ) { if ( p ) delete [] p; return 0; }
	static const wchar_t* release( const wchar_t* p ) { if ( p ) delete [] p; return 0; }
};

// copy policy
// ----------------------------------------------------------------------------
template<typename T> struct copy_policy_base
{
	static T set( T p ) { return p ? dup( p ) : 0; }
	static T copy( T p ) { return p; }

	static T release( const T& d ) { return 0; }
};
template<typename T> struct copy_policy { /* only for pointer */ };
template<typename T> struct copy_policy<T*> : public copy_policy_base<T*>
{
	static T* del( T* p ) { if ( p ) delete p; return 0; }
};
template<typename T> struct copy_policy<T[]> : public copy_policy_base<T[]>
{
	static T* del( T p[] ) { if ( p ) delete [] p; return 0; }
};
template<> struct copy_policy<char*> : public copy_policy_base<char*>
{
	static char* del( char* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct copy_policy<const char*> : public copy_policy_base<const char*>
{
	static const char* del( const char* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct copy_policy<wchar_t*> : public copy_policy_base<wchar_t*>
{
	static wchar_t del( wchar_t* p ) { if ( p ) delete [] p; return 0; }
};
template<> struct copy_policy<const wchar_t*> : public copy_policy_base<const wchar_t*>
{
	static const wchar_t del( const wchar_t* p ) { if ( p ) delete [] p; return 0; }
};

#ifndef NO_NAMESTACE
}
#endif

#endif // __OMT_PCONTAINER_H

