﻿using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;

// Online Game Starter.NET プラグイン仕様
// 仕様は固まっていないため、今後変更される可能性があります

#pragma warning disable 169

/** \brief プラグインで利用される型の定義 */
namespace OGSNET.Plugin
{
    public delegate void StartCallback(CallbackStatus status, string message); /**< プラグインからのコールバック関数   */
    public delegate IPlugin GetPlugin(string pluginId);                        /**< 登録されているプラグインを取得する */

    /** \brief コールバック関数に渡すステータス */
    public enum CallbackStatus
    {
        Notice,  /**< メッセージ */
        Warning, /**< 警告       */
        Error,   /**< エラー     */
        Finish   /**< 終了       */
    }

    /** \brief アカウント情報を表す */
    public struct Account :
        IComparable,
        IComparable<Account>,
        IEquatable<Account>
    {
        public string UserName; /**< ユーザー名 */
        public string Password; /**< パスワード */

        #region Constructor

        public Account(string username, string password)
        {
            this.UserName = username;
            this.Password = password;
        }

        #endregion
        #region Operators and Overrides

        public static bool operator ==(Account self, Account other)
        {
            return self.Equals(other);
        }

        public static bool operator !=(Account self, Account other)
        {
            return !self.Equals(other);
        }

        public static bool operator <(Account self, Account other)
        {
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(Account self, Account other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(Account self, Account other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(Account self, Account other)
        {
            return self.CompareTo(other) >= 0;
        }

        public int CompareTo(object other)
        {
            if (other is Account)
            {
                return this.CompareTo((Account)other);
            }

            throw new ArgumentException();
        }

        /** ユーザー名, パスワードの順で比較します */
        public int CompareTo(Account other)
        {
            if (this.UserName != other.UserName)
            {
                this.UserName.CompareTo(other.UserName);
            }

            if (this.Password != other.Password)
            {
                this.Password.CompareTo(other.Password);
            }

            return 0;
        }

        public bool Equals(Account other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object other)
        {
            if (other is Account)
            {
                return this.Equals((Account)other);
            }

            return false;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        #endregion
    }

    /** \brief プラグインのバージョンを表す */
    public struct Version :
        IComparable,
        IComparable<Version>,
        IEquatable<Version>
    {
        public uint Major;   /**< メジャーバージョン */
        public uint Minor;   /**< マイナーバージョン */
        public uint Release; /**< リリースバージョン */
        public uint Build;   /**< ビルドバージョン   */

        #region Constructor

        public Version(uint major = 0, uint minor = 0, uint release = 0, uint build = 0)
        {
            this.Major   = major;
            this.Minor   = minor;
            this.Release = release;
            this.Build   = build;
        }

        /** \brief ドット区切りの文字列で初期化する */
        public Version(string version) : this()
        {
            var array = version.Split('.');
            
            if (array.Length > 3)
            {
                uint.TryParse(array[3], out this.Build);
            }

            if (array.Length > 2)
            {
                uint.TryParse(array[2], out this.Release);
            }

            if (array.Length > 1)
            {
                uint.TryParse(array[1], out this.Minor);
            }

            if (array.Length > 0)
            {
                uint.TryParse(array[0], out this.Major);
            }
        }

        #endregion
        #region Operators and Overrides

        public static bool operator ==(Version self, Version other)
        {
            return self.Equals(other);
        }

        public static bool operator !=(Version self, Version other)
        {
            return !self.Equals(other);
        }

        public static bool operator <(Version self, Version other)
        {
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(Version self, Version other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(Version self, Version other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(Version self, Version other){
            return self.CompareTo(other) >= 0;
        }

        public int CompareTo(object other)
        {
            if (other is Version)
            {
                return this.CompareTo((Version)other);
            }

            throw new ArgumentException();
        }

        /** メジャー, マイナー, リリース, ビルドの順で比較する */
        public int CompareTo(Version other)
        {
            if (this.Major != other.Major)
            {
                return this.Major.CompareTo(other.Major);
            }

            if (this.Minor != other.Minor)
            {
                return this.Minor.CompareTo(other.Minor);
            }

            if (this.Release != other.Release)
            {
                return this.Release.CompareTo(other.Release);
            }

            if (this.Build != other.Build)
            {
                return this.Build.CompareTo(other.Build);
            }

            return 0;
        }

        public bool Equals(Version other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object other)
        {
            if(other is Version){
                return this.Equals((Version)other);
            }

            return false;
        }
        
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        public override string ToString()
        {
            return this.Major + "." + this.Minor + "." + this.Release + "." + this.Build;
        }

        #endregion
    }

    /** \brief ゲームの情報を表す */
    public struct GameInfo :
        IComparable,
        IComparable<GameInfo>,
        IEquatable<GameInfo>
    {
        public string OperationName; /**< 運営名 */
        public string OperationId;   /**< 運営のプラグインID。存在しない場合はnull */
        public string GameName;      /**< ゲームの名前 */
        public Uri Url;              /**< ゲームのWebサイトのURL */

        #region Constructor

        public GameInfo(string operationName, string operationId, string gameName, Uri url)
        {
            this.OperationName = operationName;
            this.OperationId   = operationId;
            this.GameName      = gameName;
            this.Url           = url;
        }

        public GameInfo(string operationName, string operationId, string gameName, string url) :
            this(operationName, operationId, gameName, new Uri(url))
        {
        }

        #endregion
        #region Operators and Overrides

        public static bool operator ==(GameInfo self, GameInfo other)
        {
            return self.Equals(other);
        }

        public static bool operator !=(GameInfo self, GameInfo other)
        {
            return !self.Equals(other);
        }

        public static bool operator <(GameInfo self, GameInfo other)
        {
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(GameInfo self, GameInfo other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(GameInfo self, GameInfo other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(GameInfo self, GameInfo other)
        {
            return self.CompareTo(other) >= 0;
        }
        
        public int CompareTo(object other)
        {
            if (other is GameInfo)
            {
                return this.CompareTo((GameInfo)other);
            }

            throw new ArgumentException();
        }

        /** 運営名, ゲーム名, URL の順で比較する */
        public int CompareTo(GameInfo other)
        {
            if (this.OperationName != other.OperationName)
            {
                return this.OperationName.CompareTo(other.OperationName);
            }

            if (this.GameName != other.GameName)
            {
                return this.GameName.CompareTo(other.GameName);
            }

            if (this.Url != other.Url)
            {
                return Uri.Compare(this.Url, other.Url, UriComponents.AbsoluteUri, UriFormat.Unescaped, StringComparison.CurrentCulture);
            }

            return 0;
        }

        public bool Equals(GameInfo other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object other)
        {
            if (other is GameInfo)
            {
                return other.Equals((GameInfo)other);
            }

            return false;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        #endregion
    }

    /** \brief 運営の情報を表す */
    public struct OperationInfo :
        IComparable,
        IComparable<OperationInfo>,
        IEquatable<OperationInfo>
    {
        public string OperationName; /**< 運営名               */
        public Uri Url;              /**< 運営のWebサイトのURL */

        #region Constructor

        public OperationInfo(string operationName, Uri url)
        {
            this.OperationName = operationName;
            this.Url           = url;
        }

        public OperationInfo(string operationName, string url)
            : this(operationName, new Uri(url))
        {
        }

        #endregion
        #region Operators and Overrides

        public static bool operator ==(OperationInfo self, OperationInfo other)
        {
            return self.Equals(other);
        }

        public static bool operator !=(OperationInfo self, OperationInfo other)
        {
            return !self.Equals(other);
        }

        public static bool operator <(OperationInfo self, OperationInfo other)
        {
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(OperationInfo self, OperationInfo other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(OperationInfo self, OperationInfo other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(OperationInfo self, OperationInfo other)
        {
            return self.CompareTo(other) >= 0;
        }

        public int CompareTo(object other)
        {
            if (other is OperationInfo)
            {
                return this.CompareTo((OperationInfo)other);
            }

            throw new ArgumentException();
        }

        /** 運営名, URL の順で比較する */
        public int CompareTo(OperationInfo other)
        {
            if (this.OperationName != other.OperationName)
            {
                return this.OperationName.CompareTo(other.OperationName);
            }

            if (this.Url != other.Url)
            {
                return Uri.Compare(this.Url, other.Url, UriComponents.AbsoluteUri, UriFormat.Unescaped, StringComparison.CurrentCulture);
            }

            return 0;
        }

        public bool Equals(OperationInfo other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object other)
        {
            if (other is OperationInfo)
            {
                return other.Equals((OperationInfo)other);
            }

            return false;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        #endregion
    }

    /** \brief プラグインの情報を表す */
    public struct PluginInfo :
        IComparable,
        IComparable<PluginInfo>,
        IEquatable<PluginInfo>
    {
        public Version Version;   /**< プラグインのバージョン     */
        public string PluginName; /**< プラグインの名前           */
        public string AuthorName; /**< プラグインの作者の名前     */
        public string Comment;    /**< プラグインに対するコメント */

        #region Constructor

        public PluginInfo(Version version, string pluginName, string authorName, string comment = "")
        {
            this.Version    = version;
            this.PluginName = pluginName;
            this.AuthorName = authorName;
            this.Comment    = comment;
        }

        public PluginInfo(string version, string pluginName, string authorName, string comment = "") :
            this(new Version(version), pluginName, authorName, comment)
        {
        }

        #endregion
        #region Operators and Overrides

        public static bool operator ==(PluginInfo self, PluginInfo other){
            return self.Equals(other);
        }

        public static bool operator !=(PluginInfo self, PluginInfo other)
        {
            return !self.Equals(other);
        }

        public static bool operator <(PluginInfo self, PluginInfo other)
        {
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(PluginInfo self, PluginInfo other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(PluginInfo self, PluginInfo other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(PluginInfo self, PluginInfo other)
        {
            return self.CompareTo(other) >= 0;
        }

        public int CompareTo(object other)
        {
            if (other is PluginInfo)
            {
                return this.CompareTo((PluginInfo)other);
            }

            throw new ArgumentException();
        }

        /** プラグイン名, 作者名, バージョン, コメントの順で比較 */
        public int CompareTo(PluginInfo other)
        {

            if (this.PluginName != other.PluginName)
            {
                return this.PluginName.CompareTo(other.PluginName);
            }

            if (this.AuthorName != other.AuthorName)
            {
                return this.AuthorName.CompareTo(other.AuthorName);
            }

            if (this.Version != other.Version)
            {
                return this.Version.CompareTo(other.Version);
            }

            if (this.Comment != other.Comment)
            {
                return this.Comment.CompareTo(other.Comment);
            }

            return 0;
        }

        public bool Equals(PluginInfo other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object other)
        {
            if (other is PluginInfo)
            {
                return this.Equals((PluginInfo)other);
            }

            return false;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        #endregion
    }
    
    /** \brief プラグインへ渡す引数の構造体 */
    public struct PluginStartInfo
    {
        public GetPlugin GetPlugin;         /**< 登録されているプラグインを取得するデリゲート */
        public WebBrowser Browser;          /**< 開始に利用されるWebBrowserオブジェクト       */
        public StartCallback StartCallback; /**< コールバック関数のデリゲート                 */
        public Account Account;             /**< 開始するアカウント                           */
    }

    /** \brief プラグインのインターフェイス */
    public interface IPlugin
    {
        PluginInfo GetPluginInfo(); /**< プラグインの情報を取得 */

        /**
         * 処理を開始する
         * \arg \c pluginStartInfo 開始時に必要な引数の構造体
         */
        void Start(PluginStartInfo pluginStartInfo);
    }

    /** \brief ゲームプラグインのインターフェイス */
    public interface IGamePlugin : IPlugin
    {
        GameInfo GetGameInfo(); /**< ゲームの情報を取得する */
    }

    /** \brief 運営プラグインのインターフェイス */
    public interface IOperationPlugin : IPlugin
    {
        OperationInfo GetOperationInfo(); /**< 運営の情報を取得 */
    }

    /** \brief プラグインのメタデータ */
    public interface IPluginMetaData
    {
        string PluginId { get; } /**< プラグインID */
    }
}

#pragma warning restore 169