/* vframe_mgr.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "csp.h"
#include "vframe_mgr.h"

#define VFRAME_NUM_MAX 64
#define VFRAME_NUM_MIN 5
#define VFRAME_MAX_LAST_FRAME_ALLOC_COUNT (64)
#define VFRAME_MAX_EXT_FRAME_NUM 256

static VFRAME_MGR vframe_mgr;

#if 0
#ifdef DEBUG
static void
print_flag(uint32_t flag)
{
  printf("vfrm_mgr:");
  if (flag & NORMAL_VFRAME)
    printf(" NORMAL_VFRAME");
  if (flag & EXT_VFRAME)
    printf(" EXT_VFRAME");
  if (flag & START_VFRAME)
    printf(" START_VFRAME");
  if (flag & END_VFRAME)
    printf(" END_VFRAME");
  if (flag & QUIT_VFRAME)
    printf(" QUIT_VFRAME");
  if (flag & DROP_VFRAME)
    printf(" DROP_VFRAME");
  printf(" !!\n");
}
#endif /* DEBUG */
#endif

static VFRAME*
alloc_vframe(VFRAME_MGR *vfrm_mgr)
{
  VFRAME *vfrm;
  const char proc_name[] = "alloc_vframe";

  if (vfrm_mgr->alloc_frame_num >= vfrm_mgr->max_frame_num)
    return NULL;

  vfrm = (VFRAME*)malloc(sizeof(VFRAME));
  if (!vfrm) {
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }
#ifdef ARCH_X86
  vfrm->bufp = (uint8_t*)malloc(vfrm_mgr->data_buf_size+15);
  vfrm->data = (uint8_t*)((((uint32_t)(vfrm->bufp)) + 15) & 0xfffffff0);
#else
  vfrm->bufp = (uint8_t*)malloc(vfrm_mgr->data_buf_size);
  vfrm->data = vfrm->bufp;
#endif /* ARCH_X86 */
  if (!vfrm->bufp) {
    free(vfrm);
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }

  vfrm->flag = NORMAL_VFRAME;
  vfrm->time = 0;
  vfrm->data_length = 0;
  vfrm->csp = CSP_UNKNOWN;
  vfrm->data_buf_size = vfrm_mgr->data_buf_size;
  vfrm->prev = vfrm->next = NULL;
  vfrm_mgr->alloc_frame_num++;
  vfrm_mgr->last_frame_alloc_count = 0;
#ifdef DEBUG
  if ((int)(vfrm->data) & 0xf) {
    fprintf(stderr, "alloc_vframe: data pointer address doesn't aligned 16.\n");
    exit(1);
  }
#endif /* DEBUG */
  return vfrm;
}

static void
free_vframe(VFRAME_MGR *vfrm_mgr, VFRAME *vfrm)
{
#ifdef DEBUG
  if (vfrm->prev || vfrm->next) {
    fprintf(stderr, "free_vframe: frame doesn't removed chain.\n");
    exit(1);
  }
#endif /* DEBUG */
  if (vfrm->bufp)
    free (vfrm->bufp);
  free(vfrm);
  vfrm_mgr->alloc_frame_num--;
  vfrm_mgr->last_frame_alloc_count = 0;
#ifdef DEBUG
  if (vfrm_mgr->alloc_frame_num < 0) {
    fprintf(stderr, "free_vframe: alloc_frame_num < 0\n");
    exit(1);
  }
#endif /* DEBUG */
}

static VFRAME*
alloc_ext_vframe(VFRAME_MGR *vfrm_mgr)
{
  VFRAME *vfrm;
  const char proc_name[] = "alloc_ext_vframe";

  if (vfrm_mgr->alloc_ext_frame_num >= vfrm_mgr->max_ext_frame_num) {
    fprintf(stderr, "%s: ext_frame_num over flow.\n", proc_name);
    return NULL;
  }

  vfrm = (VFRAME*)malloc(sizeof(VFRAME));
  if (!vfrm) {
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }
  vfrm->data = vfrm->bufp = NULL;
  vfrm->flag = EXT_VFRAME;
  vfrm->time = 0;
  vfrm->data_length = 0;
  vfrm->csp = CSP_UNKNOWN;
  vfrm->data_buf_size = 0;
  vfrm->prev = vfrm->next = NULL;
  vfrm_mgr->alloc_ext_frame_num++;
//  vfrm_mgr->last_ext_frame_alloc_count = 0;
  return vfrm;
}

static void
free_ext_vframe(VFRAME_MGR *vfrm_mgr, VFRAME *vfrm)
{
#ifdef DEBUG
  if (vfrm->prev || vfrm->next) {
    fprintf(stderr, "free_vframe: frame doesn't removed chain.\n");
    exit(1);
  }
  if (vfrm->bufp) {
    fprintf(stderr, "free_ext_vframe: data pointer not NULL.\n");
  }
#endif /* DEBUG */
  free(vfrm);
  vfrm_mgr->alloc_ext_frame_num--;
#ifdef DEBUG
  if (vfrm_mgr->alloc_ext_frame_num < 0) {
    fprintf(stderr, "free_vframe: alloc_frame_num < 0\n");
    exit(1);
  }
#endif /* DEBUG */
}

static void
vframe_free(VFRAME_MGR *vfrm_mgr, VFRAME *vfrm)
{
  if (vfrm->flag & EXT_VFRAME)
    free_ext_vframe(vfrm_mgr, vfrm);
  else if (vfrm->flag & NORMAL_VFRAME)
    free_vframe(vfrm_mgr, vfrm);
#ifdef DEBUG
  else {
    fprintf(stderr, "vframe_free: unknown frame\n");
    exit(1);
  }
#endif /* DEBUG */
}

static inline VFRAME*
add_chain(VFRAME **chain, VFRAME *vfrm)
{
  if (*chain) {
    vfrm->prev = (*chain)->prev;
    vfrm->next = (*chain);
    (*chain)->prev->next = vfrm;
    (*chain)->prev = vfrm;
  } else {
    vfrm->prev = vfrm->next = vfrm;
    *chain = vfrm;
  }
  return vfrm;
}

static inline VFRAME*
del_chain(VFRAME **chain)
{
  VFRAME *vfrm;
  if (*chain == NULL)
    return NULL;
  vfrm = *chain;
  if (vfrm->next == vfrm) {
    *chain = NULL;
  } else {
    *chain = (*chain)->next;
    vfrm->prev->next = vfrm->next;
    vfrm->next->prev = vfrm->prev;
  }
  vfrm->prev = vfrm->next = NULL;
  return vfrm;
}

static VFRAME*
get_free_frame(VFRAME_MGR *vfrm_mgr)
{
  VFRAME *vfrm;
  if (vfrm_mgr->free_chain) {
    vfrm = del_chain(&(vfrm_mgr->free_chain));
    vfrm_mgr->free_frame_num--;
  } else {
    vfrm = alloc_vframe(vfrm_mgr);
  }
  return vfrm;
}

static VFRAME*
get_ext_frame(VFRAME_MGR *vfrm_mgr)
{
  VFRAME *vfrm;
  vfrm = alloc_ext_vframe(vfrm_mgr);
  return vfrm;
}

static void
add_free_frame(VFRAME_MGR *vfrm_mgr, VFRAME *vfrm)
{
#ifdef DEBUG
  if (vfrm == NULL) {
    fprintf(stderr, "add_free_frame: invalid frame\n");
    exit(1);
  }
#if 0
  else if (!(vfrm->flag & NORMAL_VFRAME)) {
    fprintf(stderr, "add_free_frame: invalid normal video frame\n");
    exit(1);
  }
#endif
#endif /* DEBUG */
  if (vfrm->flag & EXT_VFRAME) {
    free_ext_vframe(vfrm_mgr, vfrm);
    return;
  }

  vfrm->flag = NORMAL_VFRAME;
  vfrm->data_length = 0;
  if (vfrm_mgr->vframe_flag & ENC_DST_WAIT) {
    add_chain(&vfrm_mgr->free_chain, vfrm);
    vfrm_mgr->free_frame_num++;
    if (vfrm_mgr->enc_wait_cb) {
      vfrm_mgr->enc_wait_cb();
    }
    vfrm_mgr->vframe_flag &= ~ENC_DST_WAIT;
    vfrm_mgr->last_frame_alloc_count = 0;
  } else if (vfrm_mgr->vframe_flag & SRC_WAIT) {
    add_chain(&vfrm_mgr->free_chain, vfrm);
    vfrm_mgr->free_frame_num++;
    if (vfrm_mgr->src_wait_cb) {
      vfrm_mgr->src_wait_cb();
    }
    vfrm_mgr->vframe_flag &= ~SRC_WAIT;
    vfrm_mgr->last_frame_alloc_count = 0;
  } else if (
      vfrm_mgr->alloc_frame_num > VFRAME_NUM_MIN &&
      vfrm_mgr->last_frame_alloc_count > vfrm_mgr->max_last_frame_alloc_count &&
      vfrm_mgr->free_frame_num > 0) {
    free_vframe(vfrm_mgr, vfrm);
    vfrm_mgr->last_frame_alloc_count = 0;
  } else {
    add_chain(&vfrm_mgr->free_chain, vfrm);
    vfrm_mgr->free_frame_num++;
    vfrm_mgr->last_frame_alloc_count++;
  }
}

VFRAME*
vframe_ext_frame_get(void)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  VFRAME *vfrm = NULL;
  pthread_mutex_lock(&vfrm_mgr->mutex);
  vfrm = get_ext_frame(vfrm_mgr);
  pthread_mutex_unlock(&vfrm_mgr->mutex);
  return vfrm;
}

VFRAME*
vframe_src_frame_get(void)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  VFRAME *vfrm = NULL;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm_mgr->src_frame != NULL) {
    fprintf(stderr, "vframe_src_frame_get: invalid frame\n");
  }
  if (vfrm_mgr->used_frame_num < 0) {
    fprintf(stderr, "vframe_src_frame_get: used_frame_num invalid, %d.\n", vfrm_mgr->used_frame_num);
    exit(1);
  }
#endif /* DEBUG */
  if (vfrm_mgr->used_frame_num < vfrm_mgr->max_frame_num-1) {
    vfrm = get_free_frame(vfrm_mgr);
    if (vfrm) {
      vfrm_mgr->src_frame = vfrm;
      vfrm_mgr->src_frame_num++;
      vfrm_mgr->used_frame_num++;
      vfrm_mgr->vframe_flag &= ~SRC_WAIT;
    } else {
      vfrm_mgr->src_frame = NULL;
      vfrm_mgr->vframe_flag |= SRC_WAIT;
    }
  }
#ifdef DEBUG
  if (!vfrm) {
    fprintf(stderr, "vframe_src_frame_get: src_frame_num %d, dst_frame_num %d, free_frame_num %d\n", vfrm_mgr->src_frame_num, vfrm_mgr->dst_frame_num, vfrm_mgr->free_frame_num);
  }
#endif /* DEBUG */
  pthread_mutex_unlock(&vfrm_mgr->mutex);

  return vfrm;
}

void
vframe_src_frame_update(VFRAME *vfrm)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm->flag & NORMAL_VFRAME && vfrm_mgr->src_frame != vfrm) {
    fprintf(stderr, "vframe_src_frame_update: invalid frame\n");
    exit(1);
  } else if ((vfrm->flag & EXT_VFRAME) && vfrm_mgr->src_frame) {
    fprintf(stderr, "vframe_src_frame_update: invalid frame\n");
    exit(1);
  } else if (!(vfrm->flag & (NORMAL_VFRAME|EXT_VFRAME))) {
    fprintf(stderr, "vframe_src_frame_update: invalid frame\n");
    exit(1);
  }
#endif /* DEBUG */
  add_chain(&vfrm_mgr->src_chain, vfrm);
  vfrm_mgr->src_frame = NULL;
  if (vfrm_mgr->vframe_flag & ENC_SRC_WAIT) {
    if (vfrm_mgr->enc_wait_cb) {
      vfrm_mgr->enc_wait_cb();
    }
    vfrm_mgr->vframe_flag &= ~ENC_SRC_WAIT;
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);
}

VFRAME *
vframe_enc_src_frame_get(void)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  VFRAME *vfrm = NULL;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm_mgr->enc_src_frame) {
    fprintf(stderr, "vframe_enc_src_frame_get: enc_src_frame remains\n");
    exit(1);
  }
#endif /* DEBUG */
  if (vfrm_mgr->src_chain) {
    vfrm = del_chain(&vfrm_mgr->src_chain);
    vfrm_mgr->enc_src_frame = vfrm;
    vfrm_mgr->vframe_flag &= ~ENC_SRC_WAIT;
  } else {
    //vfrm_mgr->enc_src_frame = NULL;
    vfrm_mgr->vframe_flag |= ENC_SRC_WAIT;
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);
  return vfrm;
}

void
vframe_enc_src_frame_update(VFRAME *vfrm)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm_mgr->enc_src_frame != vfrm) {
    fprintf(stderr, "vframe_enc_src_frame_update: invalid frame\n");
    exit(1);
  }
#endif /* DEBUG */
  vfrm_mgr->src_frame_num--;
  vfrm_mgr->enc_src_frame = NULL;
  if (vfrm->flag & NORMAL_VFRAME) {
    add_free_frame(vfrm_mgr, vfrm);
    vfrm_mgr->used_frame_num--;
  } else {
#ifdef DEBUG
    if (!(vfrm->flag & EXT_VFRAME)) {
      fprintf(stderr, "vframe_enc_src_frame_update: unknown frame\n");
      exit(1);
    }
#endif /* DEBUG */
    add_chain(&vfrm_mgr->dst_chain, vfrm);
    vfrm_mgr->dst_frame_num++;
    if (vfrm_mgr->vframe_flag & DST_WAIT) {
      if (vfrm_mgr->dst_wait_cb) {
        vfrm_mgr->dst_wait_cb();
      }
      vfrm_mgr->vframe_flag &= ~DST_WAIT;
    }
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);
}

VFRAME*
vframe_enc_dst_frame_get(void)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  VFRAME *vfrm = NULL;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm_mgr->enc_dst_frame != NULL) {
    fprintf(stderr, "vframe_enc_dst_frame_get: enc_dst_frame remains.\n");
    exit(1);
  }
#endif /* DEBUG */
  if (vfrm_mgr->used_frame_num < vfrm_mgr->max_frame_num-1)
    vfrm = get_free_frame(vfrm_mgr);
  if (vfrm) {
    vfrm_mgr->enc_dst_frame = vfrm;
    vfrm_mgr->dst_frame_num++;
    vfrm_mgr->used_frame_num++;
    vfrm_mgr->vframe_flag &= ~ENC_DST_WAIT;
  } else {
    //vfrm_mgr->enc_dst_frame = NULL;
    vfrm_mgr->vframe_flag |= ENC_DST_WAIT;
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);

#ifdef DEBUG
  if (!vfrm) {
    fprintf(stderr, "vframe_enc_dst_frame_get: src_frame_num %d, dst_frame_num %d, free_frame_num %d\n", vfrm_mgr->src_frame_num, vfrm_mgr->dst_frame_num, vfrm_mgr->free_frame_num);
  }
#endif /* DEBUG */
  return vfrm;
}

void
vframe_enc_dst_frame_update(VFRAME *vfrm)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (!(vfrm->flag & NORMAL_VFRAME)) {
    fprintf(stderr,"vframe_enc_dst_frame_update: frame is not NORMAL_VFRAME\n");
    exit(1);
  } else if (vfrm_mgr->enc_dst_frame != vfrm) {
    fprintf(stderr, "vframe_enc_dst_frame_update: invalid frame\n");
    exit(1);
  }
#endif /* DEBUG */
  add_chain(&vfrm_mgr->dst_chain, vfrm);
  vfrm_mgr->enc_dst_frame = NULL;
  if (vfrm_mgr->vframe_flag & DST_WAIT) {
    if (vfrm_mgr->dst_wait_cb) {
      vfrm_mgr->dst_wait_cb();
    }
    vfrm_mgr->vframe_flag &= ~DST_WAIT;
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);
}

VFRAME*
vframe_dst_frame_get(void)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  VFRAME *vfrm = NULL;

#ifdef DEBUG
  if (vfrm_mgr->dst_frame) {
    fprintf(stderr, "vframe_dst_frame_get: dst_frame remains.\n");
    exit(1);
  }
#endif /* DEBUG */
  pthread_mutex_lock(&vfrm_mgr->mutex);
  if (vfrm_mgr->dst_chain) {
    vfrm = del_chain(&vfrm_mgr->dst_chain);
    vfrm_mgr->dst_frame = vfrm;
    vfrm_mgr->vframe_flag &= ~DST_WAIT;
  } else {
    vfrm_mgr->vframe_flag |= DST_WAIT;
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);

  return vfrm;
}

void
vframe_dst_frame_update(VFRAME *vfrm)
{
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
#ifdef DEBUG
  if (vfrm_mgr->dst_frame != vfrm) {
    fprintf(stderr, "vframe_dst_frame_update: invalid frame\n");
  }
#endif /* DEBUG */
  if (vfrm->flag & EXT_VFRAME) {
    free_ext_vframe(vfrm_mgr, vfrm);
  } else {
#ifdef DEBUG
    if (!(vfrm->flag & NORMAL_VFRAME)) {
      fprintf(stderr, "vframe_dst_frame_update: invalid normal frame\n");
      exit(1);
    }
#endif /* DEBUG */
    add_free_frame(vfrm_mgr, vfrm);
    vfrm_mgr->used_frame_num--;
  }
  vfrm_mgr->dst_frame_num--;
  vfrm_mgr->dst_frame = NULL;
  pthread_mutex_unlock(&vfrm_mgr->mutex);
  //fprintf(stderr, "dst_chain %p, flag %u\n", vfrm_mgr->dst_chain, (vfrm_mgr->dst_chain) ? vfrm_mgr->dst_chain->flag : 0);
}

int
vframe_get_width(void)
{
  return vframe_mgr.width;
}

int
vframe_get_height(void)
{
  return vframe_mgr.height;
}

int
vframe_get_max_pixel_depth(void)
{
  return vframe_mgr.max_pixel_depth;
}

int
vframe_get_buf_size(void)
{
  return vframe_mgr.data_buf_size;
}

void
vframe_set_src_wait_cb(void (*ready_cb)(void))
{
  vframe_mgr.src_wait_cb = ready_cb;
}

void
vframe_set_enc_wait_cb(void (*ready_cb)(void))
{
  vframe_mgr.enc_wait_cb = ready_cb;
}

void
vframe_set_dst_wait_cb(void (*ready_cb)(void))
{
  vframe_mgr.dst_wait_cb = ready_cb;
}

int
vframe_mgr_error_quit(void)
{
  VFRAME *vfrm;
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
  if (vfrm_mgr->src_frame) {
    vframe_free(vfrm_mgr, vfrm_mgr->src_frame);
    vfrm_mgr->src_frame = NULL;
  }
  if (vfrm_mgr->dst_frame) {
    vframe_free(vfrm_mgr, vfrm_mgr->dst_frame);
    vfrm_mgr->dst_frame = NULL;
  }
  if (vfrm_mgr->enc_src_frame) {
    vframe_free(vfrm_mgr, vfrm_mgr->enc_src_frame);
    vfrm_mgr->enc_src_frame = NULL;
  }
  if (vfrm_mgr->enc_dst_frame) {
    vframe_free(vfrm_mgr, vfrm_mgr->enc_dst_frame);
    vfrm_mgr->enc_dst_frame = NULL;
  }

  while (vfrm_mgr->src_chain != NULL) {
    vfrm = del_chain(&vfrm_mgr->src_chain);
    vframe_free(vfrm_mgr, vfrm);
  }
  vfrm_mgr->src_frame_num = 0;
  while (vfrm_mgr->dst_chain != NULL) {
    vfrm = del_chain(&vfrm_mgr->dst_chain);
    vframe_free(vfrm_mgr, vfrm);
  }
  vfrm_mgr->dst_frame_num = 0;
  while (vfrm_mgr->free_chain != NULL) {
    vfrm = del_chain(&vfrm_mgr->free_chain);
    vframe_free(vfrm_mgr, vfrm);
  }
  vfrm_mgr->free_frame_num = 0;
  pthread_mutex_unlock(&vfrm_mgr->mutex);
//  pthread_mutex_destroy(&vfrm_mgr->mutex);
//  memset(vfrm_mgr, 0, sizeof(VFRAME_MGR));
  return VFRAME_MGR_OK;
}

int
vframe_mgr_quit(void)
{
  VFRAME *vfrm;
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;

  pthread_mutex_lock(&vfrm_mgr->mutex);
  if (vfrm_mgr->src_frame_num > 0 || vfrm_mgr->dst_frame_num > 0) {
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    fprintf(stderr, "vframe_mgr_quit: buf busy.\n");
    return VFRAME_MGR_BUSY;
  }
  while (vfrm_mgr->free_chain != NULL) {
    vfrm = del_chain(&vfrm_mgr->free_chain);
    free_vframe(vfrm_mgr, vfrm);
  }
  pthread_mutex_unlock(&vfrm_mgr->mutex);
//  pthread_mutex_destroy(&vfrm_mgr->mutex);
//  memset(vfrm_mgr, 0, sizeof(VFRAME_MGR));
  return VFRAME_MGR_OK;
}

int
vframe_mgr_init(int width, int height, int max_pixel_depth, int max_buf)
{
  int depth;
  int i;
  VFRAME *vfrm;
  VFRAME_MGR *vfrm_mgr = &vframe_mgr;
  int need_config = 0;
static int first = 1;

  if (first) {
    vfrm_mgr->vframe_flag = 0;
    vfrm_mgr->width = 0;
    vfrm_mgr->height = 0;
    vfrm_mgr->max_pixel_depth = 0;
    vfrm_mgr->data_buf_size = 0;
    vfrm_mgr->last_frame_alloc_count = 0;
    vfrm_mgr->max_last_frame_alloc_count = 0;
    vfrm_mgr->max_frame_num = 0;
    vfrm_mgr->max_ext_frame_num = 0;
    vfrm_mgr->alloc_frame_num = 0;
    vfrm_mgr->alloc_ext_frame_num = 0;
    vfrm_mgr->free_frame_num = 0;
    vfrm_mgr->used_frame_num = 0;
    vfrm_mgr->src_frame_num = 0;
    vfrm_mgr->dst_frame_num = 0;
    vfrm_mgr->src_chain = NULL;
    vfrm_mgr->dst_chain = NULL;
    vfrm_mgr->free_chain = NULL;
    vfrm_mgr->src_wait_cb = NULL;
    vfrm_mgr->enc_wait_cb = NULL;
    vfrm_mgr->dst_wait_cb = NULL;
    vfrm_mgr->src_frame = NULL;
    vfrm_mgr->dst_frame = NULL;
    vfrm_mgr->enc_src_frame = NULL;
    vfrm_mgr->enc_dst_frame = NULL;
    first = 0;
    need_config = 1;
    pthread_mutex_init(&vfrm_mgr->mutex, NULL);
  }

  pthread_mutex_lock(&vfrm_mgr->mutex);

  if (max_buf < VFRAME_NUM_MIN || max_buf > VFRAME_NUM_MAX)
    max_buf = VFRAME_NUM_MAX;

  if (vfrm_mgr->width != width)
    need_config = 1;
  else if (vfrm_mgr->height != height)
    need_config = 1;
  else if (vfrm_mgr->max_pixel_depth != max_pixel_depth)
    need_config = 1;
  else if (vfrm_mgr->max_frame_num != max_buf)
    need_config = 1;

  if (need_config == 0) {
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    return VFRAME_MGR_OK;
  }

  if (vfrm_mgr->src_chain != NULL || vfrm_mgr->dst_chain != NULL ||
      vfrm_mgr->src_frame_num > 0 || vfrm_mgr->dst_frame_num > 0) {
    fprintf(stderr, "vframe_mgr_init: src or dst frame busy.\n");
    pthread_mutex_unlock(&vfrm_mgr->mutex);
    return VFRAME_MGR_FAIL;
  }
  while (vfrm_mgr->free_chain != NULL) {
    vfrm = del_chain(&vfrm_mgr->free_chain);
    free_vframe(vfrm_mgr, vfrm);
  }

  depth = max_pixel_depth;

  vfrm_mgr->width = width;
  vfrm_mgr->height = height;
  vfrm_mgr->max_pixel_depth = depth;
  vfrm_mgr->data_buf_size = vfrm_mgr->width * vfrm_mgr->height * depth / 8;
  vfrm_mgr->last_frame_alloc_count = 0;
  vfrm_mgr->max_last_frame_alloc_count = VFRAME_MAX_LAST_FRAME_ALLOC_COUNT;
  vfrm_mgr->max_frame_num = max_buf;
  vfrm_mgr->max_ext_frame_num = VFRAME_MAX_EXT_FRAME_NUM;
  vfrm_mgr->alloc_frame_num = 0;
  vfrm_mgr->alloc_ext_frame_num = 0;
  vfrm_mgr->used_frame_num = 0;
  vfrm_mgr->src_frame = NULL;
  vfrm_mgr->src_frame_num = 0;
  vfrm_mgr->dst_frame = NULL;
  vfrm_mgr->dst_frame_num = 0;
  vfrm_mgr->free_chain = NULL;
  vfrm_mgr->free_frame_num = 0;
  vfrm_mgr->src_chain = NULL;
  vfrm_mgr->dst_chain = NULL;
  vfrm_mgr->free_chain = NULL;

  for (i = 0; i < VFRAME_NUM_MIN; i++) {
    vfrm = alloc_vframe(vfrm_mgr);
    add_chain(&vfrm_mgr->free_chain, vfrm);
    vfrm_mgr->free_frame_num++;
  }

#if 0
  vfrm_mgr->src_wait_cb = NULL;
  vfrm_mgr->enc_wait_cb = NULL;
  vfrm_mgr->dst_wait_cb = NULL;
#endif

  vfrm_mgr->vframe_flag = 0;

  pthread_mutex_unlock(&vfrm_mgr->mutex);

  return VFRAME_MGR_OK;
}

