/* oss.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <linux/soundcard.h>
#include <string.h>

#include "oss.h"

#define DEFAULT_DEVICE_NAME "/dev/dsp"
#define DEFAULT_MIXER_DEVICE_NAME "/dev/mixer"

static OSS_PARAM oss_param;
static const char *default_device_name = DEFAULT_DEVICE_NAME;
static const char *default_mixer_device_name = DEFAULT_MIXER_DEVICE_NAME;
static const char *mix_dev_names[] = SOUND_DEVICE_NAMES;

int oss_get_rate(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.rate;
}

int oss_get_channels(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.channels;
}

int
oss_get_bitspspl(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.bitspspl;
}

int
oss_get_bytespspl(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.bytespspl;
}

int
oss_get_fragments(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.fragments;
}

int
oss_get_fragsize(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.fragsize;
}

int
oss_get_fragstotal(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.fragstotal;
}

int
oss_get_format(void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;
  return oss_param.format;
}

#if 0
int
oss_trigger (void)
{
  int flag;
  int arg;

  if (oss_param.fd == -1)
    return OSS_FAIL;
  if (ioctl(oss_param.fd, SNDCTL_DSP_GETCAPS, &flag) < 0) {
    perror("SNDCTL_DSP_GETCAPS");
    return OSS_FAIL;
  }
  if (!(flag & DSP_CAP_TRIGGER)) {
    fprintf(stderr, "oss_trigger: Device does't support trigger.\n");
    return OSS_FAIL;
  }
  arg = 0;
  if (ioctl(oss_param.fd, SNDCTL_DSP_SETTRIGGER, &arg) < 0) {
    perror("SNDCTL_DSP_SETTRIGGER");
    return OSS_FAIL;
  }
  arg = PCM_ENABLE_INPUT;
  if (ioctl(oss_param.fd, SNDCTL_DSP_SETTRIGGER, &arg) < 0) {
    perror("SNDCTL_DSP_SETTRIGGER");
    return OSS_FAIL;
  }
  return OSS_OK;
}
#endif

int
oss_set_param(int rate, int channels, int bitspspl)
{
  audio_buf_info buf_info;
  int r = rate;
  int c = channels;
  int bits = bitspspl;
  int format;
  int frg;

  if (oss_param.fd == -1)
    return OSS_FAIL;

  switch (bitspspl) {
    case 8:
      format = AFMT_U8;
      break;
    case 16:
      format = AFMT_S16_LE;
      break;
    default:
      return OSS_FAIL;
  }

  if (ioctl(oss_param.fd, SNDCTL_DSP_RESET, NULL) < 0) {
    perror("SNDCTL_DSP_RESET");
    return OSS_FAIL;
  }

  frg = (0x7fff << 16) + 13;
  if (ioctl(oss_param.fd, SNDCTL_DSP_SETFRAGMENT, &frg) < 0) {
    perror("SNDCTL_DSP_SETFRAGMENT");
    return OSS_FAIL;
  }

  if (ioctl(oss_param.fd, SNDCTL_DSP_SETFMT, &format) < 0) {
    perror("SNDCTL_DSP_SETFMT");
    return OSS_FAIL;
  }
  switch (format) {
    case AFMT_U8:
      bits = 8;
      break;
    case AFMT_S16_LE:
      bits = 16;
      break;
    default:
      return OSS_FAIL;
  }
  if (bits != bitspspl) {
    fprintf(stderr, "oss_set_param: bitspspl error, bitspspl %d, %d\n", bitspspl, bits);
    return OSS_FAIL;
  }

  if (ioctl(oss_param.fd, SNDCTL_DSP_SPEED, &r) < 0) {
    perror("SNDCTL_DSP_SPEED");
    return OSS_FAIL;
  }
  if (r != rate) {
    fprintf(stderr, "oss_set_param: rate error, rate %d, %d\n", rate, r);
    return OSS_FAIL;
  }

  if (ioctl(oss_param.fd, SNDCTL_DSP_CHANNELS, &c) < 0) {
    perror("SNDCTL_DSP_CHANNELS");
    return OSS_FAIL;
  }
  if (c != channels) {
    fprintf(stderr, "oss_set_param: channels error, channels %d, %d\n", channels, c);
    return OSS_FAIL;
  }

  if (ioctl(oss_param.fd, SNDCTL_DSP_GETISPACE, &buf_info) < 0) {
    perror("SNDCTL_DSP_GETISPACE");
    return OSS_FAIL;
  }

  //int b;
  //if (ioctl (oss_param.fd, SNDCTL_DSP_GETBLKSIZE, &b) < 0) {
  //  perror("SNDCTL_DSP_GETBLKSIZE");
  //  return OSS_FAIL;
  //}

  oss_param.rate = r;
  oss_param.channels = c;
  oss_param.bytespspl = c * ((bits+7)/8);
  oss_param.bitspspl = bits;
  oss_param.format = format;
  oss_param.fragments = buf_info.fragments;
  oss_param.fragsize = buf_info.fragsize;
  oss_param.fragstotal = buf_info.fragstotal;
//  oss_trigger ();

  return OSS_OK;
}

int
oss_close(void)
{
  int fd;
  if (oss_param.fd == -1)
    return OSS_OK;
  fd = oss_param.fd;
  oss_param.fd = -1;
  if (close(fd) < 0) {
    perror("oss_close");
    return OSS_FAIL;
  }
  return OSS_OK;
}

int
oss_open (const char* device_name)
{
  const char *name;

  if (oss_param.fd != -1)
    oss_close();

  if (device_name == NULL)
    name = default_device_name;
  else
    name = device_name;
  oss_param.dev_name = name;
  if ((oss_param.fd = open(name, O_RDWR)) < 0) {
    fprintf(stderr, "oss_open: %s open failer ", name);
    perror ("");
    oss_param.fd = -1;
    return OSS_FAIL;
  }
  return OSS_OK;
}

int
oss_quit(void)
{
  if (oss_param.fd == -1)
    return OSS_OK;
  return oss_close();
}

int
oss_init(const char *device_name, int rate, int channels, int bitpspl)
{
static int first = 1;
  
  if (first) {
    oss_param.fd = -1;
    oss_param.mix_fd = -1;
    first = 0;
  }

  if (oss_param.fd != -1)
    oss_quit();

  if (oss_open(device_name) == OSS_FAIL)
    return OSS_FAIL;

  if (oss_set_param(rate, channels, bitpspl) == OSS_FAIL) {
    oss_close();
    return OSS_FAIL;
  }
  return OSS_OK;
}

int
oss_read_data(char* buf, int buf_size)
{
  int count;
  int result;
  int total_count;
  char *bp = buf;

  total_count = 0;
  count = buf_size;
  while (count > 0) {
    result = read (oss_param.fd, bp, count);
    if (result < 0) {
      result = 0;
      perror("oss_read_data");
    }
    bp += result;
    count -= result;
    total_count += result;
  }
  return total_count;
}

#if 0
int
oss_write (char* buf, int byte_length)
{
  return write(oss_param.fd, buf, byte_length);
}

int
oss_post (void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;

  if (ioctl (oss_param.fd, SNDCTL_DSP_POST) < 0) {
    perror ("SNDCTL_DSP_POST");
    return OSS_FAIL;
  }
  return OSS_OK;
}

int
oss_sync (void)
{
  if (oss_param.fd == -1)
    return OSS_FAIL;

  if (ioctl (oss_param.fd, SNDCTL_DSP_SYNC) < 0) {
    perror ("SNDCTL_DSP_SYNC");
    return OSS_FAIL;
  }
  return OSS_OK;
}
#endif

/* mixer functions */

static int oss_mixer_initialized = 0;

#define mchk(m,n) ((m)&(1<<(n)))

int
oss_mixer_name_to_id(const char *name)
{
  int i, id = OSS_FAIL;

  for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
    if (strcasecmp(mix_dev_names[i], name) == 0) {
      id = i;
      break;
    }
  }
  return id;
}

const char*
oss_mixer_id_to_name(int id)
{
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return NULL;
  return mix_dev_names[id];
}

int
oss_mixer_dev_id(const char *name)
{
  int i, id;

  if (!oss_mixer_initialized)
    return OSS_FAIL;

  id = OSS_FAIL;
  if (oss_param.mix_fd == -1)
    return id;
  for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
    if (mchk(oss_param.device_mask, i)) {
      if (strcasecmp(mix_dev_names[i], name) == 0) {
	id = i;
	break;
      }
    }
  }
  return id;
}

int
oss_mixer_stereo_cap(int id)
{
  int ret = OSS_FAIL;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;
  if (mchk(oss_param.stereo_mask, id))
    ret = 1;
  else
    ret = 0;
  return ret;
}

int
oss_mixer_record_cap(int id)
{
  int ret = OSS_FAIL;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;
  if (mchk(oss_param.reccap_mask, id))
    ret = 1;
  else
    ret = 0;
  return ret;
}

int
oss_mixer_record_enable(int id)
{
  int ret = OSS_FAIL;
  int mask;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;
  if (!mchk(oss_param.reccap_mask, id))
    return ret;
  mask = oss_param.recsrc_mask;
  if (oss_param.excl_input)
    mask = (1<<id);
  else
    mask |= (1<<id);
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_WRITE_RECSRC, &mask) == -1) {
    perror("SOUND_MIXER_WRITE_RECSRC");
    return ret;
  }
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_RECSRC, &mask) == -1) {
    perror("SOUND_MIXER_READ_RECSRC");
    return ret;
  }
  oss_param.recsrc_mask = mask;
  if (mchk(oss_param.recsrc_mask, id))
    ret = OSS_OK;
  return ret;
}

int
oss_mixer_record_disable(int id)
{
  int ret = OSS_FAIL;
  int mask;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;
  if (!mchk(oss_param.reccap_mask, id))
    return ret;
  mask = oss_param.recsrc_mask;
  mask &= ~(1<<id);
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_WRITE_RECSRC, &mask) == -1) {
    perror("SOUND_MIXER_WRITE_RECSRC");
    return ret;
  }
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_RECSRC, &mask) == -1) {
    perror("SOUND_MIXER_READ_RECSRC");
    return ret;
  }
  oss_param.recsrc_mask = mask;
  if ((!mchk(oss_param.recsrc_mask, id)) || (id == SOUND_MIXER_MIC))
    ret = OSS_OK;
  return ret;
}

int
oss_mixer_vol_set(int id, int left, int right)
{
  int ret = OSS_FAIL;
  int vol;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;

  vol = (left&0xff);
  if (mchk(oss_param.stereo_mask, id))
    vol |= ((right&0xff)<<8);

  if (ioctl(oss_param.mix_fd, MIXER_WRITE(id), &vol) == -1) {
    perror("MIXER_WRITE");
    return ret;
  }
  return OSS_OK;
}

int
oss_mixer_vol_get(int id, int *left, int *right)
{
  int ret = OSS_FAIL;
  int vol;

  if (!oss_mixer_initialized)
    return ret;

  if (oss_param.mix_fd == -1)
    return ret;
  if (id < 0 || id > SOUND_MIXER_NRDEVICES)
    return ret;
  if (!mchk(oss_param.device_mask, id))
    return ret;

  if (ioctl(oss_param.mix_fd, MIXER_READ(id), &vol) == -1) {
    perror("MIXER_WRITE");
    return ret;
  }
  *left = (vol&0xff);
  if (mchk(oss_param.stereo_mask, id))
    *right = ((vol>>8)&0xff);
  else if (right)
    *right = -1;
  return OSS_OK;
}

int
oss_mixer_close(void)
{
  int fd;

  if (!oss_mixer_initialized)
    return OSS_FAIL;

  if (oss_param.mix_fd == -1)
    return OSS_OK;
  fd = oss_param.mix_fd;
  oss_param.mix_fd = -1;
  if (close(fd) < 0) {
    perror("oss_mixer_close");
    return OSS_FAIL;
  }
  return OSS_OK;
}

int
oss_mixer_open (const char* device_name)
{
  const char *name;
  int cap_mask;

  if (!oss_mixer_initialized) {
    oss_param.mix_fd = -1;
    oss_mixer_initialized = 1;
  }

  if (oss_param.mix_fd != -1)
    oss_mixer_close();

  if (device_name == NULL)
    name = default_mixer_device_name;
  else
    name = device_name;
  oss_param.mix_dev_name = name;
  if ((oss_param.mix_fd = open(name, O_RDWR)) < 0) {
    fprintf(stderr, "oss_mixer_open: %s open failer ", name);
    perror ("");
    oss_param.mix_fd = -1;
    return OSS_FAIL;
  }

  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_DEVMASK, &oss_param.device_mask)
      == -1) {
    perror("SOUND_MIXER_READ_DEVMASK");
    oss_mixer_close();
    return OSS_FAIL;
  }
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_RECMASK, &oss_param.reccap_mask)
      == -1) {
    perror("SOUND_MIXER_READ_RECMASK");
    oss_mixer_close();
    return OSS_FAIL;
  }
  if (ioctl(oss_param.mix_fd,SOUND_MIXER_READ_STEREODEVS,&oss_param.stereo_mask)
      == -1) {
    perror("SOUND_MIXER_READ_STEREODEVS");
    oss_mixer_close();
    return OSS_FAIL;
  }
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_CAPS, &cap_mask)
      == -1) {
    perror("SOUND_MIXER_READ_CAPS");
    oss_mixer_close();
    return OSS_FAIL;
  }
  oss_param.excl_input = 0;
  if (cap_mask & SOUND_CAP_EXCL_INPUT) {
    oss_param.excl_input = 1;
  }
  if (ioctl(oss_param.mix_fd, SOUND_MIXER_READ_RECSRC, &oss_param.recsrc_mask)
      == -1) {
    perror("SOUND_MIXER_READ_RECSRC");
    oss_mixer_close();
    return OSS_FAIL;
  }
  return OSS_OK;
}

void
oss_print_param(void)
{
  int i;
  int first = 1;
  fprintf(stdout, "AUDIO DEVICE:            %s\n", oss_param.dev_name);
  fprintf(stdout, "AUDIO MIXER DEVICE:      %s\n", oss_param.mix_dev_name);
  fprintf(stdout, "AUDIO SOURCE:            ");
  for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
    if (mchk(oss_param.recsrc_mask, i)) {
      if (first) {
	fprintf(stdout, "%s", mix_dev_names[i]);
	first = !first;
      } else
	fprintf(stdout, ",%s", mix_dev_names[i]);
    }
  }
  fprintf(stdout, "\n");
}

